#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.


import json
import os
import time
from logging import Handler, getLevelName

from shinken.log import logger
from shinken.misc.os_utils import make_file_hidden
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.arbiter_configuration_messages import add_arbiter_messages, set_arbiter_configuration_messages_uuid, ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, init_message_arbiter
from shinkensolutions.locking.shinken_locking.shinken_interprocess_rlock import ShinkenInterProcessRLock

if TYPE_CHECKING:
    from logging import LogRecord
    from shinken.misc.type_hint import Any, Optional
    from shinken.objects.item import Item

log_handler_message_arbiter = None  # type: Optional[LogHandlerMessageArbiter]
no_arbiter_message = None  # type: Optional[NoArbiterMessage]


class LogHandlerMessageArbiter(Handler):
    def __init__(self):
        # type: () -> None
        Handler.__init__(self)
        
        self.shinken_is_enable_lock = ShinkenInterProcessRLock(make_file_hidden(os.path.join(os.path.dirname(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH), u'log_handler_message_arbiter.lock')))
        self.shinken_is_enable = True
    
    
    def emit(self, record):
        # type: (LogRecord) -> None
        
        if self.shinken_is_enable and record.levelname in ('ERROR', 'WARNING', 'CRITICAL'):
            add_arbiter_messages(json.dumps([time.time(), u'RAW', record.levelname, record.message]))


class NoArbiterMessage(object):
    def __enter__(self):
        # type: () -> None
        log_handler_message_arbiter.shinken_is_enable_lock.__enter__()
        log_handler_message_arbiter.shinken_is_enable = False
    
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        log_handler_message_arbiter.shinken_is_enable = True
        log_handler_message_arbiter.shinken_is_enable_lock.__exit__(exc_type, exc_val, exc_tb)


arbiter_configuration_messages_uuid = None


def enable_message_arbiter():
    # type: () -> None
    global arbiter_configuration_messages_uuid
    global log_handler_message_arbiter
    global no_arbiter_message
    
    log_handler_message_arbiter = LogHandlerMessageArbiter()
    no_arbiter_message = NoArbiterMessage()
    
    init_message_arbiter()
    env = os.environ
    arbiter_configuration_messages_uuid = env.get('ARBITER_CONFIGURATION_MESSAGES_UUID')  # None
    
    if arbiter_configuration_messages_uuid:
        set_arbiter_configuration_messages_uuid(arbiter_configuration_messages_uuid)
        logger.addHandler(log_handler_message_arbiter)


messages = []


def log_configuration_error(item, msg, *param):
    # type: (Item, unicode, *Any) -> None
    log_configuration(u'ERROR', item, msg, *param)


def log_configuration_warning(item, msg, *param):
    # type: ( Item, unicode, *Any) -> None
    log_configuration(u'WARNING', item, msg, *param)


def log_configuration(log_level, item, msg, *param):
    # type: (unicode, Item, unicode, *Any) -> None
    
    _msg = msg % param
    item_type_for_synchronizer = item.get_item_type_for_synchronizer()
    item_name = item.get_name()
    raw_message = u'[%s-%s] %s' % (item_type_for_synchronizer, item_name, _msg)
    with no_arbiter_message:
        log_level_as_int = getLevelName(log_level)  # noqa
        logger.log(log_level_as_int, raw_message)
    
    if arbiter_configuration_messages_uuid:
        item_uuid = item.get_uuid()
        if item_uuid is None:
            add_arbiter_messages(json.dumps([time.time(), u'RAW', log_level, raw_message]))
        else:
            add_arbiter_messages(json.dumps([time.time(), u'ELEMENT', log_level, _msg, item_type_for_synchronizer, item_name, item_uuid]))
