#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.

from collections import namedtuple

from shinken.log import PartLogger
from shinken.misc.type_hint import List, NoReturn, Optional, Tuple, Dict

from shinkensolutions.api.synchronizer import SourceTranslatePart

PRINT_VALIDATION_STATE = True

ValidationMessage = namedtuple('ValidationMessage', ['message', 'translate_key', 'params'])


class ValidationState(object):
    def __init__(self, logger, translator):
        # type:(PartLogger, SourceTranslatePart)-> None
        self.errors = []  # type: List[ValidationMessage]
        self.warnings = []  # type: List[ValidationMessage]
        self.logger = logger
        self._translator = translator
        self.extra_counter_error_number = 0
        self.extra_counter_warning_number = 0
    
    
    def is_valid(self):
        # type: () -> bool
        return not bool(self.errors or self.warnings)
    
    
    def update_from_validation_state(self, validation_state):
        # type: (ValidationState) -> NoReturn
        self.errors.extend(validation_state.errors)
        self.warnings.extend(validation_state.warnings)
        self.extra_counter_error_number += validation_state.extra_counter_error_number
        self.extra_counter_warning_number += validation_state.extra_counter_warning_number
    
    
    def add_warning(self, message='', translate_key='', params=None):
        # type: (str, str, Optional[Tuple]) -> NoReturn
        params = params or tuple()
        warning = ValidationMessage(message, translate_key, params)
        if warning not in self.warnings:
            self.warnings.append(warning)
        if PRINT_VALIDATION_STATE:
            self.logger.warning(self._message_as_string(warning))
    
    
    def add_error(self, message='', translate_key='', params=None):
        # type: (str, str, Optional[Tuple]) -> NoReturn
        params = params or tuple()
        error = ValidationMessage(message, translate_key, params)
        if error not in self.errors:
            self.errors.append(error)
        if PRINT_VALIDATION_STATE:
            self.logger.error(self._message_as_string(error))
    
    
    def _message_as_string(self, message):
        # type: (ValidationMessage) -> str
        return message.message if message.message else self._translator.translate(message.translate_key) % message.params
    
    
    def has_error(self):
        # type: () -> bool
        return bool(self.errors)
    
    
    def get_errors(self):
        # type: () -> List[str]
        return [self._message_as_string(m) for m in self.errors]
    
    
    def has_warning(self):
        # type: () -> bool
        return bool(self.warnings)
    
    
    def get_warnings(self):
        # type: () -> List[str]
        return [self._message_as_string(m) for m in self.warnings]
    
    
    def as_dict(self):
        # type: () -> Dict
        _as_dict = {
            'is_valid'      : self.is_valid(),
            'has_error'     : self.has_error(),
            'count_error'   : len(self.errors),
            'errors'        : self.get_errors(),
            'has_warning'   : self.has_warning(),
            'count_warnings': len(self.warnings),
            'warnings'      : self.get_warnings(),
        }
        return _as_dict
    
    
    def as_html(self):
        # type: () -> str
        if not self.has_error() and not self.has_warning():
            return ""
        _to_return = ['''<div class="shinken-server-message">''']
        if self.has_error():
            _to_return.append(self.as_html_by_type(self.get_errors(), "errors", extra_number=self.extra_counter_error_number - 1))
        if self.has_warning():
            _to_return.append(self.as_html_by_type(self.get_warnings(), "warnings", extra_number=self.extra_counter_warning_number - 1))
        _to_return.append("</div>")
        return "".join(_to_return)
    
    
    def as_html_by_type(self, content, type_message, extra_number=0):
        # type: (List[str], str, int) -> str
        if extra_number < 0:
            extra_number = 0
        _to_return = [
            '''<div class="shinken-message-validation shinken-%s-message" data-number="%s" data-extra-number="%s">''' % (type_message, len(content), extra_number),
            '''<div class="shinken-title">%s</div>''' % self._translator.translate("common.%s" % type_message),
            '''<div class="shinken-container-message">'''
        ]
        for _current in content:
            _to_return.append('''<div class="shinken-message">%s</div>''' % _current)
        _to_return.append('''</div></div>''')
        return "".join(_to_return)
