#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

"""
This class is for linking the WebUI with active directory,
like check passwords, or get photos.
"""

import json
import os

from shinken.modules.base_module.basemodule import BaseModule, ModuleState

from shinken.log import logger as _logger
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    import ldap
    from ldap import LDAPObject
    from shinken.log import PartLogger
    from shinken.misc.type_hint import Dict, Union, Optional
    from shinken.objects.contact import Contact
    from synchronizer.synchronizerdaemon import Synchronizer
    from webui.module import WebuiBroker
    from shinken.objects.config import Config

properties = {
    u'daemons': [u'webui', u'synchronizer-ui', u'synchronizer'],
    u'type'   : u'ad_webui'
}

# Lazy load, as import ldap can be super long
ldap = None  # type: Optional[ldap]


class LdapMode(object):
    AD = u'ad'
    OPENLDAP = u'openldap'


DEFAULT_MAPPING_FILE = u'/etc/shinken/_default/modules/auth-active-directory/mapping.json'
DEFAULT_MAPPING = {
    u'ldap_key'         : u'samaccountname',
    u'shinken_key'      : u'contact_name',
    u'login_placeholder': u''
}


# called by the plugin manager
def get_instance(plugin):
    global ldap
    try:
        import ldap as ldap
    except ImportError:
        ldap = None
    
    _logger.debug(u'Get an Active Directory/OpenLdap UI module for plugin %s' % plugin.get_name())
    
    if not ldap:
        raise Exception(u'The module python-ldap is not found. Please install it.')
    
    instance = LdapAuthenticationModule(plugin)
    return instance


class LdapAuthenticationModule(BaseModule):
    
    def __init__(self, module_config):
        # type: (Config) -> None
        
        BaseModule.__init__(self, module_config)
        self.logger_init = self.logger.get_sub_part(u'INITIALISATION')
        self.logger_auth = self.logger.get_sub_part(u'AUTHENTICATION')
        
        self.timeout = getattr(module_config, u'timeout', 5)
        self.mapping_file = getattr(module_config, u'mapping_file', u'')
        self.mode = getattr(module_config, u'mode', LdapMode.AD)
        self.ldap_uri = getattr(module_config, u'ldap_uri', None)
        self.username = getattr(module_config, u'username', None)
        self.password = getattr(module_config, u'password', None)
        self.base_dn = getattr(module_config, u'basedn', None)
        self.retrieveAttributes = []
        self.active = False
        
        # For typing
        self.name_id = u''
        self.auth_key = u''
        self.missing_parameters = []
        self.missing_parameters_output = u''
        self.invalid_credentials_for_admin = False
        self.invalid_credentials_output = u''
        self.ldap_key = u''
        self.shinken_key = u''
        self.login_placeholder = u''
        self.app = None  # type: (Union[None, Synchronizer, WebuiBroker])
    
    
    def do_loop_turn(self):
        # type: () -> None
        # for typing
        pass
    
    
    def load(self, app):
        # type: (Union[Synchronizer, WebuiBroker]) -> None
        self.app = app
    
    
    def init(self):
        # type: () -> None
        if self.mode == LdapMode.AD:
            self.retrieveAttributes = [u'userPrincipalName']
            self.name_id = u'userPrincipalName'
            self.auth_key = u'userPrincipalName'
            self.ldap_key = u'sAMAccountName'
        elif self.mode == LdapMode.OPENLDAP:
            self.retrieveAttributes = [u'dn', u'uid']
            self.name_id = u'uid'
            self.auth_key = u'dn'
            self.ldap_key = u'uid'
        else:
            raise Exception(u'[%s] The mode for this module is not correct. The value can be : "ad", "openldap"' % self.get_name())
        
        mandatory_parameters = (u'ldap_uri', u'username', u'password', u'base_dn')
        for parameter in mandatory_parameters:
            value = getattr(self, parameter, None)
            if not value:
                self.missing_parameters.append(u'basedn' if parameter == u'base_dn' else parameter)
        
        if self.missing_parameters:
            self.missing_parameters_output = u'For the module defined in the file "%s", these parameters are missing  : %s. Can\'t use this module to authenticate users.' % (self.myconf.imported_from, u', '.join(self.missing_parameters))
            self.logger_init.error(self.missing_parameters_output)
            return
        
        self._load_default_mapping()
        self._load_user_mapping_file()
        self.logger_init.info(u'EXPLANATION : The property "%s" in Shinken will be compared to the LDAP attribute "%s" to authenticate users. In the login page, the placeholder is "%s"' % (self.shinken_key, self.ldap_key, self.login_placeholder))
        self.logger_init.info(u'EXPLANATION : The authentication will be done on LDAP server "%s" on the base "%s" and this server is an %s' % (self.ldap_uri, self.base_dn, u'Active Directory' if self.mode == LdapMode.AD else u'Open Ldap'))
        
        self.retrieveAttributes.append(self.ldap_key)
        if self._connect(self.logger_init):
            self.logger_init.info(u'The username and password set in cfg file "%s" are correct and LDAP server at "%s" is available to authenticate users' % (self.myconf.imported_from, self.ldap_uri))
        elif not self.invalid_credentials_for_admin:
            self.logger_init.warning(u'The LDAP server "%s" is unavailable for the moment. The connection will be retried when a user tries to connect' % self.ldap_uri)
    
    
    def get_state(self):
        # type: () -> Dict[unicode, unicode]
        state = {
            u'status': ModuleState.OK,
            u'output': u'OK'
        }
        
        if not self.active:
            state[u'status'] = ModuleState.CRITICAL
            if self.missing_parameters:
                state[u'output'] = self.missing_parameters_output
            elif self.invalid_credentials_for_admin:
                state[u'output'] = self.invalid_credentials_output
            else:
                state[u'output'] = u'The LDAP server "%s" is unavailable for the moment. The connection will be retried when a user tries to connect' % (self.ldap_uri)
        
        return state
    
    
    def _read_mapping_file(self, file_path):
        # type: (unicode) -> Dict[unicode, unicode]
        lines = []
        try:
            with open(file_path) as mapping_file:
                for line in mapping_file.readlines():
                    line = line.strip().decode(u'utf-8')
                    if not line or line.startswith(u'#'):
                        continue
                    lines.append(line)
        except Exception as exp:
            self.logger_init.error(u'Cannot read the file "%s". The error is : %s' % (file_path, unicode(exp)))
            return {}
        try:
            conf = json.loads(u'\n'.join(lines))
        except Exception as exp:
            self.logger_init.error(u'The file "%s" is malformed. The error is : %s' % (file_path, unicode(exp)))
            return {}
        
        return conf
    
    
    def _load_default_mapping(self):
        # type: () -> None
        
        if not os.path.exists(DEFAULT_MAPPING_FILE):
            self.logger_init.warning(u'The default mapping file "%s" doesn\'t exist. The default values will be set by module' % DEFAULT_MAPPING_FILE)
            return
        
        self.logger_init.info(u'Read the default mapping file "%s" to get default values' % DEFAULT_MAPPING_FILE)
        
        default_conf_from_file = self._read_mapping_file(DEFAULT_MAPPING_FILE)
        for key, default_value in DEFAULT_MAPPING.iteritems():
            value_from_file = default_conf_from_file.get(key, None)
            if value_from_file is None:
                self.logger_init.warning(u'The default value for "%s" is not defined in file "%s". The default value "%s" is set by the module.' % (key, DEFAULT_MAPPING_FILE, default_value))
                setattr(self, key, default_value)
            else:
                setattr(self, key, value_from_file)
    
    
    def _load_user_mapping_file(self):
        # type: () -> None
        
        if not self.mapping_file:
            return
        
        if not os.path.exists(self.mapping_file):
            self.logger_init.warning(u'The parameter mapping_file is defined on "%s", but this file doesn\'t exist. The default values are used.' % self.mapping_file)
            return
        
        self.logger_init.info(u'Read the file "%s" to override default values' % self.mapping_file)
        user_conf = self._read_mapping_file(self.mapping_file)
        
        forbidden_fields = [field for field in user_conf.keys() if field not in DEFAULT_MAPPING.keys()]
        for field in forbidden_fields:
            self.logger_init.error(u'The parameter "%s" set in file "%s" is not an allowed parameter' % (field, self.mapping_file))
        
        for key in DEFAULT_MAPPING.iterkeys():
            value_from_file = user_conf.get(key, None)
            if value_from_file:
                setattr(self, key, value_from_file)
                self.logger_init.debug(u'For the parameter "%s", the new value will be "%s"' % (key, value_from_file))
    
    
    def _connect(self, logger, authentication_phase_id=None):
        # type: (PartLogger, unicode) -> Union[None, LDAPObject]
        
        if self.missing_parameters:
            return None
        logger.debug(u'Trying to get the connection to the LDAP server "%s" with the user "%s" on base_dn "%s" in mode "%s"%s' % (
            self.ldap_uri, self.username, self.base_dn, self.mode, u' ( authentication phase %s )' % authentication_phase_id if authentication_phase_id else u''))
        try:
            ldap_connection = ldap.initialize(self.ldap_uri)
            ldap_connection.set_option(ldap.OPT_REFERRALS, 0)
            
            # Any errors will throw an ldap.LDAPError exception or related exception so you can ignore the result
            ldap_connection.set_option(ldap.OPT_NETWORK_TIMEOUT, self.timeout)
            ldap_connection.simple_bind_s(self.username, self.password)
            
            self.active = True
            logger.debug(u'Connection to the LDAP server "%s" is OK%s' % (self.ldap_uri, u' ( authentication phase %s )' % authentication_phase_id if authentication_phase_id else u''))
            return ldap_connection
        except ldap.INVALID_CREDENTIALS:
            self.invalid_credentials_output = u'The username and password set for LDAP server "%s" in cfg file "%s" are invalid.' % (self.ldap_uri, self.myconf.imported_from)
            error_message = u'%s Can\'t use this module to authenticate users. ( authentication phase %s )' % (self.invalid_credentials_output, authentication_phase_id) if authentication_phase_id else self.invalid_credentials_output
            logger.error(error_message)
            self.invalid_credentials_for_admin = True
            return None
        except Exception as e:
            logger.error(
                u'Can\'t connect to the LDAP server "%s"%s => error : %s' % (self.ldap_uri, u' ( authentication phase %s )' % authentication_phase_id if authentication_phase_id else u'', e.message['desc'] if 'desc' in e.message else unicode(e)))
            return None
    
    
    # Give the entry for a contact
    def _find_contact_entry(self, contact_name, authentication_logger, authentication_phase_id):
        # type: (unicode, PartLogger, unicode) -> Union[None, Dict]
        
        search_scope = ldap.SCOPE_SUBTREE
        search_filter = (u'(%s=%s)' % (self.ldap_key, contact_name))
        
        authentication_logger.debug(
            u'Search for a user in LDAP server "%s". The LDAP attribute used is "%s" with the value "%s". The LDAP request filter is : "%s" ( authentication phase %s )' % (
                self.ldap_uri, self.ldap_key, contact_name, search_filter, authentication_phase_id))
        
        ldap_connection = self._connect(authentication_logger, authentication_phase_id)
        
        if ldap_connection is None:
            if not (self.missing_parameters or self.invalid_credentials_for_admin):
                authentication_logger.error(
                    u'The LDAP server "%s" is unavailable for the moment. Can\'t authenticate users for the moment. The connection will be retried when a user tries to connect ( authentication phase %s )' % (self.ldap_uri, authentication_phase_id))
            return None
        
        try:
            ldap_result_id = ldap_connection.search(self.base_dn, search_scope, search_filter.encode(u'utf8'), [a.encode(u'utf8') for a in self.retrieveAttributes])
            
            while True:
                result_type, result_data = ldap_connection.result(ldap_result_id, 0)
                if not result_data:
                    authentication_logger.debug(
                        u'There is no user in the LDAP server "%s" in base "%s" which has the attribute "%s=%s" ( authentication phase %s )' % (self.ldap_uri, self.base_dn, self.ldap_key, contact_name, authentication_phase_id))
                    return None
                
                if result_type == ldap.RES_SEARCH_ENTRY:
                    (_, elts) = result_data[0]
                    if self.mode == LdapMode.OPENLDAP:
                        elts[self.auth_key] = result_data[0][0].decode(u'utf8')
                    try:
                        account_name = elts[self.name_id][0].decode(u'utf8')
                    except Exception:
                        account_name = result_data[0].decode(u'utf8')
                    
                    authentication_logger.debug(
                        u'A user was found in the LDAP server "%s" in base "%s" with the attribute "%s=%s". His name is "%s" ( authentication phase %s )' % (
                            self.ldap_uri, self.base_dn, self.ldap_key, contact_name, account_name, authentication_phase_id))
                    return elts
        except ldap.LDAPError as e:
            authentication_logger.error(u'Error during requesting the LDAP server "%s" during authentication phase %s. Error message is :%s' % (self.ldap_uri, authentication_phase_id, e))
            return None
        # Always clean on exit
        finally:
            del ldap_connection
    
    
    def _try_to_reconnect(self, authentication_logger, authentication_phase_id):
        # type: (PartLogger, unicode) -> bool
        if self.missing_parameters:
            authentication_logger.error(u'Can\'t connect to the LDAP server. %s' % self.missing_parameters_output)
            return False
        
        if self.invalid_credentials_for_admin:
            
            authentication_logger.error(u'%s Can\'t use this module to authenticate users ( authentication phase %s )' % (self.invalid_credentials_output, authentication_phase_id))
            return False
        
        ldap_connection = self._connect(authentication_logger, authentication_phase_id)
        if ldap_connection is None:
            authentication_logger.error(u'The LDAP server "%s" is unavailable for the moment. The connection will be retried when a user tries to connect ( authentication phase %s )' % (self.ldap_uri, authentication_phase_id))
            return False
        return True
    
    
    # Try to auth a user in the ldap dir
    def check_auth(self, username, password, requester, authentication_phase_id):
        # type: (unicode, unicode, unicode, unicode) -> Union[bool, Dict]
        
        authentication_logger = self.logger_auth.get_sub_part(requester)
        
        if not self.active:
            connection_successful = self._try_to_reconnect(authentication_logger, authentication_phase_id)
            if not connection_successful:
                return False
        
        authentication_logger.debug(u'Trying to authenticate username "%s" on LDAP server "%s" ( authentication phase %s )' % (username, self.ldap_uri, authentication_phase_id))
        
        shinken_contact = None  # type: Optional[Dict[unicode, unicode], Contact]
        
        if self.shinken_key == DEFAULT_MAPPING[u'shinken_key']:
            shinken_contact = self.app.datamgr.get_contact_case_insensitive(username)
        else:
            for contact in self.app.datamgr.get_contacts():
                # contact is a dict   -> Module is on Synchronizer
                # contact is a Object -> Module is on WebUI
                if isinstance(contact, dict):
                    shinken_key_value = contact.get(self.shinken_key, u'')
                
                else:
                    shinken_key_value = contact.customs.get(self.shinken_key, u'') if self.shinken_key.startswith(u'_') else getattr(contact, self.shinken_key, u'')
                
                if shinken_key_value.lower() == username.lower():
                    shinken_contact = contact
                    break
        
        if not shinken_contact:
            authentication_logger.debug(u'User "%s" tried to connect but this username doesn\'t match any user in Shinken database with the property "%s" ( authentication phase %s )' % (username, self.shinken_key, authentication_phase_id))
            return False
        
        # I don't know why, but ldap automagically auth void password. That's just stupid I think so we don't allow them.
        if not password:
            authentication_logger.warning(u'User "%s" doesn\'t give the password. It\'s not allowed ( authentication phase %s )' % (username, authentication_phase_id))
            return False
        
        # first we need to find the principalname of this entry
        # because it can be a user name like j.gabes, but we should auth by ldap
        # with j.gabes@google.com for example
        contacts_in_ldap = self._find_contact_entry(username, authentication_logger, authentication_phase_id)
        
        # no user found, exit
        if contacts_in_ldap is None:
            return False
        
        try:
            if self.mode == LdapMode.AD:
                if self.auth_key in contacts_in_ldap:
                    account_name = contacts_in_ldap[self.auth_key][0].decode(u'utf8')
                else:
                    account_name = username
            else:
                account_name = contacts_in_ldap[self.auth_key]
        except KeyError:
            authentication_logger.warning(
                u'The LDAP object has not the attribute "%s". Can\'t map the attribute with the shinken key. The user value ( %s ) will be used to find the user ( authentication phase %s )' % (self.auth_key, username, authentication_phase_id))
            account_name = username
        
        connexion_for_user = ldap.initialize(self.ldap_uri)
        connexion_for_user.set_option(ldap.OPT_REFERRALS, 0)
        
        contact_name = shinken_contact[u'contact_name'] if isinstance(shinken_contact, dict) else shinken_contact.contact_name
        
        try:
            connexion_for_user.simple_bind_s(account_name.encode(u'utf8'), password.encode(u'utf8'))
            authentication_logger.debug(u'Credentials for LDAP user "%s" are OK. Authenticated as shinken user "%s" ( authentication phase %s )' % (username, contact_name, authentication_phase_id))
            return shinken_contact
        except ldap.INVALID_CREDENTIALS:
            authentication_logger.debug(u'Credentials for "%s" ( "%s" in shinken ) failed. Server "%s" refused the user\'s password ( authentication phase %s )' % (username, contact_name, self.ldap_uri, authentication_phase_id))
        
        except Exception as e:
            authentication_logger.error(u'Can\'t authenticate user  on LDAP server "%s" with error : %s ( authentication phase %s )' % (
                self.ldap_uri, e.message[u'desc'] if u'desc' in e.message else unicode(e), authentication_phase_id))
        finally:
            # Necessary for Oracle Directory Server to close connection immediately
            connexion_for_user.unbind_s()
            del connexion_for_user
        
        # No good? so no auth :)
        return False
