#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import json
import os
import shutil
import uuid

import time

from discoverymanager import DiscoveryManager
from discoveryrules import DiscoveryRulesManager, INVALID_STATE, WARNING_RULES_STATES
from shinken.log import logger, LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.basemodule import BaseModule, SOURCE_STATE
from shinken.objects.command import Command
from shinken.property import StringProp
from shinkensolutions.lib_modules.configuration_reader import read_string_in_configuration
from shinkensolutions.netaddr import valid_nmap_range, iter_nmap_range
from shinkensolutions.ssh_mongodb import ASCENDING
from shinkensolutions.ssh_mongodb.mongo_client import MongoClient
from shinkensolutions.ssh_mongodb.mongo_error import ShinkenMongoException
from .discovery_const import DiscoveryRangesStates
from .discovery_errors import DiscoveryErrors

try:
    from shinken.synchronizer.synchronizer_mongo_conf import SynchronizerMongoConf
except ImportError:
    from synchronizer.synchronizer_mongo_conf import SynchronizerMongoConf

if TYPE_CHECKING:
    from synchronizer.synchronizerdaemon import Synchronizer
    from shinken.misc.type_hint import Dict, Optional, Any, List

properties = {
    u'daemons': [u'synchronizer'],
    u'type'   : u'discovery-import',
    u'tabs'   : True
}

app = None  # type: Optional[Synchronizer]


# called by the plugin manager to get a module
def get_instance(plugin):
    logger.info(u'[Discovery Import] Get a Discovery import module for plugin %s' % plugin.get_name())
    
    instance = DiscoveryImport(plugin)
    return instance


class DiscoveryImport(BaseModule):
    SHINKEN_NMAP_MAC_PREFIXES_PATH = u'/etc/shinken/_default/sources/discovery/nmap/nmap-mac-prefixes'
    
    
    def __init__(self, mod_conf):
        BaseModule.__init__(self, mod_conf)
        self.source_name = mod_conf.get_name()[7:]
        self.logger = LoggerFactory.get_logger().get_sub_part(u'Discovery Import')
        self.mongodb_conf = SynchronizerMongoConf(mod_conf, self.logger, prefix_module_property=u'discovery-import')
        self.rules_path = read_string_in_configuration(mod_conf, u'rules_path', u'')
        self.nmap_mac_prefixes_user_path = read_string_in_configuration(mod_conf, u'nmap_mac_prefixes_path', u'')
        self.mongodb_con = None
        self.mongodb_db = None
        self.syncdaemon = None
        self.discovery_rules_manager = None
        self.discovery_confs_db = None
        self.discovery_rules_db = None
        self.global_lang = ''
        self.langs_path = ''
        self.langs = {}
        self.tabs = []
        self.known_hosts_address = []
        self.nmap_user_file_state = u'ok'
        self.old_discoveries_cleaned = False
        self.database_available = False
    
    
    # DON'T CHANGE THIS FUNCTION
    def _(self, s):
        lang = self.global_lang
        
        d = self.langs.get(lang, None)
        if d is None:
            pth = os.path.join(self.langs_path, lang + '.js')
            if not os.path.exists(pth):
                logger.error('Cannot load the lang file %s: no such file' % pth)
                return ''
            f = open(pth, 'r')
            buf = f.read()
            f.close()
            lines = buf.splitlines()
            new_lines = []
            for line in lines:
                line = line.strip()
                if line.startswith('//'):
                    continue
                if line.startswith('var lang'):
                    line = '{'
                if line.startswith('};'):
                    line = '}'
                new_lines.append(line)
            buf = '\n'.join(new_lines)
            o = json.loads(buf)
            self.langs[lang] = o
        o = self.langs[lang]
        elts = [e.strip() for e in s.split('.') if e.strip()]
        for e in elts:
            o = o.get(e, None)
            if o is None:
                self.logger.error('Traduction: cannot find %s in the lang %s' % (s, lang))
                return 'TO_TRAD(%s)' % s
        return o
    
    
    def _clean_old_discoveries(self):
        # Clean old discovery, so we will start new one
        all_previous_running_disco = list(self.discovery_confs_db.find({'state': DiscoveryRangesStates.RUNNING, 'source_name': self.get_source_name()}))
        for e in all_previous_running_disco:
            e['state'] = DiscoveryRangesStates.CRITICAL
            self.discovery_confs_db.save(e)
        self.old_discoveries_cleaned = True
    
    
    # Called by Arbiter to say 'let's prepare yourself guy'
    def init(self):
        self.logger.info('Initialization of the discovery import module')
    
    
    def load(self, daemon):
        self.syncdaemon = daemon
        
        self.global_lang = self.syncdaemon.lang
        self.langs_path = os.path.join(self.syncdaemon.modules_dir, properties['type'], 'htdocs', 'js', 'traductions')
        self.langs = {'en': None, 'fr': None}
        self.tabs = [{'name': 'rules', 'displayed_name': self._('rules.rules'), 'content': 'rules_list', 'errors': 0, 'warnings': 0}]
        self.try_database_connection()
        self.update_tabs()
        if self.database_available:
            self.reset_running_state_for_all_confs()
    
    
    def update_tabs(self):
        warnings = 0
        errors = 0
        # If we don't have discovery_rules_manager yet (probably because mongodb is unreachable) we can't update tabs correctly
        if not self.discovery_rules_manager:
            self.tabs[0]['errors'] = ''
            self.tabs[0]['warnings'] = ''
            return
        
        if self.discovery_rules_manager.status['user'] == 0:
            for rule in self.discovery_rules_manager.rules:
                if rule.state == INVALID_STATE:
                    errors += 1
                elif rule.state in WARNING_RULES_STATES:
                    warnings += 1
        if warnings == 0:
            warnings = ''
        if errors == 0:
            errors = ''
        self.tabs[0]['errors'] = errors
        self.tabs[0]['warnings'] = warnings
    
    
    def try_database_connection(self):
        # type: () -> bool
        if not self.database_available:
            try:
                self.__load_data_backend()
                self.database_available = True
            except Exception:
                self.database_available = False
        else:
            try:
                self.discovery_rules_db.find_one({})
            except Exception:
                self.database_available = False
        return self.database_available
    
    
    def get_source_name(self):
        return self.get_name()[7:]
    
    
    def _do_get_objects(self):
        self.known_hosts_address = []
        res = {u'state': u'', u'output': u'', u'objects': {}, u'warnings': [], u'errors': []}
        if self.discovery_rules_manager.status[u'default'] == 1:
            res[u'errors'].append(self.syncdaemon.t(
                u'import-discovery.critical_default_rule_file') % self.discovery_rules_manager.SHINKEN_RULES_PATH)
        if not self.__nmap_prefix_override():
            res[u'errors'].append(self.syncdaemon.t(
                u'import-discovery.critical_default_nmap_file') % self.SHINKEN_NMAP_MAC_PREFIXES_PATH)
        if self.nmap_user_file_state != u'ok':
            res[u'errors'].append(self.syncdaemon.t(
                u'nmap-user-file.%s' % self.nmap_user_file_state) % self.nmap_mac_prefixes_user_path)
        if res[u'errors']:
            res[u'state'] = SOURCE_STATE.CRITICAL
            res[u'output'] = self.syncdaemon.t(
                u'import-discovery.output_critical')
            return res
        self.discovery_rules_manager.update_user_rules()
        self.logger.info(u'Search for objects')
        self.logger.debug(u'discovery_confs [%s] ' % self.discovery_confs_db)
        
        discovery_confs_found = False
        objects = {u'host': []}
        
        for dc in self.discovery_confs_db.find({u'enabled': True, u'source_name': self.get_source_name()}):
            discovery_confs_found = True
            _id = dc[u'_id']
            state = dc[u'state']
            discovery_name = dc[u'discovery_name']
            
            self.logger.info(u'discovery_name %s state %s' % (discovery_name, state))
            
            if state in DiscoveryRangesStates.NEED_RUN_STATES:
                self.__do_launch_scan(dc)
            
            # Already launch, will be give after
            if state == DiscoveryRangesStates.RUNNING:
                continue
            
            lst = list(self.discovery_hosts_data.find({u'_DISCOVERY_ID': _id}))
            self.logger.debug(u'%d hosts from %s' % (len(lst), discovery_name))
            for o in lst:
                _SYNC_KEYS = o.get(u'_SYNC_KEYS', [])
                # If this element is missing or is void, do not export it...
                if not _SYNC_KEYS:
                    self.logger.debug(
                        u' (%s) skipping element %s as it is missing sync keys' % (discovery_name, o))
                    continue
                o[u'_SYNC_KEYS'] = (u','.join(_SYNC_KEYS)).lower()
                # Clean SE UUID inherited from discovery rules, and useless discovery id
                for k in [u'_SE_UUID', u'_SE_UUID_HASH', u'_DISCOVERY_ID']:
                    if k in o:
                        del o[k]
                objects[u'host'].append(o)
        
        if discovery_confs_found:
            res = {
                u'state'   : SOURCE_STATE.OK,
                u'output'  : self.syncdaemon.t(u'import-discovery.output_ok_load_successful'),
                u'objects' : objects,
                u'errors'  : [],
                u'warnings': []
            }
            if self.discovery_rules_manager.status[u'user'] != 0:
                res[u'state'] = SOURCE_STATE.WARNING
                res[u'warnings'].append(self.syncdaemon.t(u'import-discovery.warning_user_rules_not_parsed'))
                res[u'output'] = self.syncdaemon.t(
                    u'import-discovery.output_ok_load_sucessful_but_warnings')
        else:
            # We send OK state because we want clean old results
            res = {
                u'state'   : SOURCE_STATE.NOT_CONFIGURED,
                u'output'  : self.syncdaemon.t(u'import-discovery.output_nc_no_scan_range'),
                u'objects' : {u'host': []},
                u'errors'  : [],
                u'warnings': []
            }
        
        self.logger.info(u'import done load [%s] hosts' % len(objects[u'host']))
        self.update_tabs()
        return res
    
    
    def get_objects(self):
        if not self.try_database_connection():
            return {u'state': u'CRITICAL', u'output': self.syncdaemon.t(u'import-discovery.critical_database_access'), u'objects': {}, u'warnings': [], u'errors': []}
        try:
            return self._do_get_objects()
        except ShinkenMongoException:
            return {u'state': u'CRITICAL', u'output': self.syncdaemon.t(u'import-discovery.critical_database_access'), u'objects': {}, u'warnings': [], u'errors': []}
    
    
    def do_after_fork(self):
        # type:() -> None
        try:
            self.__load_data_backend()
        except ShinkenMongoException:
            # MongoDB client already add log about connection problems, so we don't want to add another log
            pass
        self.update_tabs()
    
    
    def compute_state(self, source):
        # Don't change the state if the source is running to prevent simultaneous runs : SEF-5260
        if source.state in (SOURCE_STATE.RUNNING, SOURCE_STATE.DIFFERENCE_COMPUTING):
            return
        source.output = ''
        source.summary_output = ''
        try:
            if self.is_all_confs_disabled(source.get_name()):
                source.state = SOURCE_STATE.NOT_CONFIGURED
                source.output = self.get_trad_no_conf_enabled()
            else:
                source.state = SOURCE_STATE.READY_FOR_IMPORT
        except Exception:
            source.state = SOURCE_STATE.CRITICAL
            source.output = self.syncdaemon.t(u'import-discovery.critical_database_access')
    
    
    def set_conf_enabled(self, conf_id, enabled):
        self.discovery_confs_db.update({'_id': conf_id}, update={'$set': {'enabled': (enabled == '1')}})
    
    
    def delete_conf(self, conf_id):
        self.discovery_confs_db.remove({'_id': conf_id})
    
    
    def _save_conf(self, conf_id, new_conf, source_name):
        conf = self.discovery_confs_db.find_one({'_id': conf_id})
        
        if conf is None:
            confs = [c['discovery_name'] for c in self.discovery_confs_db.find({'source_name': source_name})]
            if new_conf['discovery_name'] in confs:
                return DiscoveryErrors.NAME_ALREADY_EXISTS
                # return self.syncdaemon._('validator.disco_already_exist') % new_conf['discovery_name']
            
            conf = {'_id'             : uuid.uuid4().hex,
                    'state'           : DiscoveryRangesStates.PENDING,
                    'last_scan'       : 0,
                    'synchronizer'    : '',
                    'synchronizer_tag': '',
                    'scan_number'     : 0,
                    'source_name'     : source_name,
                    'last_heartbeat'  : 0}
        
        conf['discovery_name'] = new_conf['discovery_name']
        conf['iprange'] = new_conf['iprange']
        conf['scan_interval'] = new_conf['scan_interval']
        conf['notes'] = new_conf['notes']
        conf['enabled'] = new_conf['enabled']
        conf['port_range'] = new_conf['port_range']
        conf['extra_option'] = new_conf['extra_option']
        
        self.discovery_confs_db.save(conf)
        
        return 'ok'
    
    
    def save_conf(self, conf_id, new_conf, source_name):
        try:
            return self._save_conf(conf_id, new_conf, source_name)
        except ShinkenMongoException:
            return DiscoveryErrors.MONGO_UNREACHABLE
    
    
    def reset_running_state_for_all_confs(self):
        self.discovery_confs_db.update_many({'source_name': self.get_source_name(), 'state': DiscoveryRangesStates.RUNNING}, {'$set': {'state': DiscoveryRangesStates.OK}})
    
    
    def is_all_confs_disabled(self, source_name):
        enabled_conf_count = self.discovery_confs_db.find({'source_name': source_name, 'enabled': True}, only_count=True)
        return enabled_conf_count == 0
    
    
    def get_trad_no_conf_enabled(self):
        return self.syncdaemon.t('import-discovery.output_nc_no_scan_range')
    
    
    def get_confs(self):
        confs = []
        try:
            for dc in self.discovery_confs_db.find({'source_name': self.get_source_name()}):
                confs.append(dc)
        except Exception:
            return []
        return confs
    
    
    def get_extra_data(self):
        # type: () -> Dict
        discovery_confs = []
        try:
            discovery_confs_cursor = self.discovery_confs_db.find({'source_name': self.source_name}, sort=[('discovery_name', ASCENDING)])
            # add ip range as string list in each conf
            for conf in discovery_confs_cursor:
                conf_string_ip_ranges = conf.get('iprange', '')
                ranges = set()
                for conf_string_ip_range in conf_string_ip_ranges.split(' '):
                    if valid_nmap_range(conf_string_ip_range):
                        ip_range = [str(s) for s in iter_nmap_range(conf_string_ip_range)]
                        if '/' in conf_string_ip_range and len(ip_range) > 2:
                            # remove the first object and le last one (network and broadcast)
                            ip_range.pop(0)  # pop the first
                            ip_range.pop()  # pop the last
                        
                        ranges = ranges.union(set(ip_range))
                conf['string_iprange'] = list(ranges)
                discovery_confs.append(conf)
        except Exception:
            pass
        # get activate ratio
        
        nb = len([i for i in discovery_confs if i['enabled']])
        nb_total = len(discovery_confs)
        return {
            'discovery_confs'   : discovery_confs,
            'number_active_conf': nb,
            'number_total_conf' : nb_total
        }
    
    
    # Give try to load the mongodb and turn until we are killed or the monogo goes up
    def __load_data_backend(self):
        while not self.interrupted:
            self.__load_data_backend_try()
            # connect was ok? so bail out :)
            return
        if not self.old_discoveries_cleaned:
            self._clean_old_discoveries()
    
    
    def __load_data_backend_try(self):
        if self.mongodb_db is None:
            self.mongodb_db = MongoClient(self.mongodb_conf, self.logger)
            self.mongodb_db.init(u'Discovery Import')
        else:
            self.mongodb_db.recreate_connection(u'Discovery Import')
        self.discovery_confs_db = self.mongodb_db.get_collection(u'discovery_confs')
        self.discovery_rules_db = self.mongodb_db.get_collection(u'%s_rules' % self.source_name)
        if not self.discovery_rules_manager:
            self.discovery_rules_manager = DiscoveryRulesManager(self.mongodb_db, u'%s_rules' % self.source_name, self.rules_path)
        else:
            self.discovery_rules_manager.set_mongo_db(self.mongodb_db)
            self.discovery_rules_manager.update_user_rules()
        self.discovery_hosts_data = self.mongodb_db.get_collection(u'discovery_hosts_data')
        self.logger.debug('Connection OK for %s' % self.mongodb_db)
    
    
    def __nmap_prefix_override(self):
        default_path = self.SHINKEN_NMAP_MAC_PREFIXES_PATH
        user_path = self.nmap_mac_prefixes_user_path
        final_dest = '/usr/share/nmap/nmap-mac-prefixes'
        self.nmap_user_file_state = 'ok'
        if not os.path.isfile(default_path):
            self.logger.error('Default file nmap-mac-prefixes was not found at %s.' % default_path)
            return False
        if user_path and not os.path.isfile(user_path):
            if user_path:
                self.logger.error('User file nmap-mac-prefixes was not found at %s.' % user_path)
            self.nmap_user_file_state = 'not_found'
            shutil.copyfile(default_path, final_dest)
            return True
        shinken_file = open(default_path, 'r').readlines()
        user_file = []
        if user_path:
            user_file = open(user_path, 'r').readlines()
        
        to_write = []
        index = 0
        while index < len(shinken_file) and shinken_file[index].startswith('#'):
            to_write.append(shinken_file[index])
            index += 1
        for line in user_file:
            line = line.strip(' ')
            if not line.startswith('#') and line != '\n':
                if len(line.split(' ')[0]) != 6:
                    self.logger.error('Error in your nmap-mac-prefixes file, please verify it.')
                    shutil.copyfile(default_path, final_dest)
                    self.nmap_user_file_state = 'problem_in_file'
                    return True
                to_write.append(line)
        while index < len(shinken_file):
            to_write.append(shinken_file[index])
            index += 1
        final_file = open(final_dest, 'w+')
        final_file.write(''.join(to_write))
        return True
    
    
    def __do_launch_scan(self, dc):
        self.logger.info('do launch scan for %s' % dc['discovery_name'])
        
        now = int(time.time())
        _id = dc['_id']
        
        dc['state'] = DiscoveryRangesStates.RUNNING
        dc['last_heartbeat'] = now
        dc['last_scan'] = now
        dc['synchronizer'] = self.syncdaemon.me.get_name()
        port_range = dc.get('port_range', '')
        extra_option = dc.get('extra_option', '')
        
        self.discovery_confs_db.save(dc)
        
        backend = DiscoveryDummyBackend()
        macros = [('NMAPTARGETS', dc['iprange'])]
        
        setattr(self.syncdaemon.conf, '$NMAPTARGETS$', dc['iprange'])
        
        nmap_extra_option = ''
        if port_range:
            nmap_extra_option += ' -p %s' % port_range
        if extra_option:
            nmap_extra_option += ' %s' % extra_option
        
        nmapextraoption = 'NMAPEXTRAOPTION'
        # Add the macro value, you must set all this.
        self.syncdaemon.conf.__class__.properties['$' + nmapextraoption + '$'] = StringProp(default='')
        self.syncdaemon.conf.__class__.macros[nmapextraoption] = '$' + nmapextraoption + '$'
        self.syncdaemon.conf.resource_macros_names.append(nmapextraoption)
        setattr(self.syncdaemon.conf, '$' + nmapextraoption + '$', nmap_extra_option)
        macros.append((nmapextraoption, nmap_extra_option))
        
        # HACK: I don't understand why the nmap_discovery command is not linked here, it's a nonsense,
        # so temporary fix it by linking in HARD way a new command defined on the way
        nmap_discovery = Command(
            {
                'command_name'   : 'nmap_discovery',
                'shell_execution': '1',
                'command_line'   : '''$PLUGINSDIR$/discovery/nmap_discovery_runner.py --min-rate $NMAPMINRATE$ --max-retries $NMAPMAXRETRIES$ -t \'%s\' -o \'%s\'''' % (dc['iprange'], nmap_extra_option)
            }
        )
        
        # Hook the runner nmap and force the command
        for r in self.syncdaemon.conf.discoveryruns:
            r.discoveryrun_command.command = nmap_discovery
        
        d = DiscoveryManager(
            path=None,
            macros=macros,
            overwrite=True,
            runners=['nmap'],
            output_dir='',
            backend=backend,
            modules_path='',
            merge=False,
            conf=self.syncdaemon.conf,
            first_level_only=True,
            trad=self.syncdaemon.t,
            rulemanager=self.discovery_rules_manager
        )
        
        # Ok, let start the plugins that will give us the data
        d.launch_runners()
        all_ok = False
        while not all_ok:
            all_ok = d.is_all_ok()
            # Maybe dc was updated by configuration, if so
            # get the last version
            dc = self.discovery_confs_db.find_one({'_id': _id})
            dc['last_heartbeat'] = now
            self.logger.debug('The discovery range scan %s is still in progress' % dc['discovery_name'])
            self.discovery_confs_db.save(dc)
            time.sleep(0.5)
        
        d.wait_for_runners_ends()
        
        # We get the results, now we can reap the data
        d.get_runners_outputs()
        
        # and parse them
        d.read_disco_buf()
        
        # Now look for rules
        d.match_rules()
        
        d.loop_discovery()
        
        d.write_config()
        
        hosts_index = 0
        for h in backend.hosts:
            h['_id'] = uuid.uuid4().hex  # This is mandatory for insert_many purpose
            h['_DISCOVERY_ID'] = _id
            hname = h.get('host_name', '')
            address = h.get('address', '')
            possible_keys = [hname, address]
            h['_SYNC_KEYS'] = [k.lower() for k in possible_keys if k]
            if address in self.known_hosts_address:
                backend.hosts.pop(hosts_index)
            else:
                self.known_hosts_address.append(address)
            hosts_index += 1
        
        # First remove in the discovery_data col the
        self.discovery_hosts_data.remove({'_DISCOVERY_ID': _id})
        self.logger.debug('the discovery range %s did discover %d hosts' % (dc['discovery_name'], len(backend.hosts)))
        if len(backend.hosts) >= 1:
            self.discovery_hosts_data.insert_many(backend.hosts)
        
        # And save the discovery entry
        dc['state'] = DiscoveryRangesStates.OK
        self.discovery_confs_db.save(dc)
    
    
    def do_loop_turn(self):
        pass


class DiscoveryDummyBackend:
    def __init__(self):
        self.hosts = []
    
    
    def write_host_config_to_db(self, hname, h):
        print 'write_host_config_to_db::', hname, h
        self.hosts.append(h)


def _get_own_module(source_name):  # This function return your module class | MANDATORY
    # type:(unicode) -> DiscoveryImport
    return next((mod for mod in app.source_module_instances if mod.name[7:] == source_name), None)


def rules(source_name):
    # type:(unicode) -> Dict[unicode, Any]
    mod = _get_own_module(source_name)
    mod.update_tabs()
    
    rules_path = mod.rules_path
    display_rules = []
    
    if not mod.try_database_connection():
        return {
            u'sname'        : source_name,  # This is the name of your source
            u'app'          : app,  # This is the SynchronizerDaemon
            u'mod'          : mod,  # This is your  class for access to lang and whatever you want
            u'display_rules': display_rules,  # These are the rules in a list ready to be displayed
            u'status'       : 0,  # This is the user file status which allows us to show an error if the status isn't 0
            u'status_trad'  : 0,
            u'rules_path'   : rules_path,
            u'module_state' : 1
        }
    
    reload_rules(source_name)
    
    _rules = mod.discovery_rules_db.find({}, sort=(u'order', 1))
    
    for rule in _rules:
        display_rules.append(rule)
    
    status = mod.discovery_rules_manager.status[u'user']
    status_trad = mod.discovery_rules_manager.status[u'user_error_trad']
    
    return {
        u'sname'        : source_name,  # This is the name of your source
        u'app'          : app,  # This is the synchronizerdaemon
        u'mod'          : mod,  # This is your  class for access to lang and whatever you want
        u'display_rules': display_rules,  # These are the rules in a list ready to be displayed
        u'status'       : status,  # This is the user file status which allows us to show an error if the status isn't 0
        u'status_trad'  : status_trad,
        u'rules_path'   : rules_path,
        u'module_state' : 0
    }


def reload_rules(source_name):
    # type: (unicode) -> Dict[unicode,List[unicode]]
    mod = _get_own_module(source_name)
    if mod.discovery_rules_manager:
        mod.discovery_rules_manager.update_user_rules()
    return {u'warnings': mod.tabs[0][u'warnings'], u'errors': mod.tabs[0][u'errors']}


pages = {
    # rules routes
    rules       : {u'routes': [u'/sources/:source_name/rules/rules_list'], u'view': u'rules_list', u'static': True, u'wrappers': [u'json']},
    reload_rules: {u'routes': [u'/sources/:source_name/rules/rules_list/reload_rules/'], u'view': None, u'static': True, u'wrappers': [u'json']},
}
