#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2016-2022:
#    Shinken solutions

import json
import time

from event_container.ec_module_webui import EVENTS_EXPORT_MAX_RETRY_DEFAULT_VALUE
from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.localinstall import get_shinken_current_short_version, get_shinken_current_version_and_patch
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from webui.module import WebuiBroker

CONTEXT_PATH = '/var/lib/shinken/context.json'

app = None  # type: Optional[WebuiBroker]


def get_user():
    logger.debug(u'call get_user')
    user = app.get_user_auth()
    
    # Second we look for bdd access.
    collection = app.get_user_collection()
    if not collection:
        logger.info(u'get_user No more backend for saving the data')
        app.abort(500, u'No more backend for saving the data')
        return json.dumps(False)
    
    user_id = user.uuid
    # Our main key is the uuid
    user_info = collection.find_one({u'_id': user_id}, {u'_id': 0})
    
    # if no previous entries, None is good
    if user_info:
        user_group = app.get_contact_groups(user)
        if user_group:
            user_info[u'groups'] = []
            for group in user_group:
                user_info[u'groups'].append({u'uuid': group.uuid, u'name': group.get_name()})
        
        user_info[u'acl'] = {}
        for key, value in user.__dict__.iteritems():
            if key.startswith(u'acl_'):
                user_info[u'acl'][key[4:]] = value
        
        is_dst = time.daylight and time.localtime().tm_isdst > 0
        
        user_info[u'tiles_background'] = app.tiles_background
        user_info[u'apply_filter_method'] = app.apply_filter_method
        user_info[u'userName'] = user.contact_name
        user_info[u'is_admin'] = user.is_admin
        user_info[u'backend_localtime_epoch'] = int(time.time())
        user_info[u'time_zone_backend'] = int(time.altzone if is_dst else time.timezone)
        user_info[u'known_modules'] = list(set([mod.properties[u'type'] for mod in app.modules_manager.get_all_instances()]))
        user_info[u'show_external_url'] = user.acl_show_external_url
        
        if u'prefs' not in user_info:
            user_info[u'prefs'] = {}
        
        if u'historicTab' not in user_info[u'prefs']:
            user_info.get(u'prefs')[u'historicTab'] = {}
        
        if u'show_historic_outputs' not in user_info[u'prefs'][u'historicTab']:
            user_info.get(u'prefs').get(u'historicTab')[u'show_historic_outputs'] = app.history__display_outputs
        
        if u'collapse_all_historic_outputs' not in user_info[u'prefs'][u'historicTab']:
            user_info.get(u'prefs').get(u'historicTab')[u'collapse_all_historic_outputs'] = app.history__collapse_outputs
        
        if u'size_sla_column' not in user_info[u'prefs'][u'historicTab']:
            user_info.get(u'prefs').get(u'historicTab')[u'size_sla_column'] = app.history__size_sla_pane
        
        user_info[u'shinken_version'] = ToolsBoxString.convert_newlines_to_html_line_break(get_shinken_current_version_and_patch())
        user_info[u'shinken_short_version'] = get_shinken_current_short_version()
        user_info[u'configuration_id'] = app.datamgr.get_configuration_id()
        
        # The UI need the check plugin styles css hash: if does change, need to reload the UI
        user_info[u'check_plugin_styles_css_hash'] = app.get_check_plugin_styles_css_hash()
        
        # for admin only we will add the architecture map
        if user.is_admin:
            user_info[u'architecture_map'] = app.arch_export_handler.get_api_architecture_export()
        
        event_container_modules__days_keep_data = [mod.reader_stats.get_day_keep_data() for mod in app.modules_manager.get_all_instances() if mod.properties[u'type'] == u'event_container']
        if event_container_modules__days_keep_data:
            user_info[u'event_container_day_keep_data'] = min(event_container_modules__days_keep_data)
        
        user_info[u'events_export_max_retry'] = next((mod.events_export_max_retry for mod in app.modules_manager.get_all_instances() if mod.properties[u'type'] == u'event_container'), EVENTS_EXPORT_MAX_RETRY_DEFAULT_VALUE)
        user_info[u'events_csv_export_limit'] = getattr(app, u'csv_export_limit', 500)
        user_info[u'properties_display_text'] = app.get_properties_display_text()
        return json.dumps(user_info)
    else:
        logger.info(u'get_user user not found')
        app.abort(404, u'User not found')
        return json.dumps(False)


def save_user():
    user = app.get_user_auth()
    
    # Second we look for bdd access.
    user_collection = app.get_user_collection()
    if not user_collection:
        logger.info('save_user No more backend for saving the data')
        app.abort(500, 'No more backend for saving the data')
        return json.dumps(False)
    
    # Third load the data to save.
    if hasattr(app.request.body, 'getvalue'):
        data = json.loads(app.request.body.getvalue())
    else:  # file?
        data = json.loads(app.request.body.read())
    
    user_id = user.uuid
    # Eventually we save the data.
    logger.debug('hive data to save %s', data)
    data['uuid'] = user_id
    data['groups'] = []  # We clean group we don't want save this.
    user_collection.update(filter={'_id': user_id}, update=data, upsert=True)
    return json.dumps({'uuid': user_id})


# Remove all user screens & shares. Also delete user entry.
# Allow to recreate from old userdata.
def clean_user(uuid):
    shares = app.get_share_collection()
    hives = app.get_hive_collection()
    lists = app.get_list_collection()
    dashboards = app.get_dashboard_collection()
    removed_hives = hives.remove({'owner.uuid': uuid})
    removed_lists = lists.remove({'owner.uuid': uuid})
    removed_dashboards = dashboards.remove({'owner.uuid': uuid})
    removed_shares = shares.remove({'owner.uuid': uuid})
    removed = {
        'screen': {
            'hives'     : removed_hives.get('n', 0),
            'lists'     : removed_lists.get('n', 0),
            'dashboards': removed_dashboards.get('n', 0)
        },
        'shares': removed_shares.get('n', 0)
    }
    return json.dumps(removed)


pages = {
    get_user  : {'routes': ['/user'], 'method': 'GET', 'wrappers': ['auth', 'json']},
    save_user : {'routes': ['/user'], 'method': 'POST', 'wrappers': ['auth', 'json']},
    clean_user: {'routes': ['/user/:uuid/clean'], 'method': 'DELETE', 'wrappers': ['auth', 'json']}
}
