#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import itertools
import re
import time
import types
from collections import namedtuple

from shinken.commandcall import UNSET_POLLER_REACTIONNER_TAG_VALUE
from shinken.log import logger
from shinken.misc.fast_copy import fast_deepcopy, NO_COPY
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.beacon import Beacon
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString
from ...dao.def_items import can_property_contain_links, STOP_INHERITANCE_VALUES
from ...dao.helpers import get_default_value
from ...front_end.component_select_options import ComponentOption
from ...plugins.elements.apply_filters import filter_values, parse_filter, FilteringError

try:
    from ordereddict import OrderedDict
except ImportError:
    from collections import OrderedDict
from ...business.sync_ui_common import syncuicommon
from ...dao.def_items import ITEM_STATE, ITEM_TYPE, WORKING_AREA_STATUS, WORKING_AREA_LAST_ACTION, METADATA, SOURCE_STATE
from ...dao.helpers import item_for_link

if TYPE_CHECKING:
    pass

ItemValue = namedtuple('ItemValue', ['displayed_value', 'flat_value'])
VALUE_CACHE = {}

# These vars are set in elements.py at runtime through the set_app() function
app = None


class COLUMN(object):
    ALL = 'ALL'
    CUSTOM = 'CUSTOM'
    DEBUG = 'DEBUG'
    WITH_SOURCE_INFO = 'WITH_SOURCE_INFO'
    SI_ADMIN = 'SI_ADMIN'


class FORMATS(object):
    DEFAULT_STRING = 'list_column_formats/default_string'
    STRING_ACCEPT_XSS = 'list_column_formats/string_accept_xss'
    SIMPLE_STRING = 'list_column_formats/simple_string'
    TYPE_ELEMENT = 'list_column_formats/type_element'
    WORKING_AREA_STATUS = 'list_column_formats/working_area_status'
    NAME = 'list_column_formats/name'
    EDITION_RIGHTS = 'list_column_formats/edition_rights'
    DATA = 'list_column_formats/data'
    LIST = 'list_column_formats/list'
    USE = 'list_column_formats/use'
    DEBUG_CHECKS = 'list_column_formats/debug_checks'
    DEBUG_SONS = 'list_column_formats/debug_sons'
    DEBUG_HOSTS = 'list_column_formats/debug_hosts'
    SOURCES = 'list_column_formats/sources'
    TYPE = 'list_column_formats/type'
    USER = 'list_column_formats/user'
    STATUS = 'list_column_formats/status'


class ValueGetter:
    
    def __init__(self, value_name, item_type):
        self.value_name = value_name
        self._item_type = item_type
    
    
    def default_value_getter(self, item):
        if self._item_type != ITEM_TYPE.ELEMENTS and can_property_contain_links(self._item_type, self.value_name):
            value = item.flatten_prop(self.value_name)
        else:
            value = item.get(self.value_name, '')
        if value in STOP_INHERITANCE_VALUES:
            value = get_default_value(item.get_type(), self.value_name, app=app, same_as_host=True)
        if not value:
            value = ''
        
        return ItemValue(value, value)
    
    
    @staticmethod
    def shell_execution_getter(item):
        flat_value = item.get('shell_execution', '0')
        if flat_value in STOP_INHERITANCE_VALUES:
            flat_value = get_default_value(item.get_type(), 'shell_execution', app=app)
        value = app._('element.true') if flat_value != '0' else app._('element.false')
        return ItemValue(value, flat_value)
    
    
    @staticmethod
    def poller_tag_getter(item):
        flat_value = None
        value = item.get('poller_tag', 'None')
        if value in STOP_INHERITANCE_VALUES:
            if item.get_type() in (ITEM_TYPE.SERVICESHOSTS, ITEM_TYPE.SERVICESHOSTTPLS, ITEM_TYPE.SERVICESCLUSTERTPLS, ITEM_TYPE.SERVICESCLUSTERS, ITEM_TYPE.SERVICETPLS, ITEM_TYPE.COMMANDS):
                value = get_default_value(item.get_type(), 'poller_tag', app=app, same_as_host=True)
            else:
                value = app._('element.untagged')
        elif value == UNSET_POLLER_REACTIONNER_TAG_VALUE:
            if item.get_type() in (ITEM_TYPE.SERVICESHOSTS, ITEM_TYPE.SERVICESHOSTTPLS, ITEM_TYPE.SERVICESCLUSTERTPLS, ITEM_TYPE.SERVICESCLUSTERS, ITEM_TYPE.SERVICETPLS, ITEM_TYPE.COMMANDS):
                value = '%s [%s]' % (app._('element.default'), get_default_value(item.get_type(), 'poller_tag', app=app, same_as_host=True))
            else:
                value = '%s [%s]' % (app._('element.default'), app._('element.untagged'))
            flat_value = UNSET_POLLER_REACTIONNER_TAG_VALUE
        elif value == 'None':
            value = app._('element.untagged')
        if flat_value is None:
            flat_value = value
        return ItemValue(value, flat_value)
    
    
    @staticmethod
    def use_getter(item):
        flat_value = item.flatten_prop('use')
        value = '' if not flat_value else flat_value
        return ItemValue(value, flat_value)
    
    
    @staticmethod
    def enabled_getter(item):
        flat_value = item.get('enabled', '1')
        value = app._('element.enabled') if flat_value == '1' else app._('element.disabled')
        return ItemValue(value, flat_value)
    
    
    @staticmethod
    def wa_status_getter(item):
        if not ITEM_TYPE.has_work_area(item.get_type()):
            return ItemValue('', '')
        flat_value = WORKING_AREA_LAST_ACTION.get_work_area_action(item, METADATA.get_metadata(item, METADATA.STATE))
        value = VALUE_CACHE.get('working_area_state-%s' % flat_value, None)
        if value:
            return value
        
        if flat_value and '_' in flat_value:
            value = flat_value.lower()
            flat_value = flat_value.split('_')[0]
            value = app._('element.%s' % value)
        else:
            value = app._('element.%s' % flat_value) if flat_value else flat_value
        
        v = ItemValue(value, flat_value)
        VALUE_CACHE['working_area_state-%s' % flat_value] = v
        return v
    
    
    @staticmethod
    def element_type_getter(item):
        flat_value = METADATA.get_metadata(item, METADATA.ITEM_TYPE)
        value = app._('type.%s' % flat_value[:-1])
        return ItemValue(value, flat_value)
    
    
    @staticmethod
    def name_getter(item):
        value = METADATA.get_metadata(item, METADATA.NAME)
        return ItemValue(value, value)
    
    
    @staticmethod
    def last_modification_who_getter(item):
        last_modification = item.get('last_modification', {})
        value = ''
        if last_modification and 'date' in last_modification:
            last_modification_contact_links = last_modification.get('contact', {}).get('links', [])
            value = 'User not found'
            if last_modification_contact_links:
                contact = item_for_link(last_modification_contact_links[0], states=[ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION])
                value = contact.get_name()
        return ItemValue(value, value)
    
    
    @staticmethod
    def last_modification_date_getter(item):
        last_modification = item.get('last_modification', {})
        value = ''
        if last_modification and 'date' in last_modification:
            value = app.helper.print_date(float(last_modification.get('date')))
        return ItemValue(value, value)
    
    
    @staticmethod
    def bp_rule_getter(item):
        value = item.flatten_prop('bp_rule')
        if value is None or value in STOP_INHERITANCE_VALUES:
            value = ''
        return ItemValue(value, value)
    
    
    @staticmethod
    def can_edit_getter(item, user):
        flat_value = syncuicommon.look_for_host_acl(user, item, what_for='edition')
        value = app._('element.true') if flat_value else app._('element.false')
        return ItemValue(value, flat_value)
    
    
    @staticmethod
    def type_getter(item):
        flat_value = METADATA.get_metadata(item, METADATA.SOURCE_STATE)
        source_state = flat_value if flat_value != SOURCE_STATE.CHANGE else 'difference'
        value = app._('element.%s' % source_state) if source_state else ''
        return ItemValue(value, flat_value)
    
    
    @staticmethod
    def debug_checks_getter(item):
        value = item.get_link_checks()
        return ItemValue(value, value)
    
    
    @staticmethod
    def debug_hosts_getter(item):
        value = METADATA.get_metadata(item, METADATA.REVERSE_LINKS_CHECKS, [])
        return ItemValue(value, value)
    
    
    @staticmethod
    def debug_sons_getter(item):
        value = item.get_sons()
        return ItemValue(value, value)
    
    
    @staticmethod
    def admin_level_getter(item):
        value = item.get_admin_level()
        value = value if not value else app._('element.%s' % value)
        return ItemValue(value, value)
    
    
    @staticmethod
    def comments_getter(item):
        last_comment = app.helper.get_last_comment(item)
        value = '' if not last_comment else app._('element.comment_in_item') % (last_comment['user_name'], app.helper.print_date(last_comment['at']), app._('element.%s' % last_comment['on']), last_comment['text'])
        return ItemValue(value, value)
    
    
    @staticmethod
    def macromodulation_data_getter(item):
        value = ",".join(["%s: %s" % (k[1:], v) for (k, v) in item.iteritems() if k.startswith('_') and k.lower() not in ('_id', '_sync_keys', '__sync_idx__', '_se_uuid', '_se_uuid_hash')])
        return ItemValue(value, value)


column_configuration = {}
_column_configuration_cache = {}


def load_column_configuration():
    VALUE_CACHE['can_edit_ok'] = ItemValue(app._('element.true'), '1')
    for item_type in itertools.chain(ITEM_TYPE.ALL_TYPES, (ITEM_TYPE.ELEMENTS,)):
        column_configuration[item_type] = OrderedDict()
    
    _add_common_first_column()
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.ELEMENTS, 'type_element', app._('element.class'), size=120, have_value_list=True, get_value=ValueGetter.element_type_getter, value_formatter=FORMATS.TYPE_ELEMENT)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.ELEMENTS, 'working_area_status', app._('element.working_area_status'), size=120, have_value_list=True, get_value=ValueGetter.wa_status_getter, value_formatter=FORMATS.WORKING_AREA_STATUS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.ELEMENTS, 'name', app._('element.name'), get_value=ValueGetter.name_getter, value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.ELEMENTS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CLUSTERS, 'host_name', app._('element.name'), value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CLUSTERS, 'bp_rule', app._('element.definition'), get_value=ValueGetter.bp_rule_getter, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CLUSTERS, 'realm', app._('element.realm'), have_value_list=True, size=120, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CLUSTERS, 'poller_tag', app._('element.poller_tag'), have_value_list=True, get_value=ValueGetter.poller_tag_getter, size=100, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CLUSTERS, 'use', app._('type.clustertpl'), have_value_list=True, get_value=ValueGetter.use_getter, value_formatter=FORMATS.USE)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CLUSTERS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.CLUSTERS, 'notes_url', 'notes_url')
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.CLUSTERS, 'notes_multi_url', 'notes_multi_url')
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CLUSTERTPLS, 'name', app._('element.name'), get_value=ValueGetter.name_getter, value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CLUSTERTPLS, 'bp_rule', app._('element.definition'), get_value=ValueGetter.bp_rule_getter, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CLUSTERTPLS, 'realm', app._('element.realm'), have_value_list=True, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CLUSTERTPLS, 'poller_tag', app._('element.poller_tag'), have_value_list=True, get_value=ValueGetter.poller_tag_getter, size=100, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CLUSTERTPLS, 'use', app._('type.clustertpl'), have_value_list=True, get_value=ValueGetter.use_getter, value_formatter=FORMATS.USE)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CLUSTERTPLS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.CLUSTERTPLS, 'notes_url', 'notes_url')
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.CLUSTERTPLS, 'notes_multi_url', 'notes_multi_url')
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTS, 'working_area_status', app._('element.working_area_status'), have_value_list=True, get_value=ValueGetter.wa_status_getter, value_formatter=FORMATS.WORKING_AREA_STATUS)
    _column_configuration_factory(COLUMN.SI_ADMIN, ITEM_TYPE.HOSTS, 'can_edit', app._('element.can_edit'), size=70, have_value_list=True, value_formatter=FORMATS.EDITION_RIGHTS, get_value=ValueGetter.can_edit_getter)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTS, 'host_name', app._('element.name'), get_value=ValueGetter.name_getter, value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTS, 'display_name', app._('element.description'), value_formatter=FORMATS.DEFAULT_STRING)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTS, 'address', app._('element.address'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTS, 'realm', app._('element.realm'), have_value_list=True, size=80, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTS, 'poller_tag', app._('element.poller_tag'), have_value_list=True, get_value=ValueGetter.poller_tag_getter, size=80, value_formatter=FORMATS.DEFAULT_STRING)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTS, 'use', app._('element.templates'), have_value_list=True, get_value=ValueGetter.use_getter, value_formatter=FORMATS.USE)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTS, 'comments', app._('element.comment'), get_value=ValueGetter.comments_getter, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.DEBUG, ITEM_TYPE.HOSTS, 'checks', 'checks', get_value=ValueGetter.debug_checks_getter, value_formatter=FORMATS.DEBUG_CHECKS)
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.HOSTS, 'notes_url', 'notes_url')
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.HOSTS, 'notes_multi_url', 'notes_multi_url')
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTTPLS, 'name', app._('element.name'), get_value=ValueGetter.name_getter, value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTTPLS, 'realm', app._('element.realm'), have_value_list=True, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTTPLS, 'poller_tag', app._('element.poller_tag'), get_value=ValueGetter.poller_tag_getter, have_value_list=True, value_formatter=FORMATS.DEFAULT_STRING)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTTPLS, 'use', app._('type.hosttpl'), have_value_list=True, get_value=ValueGetter.use_getter, value_formatter=FORMATS.USE)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTTPLS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.DEBUG, ITEM_TYPE.HOSTTPLS, 'checks', 'checks', get_value=ValueGetter.debug_checks_getter, value_formatter=FORMATS.DEBUG_CHECKS)
    _column_configuration_factory(COLUMN.DEBUG, ITEM_TYPE.HOSTTPLS, 'sons', 'sons', get_value=ValueGetter.debug_sons_getter, value_formatter=FORMATS.DEBUG_SONS)
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.HOSTTPLS, 'notes_url', 'notes_url')
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.HOSTTPLS, 'notes_multi_url', 'notes_multi_url')
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTGROUPS, 'hostgroup_name', app._('element.name'), value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTGROUPS, 'members', app._('element.members'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTGROUPS, 'template_members', app._('element.template_members'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTGROUPS, 'synchronizer_internal_description', app._('hostgroups.synchronizer_internal_description'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.HOSTGROUPS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESHOSTS, 'service_description', app._('element.name'), value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESHOSTS, 'host_name', app._('element.apply_hosts'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESHOSTS, 'hostgroup_name', app._('element.apply_host_groups'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESHOSTS, 'use', app._('element.check_templates'), have_value_list=True, get_value=ValueGetter.use_getter, value_formatter=FORMATS.USE)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESHOSTS, 'poller_tag', app._('element.poller_tag'), have_value_list=True, get_value=ValueGetter.poller_tag_getter, size=90, value_formatter=FORMATS.DEFAULT_STRING)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESHOSTS, 'pack', app._('element.pack'), size=120, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.DEBUG, ITEM_TYPE.SERVICESHOSTS, 'hosts', 'hosts', get_value=ValueGetter.debug_hosts_getter, value_formatter=FORMATS.DEBUG_HOSTS)
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.SERVICESHOSTS, 'notes_url', 'notes_url')
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.SERVICESHOSTS, 'notes_multi_url', 'notes_multi_url')
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESHOSTTPLS, 'service_description', app._('element.name'), value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESHOSTTPLS, 'host_name', app._('element.apply_host_templates'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESHOSTTPLS, 'hostgroup_name', app._('element.apply_host_groups'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESHOSTTPLS, 'use', app._('element.check_templates'), have_value_list=True, get_value=ValueGetter.use_getter, value_formatter=FORMATS.USE)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESHOSTTPLS, 'poller_tag', app._('element.poller_tag'), have_value_list=True, get_value=ValueGetter.poller_tag_getter, size=90, value_formatter=FORMATS.DEFAULT_STRING)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESHOSTTPLS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.DEBUG, ITEM_TYPE.SERVICESHOSTTPLS, 'hosts', 'hosts', get_value=ValueGetter.debug_hosts_getter, value_formatter=FORMATS.DEBUG_HOSTS)
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.SERVICESHOSTTPLS, 'notes_url', 'notes_url')
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.SERVICESHOSTTPLS, 'notes_multi_url', 'notes_multi_url')
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESCLUSTERS, 'service_description', app._('element.name'), value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESCLUSTERS, 'host_name', app._('element.apply_clusters'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESCLUSTERS, 'use', app._('element.check_templates'), have_value_list=True, get_value=ValueGetter.use_getter, value_formatter=FORMATS.USE)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESCLUSTERS, 'poller_tag', app._('element.poller_tag'), have_value_list=True, get_value=ValueGetter.poller_tag_getter, value_formatter=FORMATS.DEFAULT_STRING)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESCLUSTERS, 'pack', app._('element.pack'), size=120, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.SERVICESCLUSTERS, 'notes_url', 'notes_url')
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.SERVICESCLUSTERS, 'notes_multi_url', 'notes_multi_url')
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESCLUSTERTPLS, 'service_description', app._('element.name'), value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESCLUSTERTPLS, 'host_name', app._('element.apply_cluster_templates'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESCLUSTERTPLS, 'use', app._('element.check_templates'), have_value_list=True, get_value=ValueGetter.use_getter, value_formatter=FORMATS.USE)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESCLUSTERTPLS, 'poller_tag', app._('element.poller_tag'), have_value_list=True, get_value=ValueGetter.poller_tag_getter, size=90, value_formatter=FORMATS.DEFAULT_STRING)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICESCLUSTERTPLS, 'pack', app._('element.pack'), size=120, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.SERVICESCLUSTERTPLS, 'notes_url', 'notes_url')
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.SERVICESCLUSTERTPLS, 'notes_multi_url', 'notes_multi_url')
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICETPLS, 'name', app._('element.name'), get_value=ValueGetter.name_getter, value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICETPLS, 'use', app._('element.check_templates'), get_value=ValueGetter.use_getter, value_formatter=FORMATS.USE, have_value_list=True)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICETPLS, 'poller_tag', app._('element.poller_tag'), size=90, have_value_list=True, get_value=ValueGetter.poller_tag_getter, value_formatter=FORMATS.DEFAULT_STRING)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.SERVICETPLS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.SERVICETPLS, 'notes_url', 'notes_url')
    _column_configuration_factory(COLUMN.CUSTOM, ITEM_TYPE.SERVICETPLS, 'notes_multi_url', 'notes_multi_url')
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTS, 'contact_name', app._('element.name'), value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTS, 'display_name', app._('element.display_name'), value_formatter=FORMATS.DEFAULT_STRING)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTS, 'notificationways', app._('element.notification_ways'), have_value_list=True, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTS, 'use', app._('element.contact_templates'), have_value_list=True, get_value=ValueGetter.use_getter, value_formatter=FORMATS.USE)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTS, 'contactgroups', app._('type.contactgroups'), have_value_list=True, value_formatter=FORMATS.LIST)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTS, 'admin_level', app._('element.admin_level'), get_value=ValueGetter.admin_level_getter, have_value_list=True, size=130, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTTPLS, 'name', app._('element.name'), get_value=ValueGetter.name_getter, value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTTPLS, 'notificationways', app._('element.notification_ways'), have_value_list=True, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTTPLS, 'use', app._('element.contact_templates'), have_value_list=True, get_value=ValueGetter.use_getter, value_formatter=FORMATS.USE)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTTPLS, 'contactgroups', app._('type.contactgroups'), have_value_list=True, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTTPLS, 'admin_level', app._('element.admin_level'), get_value=ValueGetter.admin_level_getter, have_value_list=True, size=130, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTTPLS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTGROUPS, 'contactgroup_name', app._('element.name'), value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTGROUPS, 'members', app._('element.members'))
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTGROUPS, 'synchronizer_internal_description', app._('contactgroups.synchronizer_internal_description'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.CONTACTGROUPS, 'pack', app._('element.pack'))
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.ESCALATIONS, 'escalation_name', app._('element.name'), value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.ESCALATIONS, 'first_notification_time', app._('element.escalation_start'), size=90, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.ESCALATIONS, 'last_notification_time', app._('element.escalation_end'), size=90, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.ESCALATIONS, 'contacts', app._('type.contact'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.ESCALATIONS, 'contact_groups', app._('type.contactgroup'))
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.ESCALATIONS, 'pack', app._('type.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.NOTIFICATIONWAYS, 'notificationway_name', app._('element.name'), value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.NOTIFICATIONWAYS, 'host_notification_period', app._('notificationways.host_notification_period'), have_value_list=True, size=120, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.NOTIFICATIONWAYS, 'service_notification_period', app._('notificationways.service_notification_period'), have_value_list=True, size=120, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.NOTIFICATIONWAYS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.COMMANDS, 'command_name', app._('element.name'), value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.COMMANDS, 'poller_tag', app._('element.poller_tag'), have_value_list=True, get_value=ValueGetter.poller_tag_getter, size=90, value_formatter=FORMATS.DEFAULT_STRING)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.COMMANDS, 'shell_execution', app._('element.run_in_a_shell'), get_value=ValueGetter.shell_execution_getter, value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.COMMANDS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.BUSINESSIMPACTMODULATIONS, 'business_impact_modulation_name', app._('element.name'), value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.BUSINESSIMPACTMODULATIONS, 'business_impact', app._('element.priority'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.BUSINESSIMPACTMODULATIONS, 'modulation_period', app._('element.modulation_period'), have_value_list=True, value_formatter=FORMATS.LIST)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.BUSINESSIMPACTMODULATIONS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.MACROMODULATIONS, 'macromodulation_name', app._('element.name'), value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.MACROMODULATIONS, 'custom', app._('element.datas'), get_value=ValueGetter.macromodulation_data_getter, value_formatter=FORMATS.DATA)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.MACROMODULATIONS, 'modulation_period', app._('element.modulation_period'), have_value_list=True, value_formatter=FORMATS.LIST)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.MACROMODULATIONS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.RESULTMODULATIONS, 'resultmodulation_name', app._('element.name'), value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.RESULTMODULATIONS, 'modulation_period', app._('element.modulation_period'), have_value_list=True, value_formatter=FORMATS.LIST)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.RESULTMODULATIONS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.TIMEPERIODS, 'timeperiod_name', app._('element.name'), value_formatter=FORMATS.NAME)
    _column_configuration_factory(COLUMN.ALL, ITEM_TYPE.TIMEPERIODS, 'pack', app._('element.pack'), value_formatter=FORMATS.STRING_ACCEPT_XSS)
    
    _add_common_last_column()


def elements_generic(item_type, items_state=ITEM_STATE.STAGGING, only_my_item=False):
    error_log_messages = []  # Fill this list to see errors back to front-end
    
    beacon = Beacon('item_list', enable=False, quiet=True)
    elements_time_start = time.time()
    
    in_working_area = items_state == ITEM_STATE.WORKING_AREA
    user = app.get_user_auth()
    is_admin = user.is_admin()
    log_perf_tag = 'item_lists [%s]' % user['contact_name']
    
    if not is_admin and item_type not in [ITEM_TYPE.HOSTS, ITEM_TYPE.CONTACTS]:
        return app.abort(403, app.t(u'element.no_edit'))
    
    # We want to limit the number of elements
    start = int(app.request.GET.get('start', '0'))
    end = int(app.request.GET.get('end', '300'))
    req_filter = app.request.GET.get('filter', '').strip()
    is_shinken_debug = app.request.GET.get('shinken_debug', '').strip()
    shinken_custom_list = app.request.GET.get('list_extra_columns', '').strip().split(',')
    
    if item_type not in ITEM_TYPE.ALL_TYPES and item_type != ITEM_TYPE.ELEMENTS:
        return app.abort(404, '404 Not Found')
    
    column_conf = _get_column_configuration(item_type, items_state, is_admin, is_shinken_debug, shinken_custom_list=shinken_custom_list)
    
    step = end - start if end > start else 300
    datamanagerv2 = app.datamanagerV2
    
    beacon.beacon('controller', 'arguments_parsing')
    items = _query_data(item_type, items_state, datamanagerv2, is_admin)
    beacon.beacon('controller', 'query_data')
    _filter, _nodes_filter = _parse_filter(req_filter, column_conf, error_log_messages)
    beacon.beacon('controller', 'parse_filter')
    
    items_extra_info = {}
    nb_of_elts_before_filter = 0
    start_time = time.time()
    in_page_items = []
    
    for item in items:
        _item_state = item.get_state()
        _item_type = item.get_type()
        _item_id = item['_id']
        
        if _item_state == ITEM_STATE.NEW:
            # The non admin user can't view new, skip the object
            if not is_admin:
                continue
            
            # The new hosts are in working area not in staging, skip the object
            if items_state == ITEM_STATE.STAGGING and ITEM_TYPE.has_work_area(_item_type):
                continue
        
        item_extra_info = {}
        items_extra_info[item.get_key()] = item_extra_info
        
        if items_state == ITEM_STATE.WORKING_AREA and only_my_item:
            working_area_status_value = ValueGetter.wa_status_getter(item)
            item_extra_info['working_area_status'] = working_area_status_value
            work_area_status = working_area_status_value.flat_value
            work_area_user_list = item.get('work_area_info', {}).get('get_by_user', [])
            if work_area_status == WORKING_AREA_STATUS.PROPOSED:
                continue
            elif not ((work_area_status == WORKING_AREA_STATUS.WORKING or work_area_status == WORKING_AREA_STATUS.REJECTED) and user['_id'] in work_area_user_list):
                continue
        beacon.beacon('controller', 'filter_elements', 'pre_filter')
        
        can_view = _compute_acl(_item_type, item, user, column_conf, item_extra_info)
        beacon.beacon('controller', 'filter_elements', 'compute_acl')
        
        if not can_view:
            continue
        
        nb_of_elts_before_filter += 1
        
        match_filter = True
        for filter_tag, filter_value in _nodes_filter.iteritems():
            # Show diffs and new for element who has a work area is not allowed on "all elements" page but they have the value, so we need to skip it.
            if filter_tag == 'type' and item_type == ITEM_TYPE.ELEMENTS and ITEM_TYPE.has_work_area(_item_type):
                match_filter = False
                break
            
            if filter_tag in item_extra_info:
                property_value = item_extra_info[filter_tag]
            else:
                property_value = column_conf[filter_tag]['get_value'](item)
                item_extra_info[filter_tag] = property_value
            
            if filter_value is not None and not filter_values(filter_value, ToolsBoxString.unescape_XSS(property_value.displayed_value)):
                match_filter = False
                break
        beacon.beacon('controller', 'filter_elements', 'apply_filter_on_one_element')
        
        if not match_filter:
            continue
        
        in_page_items.append(item)
    
    if _nodes_filter:
        logger.log_perf(start_time, '%s filter_element with %d filters' % (log_perf_tag, len(_filter)), 'Total time')
    
    if in_page_items:
        in_page_items.sort(key=lambda src: _natural_keys(src.get_type() + src.get_name().lower()))
    beacon.beacon('controller', 'sorting_elements')
    
    nb_all_elts = len(in_page_items)
    # If we overflow, came back as normal
    if start > nb_all_elts:
        start = 0
        end = 300
        step = 300
    
    navi = app.helper.get_navi(nb_all_elts, start, step=step)
    
    all_elts = in_page_items
    in_page_items = in_page_items[start:end]
    beacon.beacon('controller', 'pagination')
    
    for in_page_item in in_page_items:
        _item_id = in_page_item['_id']
        _build_extra_info(in_page_item, items_extra_info[in_page_item.get_key()], column_conf, in_working_area, beacon)
    
    nb_new_elts = len([e for e in all_elts if METADATA.get_metadata(e, METADATA.SOURCE_STATE) == SOURCE_STATE.NEW])
    if items_state == ITEM_STATE.STAGGING:
        nb_change_elts = len([e for e in all_elts if METADATA.get_metadata(e, METADATA.SOURCE_STATE) == SOURCE_STATE.CHANGE and not ITEM_TYPE.has_work_area(e.get_type())])
    else:
        nb_change_elts = len([e for e in all_elts if METADATA.get_metadata(e, METADATA.SOURCE_STATE) == SOURCE_STATE.CHANGE])
    nb_proposed_elts = len([e for e in all_elts if WORKING_AREA_STATUS.get_work_area_status(e, e.get_type()) == WORKING_AREA_STATUS.PROPOSED])
    
    values = _list_all_values(item_type, datamanagerv2, items_state, column_conf, only_my_item=False)
    beacon.beacon('controller', 'list_all_values_for_filter_dropdown')
    analyzer_enabled = False
    if item_type == ITEM_TYPE.HOSTS:
        sources = app.get_api_sources_from_backend()
        analyzer_enabled = sources.get('server-analyzer', {}).get('enabled', False)
    
    logger.log_perf(elements_time_start, '%s with %d items and %d filters' % (log_perf_tag, nb_of_elts_before_filter, len(_filter)), 'Total time')
    
    return {
        'title'               : '%s %s' % (app.t('element.elt_list_the'), app.t('type.%s' % item_type)),  # browser tab title
        'mypage'              : item_type,  # edit url
        'in_working_area'     : in_working_area,
        'addpage'             : item_type[:-1],  # add url
        'app'                 : app,
        'user'                : user,
        'items'               : in_page_items,
        'items_extra_info'    : items_extra_info,
        'item_type'           : item_type,
        'helper'              : app.helper,
        'navi'                : navi,
        'filter'              : _filter,
        'list_values'         : values,
        'nb_all_items'        : nb_all_elts,
        'nb_change_items'     : nb_change_elts,
        'nb_new_items'        : nb_new_elts,
        'nb_proposed_items'   : nb_proposed_elts,
        '_column_conf'        : column_conf,
        'analyzer_enabled'    : analyzer_enabled,
        'is_admin'            : user.is_admin(),
        'time_start'          : elements_time_start,
        'beacon'              : beacon,
        'backend_log_title'   : 'Error occurred when retrieving %s' % item_type,
        'backend_log_messages': error_log_messages,
    }


def _build_extra_info(item, item_extra_info, column_conf, in_working_area, beacon):
    for item_property, column_def in column_conf.iteritems():
        if item_property in item_extra_info:
            beacon.beacon('controller', 'build_extra_info', 'get_extra_info_for_one_element', item_property)
            continue
        property_value = column_def['get_value'](item)
        item_extra_info[item_property] = property_value
        beacon.beacon('controller', 'build_extra_info', 'get_extra_info_for_one_element', item_property)
    _add_data_to_extra_info(item_extra_info, item, in_working_area)
    beacon.beacon('controller', 'build_extra_info', 'get_extra_info_for_one_element')


def _compute_acl(_item_type, item, user, column_conf, item_extra_info):
    can_view = True
    if column_conf.get('can_edit', None):
        item_extra_info['can_edit'] = column_conf['can_edit']['get_value'](item, user)
        # If acl is enabled, maybe we should NOT look at them? :)
        can_edit = item_extra_info['can_edit'].flat_value
        
        if _item_type in (ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS, ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS):
            # First look at edition
            if can_edit:  # if you can edit, you can see, auto
                can_view = True
            else:  # you can't edit but maybe you can just see it
                can_view = syncuicommon.look_for_host_acl(user, item, what_for='view')
    elif _item_type in (ITEM_TYPE.CONTACTS, ITEM_TYPE.CONTACTTPLS):
        can_view = syncuicommon.look_for_contact_acl(user, item)
        item_extra_info['can_edit'] = VALUE_CACHE['can_edit_ok']
    else:
        item_extra_info['can_edit'] = VALUE_CACHE['can_edit_ok']
    return can_view


def _parse_filter(req_filter, column_conf, error_log_messages):
    _filter = {}
    _nodes_filter = {}
    if req_filter:
        for search_type in req_filter.split('~'):
            split_value = search_type.split(':', 1)
            if len(split_value) == 1:
                continue
            filter_tag = split_value[0].lower()
            if not column_conf.get(filter_tag, None):
                continue
            filter_value = split_value[1].lower().decode('utf8', 'ignore').replace('%fq%', '~').replace('%fg%', ':')
            _filter[filter_tag] = filter_value
        for filter_tag, filter_value in _filter.items():
            if not filter_value:
                continue
            try:
                _nodes_filter[filter_tag] = parse_filter(filter_value)
            except FilteringError as exc:
                message = u'The filter expression [ %s ] was refused: [ %s ]' % (filter_value, str(exc))
                logger.warning(message)
                error_log_messages.append(message)
                _nodes_filter[filter_tag] = None
    
    return _filter, _nodes_filter


def _query_data(item_type, items_state, datamanagerv2, is_admin):
    # special case for host Host in stagging because they need to manage the proposed
    lookup_states = [items_state]
    if is_admin:
        lookup_states.append(ITEM_STATE.NEW)
    if item_type in (ITEM_TYPE.HOSTS, ITEM_TYPE.ELEMENTS):
        if items_state == ITEM_STATE.STAGGING:
            proposed_items = datamanagerv2.find_merge_state_items(ITEM_TYPE.HOSTS, (ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA), where={'work_area_info.status': WORKING_AREA_STATUS.PROPOSED})
            proposed_ids = [i['_id'] for i in proposed_items]
            # exclude all the proposed hosts id's
            staging_items = datamanagerv2.find_merge_state_items(item_type, lookup_states, where={'_id': {'$nin': proposed_ids}})
            items = proposed_items + staging_items
        elif items_state == ITEM_STATE.WORKING_AREA:
            items = datamanagerv2.find_merge_state_items(item_type, item_states=[ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING, ITEM_STATE.NEW])
        else:
            items = []
    else:
        if len(lookup_states) > 1:
            items = datamanagerv2.find_merge_state_items(item_type, item_states=lookup_states)
        else:
            items = datamanagerv2.find_items(item_type, item_state=lookup_states[0])
    return items


def _get_column_configuration(item_type, items_state, is_shinken_admin, is_shinken_debug, shinken_custom_list=None):
    if shinken_custom_list is None:
        shinken_custom_list = []
    accepted_columns = [COLUMN.ALL]
    if is_shinken_debug:
        accepted_columns.append(COLUMN.DEBUG)
    if shinken_custom_list:
        accepted_columns.append(COLUMN.CUSTOM)
    if is_shinken_admin and (item_type == ITEM_TYPE.ELEMENTS or ITEM_STATE.WORKING_AREA in items_state or not ITEM_TYPE.has_work_area(item_type)):
        accepted_columns.append(COLUMN.WITH_SOURCE_INFO)
    if not is_shinken_admin:
        accepted_columns.append(COLUMN.SI_ADMIN)
    
    accepted_columns = tuple(accepted_columns)
    
    copy_dispatcher = {types.FunctionType: NO_COPY, types.MethodType: NO_COPY}
    if shinken_custom_list:
        column_conf = fast_deepcopy(column_configuration, additional_dispatcher=copy_dispatcher)
        for cc_type in column_conf.itervalues():
            for column_name, column_def in cc_type.items():
                if column_def['column_def'] not in accepted_columns or (column_def['column_def'] == COLUMN.CUSTOM and column_name not in shinken_custom_list):
                    cc_type.pop(column_name, None)
    else:
        column_conf = _column_configuration_cache.get(accepted_columns, None)
        if not column_conf:
            column_conf = fast_deepcopy(column_configuration, additional_dispatcher=copy_dispatcher)
            for cc_type in column_conf.itervalues():
                for column_name, column_def in cc_type.items():
                    if column_def['column_def'] not in accepted_columns:
                        cc_type.pop(column_name, None)
            
            _column_configuration_cache[accepted_columns] = column_conf
    
    return column_conf[item_type]


def _add_common_first_column():
    for item_type in itertools.chain(ITEM_TYPE.ALL_TYPES, (ITEM_TYPE.ELEMENTS,)):
        _column_configuration_factory(COLUMN.WITH_SOURCE_INFO, item_type, 'type', app._('element.type'), get_value=ValueGetter.type_getter, value_formatter=FORMATS.TYPE, extra_class='block-ok filter', have_value_list=True, size=84)
        _column_configuration_factory(COLUMN.ALL, item_type, 'status', app._('element.active'), get_value=ValueGetter.enabled_getter, value_formatter=FORMATS.STATUS, have_value_list=True, size=84)


def _add_common_last_column():
    for item_type in itertools.chain(ITEM_TYPE.ALL_TYPES, (ITEM_TYPE.ELEMENTS,)):
        _column_configuration_factory(COLUMN.ALL, item_type, 'last_modification_who', app._('element.who'), get_value=ValueGetter.last_modification_who_getter, value_formatter=FORMATS.USER)
        _column_configuration_factory(COLUMN.ALL, item_type, 'last_modification_date', app._('element.last_update'), get_value=ValueGetter.last_modification_date_getter, value_formatter=FORMATS.SIMPLE_STRING)
        _column_configuration_factory(COLUMN.ALL, item_type, 'sources', app._('element.sources'), have_value_list=True, size=120, value_formatter=FORMATS.SOURCES)


def _column_configuration_factory(column_def, item_type, column_name, label, size=0, have_value_list=False, value_formatter=FORMATS.DEFAULT_STRING, get_value=None, extra_class=''):
    if value_formatter is None:
        value_formatter = lambda value: value
    if get_value is None:
        get_value = ValueGetter(column_name, item_type).default_value_getter
    column_configuration[item_type][column_name] = {
        'label'          : label,
        'size'           : size,
        'have_value_list': have_value_list,
        'value_formatter': value_formatter,
        'get_value'      : get_value,
        'extra_class'    : extra_class,
        'column_def'     : column_def,
    }


def _add_data_to_extra_info(item_extra_info, item, in_working_area):
    _can_edit = 1 if item_extra_info['can_edit'].flat_value else 0
    work_area_status = item_extra_info.get('working_area_status', ItemValue('', '')).flat_value
    
    cannot_edit = '' if _can_edit else 'cannot-edit'
    work_area_class = ''
    # We allow a submit to stagging if the element is modified or was rejected (so direct resubmit without modification)
    if in_working_area and work_area_status in ['WORKING', 'REJECTED'] and not METADATA.get_metadata(item, METADATA.SOURCE_STATE) == ITEM_STATE.NEW:
        work_area_class = 'work-area-can-submit'
    elif work_area_status == 'PROPOSED':
        work_area_class = 'work-area-proposed'
    elif work_area_status == WORKING_AREA_STATUS.VALIDATED:
        work_area_class = 'work-area-validated'
    
    list_datas = 'data-status="%s" data-table="%s" data-item-type="%s" data-item-state="%s" data-item-name="%s" data-is-change="%s" data-can-edit="%s" data-item-work-last-action="%s"' % (
        item_extra_info['status'].flat_value,
        item.get_class(),
        item.get_type(),
        item.get_state(),
        item.get_name(),
        ('1' if item.have_manual_changes() and (in_working_area or item.get_type() != ITEM_TYPE.HOSTS) else '0'),
        _can_edit,
        WORKING_AREA_LAST_ACTION.get_work_area_action(item, item.get_state()).split('_')[-1])
    item_extra_info['data_infos'] = list_datas
    item_extra_info['additional_classes'] = '%s %s' % (cannot_edit, work_area_class)
    item_extra_info['changes'] = METADATA.get_metadata(item, METADATA.CHANGES)


def _list_all_values(_type, datamanagerv2, items_state, column_conf, only_my_item=False):
    values = {}
    column_which_needs_list = [name for name in column_conf if column_conf[name]['have_value_list']]
    for _key in column_which_needs_list:
        values[_key] = _list_value(_key, _type, datamanagerv2, items_state, only_my_item)
    return values


def _create_list_value(display, value=None, is_equal=False, always_visible=False, extra_class='', is_empty=False, can_have_xss_failure=False):
    if value is None and not is_equal:
        value = '&quot;%s&quot;' % display
    elif value is None and is_equal:
        value = '=&quot;%s&quot;' % display
    elif is_equal:
        value = '=&quot;%s&quot;' % value
    if is_empty:
        value = '=&quot&quot;'
    return ComponentOption('', True, value, False, 0, label=display, always_visible=always_visible, extra_class=extra_class, can_have_xss_failure=can_have_xss_failure)


def _list_value(_key, _type, datamanagerv2, items_state, only_my_item=False):
    # logger.debug('[list_elements] _key:%s' % _key)
    return_value = _create_list_value(['elements not found'])
    default_value = [
        _create_list_value('-- %s --' % app.t('element.all'), '', always_visible=True),
        _create_list_value('-- %s --' % app.t('element.empty'), is_empty=True),
    ]
    if _key == 'type':
        return_value = [
            _create_list_value(app.t('element.new'), is_equal=True),
            _create_list_value(app.t('element.difference'), is_equal=True)
        ]
    elif _key == 'status':
        return_value = [
            _create_list_value('-- %s --' % app.t('element.all'), '', always_visible=True),
            _create_list_value(app.t('element.enabled'), is_equal=True),
            _create_list_value(app.t('element.disabled'), is_equal=True)
        ]
    elif _key == 'shell_execution':
        return_value = [
            _create_list_value('-- %s --' % app.t('element.all'), '', always_visible=True),
            _create_list_value(app.t('element.true'), is_equal=True),
            _create_list_value(app.t('element.false'), is_equal=True)
        ]
    elif _key == 'admin_level':
        return_value = [
            _create_list_value('-- %s --' % app.t('element.all'), '', always_visible=True),
            _create_list_value(app.t('element.shinken_admin'), is_equal=True),
            _create_list_value(app.t('element.si_admin'), is_equal=True),
            _create_list_value(app.t('element.user'), is_equal=True)
        ]
    elif _key == 'can_edit':
        return_value = [
            _create_list_value('-- %s --' % app.t('element.all'), '', always_visible=True),
            _create_list_value(app.t('element.true'), is_equal=True),
            _create_list_value(app.t('element.false'), is_equal=True)
        ]
    elif _key == 'class':
        return_value = [
            _create_list_value(app.t('element.check_dedicated')),
            _create_list_value(app.t('element.template'))
        ]
    elif _key == 'working_area_status':
        if items_state == ITEM_STATE.WORKING_AREA and only_my_item:
            return_value = [
                _create_list_value('-- %s --' % app.t('element.all'), '', always_visible=True),
                _create_list_value(app.t('element.%s' % WORKING_AREA_STATUS.WORKING)),
                _create_list_value(app.t('element.%s' % WORKING_AREA_STATUS.REJECTED)),
            ]
        else:
            return_value = [
                _create_list_value('-- %s --' % app.t('element.all'), '', always_visible=True),
                _create_list_value(app.t('element.%s' % WORKING_AREA_STATUS.WORKING)),
                _create_list_value(app.t('element.working_new'), is_equal=True),
                _create_list_value(app.t('element.working_change'), is_equal=True),
                _create_list_value(app.t('element.working_delete'), is_equal=True),
                _create_list_value(app.t('element.%s' % WORKING_AREA_STATUS.REJECTED)),
                _create_list_value(app.t('element.rejected_new'), is_equal=True),
                _create_list_value(app.t('element.rejected_change'), is_equal=True),
                _create_list_value(app.t('element.rejected_delete'), is_equal=True),
                _create_list_value(app.t('element.%s' % WORKING_AREA_STATUS.PROPOSED)),
                _create_list_value(app.t('element.proposed_new'), is_equal=True),
                _create_list_value(app.t('element.proposed_change'), is_equal=True),
                _create_list_value(app.t('element.proposed_delete'), is_equal=True),
                _create_list_value(app.t('element.%s' % WORKING_AREA_STATUS.VALIDATED)),
            ]
    elif _key == 'realm':
        values = []
        for realm in app.conf.realms:
            realm = realm.get_name().strip()
            if realm not in values:
                values.append(_create_list_value(realm, is_equal=True))
        return_value = values
    elif _key == 'type_element':
        return_value = [
            _create_list_value('-- %s --' % app.t('element.all'), '', always_visible=True),
            _create_list_value(app.t('type.cluster'), is_equal=True),
            _create_list_value(app.t('type.clustertpl'), is_equal=True),
            _create_list_value(app.t('type.host'), is_equal=True),
            _create_list_value(app.t('type.hosttpl'), is_equal=True),
            _create_list_value(app.t('type.hostgroup'), is_equal=True),
            _create_list_value(app.t('type.serviceshost'), is_equal=True),
            _create_list_value(app.t('type.serviceshosttpl'), is_equal=True),
            _create_list_value(app.t('type.servicescluster'), is_equal=True),
            _create_list_value(app.t('type.servicesclustertpl'), is_equal=True),
            _create_list_value(app.t('type.servicetpl'), is_equal=True),
            _create_list_value(app.t('type.contact'), is_equal=True),
            _create_list_value(app.t('type.contacttpl'), is_equal=True),
            _create_list_value(app.t('type.contactgroup'), is_equal=True),
            _create_list_value(app.t('type.escalation'), is_equal=True),
            _create_list_value(app.t('type.notificationway'), is_equal=True),
            _create_list_value(app.t('type.businessimpactmodulation'), is_equal=True),
            _create_list_value(app.t('type.macromodulation'), is_equal=True),
            _create_list_value(app.t('type.resultmodulation'), is_equal=True),
            _create_list_value(app.t('type.command'), is_equal=True),
            _create_list_value(app.t('type.timeperiod'), is_equal=True)
        ]
    elif _key == 'poller_tag':
        values = []
        tags = set()
        for poller in app.conf.pollers:
            for poller_tag in getattr(poller, 'poller_tags', []):
                if poller_tag is None or poller_tag == u'None' or poller_tag in tags:
                    continue
                list_value = _create_list_value(ToolsBoxString.escape_XSS(poller_tag), is_equal=True, can_have_xss_failure=True)
                values.append(list_value)
                tags.add(poller_tag)
        
        return_value = values
        default_value = [_create_list_value('-- %s --' % app._('element.all'), '', always_visible=True)]
        if _type in (ITEM_TYPE.SERVICESHOSTS, ITEM_TYPE.SERVICESHOSTTPLS):
            default_value.append(_create_list_value('-- %s [%s] --' % (app.t('element.default'), app._('element.same_as_host')), '%s [%s]' % (app.t('element.default'), app.t('element.same_as_host')), is_equal=True, always_visible=True,
                                                    extra_class="shinken-inheritance"))
            default_value.append(_create_list_value('-- %s --' % app.t('element.same_as_host'), app._('element.same_as_host'), is_equal=True, always_visible=True))
            default_value.append(_create_list_value('-- %s --' % app._('element.untagged'), app._('element.untagged'), is_equal=True, always_visible=True))
        elif _type in (ITEM_TYPE.SERVICESCLUSTERS, ITEM_TYPE.SERVICESCLUSTERTPLS):
            default_value.append(_create_list_value('-- %s [%s] --' % (app.t('element.default'), app._('element.same_as_cluster')), '%s [%s]' % (app.t('element.default'), app.t('element.same_as_cluster')), is_equal=True, always_visible=True,
                                                    extra_class="shinken-inheritance"))
            default_value.append(_create_list_value('-- %s --' % app.t('element.same_as_cluster'), app._('element.same_as_cluster'), is_equal=True, always_visible=True))
            default_value.append(_create_list_value('-- %s --' % app._('element.untagged'), app._('element.untagged'), is_equal=True, always_visible=True))
        elif _type in (ITEM_TYPE.SERVICETPLS, ITEM_TYPE.COMMANDS):
            default_value.append(_create_list_value('-- %s [%s] --' % (app.t('element.default'), app._('element.same_as_host_parent')), '%s [%s]' % (app.t('element.default'), app.t('element.same_as_host_parent')), is_equal=True, always_visible=True,
                                                    extra_class="shinken-inheritance"))
            default_value.append(_create_list_value('-- %s --' % app.t('element.same_as_host_parent'), app._('element.same_as_host_parent'), is_equal=True, always_visible=True))
            default_value.append(_create_list_value('-- %s --' % app._('element.untagged'), app._('element.untagged'), is_equal=True, always_visible=True))
        else:
            default_value.append(
                _create_list_value('-- %s [%s] --' % (app.t('element.default'), app._('element.untagged')), '%s [%s]' % (app.t('element.default'), app.t('element.untagged')), is_equal=True, always_visible=True, extra_class="shinken-inheritance"))
            default_value.append(_create_list_value('-- %s --' % app.t('element.untagged'), app._('element.untagged'), is_equal=True, always_visible=True))
    elif _key == 'sources':
        values = []
        values.append(_create_list_value('shinken-core', is_equal=False))
        for source in app.sources:
            values.append(_create_list_value(source.get_name(), is_equal=False))
        return_value = values
    elif _key == 'modulation_period' or _key == 'host_notification_period' or _key == 'service_notification_period':
        values = []
        for timeperiod in datamanagerv2.find_items('timeperiods', items_state):
            tname_ = timeperiod.get_name()
            if tname_:
                values.append(_create_list_value(tname_, is_equal=True))
        return_value = values
    elif _key == 'use' and (_type in ITEM_TYPE.ALL_HOST_CLASS):
        values = []
        
        if items_state == ITEM_STATE.WORKING_AREA:
            for host_template in app.datamanagerV2.find_merge_state_items(ITEM_TYPE.HOSTTPLS, [ITEM_STATE.WORKING_AREA, ITEM_STATE.NEW, ITEM_STATE.STAGGING]):
                hname_ = host_template['name']
                if hname_:
                    values.append(_create_list_value(hname_, is_equal=False))
        else:
            _item_type = ITEM_TYPE.HOSTTPLS
            if _type == ITEM_TYPE.CLUSTERS or _type == ITEM_TYPE.CLUSTERTPLS:
                _item_type = ITEM_TYPE.CLUSTERTPLS
            for host_template in datamanagerv2.find_items(_item_type, items_state):
                hname_ = host_template.get_name()
                if hname_:
                    values.append(_create_list_value(hname_, is_equal=False))
        
        return_value = values
    elif _key == 'use' and _type in ITEM_TYPE.ALL_SERVICES:
        values = []
        for service_template in datamanagerv2.find_items(ITEM_TYPE.SERVICETPLS, items_state):
            sname_ = service_template.get_name()
            if sname_:
                values.append(_create_list_value(sname_, is_equal=False))
        return_value = values
    elif _key == 'use' and _type in ('contacts', 'contacttpls'):
        values = []
        for contact_template in datamanagerv2.find_items('contacttpls', items_state):
            cname_ = contact_template.get_name()
            if cname_:
                values.append(_create_list_value(cname_, is_equal=False))
        return_value = values
    elif _key == 'notificationways':
        values = []
        for notificationway in datamanagerv2.find_items(ITEM_TYPE.NOTIFICATIONWAYS, items_state):
            cname_ = notificationway.get_name()
            if cname_:
                values.append(_create_list_value(cname_, is_equal=False))
        return_value = values
    elif _key == 'contactgroups':
        values = []
        for notificationway in datamanagerv2.find_items(ITEM_TYPE.CONTACTGROUPS, items_state):
            cname_ = notificationway.get_name()
            if cname_:
                values.append(_create_list_value(cname_, is_equal=False))
        return_value = values
    
    if _key != 'admin_level':
        return_value = sorted(return_value, key=lambda k: k.label.lower())
    
    if _key in ['status', 'type_element', 'shell_execution', 'can_edit', 'working_area_status', 'admin_level']:
        default_value = return_value
    else:
        default_value.extend(return_value)
    return default_value


def _natural_keys(text):
    '''
    alist.sort(key=natural_keys) sorts in human order
    http://nedbatchelder.com/blog/200712/human_sorting.html
    '''
    return [_atoi(c.lower()) for c in re.split('(\d+)', text)]


def _atoi(text):
    return int(text) if text.isdigit() else text
