#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import base64
import copy
import json
import random
import re
import threading
import time
import zlib

from shinken.action import Action
from shinken.check import Check
from shinken.commandcall import UNSET_POLLER_REACTIONNER_TAG_VALUE
from shinken.http_client import HTTPClient, HTTPException
from shinken.log import logger
from shinken.macroresolver import COMMAND_LINE_LIMIT, MACRO_NB_LIMIT
from shinken.misc.type_hint import List, TYPE_CHECKING
from shinken.objects.service import Service
from shinken.objects.config import Config
from shinken.objects.resultmodulation import ModulationParser, ModulationSyntaxError
from shinken.safepickle import SafeUnpickler
from shinken.toolbox.string_helper import safe_replace
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString
from ...business.sync_ui_common import syncuicommon
from ...dao import item_saving_formatter
from ...dao.crypto import get_frontend_cipher_from_request
from ...dao.data_resolver import resolve_macros, truncate_long_command_line
from ...dao.def_items import ITEM_STATE, ITEM_TYPE, STOP_INHERITANCE_VALUES, VALUE_FORCE_DEFAULT, SERVICE_OVERRIDE, DEF_ITEMS, METADATA, PROP_DEFAULT_VALUE
from ...dao.from_info import DEFAULT_FROM_INFO
from ...dao.helpers import update_frontend_links_into_links, get_default_value, get_origin, item_for_link
from ...dao.items import get_item_instance

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Dict, Any, Union
    from ...synchronizerdaemon import Synchronizer
    from ...dao.datamanagerV2 import DataManagerV2
    from ...dao.items import BaseItem, RawItem
    from ...dao.items import InstanceItem

POLLER_HAVE_NO_CONFIGURATION = 2048

app = None  # type:Optional[Synchronizer]
datamanagerV2 = None  # type: Optional[DataManagerV2]

# We will have a result dict where our checks will be put back, but beware, multiple threads will access it (read+write)
check_done_by_poller_lock = threading.RLock()
check_done_by_poller = {}


class PollerException(Exception):
    def __init__(self, message=None):
        # type: (basestring) -> None
        super(PollerException, self).__init__(message)
        self.message = message


def _select_poller(command, cmd_poller_tag, realm):
    logger.debug("[try_check] selecting poller for command [%s] and poller_tag [%s] and realm [%s]" % (command.get_name(), cmd_poller_tag, realm.realm_name))
    logger.debug("[try_check] %s" % ('-' * 75))
    logger.debug("[try_check] list all pollers : ")
    logger.debug("[try_check] % 20s | % 6s | % 6s | % 20s | % 20s |" % ('poller_name', 'enable', 'spare', 'poller_tag', 'poller_mod'))
    logger.debug("[try_check] %s" % ('-' * 75))
    for poller in app.conf.pollers:
        logger.debug("[try_check] % 20s | % 6s | % 6s | % 20s | % 20s |" % (poller.get_name(), poller.enabled, poller.spare, poller.poller_tags, ','.join([getattr(mod, 'module_type') for mod in poller.modules])))
    logger.debug("[try_check] %s" % ('-' * 75))
    
    # ALL ENABLED POLLERS
    all_pollers = list([p for p in app.conf.pollers if p.enabled])
    # Err Case 1: no pollers defined at all
    if len(all_pollers) == 0:
        err = app._('element.run_check_error_no_pollers')
        return None, None, err
    
    # Pollers in the Good Realm
    # Note: in potential_pollers we have this realm pollers and higher realm pollers
    this_realm_pollers = list([p for p in realm.potential_pollers if p.enabled])
    # Err case 2: no poller in this realm
    if len(this_realm_pollers) == 0:
        err = app._('element.run_check_error_no_pollers_in_realm_or_higer_realm') % realm.get_name()
        return None, None, err
    
    # Pollers with good poller tag
    no_with_poller_tag = False
    pollers_by_tags = {'None': set()}
    for p in this_realm_pollers:
        # If no poller tags, None is the default
        if len(p.poller_tags) == 0:
            pollers_by_tags['None'].add(p)
            continue
        for tag in p.poller_tags:
            if tag not in pollers_by_tags:
                pollers_by_tags[tag] = set()
            pollers_by_tags[tag].add(p)
    
    # logger.debug('[try_check] pollers by tags [%s]' % pollers_by_tags)
    pollers_with_valid_tag = pollers_by_tags.get(cmd_poller_tag, set())
    if len(pollers_with_valid_tag) == 0:
        no_with_poller_tag = True
    
    # Pollers with good module
    no_with_module = False
    
    cmd_module_type = command.get('module_type', 'fork')
    cmd_module_type_escape_xss = ToolsBoxString.escape_XSS(cmd_module_type)
    cmd_poller_tag_escape_xss = ToolsBoxString.escape_XSS(cmd_poller_tag)
    
    pollers_by_modules = {'fork': set()}
    for p in this_realm_pollers:
        # logger.debug('[try_check] pollers possible: [%s]' % p)
        pollers_by_modules['fork'].add(p)
        for mod in p.modules:
            m_type = getattr(mod, 'module_type')
            if m_type not in pollers_by_modules:
                pollers_by_modules[m_type] = set()
            pollers_by_modules[m_type].add(p)
    
    # logger.debug('[try_check] pollers by modules: [%s]' % pollers_by_modules)
    pollers_with_valid_module = pollers_by_modules.get(cmd_module_type, set())
    if len(pollers_with_valid_module) == 0:
        no_with_module = True
    
    # ERRORS: we want specific and more precise message possible, so maybe we have both errors.
    # If only poller fail
    realm_name = ToolsBoxString.escape_XSS(realm.get_name())
    if no_with_poller_tag and not no_with_module:
        err = app._('element.run_check_error_no_pollers_with_poller_tag') % (cmd_poller_tag_escape_xss, realm_name)
        return None, None, err
    
    # if only module fail
    if no_with_module and not no_with_poller_tag:
        err = app._('element.run_check_error_no_pollers_with_module') % (cmd_module_type_escape_xss, realm_name)
        return None, None, err
    
    # Maybe both?
    if no_with_module and no_with_poller_tag:
        err = app._('element.run_check_error_no_pollers_with_module_and_pollers_with_module') % (cmd_poller_tag_escape_xss, cmd_module_type_escape_xss, realm_name)
        return None, None, err
    
    # Valid pollers must match both
    valid_pollers = list(pollers_with_valid_tag.intersection(pollers_with_valid_module))
    # Randomize the list of pollers, so we don't have always the same
    random.shuffle(valid_pollers)  # shuffle do it in place
    logger.debug('[try check] List of pollers that can manage this check: %s' % ', '.join([poller.get_name() for poller in valid_pollers]))
    # Ok so we have at least one matching poller :)
    
    if len(valid_pollers) == 0:
        err = app._('element.run_check_error_no_pollers_with_module_and_pollers_with_module') % (cmd_poller_tag_escape_xss, cmd_module_type_escape_xss, realm_name)
        return None, None, err
    
    # Get the first poller that match
    ret_val = None
    while True:
        connection = None
        err = ''
        p = valid_pollers.pop(0)  # we already look that it cannot be void
        poller_name = p.get_name()
        # Use a specific connection because such connections cannot be shared between threads.
        try:
            connection = _get_poller_connection(p)
            ret_val = (p, connection, None)
        except Exception as exp:
            if getattr(exp, 'errno', -1) == HTTPException.SSL_FAILED:
                err = app._('element.run_check_error_connection_error_ssl_failed') % poller_name
            elif getattr(exp, 'errno', -1) == POLLER_HAVE_NO_CONFIGURATION:
                err = app._('element.poller_no_configuration') % poller_name
            else:
                
                err = app._('element.run_check_error_connection_error') % (poller_name, getattr(exp, 'errstr', ''))
            ret_val = (None, None, err)
        
        if connection or not valid_pollers:
            break
    
    logger.debug('[try_check] poller: give back poller for execution: [%s]' % poller_name)
    return ret_val


def _get_poller_connection(poller):
    proto = 'https' if poller.use_ssl else 'http'
    uri = '%s://%s:%s/' % (proto, poller.address, poller.port)
    poller.uri = uri
    poller_name = poller.get_name()
    
    timeout = getattr(poller, 'timeout', 3)
    data_timeout = getattr(poller, 'data_timeout', 120)
    
    logger.debug("[try_check] Init connection with poller [%s] to [%s]" % (poller_name, uri))
    try:
        connection = HTTPClient(uri=uri, strong_ssl=poller.hard_ssl_name_check, timeout=timeout, data_timeout=data_timeout)
    except HTTPException as exp:
        logger.warning("[try_check] Connection problem with poller [%s] to [%s] : [%s]" % (poller_name, uri, exp))
        raise exp
    
    try:
        # initial ping must be quick
        managed_configuration = connection.get('get_currently_managed_configuration')
        logger.debug('[try_check] poller [%s] manage [%s]' % (poller.get_name(), managed_configuration))
        if not managed_configuration:
            logger.debug('[try check] The poller %s is not managing any configuration currently' % poller.get_name())
            pe = PollerException()
            pe.errno = POLLER_HAVE_NO_CONFIGURATION
            raise pe
        # Maybe it's not activated? if so it's like a no conf poller
        is_activated = managed_configuration.get('activated', False)  # old poller? i don't want to talk to you!
        if not is_activated:
            logger.debug('[try check] The poller %s is currently idle, cannot send it checks' % poller.get_name())
            pe = PollerException()
            pe.errno = POLLER_HAVE_NO_CONFIGURATION
            raise pe
    except HTTPException as exp:
        logger.warning("[try_check] Connection problem with poller [%s] to [%s] : [%s]" % (poller_name, uri, exp))
        raise exp
    
    logger.info("[try_check] Connection with poller [%s] to [%s] OK" % (poller_name, uri))
    return connection


def _execute_command_on_poller(poller, connection, command, command_name, exec_command_line, shell_execution, timeout, timeout_from):
    check = Check(
        'inpoller',
        exec_command_line,
        None,
        time.time(),
        depend_on_me=None,
        uuid=None,
        timeout=timeout,
        timeout_from=timeout_from,
        poller_tag=command.get('poller_tags', 'None'),
        reactionner_tag='None',
        env={},
        module_type=command.get('module_type', 'fork'),
        from_trigger=False,
        dependency_check=False,
        shell_execution=shell_execution,
        command_name=command_name)
    sync_id = 'sync_' + app.me.get_name()
    poller_name = poller.get_name()
    try:
        connection.post('synchronizer_push_actions', {'actions': [check], 'sync_id': sync_id})
    except Exception as exp:
        
        if exp.errno == 404:
            _message = '[try_check] The Poller [%s] is not up to date : %s' % (poller_name, exp)
            _user_err = app._('element.run_check_error_poller_not_up_to_date') % (poller_name, exp.errstr)
        else:
            _message = '[try_check] Connection problem with poller [%s] to [%s] : [%s]' % (poller_name, poller.uri, exp)
            _user_err = app._('element.run_check_error_poller_connection_problem')
        logger.warning(_message)
        raise PollerException(_user_err)
    
    with check_done_by_poller_lock:
        if poller_name not in check_done_by_poller:
            check_done_by_poller[poller_name] = {}
    
    logger.debug('[try_check]: execution timeout: %d' % timeout)
    send_time = int(time.time())
    kill_date = send_time + timeout + 15  # kill 15s after the timeout, quite large
    must_loop = True
    # We will get distant check only every 1s, but look at global dict every 100ms if another thread did get it
    loop_idx = -1
    while must_loop:
        loop_idx += 1
        with check_done_by_poller_lock:
            must_loop = check.get_uuid() not in check_done_by_poller[poller_name]
        
        if not must_loop:
            break
        # Maybe we are too late to get a result?
        now = int(time.time())
        if now > kill_date:
            _message = '[try_check] Execution was too slow on the poller [%s] [%s] : %d > %d (timeout)' % (poller_name, poller.uri, (now - send_time), timeout)
            logger.warning(_message)
            _user_err = app._('element.run_check_error_poller_execution_timeout') % poller_name
            raise PollerException(_user_err)
        # Only do http GET for return every 1s to not hammer the poller daemon
        if loop_idx % 10 == 0:
            # Ok we need to update our thread list
            try:
                results = connection.get('synchronizer_get_returns', {'sync_id': sync_id}, wait='long')
                results = base64.b64decode(results)
                results = zlib.decompress(results)
                results = SafeUnpickler.loads(str(results), u'Try check result from poller %s' % poller_name)
                
                for result in results:
                    with check_done_by_poller_lock:
                        check_done_by_poller[poller_name][result.get_uuid()] = result
            
            except Exception as exp:
                _message = '[try_check] Connection problem with poller [%s] to [%s] : [%s]' % (poller_name, poller.uri, exp)
                logger.warning(_message)
                _user_err = app._('element.run_check_error_poller_connection_problem')
                raise PollerException(_user_err)
        time.sleep(0.1)
    with check_done_by_poller_lock:
        result = check_done_by_poller[poller_name][check.get_uuid()]
        del check_done_by_poller[poller_name][check.get_uuid()]
    # Now parse the result and get a proper check with all managed/adapted to lang
    result.get_return_from(result, run_from='poller', lang=app.lang)
    return result


def _get_all_elt_template(elt, elt_type):
    # type: (Dict[unicode, Any], unicode) -> List[BaseItem]
    flattened_use = datamanagerV2.flatten_prop(elt, elt_type, 'use')
    elt_use = [t.strip() for t in flattened_use.split(',') if t.strip()]
    elt_templates = [datamanagerV2.find_item_by_name(tname, elt_type, ITEM_STATE.STAGGING) for tname in elt_use]
    return [t for t in elt_templates if t is not None]


def _expand_macro_in_command_line(expand_macros_infos, command_line):
    # type: (Dict[Any], unicode) -> (unicode, unicode, bool)
    global app
    logger.debug('[try_check] command line, before expand macros: %s' % command_line)
    
    command_line_protected = command_line
    for _, original_macro_name, macro_infos in expand_macros_infos.itervalues():
        if len(command_line) >= COMMAND_LINE_LIMIT or len(command_line_protected) >= COMMAND_LINE_LIMIT:
            break
        if macro_infos is None:
            macro_infos = {'value': '__UNKNOWN_DATA__', 'from': '', 'founded_prop': ''}
        # Hook 'null' properties to be change into '' so they can override templates
        if macro_infos['unprotected_value'] in STOP_INHERITANCE_VALUES:
            macro_infos['value'] = ''
            macro_infos['unprotected_value'] = ''
        
        macro_to_replace = u'$%s$' % original_macro_name
        if command_line.count(macro_to_replace) >= MACRO_NB_LIMIT or command_line_protected.count(macro_to_replace) >= MACRO_NB_LIMIT:
            translate = getattr(app, '_')
            command_line = "%s\n\n[ %s ]" % (translate('element.command_with_too_much_macros') % (MACRO_NB_LIMIT,), command_line)
            command_line_protected = "%s\n\n[ %s ]" % (translate('element.command_with_too_much_macros') % (MACRO_NB_LIMIT,), command_line_protected)
            return command_line, command_line_protected, True
        
        command_line, _ = safe_replace(command_line, macro_to_replace, macro_infos['unprotected_value'], COMMAND_LINE_LIMIT)
        command_line_protected, _ = safe_replace(command_line_protected, macro_to_replace, macro_infos['value'], COMMAND_LINE_LIMIT)
        del macro_infos['unprotected_value']
        macro_infos['value'] = macro_infos['value'].replace('$$', '$')
    
    # get back double dollar
    command_line = command_line.replace('$$', '$')
    command_line_protected = command_line_protected.replace('$$', '$')
    
    command_line, has_truncate = truncate_long_command_line(app, command_line)
    command_line_protected, has_truncate_protected = truncate_long_command_line(app, command_line_protected)
    
    has_error = has_truncate or has_truncate_protected
    
    return command_line, command_line_protected, has_error


def _apply_service_overrides(check, host, host_cluster_templates, dfe):
    # type: (BaseItem, Dict, List, Dict) -> None
    
    # Step : Build service_override link
    if SERVICE_OVERRIDE in host:
        item_saving_formatter._build_link_service_overrides(host, ITEM_TYPE.HOSTS, ITEM_STATE.STAGGING, datamanagerV2, add_link=True)
    
    # Step : Compute service_override inheritance
    host_service_overrides = host.get(SERVICE_OVERRIDE, {})
    service_overrides = {} if host_service_overrides.get(u'has_error', False) or host_service_overrides.get(u'has_errors', False) else host_service_overrides
    
    for tpl in host_cluster_templates:
        resolve_template_value = tpl.get(SERVICE_OVERRIDE, {})
        resolve_template_value = {} if resolve_template_value.get(u'has_error', False) or resolve_template_value.get(u'has_errors', False) else resolve_template_value
        
        if not service_overrides and resolve_template_value:
            service_overrides = resolve_template_value
        elif service_overrides and resolve_template_value:
            for tpl_link in resolve_template_value[u'links']:
                if not [i for i in service_overrides[u'links'] if (i[u'key'] == tpl_link[u'key'] and i[u'check_link'] == tpl_link[u'check_link'] and i.get(u'dfe_key', u'') == tpl_link.get(u'dfe_key', u''))]:
                    service_overrides[u'links'].append(tpl_link.copy())
    
    if not service_overrides:
        return
    
    dfe_key = dfe.get('key', '') if dfe else ''
    # Step : Apply full service_override
    for override in service_overrides.get('links', []):
        service_link = override.get('check_link', {})
        check_name = check.get_name()
        if service_link.get('_id') == check['_id'] and (not dfe_key or dfe_key == override.get('dfe_key', '')) or service_link.get('name') == check_name:
            if override['key'] == 'check_command_args':
                check['check_command'] = check['check_command'].copy()
                check['check_command']['node'] = check['check_command']['node'].copy()
                check['check_command']['node']['args'] = override['value']
            else:
                value = override['value']
                check_value = check.get(override['key'], {})
                if value in STOP_INHERITANCE_VALUES:
                    value = {
                        'has_plus': False,
                        'links'   : [{'exists': False, 'name': value}]
                    }
                elif override['key'] in DEF_ITEMS[check.get_type()]['fields_with_plus'] and override['value']['has_plus'] and check_value:
                    value = {
                        'has_plus': True,
                        'links'   : override['value']['links'] + check_value.get('links', [])
                    }
                
                check._raw_set_value(override['key'], value)


def _prepare_modulate_data(item):
    # type: (Union[InstanceItem, RawItem]) -> None
    if hasattr(item, 'prepare_modulate_data'):
        timeperiods = _make_timeperiods()
        item.prepare_modulate_data(timeperiods)
        return
    
    if 'macromodulations' not in item:
        return
    
    item['@modulation'] = {}
    
    timeperiods = None
    modulations = []
    for modulation_link in item['macromodulations']['links']:
        if modulation_link['exists']:
            found_modulation = datamanagerV2.find_item_by_id(modulation_link['_id'], item_type=ITEM_TYPE.MACROMODULATIONS, item_state=ITEM_STATE.STAGGING)
            if found_modulation.is_enabled():
                modulations.append(found_modulation)
    
    now = time.time()
    for modulation in reversed(modulations):
        if not modulation.is_enabled():
            continue
        mod_name = modulation.get_name()
        
        modulation_period = modulation.get_link_item('modulation_period', only_exist=True)
        if modulation_period:
            if not timeperiods:
                timeperiods = _make_timeperiods()
            timeperiod = timeperiods.find_by_name(modulation_period.get_name())
            if not timeperiod or not timeperiod.is_time_valid(now):
                continue
        
        for data_name, data_value in modulation.iteritems():
            if not data_name.startswith('_'):
                continue
            item['@modulation'][data_name] = (data_value, mod_name)
            # if the data modulate do not exist in the item,
            # we add a none value to show a empty item value in revolve data table
            if data_name not in item:
                item[data_name] = None


def _modulate_result(response_json, check):
    resultmodulations = check['@result_modulation']
    if not resultmodulations:
        return
    
    now = time.time()
    timeperiods = None
    tokenizer = ModulationParser()
    
    response_json['resultmodulations'] = []
    response_json['applied_resultmodulation'] = {}
    
    modulated = False
    for resultmodulation in resultmodulations:
        if not resultmodulation.is_existing():
            # If we don't have the link, maybe it's still in new state
            _resultmodulation = app.datamanagerV2.find_item_by_name(resultmodulation.get_name(), item_type=ITEM_TYPE.RESULTMODULATIONS, item_state=ITEM_STATE.NEW)
            
            # If not then it is really unknown and we skip it
            if not _resultmodulation:
                modulation_info = {
                    'missing': True,
                    'item'   : {'resultmodulation_name': resultmodulation.get_name()}
                }
                response_json['resultmodulations'].append(modulation_info)
                continue
        
        json_rules = []
        modulation_info = {
            'timeperiod': None,
            'rules'     : json_rules,
            'item'      : resultmodulation.get_raw_item(keep_metadata=(METADATA.STATE,))
        }
        
        # TIMEPERIOD
        valid_tp = True
        modulation_period = resultmodulation.get_link_item('modulation_period')
        if modulation_period:
            if not timeperiods:
                timeperiods = _make_timeperiods()
            timeperiod = timeperiods.find_by_name(modulation_period.get_name())
            if modulation_period.is_existing() and timeperiod:
                valid_tp = timeperiod.is_time_valid(now)
                tp_infos = {'eval': valid_tp}
            else:
                valid_tp = False
                tp_infos = {'missing': True}
            
            modulation_info['timeperiod'] = tp_infos
        
        response_json['resultmodulations'].append(modulation_info)
        if resultmodulation.get('output_rules', ''):
            # RULES
            try:
                rules = tokenizer.parse(tokenizer.tokenize(resultmodulation['output_rules']))
            except ModulationSyntaxError as e:
                # TODO This should never happen -> SEF-4074
                continue
            
            for rule in rules:
                if modulated or not valid_tp or not resultmodulation.is_enabled():
                    in_state_eval = 'no eval'
                    pattern_eval = 'no eval'
                elif rule.in_states and response_json['execution']['rc'] not in rule.in_states:
                    in_state_eval = 'false'
                    pattern_eval = 'no eval'
                else:
                    in_state_eval = 'true'
                    try:
                        if not rule.regexp or re.search(rule.regexp, response_json['execution']['stdout']):
                            pattern_eval = 'true'
                            resultmodulation_name = resultmodulation.get_name() if resultmodulation else ''
                            response_json['applied_resultmodulation'] = {
                                'name'  : resultmodulation_name,
                                'rc'    : rule.out_state,
                                'origin': check.get_type()
                            }
                        else:
                            pattern_eval = 'false'
                    except:
                        pattern_eval = 'false'  # Some invalid regex here
                    
                    modulated = modulated or pattern_eval == 'true'
                result_mod_result = {
                    'in_state'     : rule.in_states,
                    'in_state_eval': in_state_eval,
                    'pattern'      : rule.regexp,
                    'pattern_eval' : pattern_eval,
                    'out_state'    : rule.out_state
                }
                json_rules.append(result_mod_result)


def _make_timeperiods():
    timeperiods = datamanagerV2.find_merge_state_items(ITEM_TYPE.TIMEPERIODS, [ITEM_STATE.STAGGING, ITEM_STATE.NEW])
    raw_objects = {'timeperiod': [timeperiod.get_raw_item() for timeperiod in timeperiods]}
    
    for timeperiod in raw_objects['timeperiod']:
        todel = [k for k in timeperiod if not isinstance(timeperiod[k], basestring)]
        for k in todel:
            del timeperiod[k]
    
    conf = Config()
    conf.create_objects_for_type(raw_objects, 'timeperiod')
    conf.timeperiods.explode()
    conf.timeperiods.create_reversed_list()
    conf.timeperiods.linkify()
    return conf.timeperiods


def _get_command_timeout(command, check, host):
    # type: (BaseItem, BaseItem, dict) -> (int, str)
    # Return the "command" timeout first if available..
    _timeout = command.get('timeout', '-1')
    _from = 'command'
    if _timeout in ('-1', ''):
        _timeout = check.get('check_running_timeout', '-1')
        _from = 'check'
        if _timeout in ('-1', ''):
            _timeout = host.get('check_running_timeout', '-1')
            _from = 'host'
            if _timeout in ('-1', ''):
                _timeout = app.global_conf.check_running_timeout
                _from = 'global'
    
    if not _timeout.isdigit():
        raise PollerException(app._('element.run_check_error_no_valid_timeout') % (_timeout, _from))
    return int(_timeout), _from


def _prepare_result_modulation(check, host, templates):
    # type: (BaseItem, dict, list) -> None
    actual_resultmodulations = None
    
    # If there are only a "+" in check we do not get host value
    check_resultmodulations_links = check.get_link_items('resultmodulations')
    check_plus_in_resultmodulations = check.get('resultmodulations', {}).get('has_plus', False)
    
    if check_resultmodulations_links and check_resultmodulations_links[0].get_name() in STOP_INHERITANCE_VALUES:
        # Here the STOP_INHERITANCE_VALUES means "same_as_host" so we don't want stop the inheritance in this case.
        # When adding the correct operation of STOP_INHERITANCE_VALUES, use the following line here and the line with "pass" with the new keyword for same_as_host
        # actual_resultmodulations = []
        pass
    elif not check_resultmodulations_links and check_plus_in_resultmodulations:
        actual_resultmodulations = []
    elif check_resultmodulations_links:
        actual_resultmodulations = check_resultmodulations_links
    
    if actual_resultmodulations is not None:
        check['@result_modulation'] = actual_resultmodulations
        return
    
    host_plus_in_resultmodulations = False
    host_result_modulation = host.get('resultmodulations', {})
    if host_result_modulation in STOP_INHERITANCE_VALUES:
        actual_resultmodulations = []
    elif host_result_modulation:
        host_resultmodulations_links = host_result_modulation.get('links', [])
        host_plus_in_resultmodulations = host_result_modulation.get('has_plus', False)
        
        if host_resultmodulations_links:
            actual_resultmodulations = []
            for host_resultmodulations_link in host_resultmodulations_links:
                host_resultmodulation = item_for_link(host_resultmodulations_link, only_exist=True)
                if host_resultmodulation is not None:
                    actual_resultmodulations.append(host_resultmodulation)
    
    if actual_resultmodulations is not None and not host_plus_in_resultmodulations:
        check['@result_modulation'] = actual_resultmodulations
        return
    
    for template in templates:
        hosttpl_resultmodulations_links = template.get_link_items('resultmodulations')
        host_plus_in_resultmodulations = template.get('resultmodulations', {}).get('has_plus', False)
        
        if hosttpl_resultmodulations_links and hosttpl_resultmodulations_links[0].get_name() in STOP_INHERITANCE_VALUES:
            actual_resultmodulations = [] if actual_resultmodulations is None else actual_resultmodulations
        elif hosttpl_resultmodulations_links:
            if actual_resultmodulations is None:
                actual_resultmodulations = hosttpl_resultmodulations_links
            else:
                actual_resultmodulations.extend(hosttpl_resultmodulations_links)
        
        if actual_resultmodulations is not None and not host_plus_in_resultmodulations:
            check['@result_modulation'] = actual_resultmodulations
            return
    
    check['@result_modulation'] = actual_resultmodulations


def run_check():
    # type: () -> basestring
    app.response.content_type = 'application/json'
    user = app.get_user_auth()
    return_val = {}  # type: Dict[unicode,Any]
    
    forms = json.loads(app.request.forms.get('host', '{}'))
    check_id = app.request.forms.get('check_id', '')
    do_exec = app.request.forms.get('do_exec', '0') == '1'
    on_poller = app.request.forms.get('on_poller', '0') == '1'
    dfe = json.loads(app.request.forms.get('dfe', '{}'))
    item_type = app.request.forms.get('item_type', ITEM_TYPE.HOSTS)
    
    if not check_id:
        return json.dumps({'error': app._('element.missing_id_check')})
    
    check = datamanagerV2.find_item_by_id(check_id, ITEM_TYPE.SERVICESCLUSTERS, ITEM_STATE.STAGGING) or \
            datamanagerV2.find_item_by_id(check_id, ITEM_TYPE.SERVICETPLS, ITEM_STATE.STAGGING) or \
            datamanagerV2.find_item_by_id(check_id, ITEM_TYPE.SERVICESCLUSTERTPLS, ITEM_STATE.STAGGING) or \
            datamanagerV2.find_item_by_id(check_id, ITEM_TYPE.SERVICESHOSTS, ITEM_STATE.STAGGING) or \
            datamanagerV2.find_item_by_id(check_id, ITEM_TYPE.SERVICESHOSTTPLS, ITEM_STATE.STAGGING)
    
    if check is None:
        return json.dumps({'error': app._('element.no_check')})
    
    host = json.loads(forms['item'])
    
    # Transforms frontend links into real links
    # As there is no static get_links() method, we need a minimal BaseItem with the required data,
    # in order to call its build_link() method
    base_item = get_item_instance(item_type, raw_item=host)
    METADATA.update_metadata(base_item, METADATA.ITEM_TYPE, item_type)
    for linking_property in DEF_ITEMS[item_type]['props_links']:
        update_frontend_links_into_links(linking_property, base_item.get_links(linking_property), item_type, datamanagerV2)
    
    db_host = datamanagerV2.find_merge_state_items(ITEM_TYPE.HOSTS, [ITEM_STATE.NEW, ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING], where={'_id': host['_id']})
    if not db_host:  # We didn't found any host, maybe it's a cluster !
        db_host = datamanagerV2.find_merge_state_items(ITEM_TYPE.CLUSTERS, [ITEM_STATE.NEW, ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING], where={'_id': host['_id']})
    db_host = db_host[0] if db_host else None
    
    if db_host:
        acl = syncuicommon.check_acl(db_host, user, u'')
        if not acl[u'can_view']:
            return app.abort(401)
    
    frontend_cipher = get_frontend_cipher_from_request(app.request.forms.get('host', '{}'))
    
    host = frontend_cipher.uncipher(host, item_type=ITEM_TYPE.HOSTS, old_item=db_host, user=user)
    host_name = host.get(u'host_name', host.get(u'address', u'name not found'))
    check_name = check.get_name()
    
    logger.debug(u'[try_check] try check [%s]' % check_name)
    if dfe:
        logger.debug(u'[try_check] dfe ARGS key [%s] / value [%s]' % (dfe.get(u'key', u''), dfe.get(u'value', u'')))
    
    # We need the poller_tag and realm value, but we must look at the templates values if need
    templates = _get_all_elt_template(host, DEF_ITEMS[item_type][u'template'])
    
    check = _clone_for_try_check(check)
    _apply_service_overrides(check, host, templates, dfe)
    _prepare_result_modulation(check, host, templates)
    _prepare_modulate_data(check)
    _prepare_modulate_data(host)
    
    # Get command_name and args
    command = check.get_link_item(u'check_command')
    if not command:
        return json.dumps({u'error': app.t(u'element.no_such_command')})
    
    poller_tag_for_execution = _get_poller_tag_for_execution(check, command, host, templates, item_type)
    
    # Save for realm, we need a final value
    host_realm = host.get(u'realm', u'')
    logger.debug(u'[try check] LOOKING FOR REALM The host setting realm is "%s"' % u''.join(host_realm))
    if not host_realm:
        logger.debug(u'[try check] LOOKING FOR REALM will look into templates: %s' % u','.join([i.get_name() for i in templates]))
        for tpl in templates:
            host_realm = tpl.get(u'realm', u'')
            # As the template is already resolved maye it give us the default value, we don't need that
            is_in_fact_the_default_value = tpl.is_prop_default_value(u'realm')
            logger.debug(u'[try check] LOOKING FOR REALM The host template %s realm => "%s" (is the default value?=%s)' % (tpl, host_realm, is_in_fact_the_default_value))
            if host_realm and not is_in_fact_the_default_value:
                break
    
    command_name = command.get_name()
    
    if command_name == u'bp_rule':
        return json.dumps({u'error': app.t(u'element.no_run_bp_rule')})
    
    check_command_arg = check[u'check_command'][u'node'][u'args']
    
    # Get command_line
    command_line = command.get(u'command_line', u'')
    if not command_line:
        return json.dumps({u'error': app.t(u'element.no_cmdline')})
    original_command_line = command_line
    
    # Get all value of the command line macros
    _expand_macros, error_msg = resolve_macros(app, host, templates, check, check_command_arg, command_line, dfe, item_type, frontend_cipher)
    
    if error_msg:  # in case of error (recursive limit)
        expanded_command_is_invalid = True
        expanded_command_line = error_msg
        expanded_command_line_protected = error_msg
    else:
        # Replace all macros in command
        expanded_command_line, expanded_command_line_protected, expanded_command_is_invalid = _expand_macro_in_command_line(_expand_macros, command_line)
    
    # We want a result with only higher macros
    argns = [(macro_name, macro_info) for macro_name, (_, _, macro_info) in _expand_macros.iteritems()]
    argns_sorted = sorted(argns, key=lambda (_, macro): macro['resolve_order'])
    return_val['macros'] = argns_sorted
    return_val['command_line'] = original_command_line
    # non-admin will have a protected commande line display
    # uncomment the next line to let the admin see the 'real' command line
    # return_val['expanded_command_line'] = expanded_command_line if is_admin else expanded_command_line_protected
    return_val['expanded_command_line'] = expanded_command_line_protected
    logger.debug('[try_check] new command line, to be returned: [%s]' % return_val['expanded_command_line'])
    # really exec it
    if do_exec:
        # before go to exec, be sure to not launch '__UNKNOWN_DATA__' into it, because scheduler replace it by '' at execution
        exec_command_line = expanded_command_line.replace('__UNKNOWN_DATA__', '')
        shell_execution = (command.get('shell_execution', '0') == '1')
        
        try:
            (timeout, timeout_from) = _get_command_timeout(command, check, host)
        except PollerException as exp:
            return_val['execution'] = {
                'stdout'         : exp.message,
                'stderr'         : exp.message,
                'execution_time' : 0,
                'rc'             : 2,
                'u_time'         : 0,
                's_time'         : 0,
                'cpu_time'       : 0,
                'executor_type'  : 'none',
                'shell_execution': False
            }
            return json.dumps(return_val)
        
        if on_poller:
            formatted_command_name = "%s-//-%s-//-%s" % (host_name, check_name, command_name)
            
            # We search a poller to run our command if we don't found it we try to run on local machine
            # We have the realm as string, search the final object
            if host_realm:
                host_realm = app.conf.realms.find_by_name(host_realm)
                if not host_realm:
                    return json.dumps({'error': app._('element.run_check_error_invalide_realm') % host.get('realm', '')})
            else:
                host_realm = app.conf.realms.get_default()
            
            poller, connection, err = _select_poller(command, poller_tag_for_execution, host_realm)
            if err is not None:
                return json.dumps({'error': err})
            if expanded_command_is_invalid:
                # The command is invalid, we set result directly
                result = Check(
                    status=u'',
                    command=exec_command_line,
                    ref=None,
                    t_to_go=None
                )
                result.exit_status = 3
                result.output = expanded_command_line
            else:
                try:
                    result = _execute_command_on_poller(poller, connection, command, formatted_command_name, exec_command_line, shell_execution, timeout, timeout_from)
                except PollerException as exp:
                    err = unicode(exp)
                    return json.dumps({'error': err})
            
            if result.perf_data:
                result.perf_data = ' | ' + result.perf_data
            
            return_val['execution'] = {
                'stdout'         : Service.get_unknown_return_code_output(result.exit_status, result.output + result.long_output + result.perf_data),
                'stderr'         : Service.get_unknown_return_code_output(result.exit_status, result.output + result.long_output + result.perf_data),
                'output'         : Service.get_unknown_return_code_output(result.exit_status, result.output),
                'long_output'    : result.long_output,
                'perf_data'      : result.perf_data,
                'execution_time' : result.execution_time,
                'rc'             : Service.map_state_from_return_code(result.exit_status),
                'u_time'         : result.u_time,
                's_time'         : result.s_time,
                'cpu_time'       : result.cpu_time,
                'shell_execution': shell_execution,
                'executor_type'  : 'poller',
                'executor_name'  : poller.get_name(),
            }
        
        else:
            start = time.time()
            if expanded_command_is_invalid:
                # The command is invalid, we set result directly
                a = Check(
                    status=u'',
                    command=exec_command_line,
                    ref=None,
                    t_to_go=None
                )
                a.executor_id = app.me.display_name
                a.exit_status = 3
                a.output = expanded_command_line
            else:
                a = Action(None, exec_command_line, command_name, timeout, 'none', timeout_from=timeout_from)
                a.executor_id = app.me.display_name
                a.shell_execution = shell_execution
                a.execute()
                while not a.is_finish():
                    a.check_finished(app.global_conf.max_plugins_output_length)
                    time.sleep(a.check_finished_period)
                # Now parse the result and get a proper check with all managed/adapted to lang
                a.get_return_from(a, run_from='synchronizer', lang=app.lang)
                
                if a.perf_data:
                    a.perf_data = ' | ' + a.perf_data
            return_val['execution'] = {
                'stdout'         : Service.get_unknown_return_code_output(a.exit_status, a.output + a.long_output + a.perf_data),
                'stderr'         : Service.get_unknown_return_code_output(a.exit_status, a.output + a.long_output + a.perf_data),
                'output'         : Service.get_unknown_return_code_output(a.exit_status, a.output),
                'long_output'    : a.long_output,
                'perf_data'      : a.perf_data,
                'execution_time' : time.time() - start,
                'rc'             : Service.map_state_from_return_code(a.exit_status),
                'u_time'         : a.u_time,
                's_time'         : a.s_time,
                'cpu_time'       : a.cpu_time,
                'shell_execution': shell_execution,
                'executor_type'  : 'synchronizer',
                'executor_name'  : app.me.get_name(),
            }
        
        _modulate_result(return_val, check)
    
    else:
        return_val['execution'] = None
    
    return json.dumps(return_val)


def _get_poller_tag_for_execution(check, command, host, host_cluster_templates, item_type):
    # Choose the right poller tag to use
    # Priority is :
    # 1. the one defined in the check
    # 2. the one defined in the host
    # 3. the one defined in the command
    poller_tag_to_use = get_default_value(item_type, 'poller_tag')
    command_poller_tag = command.get('poller_tag', '')
    check_poller_tag = check.get('poller_tag', '')
    host_poller_tag = host.get('poller_tag', '')
    tag_found = False
    # Check
    if check_poller_tag in STOP_INHERITANCE_VALUES:
        poller_tag_to_use = get_default_value(ITEM_TYPE.SERVICESHOSTS, 'poller_tag')
        tag_found = True
    elif get_origin(check, 'poller_tag') != DEFAULT_FROM_INFO:
        poller_tag_to_use = check_poller_tag
        tag_found = True
    # Host
    elif host_poller_tag in STOP_INHERITANCE_VALUES:
        poller_tag_to_use = get_default_value(item_type, 'poller_tag')
        tag_found = True
    elif host_poller_tag:
        poller_tag_to_use = host_poller_tag
        tag_found = True
    # Host templates because the host does not have its inheritance here
    else:
        host_tpl_poller_tag = ''
        for tpl in host_cluster_templates:
            tmp_host_tpl_poller_tag = tpl.get('poller_tag', '')
            origin = get_origin(tpl, 'poller_tag')
            if tmp_host_tpl_poller_tag and origin != DEFAULT_FROM_INFO:
                host_tpl_poller_tag = tmp_host_tpl_poller_tag
        # If we still haven't found a poller tag defined in the templates or if it is forced to default value
        # The resulting value is set to the default value
        if host_tpl_poller_tag in (PROP_DEFAULT_VALUE, VALUE_FORCE_DEFAULT):
            poller_tag_to_use = get_default_value(item_type, 'poller_tag')
            tag_found = True
        elif host_tpl_poller_tag:
            poller_tag_to_use = host_tpl_poller_tag
            tag_found = True
    
    # Command
    if not tag_found and command_poller_tag in STOP_INHERITANCE_VALUES:
        poller_tag_to_use = get_default_value(ITEM_TYPE.COMMANDS, 'poller_tag')
        tag_found = True
    elif not tag_found and get_origin(command, 'poller_tag') != DEFAULT_FROM_INFO:
        poller_tag_to_use = command_poller_tag
        tag_found = True
    
    if not tag_found:
        poller_tag_to_use = get_default_value(item_type, 'poller_tag')
    
    # The default value in the class is UNSET_POLLER_REACTIONNER_TAG_VALUE but we need
    # to search for None when looking for real poller
    if poller_tag_to_use == UNSET_POLLER_REACTIONNER_TAG_VALUE:
        poller_tag_to_use = 'None'
    
    return poller_tag_to_use


def _clone_for_try_check(check):
    # Warning this method do a simple copy of the check.
    # DO NOT MODIFY INNER DICT
    new_item = get_item_instance(check.get_type())
    new_item['@metadata'] = copy.copy(check["@metadata"])
    raw_item = METADATA.get_metadata(check, METADATA.RAW_ITEM)
    if raw_item:
        METADATA.update_metadata(new_item, METADATA.RAW_ITEM, copy.deepcopy(raw_item))
    for prop, value in check.iteritems():
        if prop == "@metadata":
            continue
        new_item._raw_set_value(prop, value)
    return new_item
