#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

import Cookie
import codecs
import re
import uuid

import shinken.misc.configuration_error_log as arbiter_configuration_messages
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.common_ui import COMMON_UI_HTDOCS_DIRECTORY, COMMON_UI_VIEWS_DIRECTORY
from shinkensolutions.logs.logger_authentication import LoggerAuthentication
from shinkensolutions.ssh_mongodb import ASCENDING
from shinkensolutions.ssh_mongodb.mongo_error import ShinkenMongoException
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString

if TYPE_CHECKING:
    from typing import Optional


def monkey_patch_cookie():
    _LegalCharsPatt = r"\w\d!#%&'~_`><@,:/\$\*\+\-\.\^\|\)\(\?\}\{\="
    _MorselAndNextKeyPattern = re.compile(r"""
        ;?,?\s*                     # Possible separator from preceding morsel.
        (?P<key>
            [%s]+?                  # Any word of at least one letter, nongreedy
        )
        \s*=\s*                     # Equal Sign
        (?P<val>
            "(?:[^\\"]|\\.)*"       # Any doublequoted string
            |                       # or
            [%s\ ]*?                # Any word or empty string
        )
        # Match against the start of the next morsel (or end of the string) to
        # be able to distinguish between ' ' and ',' being part of this morsel's
        # value or the separator for the next morsel.
        \s*
        (?P<next>                   # The start of the next morsel
            (;,|;|,)                # ... a separator
            \s*[%s]+?               # ... the key
            \s*=\s*                 # ... the equal sign
            |                       # or
            ;?,?
            \Z                      # the end of the string.
        )
        """ % (_LegalCharsPatt, _LegalCharsPatt, _LegalCharsPatt), re.VERBOSE)
    
    
    def __ParseString(self, str, patt=_MorselAndNextKeyPattern):
        i = 0  # Our starting point
        n = len(str)  # Length of string
        M = None  # current morsel
        
        while 0 <= i < n:
            # Start looking for a cookie
            match = patt.match(str, i)
            if not match:
                break  # No more cookies
            
            K, V = match.group("key").strip(), match.group("val").strip()
            i = match.start("next")
            
            # Parse the key, value in case it's metainfo
            if K[0] == "$":
                # We ignore attributes which pertain to the cookie
                # mechanism as a whole.  See RFC 2109.
                # (Does anyone care?)
                if M:
                    M[K[1:]] = V
            elif K.lower() in Cookie.Morsel._reserved:
                if M:
                    if V is None:
                        if K.lower() in Cookie.Morsel._flags:
                            M[K] = True
                    else:
                        M[K] = Cookie._unquote(V)
            elif V is not None:
                rval, cval = self.value_decode(V)
                self._BaseCookie__set(K, rval, cval)
                M = self[K]
    
    
    setattr(Cookie, '_LegalCharsPatt', _LegalCharsPatt)
    setattr(Cookie, '_MorselAndNextKeyPattern', _MorselAndNextKeyPattern)
    setattr(Cookie.BaseCookie, '_BaseCookie__ParseString', __ParseString)


if not hasattr(Cookie, '_MorselAndNextKeyPattern'):
    monkey_patch_cookie()

import hmac
import urllib
import errno
import httplib
import imp
import shinkensolutions.shinkenjson as json
import os
import shutil
import signal
import socket
import string
import sys
import traceback
import time
import multiprocessing
import ctypes
import gc
import threading
import ssl
import base64
import binascii
import inspect
from Cookie import SimpleCookie
from multiprocessing import Process, Event
from pymongo.errors import OperationFailure, ConnectionFailure
from shinkensolutions.ssh_mongodb.sshtunnelmongomgr import mongo_by_ssh_mgr
from . import route_wrappers
from .cherrypybackend import CherryPyServerHTTP
from .front_end.helper import Helper
from .business.item_controller import exceptions
from .route_wrappers import format_validator_message_if_not_admin, format_validator_message_if_base_exception, abort_if_not_admin_nor_admin_si, abort_if_not_admin, protect_transaction, protect_db_transaction, protect_internal_call
from .synchronizerconfig import SynchronizerConfig
from .dao.def_items import (DEF_ITEMS, ITEM_STATE, ITEM_TYPE, NAGIOS_TABLES, SRC_NAME_MAX_LENGTH)
from .dao.datamanagerV2 import DataManagerV2
from .dao.dataprovider.dataprovider_mongo import DataProviderMongo
from .dao.dataprovider.dataprovider_metadata import DataProviderMetadata
from .dao.trash_manager.trash_manager import trash_manager
from .dao.database_migration import migrate_database
from .dao.crypto import DatabaseCipher, FrontendCipher
from .dao.validators.validator import Validator
from .dao.helpers import set_item_property

from .business.source.source import Source
from .dao.helpers import split_and_strip_list
from .business.source.source_controller import SourceController
from .business.analyzer_controller import AnalyzerController
from .business.sync_ui_common import syncuicommon
from .business.item_controller.state_controller import StateController
from .business.item_controller.massive_change.massive_change_controller import MassiveChangeController
from .business.arbiter_controler import ArbiterController
from .business.core_item import add_core_items
from shinken.modules.base_module.basemodule import SOURCE_STATE
from .component.component_manager import component_manager
from shinken.daemon import Daemon, Interface, IStatsInterface
from shinken.log import logger, LoggerFactory, INFO
from shinken.util import start_malloc_trim_thread
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.config import Config
from shinken.objects.host import VIEW_CONTACTS_DEFAULT_VALUE
from shinken.property import BoolProp
from shinken.transactioner import transaction_protecter
from shinkensolutions.external_resources.external_resources import external_resources
from .htdocs.translate.links_documentation import DocumentationLinks
from shinkensolutions.localinstall import get_shinken_current_short_version, get_shinken_current_version_and_patch
from shinkensolutions.shinken_time_helper import print_human_readable_period
import shinkensolutions.api.synchronizer.http_lib_external as http_lib_external
import shinkensolutions.api.synchronizer.source.translation as source_translation
from shinkensolutions.lib_modules.configuration_reader import read_int_in_configuration
from shinken.runtime_stats.cpu_stats import cpu_stats_helper

if TYPE_CHECKING:
    from shinken.misc.type_hint import Union, List, Dict, Any, NoReturn
    from synchronizer.dao.items.contactitem import ContactItem

try:
    libc6 = ctypes.CDLL('libc.so.6')
except:
    pass

orig_collect = gc.collect

CONTEXT_PATH = '/var/lib/shinken/context.json'
CURRENT_VERSION = '02.06.00'

if os.path.exists(CONTEXT_PATH):
    context = json.loads(open(CONTEXT_PATH, 'r').read())
    CURRENT_VERSION = context.get('current_version', CURRENT_VERSION)


def monky_collect(*arg):
    try:
        orig_collect(*arg)
        time_start = time.time()
        libc6.malloc_trim(0)
        time_spend = time.time() - time_start
        if time_spend > 0.01:
            print "pacthed gc.collect (malloc_trim time %.3f)" % time_spend
    except:
        pass


gc.collect = monky_collect

FALLBACK_USER = {
    'contact_name': 'shinken-core',
    'is_admin'    : '1',
    '_id'         : '-1',
}


# Public route for modified item can be call directly by the customer
# DO NOT USE THIS API FOR INTERNAL USE
# Param api_item_enabled must be set to enable this routes
class IForItemApi(Interface):
    # will be set in url /item-api/v1/methods ...
    API_VERSION = 1
    
    
    def __init__(self, app):
        super(IForItemApi, self).__init__(app, route_prefix='item-api', public_api=True, default_lock=False)
    
    
    def delete_items_in_staging_and_working_area(self, login, password, uuids, item_type):
        # Open API use by client see SEF-4500
        missing_parameters = set()
        if not login:
            missing_parameters.add('login')
        if not password:
            missing_parameters.add('password')
        if not uuids:
            missing_parameters.add('uuids')
        if not item_type:
            missing_parameters.add('item_type')
        if missing_parameters:
            return self.app.abort(400, 'Missing parameters : %s' % (', '.join(missing_parameters)))
        
        conn = self.app.get_synchronizer_syncui_connection()
        if not self.app.http_daemon_ready:
            self.app.abort(504, 'Syncui is not ready for the moment, need to retry in a few moment')
        
        user = self._check_auth(conn, login, password)
        
        if item_type not in DEF_ITEMS:
            return self.app.abort(400, 'Invalid parameter item_type')
        
        uuids = uuids.split(',')
        cookies = SimpleCookie()
        
        
        def touni(s, enc='utf8', err='strict'):
            return s.decode(enc, err) if isinstance(s, bytes) else unicode(s)
        
        
        def tob(s, enc='utf8'):
            return s.encode(enc) if isinstance(s, unicode) else bytes(s)
        
        
        def cookie_encode(data, key):
            msg = base64.b64encode(json.dumps(data))
            sig = base64.b64encode(hmac.new(tob(key), msg).digest())
            return tob('!') + sig + tob('?') + msg
        
        
        value = touni(cookie_encode(('user', user['_id']), self.app.auth_secret))
        cookies['user'] = value
        
        headers = {"Cookie": "%s" % str(cookies).split(':')[1]}
        params = urllib.urlencode({
            'bypass_work_area': '1'
        })
        result = {
            'deleted_items'    : {
                'list': [],
                'nb'  : 0,
            },
            'not_deleted_items': {
                'list': [],
                'nb'  : 0,
            }
        }
        for item_id in uuids:
            conn.request("POST", "/element/q/%s/delete/%s?%s" % (item_type, item_id, params), headers=headers)
            response = conn.getresponse()
            buf = response.read()
            try:
                call_result = json.loads(buf)
                if call_result['rc'] == 200:
                    result['deleted_items']['nb'] += 1
                    result['deleted_items']['list'].append(item_id)
                else:
                    result['not_deleted_items']['nb'] += 1
                    result['not_deleted_items']['list'].append(item_id)
            
            except:
                pass
        
        if result['deleted_items']['nb'] == len(uuids):
            result['msg'] = 'OK all items were deleted'
        else:
            result['msg'] = 'OK but %s items were not deleted' % result['not_deleted_items']['nb']
        
        conn.close()
        return result
    
    
    def delete_items_in_production(self, login, password, uuids, item_type):
        # Open API use by client see SEF-4500
        missing_parameters = set()
        if not login:
            missing_parameters.add('login')
        if not password:
            missing_parameters.add('password')
        if not uuids:
            missing_parameters.add('uuids')
        if not item_type:
            missing_parameters.add('item_type')
        if missing_parameters:
            return self.app.abort(400, 'Missing parameters : %s' % (', '.join(missing_parameters)))
        
        conn = self.app.get_synchronizer_syncui_connection()
        if not self.app.http_daemon_ready:
            self.app.abort(504, 'Syncui is not ready for the moment, need to retry in a few moment')
        user = self._check_auth(conn, login, password)
        
        if item_type != ITEM_TYPE.HOSTS:
            return self.app.abort(500, 'This item_type is not yet implemented')
        
        params = urllib.urlencode({
            'private_key': self.app.get_private_http_key(),
            'user_id'    : user['_id'],
            'uuids'      : uuids,
            'item_type'  : item_type,
        })
        conn.request("GET", "/internal/delete_in_production?%s" % params)
        response = conn.getresponse()
        buf = response.read()
        result = json.loads(buf)
        conn.close()
        if result['code'] != 200:
            self.app.abort(result['code'], result['data']['msg'])
        return result['data']
    
    
    def _check_auth(self, sync_con, login, password):
        params = urllib.urlencode({
            u'private_key'  : self.app.get_private_http_key(),
            u'shinken_login': login,
            u'password'     : password,
            u'requester'    : u'API_ITEM'
        })
        sync_con.request("GET", "/get_auth?%s" % params)
        response = sync_con.getresponse()
        buf = response.read()
        result = json.loads(buf)
        if result['code'] != 200:
            self.app.abort(result['code'], result['msg'])
        
        if not result['enabled']:
            self.app.abort(403, 'The user %s is not enabled' % login)
        
        if result['admin_level'] != 'shinken_admin':
            self.app.abort(403, 'The user %s is not a shinken admin' % login)
        
        return result['user']


# Param api_trusted_source_enabled must be set to enable this routes
class IForTrustedSource(Interface):
    # will be set in url /trusted-source/v1/methods ...
    API_VERSION = 1
    
    
    def __init__(self, app):
        super(IForTrustedSource, self).__init__(app, route_prefix='trusted-source', public_api=True, default_lock=False)
    
    
    def force_source_import(self, login, password, source_name):
        # Open API use by client see SEF-4354
        missing_parameters = set()
        if not login:
            missing_parameters.add('login')
        if not password:
            missing_parameters.add('password')
        if not source_name:
            missing_parameters.add('source_name')
        if missing_parameters:
            return self.app.abort(400, 'Missing parameters : %s' % (', '.join(missing_parameters)))
        
        if source_name == 'syncui':
            return self.app.abort(500, 'Source [%s] is a internal source, we cannot import elements from this source.' % source_name)
        
        conn = self.app.get_synchronizer_syncui_connection()
        if not self.app.http_daemon_ready:
            self.app.abort(504, 'Syncui is not ready for the moment, need to retry in a few moment')
        self._check_auth(conn, login, password)
        
        source = next((s for s in self.app.sources if s.get_name() == source_name), None)
        if source is None:
            return self.app.abort(400, 'Unknown source')
        
        if source.enabled:
            source.set_force_import()
            return 'OK'
        else:
            return self.app.abort(503, 'Source [%s] is disabled' % source_name)
    
    
    def force_trusted_source_behaviour(self, login, password, filter):
        # Open API use by client see SEF-4354
        missing_parameters = set()
        if not login:
            missing_parameters.add('login')
        if not password:
            missing_parameters.add('password')
        if not filter:
            missing_parameters.add('filter')
        if missing_parameters:
            return self.app.abort(400, 'Missing parameters : %s' % (', '.join(missing_parameters)))
        
        if not self.app.http_daemon_ready:
            self.app.abort(504, 'Syncui is not ready for the moment, need to retry in a few moment')
        
        _filter_split = filter.split(':')
        if len(_filter_split) == 2:
            if _filter_split[0] != 'sources':
                return self.app.abort(500, 'Invalid filter')
            source_name = _filter_split[1]
        else:
            return self.app.abort(500, 'Invalid filter')
        
        source = next((s for s in self.app.sources if s.get_name() == source_name), None)
        if not source:
            return self.app.abort(500, 'Source [%s] in filter was unknown' % source_name)
        if source.get_name() == 'syncui':
            return self.app.abort(500, 'Source [%s] is a internal source, we cannot import elements from this source.' % source_name)
        if not source.enabled:
            return self.app.abort(503, 'Source [%s] is disabled' % source_name)
        if source.state == SOURCE_STATE.CRITICAL:
            return self.app.abort(500, 'Source [%s] has errors, we cannot import elements from this source.' % source_name)
        if source.state == SOURCE_STATE.NOT_CONFIGURED:
            return self.app.abort(500, 'Source [%s] is not configured.' % source_name)
        
        result = self.app.call_internal_put_source_in_staging(login, source_name)
        
        if result['code'] != 200:
            self.app.abort(result['code'], result['msg'].encode('ascii', 'replace'))
        
        del result['code']
        del result['msg']
        
        return result
    
    
    def get_api_source_controller_importing(self):
        return self.app.source_controller.import_or_merge_needed_or_doing()
    
    
    def put_in_production(self, login, password, filter, item_type):
        # Open API use by client see SEF-4501
        missing_parameters = set()
        if not login:
            missing_parameters.add('login')
        if not password:
            missing_parameters.add('password')
        if not filter:
            missing_parameters.add('filter')
        if not item_type:
            missing_parameters.add('item_type')
        if missing_parameters:
            return self.app.abort(400, 'Missing parameters : %s' % (', '.join(missing_parameters)))
        
        conn = self.app.get_synchronizer_syncui_connection()
        if not self.app.http_daemon_ready:
            self.app.abort(504, 'Syncui is not ready for the moment, need to retry in a few moment')
        user = self._check_auth(conn, login, password)
        
        if item_type != ITEM_TYPE.HOSTS:
            return self.app.abort(500, 'This item_type is not yet implemented')
        
        source_name = ''
        if len(filter.split(':')) == 2:
            _filter_split = filter.split(':')
            if _filter_split[0] != 'sources':
                return self.app.abort(500, 'Invalid filter')
            source_name = _filter_split[1]
        else:
            return self.app.abort(500, 'Invalid filter')
        
        source = next((s for s in self.app.sources if s.get_name() == source_name), None)
        if not source:
            return self.app.abort(500, 'Source [%s] in filter was unknown' % source_name)
        if source.get_name() == 'syncui':
            return self.app.abort(500, 'Source [%s] is a internal source, we cannot import elements from this source.' % source_name)
        if source.state == SOURCE_STATE.CRITICAL:
            return self.app.abort(500, 'Source [%s] has errors, we cannot import elements from this source.' % source_name)
        
        params = urllib.urlencode({
            'private_key': self.app.get_private_http_key(),
            'user_id'    : user['_id'],
            'filter'     : filter,
            'item_type'  : item_type,
        })
        conn.request("GET", "/internal/put_in_production?%s" % params)
        response = conn.getresponse()
        buf = response.read()
        result = json.loads(buf)
        conn.close()
        if result['code'] != 200:
            self.app.abort(result['code'], result['msg'])
        return result['msg']
    
    
    def _check_auth(self, sync_con, login, password):
        
        params = urllib.urlencode({
            u'private_key'  : self.app.get_private_http_key(),
            u'shinken_login': login,
            u'password'     : password,
            u'requester'    : u'API_TRUSTED_SOURCE'
        })
        sync_con.request("GET", "/get_auth?%s" % params)
        response = sync_con.getresponse()
        buf = response.read()
        result = json.loads(buf)
        if result['code'] != 200:
            self.app.abort(result['code'], result['msg'])
        
        if not result['enabled']:
            self.app.abort(403, 'The user %s is not enabled' % login)
        
        if result['admin_level'] != 'shinken_admin':
            self.app.abort(403, 'The user %s is not a shinken admin' % login)
        
        return result['user']


# Interface for the other Arbiter
# It connects, and together we decide who's the Master and who's the Slave, etc.
# Here is a also a function to get a new conf from the master
class IForArbiter(Interface):
    WHOLE_CONFIGURATION_MAX_RETRIES = 3
    
    
    def __init__(self, app):
        # type: (Synchronizer) -> None
        super(IForArbiter, self).__init__(app)
        self.app = app
        self._logger_arbiter_get_configuration = LoggerFactory.get_logger(u'ARBITER GET CONFIGURATION')
        self._count_get_whole_configuration_fail = 0
    
    
    def _log_try_get_whole_configuration(self, nb_try, max_try):
        # type: (int, int) -> NoReturn
        
        warn_log_str = u'Arbiter asks me for the configuration but the Synchronizer is not ready for the moment after %d attempts. Max attempts before Arbiter give up: %d'
        error_log_str = u'Arbiter asks me for the configuration but the Synchronizer is not ready, Arbiter will stop to ask us a configuration. You will need to restart manually your Arbiter when Synchronizer is ready. (attempt %d)'
        max_try = max_try or self.app.arbiter_synchronizer_import_max_try
        count = nb_try or self._count_get_whole_configuration_fail
        third = max_try / 3
        
        if count < third:
            self._logger_arbiter_get_configuration.debug(warn_log_str % (count, max_try))
        elif third <= count < third * 2:
            self._logger_arbiter_get_configuration.info(warn_log_str % (count, max_try))
        elif third * 2 <= count < max_try:
            self._logger_arbiter_get_configuration.warning(warn_log_str % (count, max_try))
        else:
            self._logger_arbiter_get_configuration.error(error_log_str % count)
    
    
    def get_whole_configuration(self, states, auth_key, keep_overrides_on_disabled_checks=u'false', nb_try=0, max_try=0):
        if auth_key != self.app.conf.master_key:
            result = {u'rc': 403}
            return result
        
        conn = self.app.get_synchronizer_syncui_connection()
        result = {u'rc': 200, u'data': {}}
        try:
            params = urllib.urlencode({
                u'states'                           : states,
                u'keep_overrides_on_disabled_checks': keep_overrides_on_disabled_checks,
                u'private_key'                      : self.app.get_private_http_key(),
            })
            conn.request(u'GET', u'/internal/get_whole_configuration?%s' % params)
            response = conn.getresponse()
            buf = response.read()
            conn.close()
            buf_load = json.loads(buf)
            result[u'data'] = buf_load[u'configuration']
            self._count_get_whole_configuration_fail = 0
        except:
            self._count_get_whole_configuration_fail += 1
            self._log_try_get_whole_configuration(nb_try, max_try)
            result[u'rc'] = 504
        return result
    
    
    get_whole_configuration.method = u'POST'
    get_whole_configuration.need_lock = False
    
    
    def have_conf(self, magic_hash):
        # I've got a conf and a good one
        if self.app.cur_conf and self.app.cur_conf.magic_hash == magic_hash:
            return True
        else:  # I've no conf or a bad one
            return False
    
    
    # The master Arbiter is sending us a new conf. Ok, we take it
    def put_conf(self, conf):
        super(IForArbiter, self).put_conf(conf)
        self.app.must_run = False
    
    
    def get_config(self):
        return self.app.conf
    
    
    # The master arbiter asks me not to run!
    def do_not_run(self):
        # If I'm the master, then F**K YOU!
        if self.app.is_master:
            print "Some f***ing idiot asks me not to run. I'm a proud master, so I decide to run anyway"
        # Else, I'm just a spare, so I listen to my master
        else:
            print "Someone asks me not to run"
            self.app.last_master_speack = time.time()
            self.app.must_run = False
    
    
    # Here a function called by check_shinken to get daemon status
    def get_satellite_status(self, daemon_type, daemon_name):
        daemon_name_attr = daemon_type + '_name'
        daemons = self.app.get_daemons(daemon_type)
        if daemons:
            for dae in daemons:
                if hasattr(dae, daemon_name_attr) and getattr(dae, daemon_name_attr) == daemon_name:
                    if hasattr(dae, 'alive') and hasattr(dae, 'spare'):
                        return {'alive': dae.alive, 'spare': dae.spare}
        return None
    
    
    # Here a function called by check_shinken to get daemons list
    def get_satellite_list(self, daemon_type):
        satellite_list = []
        daemon_name_attr = daemon_type + "_name"
        daemons = self.app.get_daemons(daemon_type)
        if daemons:
            for dae in daemons:
                if hasattr(dae, daemon_name_attr):
                    satellite_list.append(getattr(dae, daemon_name_attr))
                else:
                    # If one daemon has no name... ouch!
                    return None
            return satellite_list
        return None
    
    
    # Dummy call. We are a master, we managed what we want
    def what_i_managed(self):
        return []
    
    
    def get_all_states(self):
        res = {'arbiter'    : self.app.conf.arbiters,
               'scheduler'  : self.app.conf.schedulers,
               'poller'     : self.app.conf.pollers,
               'reactionner': self.app.conf.reactionners,
               'receiver'   : self.app.conf.receivers,
               'broker'     : self.app.conf.brokers}
        return res
    
    
    def validate_all_elements(self):
        conn = self.app.get_synchronizer_syncui_connection()
        result = {'rc': 200, 'result': {}}
        try:
            params = urllib.urlencode({
                'private_key': self.app.get_private_http_key(),
            })
            conn.request("GET", "/internal/validate_all_elements?%s" % params)
            response = conn.getresponse()
            buf = response.read()
            conn.close()
            result['result'] = json.loads(buf)
        except:
            logger.error('[validate_all_elements] Syncui is not ready for the moment, need to retry in a few moment')
            result['rc'] = 504
        return result
    
    
    validate_all_elements.need_lock = False
    
    
    def apply_all_diffs(self):
        conn = self.app.get_synchronizer_syncui_connection()
        result = {'rc': 200, 'data': {}}
        try:
            params = urllib.urlencode({
                'private_key': self.app.get_private_http_key(),
            })
            conn.request("GET", "/internal/apply_all_diffs?%s" % params)
            response = conn.getresponse()
            buf = response.read()
            conn.close()
            result['data'] = json.loads(buf)
        except:
            logger.error('[apply_all_diffs] Syncui is not ready for the moment, need to retry in a few moment')
            result['rc'] = 504
            result['data'] = 'Syncui is not ready for the moment, need to retry in a few moment'
        return result['data']
    
    
    apply_all_diffs.need_lock = False
    
    
    def commit_all(self):
        conn = self.app.get_synchronizer_syncui_connection()
        try:
            params = urllib.urlencode({
                'private_key': self.app.get_private_http_key(),
            })
            conn.request("GET", "/internal/commit_all?%s" % params)
            conn.close()
        except:
            logger.error('[commit_all] Syncui is not ready for the moment, need to retry in a few moment')
    
    
    commit_all.need_lock = False


class IStats(IStatsInterface):
    # Interface for various stats about synchronizer activity
    
    def _get_protected_fields_stats_from_database(self):
        # type: () -> Dict[unicode, Any]
        mongo_component = component_manager.get_mongo_component_without_auto_reconnect()
        try:
            current_protected_fields = mongo_component.col_synchronizer_info.find_one({u'_id': u'protected_fields_info'})
            return current_protected_fields if current_protected_fields else {}
        except ShinkenMongoException:
            return None
    
    
    def get_raw_stats(self, param=u''):
        # type: (unicode) -> Dict[unicode, Any]
        return super(IStats, self).get_raw_stats(param=param)
    
    
    get_raw_stats.doc = u'get stats of the daemon'
    get_raw_stats.need_lock = False
    
    
    def _get_call_stats(self):
        # type: () -> Dict[unicode, Any]
        log = LoggerFactory.get_logger(u'DAEMON STATS')
        
        call_stats_dump = {}
        conn = self.app.get_synchronizer_syncui_connection()
        try:
            params = urllib.urlencode({
                u'private_key': self.app.get_private_http_key(),
            })
            conn.request(u'GET', u'/internal/get_call_stats?%s' % params)
            response = conn.getresponse()
            buf = response.read()
            conn.close()
            call_stats_dump = json.loads(buf)
        except Exception:
            log.warning(u'Cannot get call stats of the ui configuration.')
        
        return call_stats_dump
    
    
    def _daemon_get_raw_stats(self, param=''):
        # type: (unicode) -> Dict[unicode, Any]
        try:
            mongodb_stats = mongo_by_ssh_mgr.check_connexion_mongodb(self.app.mongodb_conf.uri)
        except ShinkenMongoException:
            mongodb_stats = None
        
        raw_stats = {
            u'http_errors_count': self.app.http_errors_count,
            u'have_conf'        : True,
            u'api_version'      : Interface.RAW_STATS_API_VERSION,
            u'activated'        : True,
            u'spare'            : False,
            u'call_stats_dump'  : self._get_call_stats(),
            u'mongodb_stats'    : mongodb_stats,
            u'encryption_stats' : self._get_protected_fields_stats_from_database()
        }
        return raw_stats
    
    
    # Get the number of count nodes for  the licence (so host that are not disabled)
    def get_host_enabled_count(self):
        mongo_component_without_auto_reconnect = component_manager.get_mongo_component_without_auto_reconnect()
        try:
            data_provider = DataProviderMongo(mongo_component_without_auto_reconnect, self.app.database_cipher)
            where = {'enabled': {'$ne': '0'}}
            return data_provider.count_items(ITEM_TYPE.HOSTS, ITEM_STATE.PRODUCTION, where=where)
        except ShinkenMongoException:
            return None
    
    
    get_host_enabled_count.need_lock = False


# this routes are used by syncui process to communicate this the synchronizer which have the source controller
class IForInternalSource(Interface):
    def __init__(self, app):
        super(IForInternalSource, self).__init__(app, default_lock=False)
    
    
    def get_api_source_controller_importing(self):
        return self.app.source_controller.import_or_merge_needed_or_doing()
    
    
    def get_api_sources(self):
        response = {}
        for source in self.app.sources:
            source_name = source.get_name()
            if not source_name:
                continue
            
            source_info = source.get_api_infos()
            try:
                source_info.update(source.get_ip_ranges(source_name, source))
            except ShinkenMongoException:
                source_info[u'output'] = u'Impossible de récupérer la configuration en raison d\'un problème de connexion a mongo. L\'état sera rétabli lorsque la connexion sera à nouveau disponible'
                source_info[u'state'] = SOURCE_STATE.NOT_CONFIGURED
                source_info[u'nb_elements'] = 0
            
            response[source_name] = source_info
        return response
    
    
    def get_api_source_run_number(self, source_name):
        if not source_name:
            return self.app.abort(400, 'Missing parameter')
        
        source = next((src for src in self.app.sources if src.get_name() == source_name), None)
        if source:
            return source.current_import_nb
        else:
            return self.app.abort(400, 'Source %s not found' % source_name)
    
    
    def set_api_sources_order(self, order):
        if not order:
            return self.app.abort(400, 'Missing parameter')
        
        msg_error = ''
        for (order_index, source_name) in enumerate(order.split(',')):
            
            source = next((s for s in self.app.sources if s.get_name() == source_name), None)
            if source is None:
                continue
            
            order_index += 1
            prev_order = source.order
            if prev_order != order_index:
                source.order = order_index
                if source.enabled:
                    source.merge_asked = True
                if source.internal:
                    ret = self.app.save_source_order_for_internal_source(source)
                else:
                    ret = Synchronizer.save_source_order_into_file(source)
                if ret:
                    msg_error = ret
                self.app.source_controller.merged_asked = True  # when changing a source state, recompute diff and co
        return {'msg_error': msg_error}
    
    
    def set_api_sources(self, sname, enabled):
        if not sname or not enabled:
            return self.app.abort(400, 'Missing parameter')
        
        source = next((s for s in self.app.sources if s.get_name() == sname), None)
        if source is None:
            return self.app.abort(400, 'Unknown source')
        if enabled not in ['0', '1']:
            return self.app.abort(400, 'Unknown enabled parameter')
        
        b = (enabled == '1')
        
        # Analyzer should be managed by the analyzer controller so it can stop jobs and such things
        if source.my_type == 'analyzer':
            did_change = self.app.analyzer_controller.set_enabled(sname, b)
        else:  # other sources (collectors & listeners) can be directly notified
            did_change = source.set_enabled(b)
        
        if did_change:
            source.compute_state()
            self.app.source_controller.merged_asked = True  # when changing a source state, recompute diff and co
            # We should change the 'enabled' boolean into the source cfg file if present
            if source.internal:
                self.app.save_source_enabled_into_mongo(source)
            else:
                Synchronizer.save_source_enabled_into_file(source)
        
        return source.enabled
    
    
    def set_merge_asked(self):
        self.app.source_controller.merged_asked = True  # when changing a source state, recompute diff and co
        return True
    
    
    def set_api_sources_conf(self, source_name, conf_id, enabled):
        if not source_name or not conf_id or not enabled:
            return self.app.abort(400, 'Missing parameter')
        
        if enabled not in ['0', '1']:
            return self.app.abort(400, 'Unknown enabled parameter')
        
        _source = next((s for s in self.app.sources if s.get_name() == source_name), None)
        if _source is None:
            return self.app.abort(400, 'Unknown source')
        
        _source.set_conf_enabled(conf_id, enabled)
        
        return enabled
    
    
    def delete_sources_conf(self, source_name, conf_id):
        if not source_name or not conf_id:
            return self.app.abort(400, 'Missing parameter')
        
        _source = next((s for s in self.app.sources if s.get_name() == source_name), None)
        if _source is None:
            return self.app.abort(400, 'Unknown source')
        
        _source.delete_conf(conf_id)
        
        return
    
    
    def save_sources_conf(self, source_name, conf_id, new_conf):
        if not source_name or not conf_id or not new_conf:
            return self.app.abort(400, 'Missing parameter')
        
        new_conf = json.loads(base64.b64decode(new_conf))
        _source = next((s for s in self.app.sources if s.get_name() == source_name), None)
        if _source is None:
            return self.app.abort(400, 'Unknown source')
        
        return _source.save_conf(conf_id, new_conf)
    
    
    def api_force_source(self, sname):
        if not sname:
            return self.app.abort(400, 'Missing parameter')
        source = next((s for s in self.app.sources if s.get_name() == sname), None)
        if source is None:
            return self.app.abort(400, 'Unknown source')
        if source.enabled:
            source.set_force_import()
            return 'OK'
        else:
            return 'DISABLED'
    
    
    def api_restart_source(self, sname):
        if not sname:
            return self.app.abort(400, 'Missing parameter')
        source = next((s for s in self.app.sources if s.get_name() == sname), None)
        if source is None:
            return self.app.abort(400, 'Unknown source')
        source.restart()
        return source.state
    
    
    def api_clean_source(self, sname):
        if not sname:
            return self.app.abort(400, 'Missing parameter')
        source = next((s for s in self.app.sources if s.get_name() == sname), None)
        if source is None:
            return self.app.abort(400, 'Unknown source')
        
        self.app.source_controller.clean_source(source)
        return source.state
    
    
    def api_remove_source_item(self, source_name, info_items):
        source = next((s for s in self.app.sources if s.get_name() == source_name), None)
        if source is None:
            return self.app.abort(400, 'Unknown source')
        
        if not hasattr(source, 'remove_source_item'):
            return self.app.abort(400, 'source [%s] do not implement remove_source_item method' % source.get_name())
        
        info_items = json.loads(base64.urlsafe_b64decode(info_items))
        for item_type, item_id in info_items:
            source.remove_source_item(item_type, item_id)
        
        if source.enabled and source.state != SOURCE_STATE.CRITICAL:
            # ask the back to force a new import
            source.set_force_import()
            # wait for the end of merge so the new page will have the right informations on counter
            while self.app.source_controller.import_or_merge_needed_or_doing():
                logger.debug('waiting for import and merge to be done')
                time.sleep(.5)
    
    
    def remove_delete_item_by_sync_keys_from_import_cache(self, deleted_item_sync_keys):
        # type: (Str) -> NoReturn
        self.app.source_controller.remove_delete_item_by_sync_keys_from_import_cache(deleted_item_sync_keys)


# this routes are used by syncui process for analyzer
class IForInternalAnalyser(Interface):
    def __init__(self, app):
        # type: (Synchronizer) -> None
        super(IForInternalAnalyser, self).__init__(app, default_lock=False)
        self.app = app  # type: Synchronizer
    
    
    def add_hosts_to_analyzer(self, analyzer_name, hosts_uuids, hosts_states, overload_parameters, launch_batch_uuid):
        # NOTE: overload_parameters is a base64 of the original overload_parameters
        overload_parameters = json.loads(base64.b64decode(overload_parameters))
        
        # we have a part.key=value, split them into {'part': {'key':value}}
        new_overload_parameters = {}
        for k, v in overload_parameters.iteritems():
            _part, _prop = k.split('.', 1)
            if _part not in new_overload_parameters:
                new_overload_parameters[_part] = {}
            new_overload_parameters[_part][_prop] = v
        
        logger.debug('[analyzer] add_hosts_to_analyzer:: %s %s %s %s' % (analyzer_name, hosts_uuids, new_overload_parameters, launch_batch_uuid))
        if not analyzer_name or not hosts_uuids:
            return self.app.abort(400, 'Missing parameter')
        
        try:
            r = self.app.analyzer_controller.add_hosts_to_analyze(analyzer_name, hosts_states, hosts_uuids, new_overload_parameters, launch_batch_uuid)
        except KeyError:  # analyzer is missing
            return self.app.abort(400, 'Unknown analyzer')
        return r
    
    
    def get_analyze_jobs_result(self, launch_batch_uuid):
        r = self.app.analyzer_controller.get_analyze_jobs_result(launch_batch_uuid)
        return r
    
    
    def stop_analyze_batch(self, launch_batch_uuid):
        r = self.app.analyzer_controller.stop_analyze_batch(launch_batch_uuid)
        return r


class Synchronizer(Daemon):
    # used by the authentication
    class _ContactDataManager:
        def __init__(self, synchronizer):
            self.synchronizer = synchronizer
        
        
        def get_contact(self, cname, key='contact_name', item_is_new=False, by_id=False):
            if by_id:
                contact = self.synchronizer.datamanagerV2.find_item_by_id(cname, ITEM_TYPE.CONTACTS, ITEM_STATE.NEW if item_is_new else ITEM_STATE.STAGGING)
            else:
                contact = self.synchronizer.datamanagerV2.find_item_by_name(cname, ITEM_TYPE.CONTACTS, ITEM_STATE.NEW if item_is_new else ITEM_STATE.STAGGING)
            
            return contact
        
        
        def get_contact_case_insensitive(self, cname, key='contact_name', item_is_new=False):
            contact = self.synchronizer.datamanagerV2.find_item_by_name(cname, ITEM_TYPE.CONTACTS, ITEM_STATE.NEW if item_is_new else ITEM_STATE.STAGGING)
            
            return contact
        
        
        def get_contacts(self):
            return self.synchronizer.datamanagerV2.find_items(ITEM_TYPE.CONTACTS, ITEM_STATE.STAGGING)
    
    def __init__(self, config_files, is_daemon, do_replace, verify_only, debug, debug_file, sync_name='', daemon_id=0):
        arbiter_configuration_messages.enable_message_arbiter()
        super(Synchronizer, self).__init__('synchronizer', config_files[0], is_daemon, do_replace, debug, debug_file, daemon_id)
        
        # Here we init the debugger with tee to write log in file and print on stdout/stderr.
        # So on Start, the init script know if the start run correctly or crash. If not, the init script never stop and show the /tmp/bad_start
        # The tee will be disable at the daemonize in the main function
        self.init_debug_logger(tee=True)
        
        # We need a private keys to exchange between this process and the future sub HTTP
        self._private_http_key = binascii.b2a_hex(os.urandom(128))
        
        # Keep only the cfg files, not the ini
        self.__config_files = [c for c in config_files if c.endswith('.cfg')]
        
        # The ini specific configuration file, should be only one file
        self.config_file = None
        for c in config_files:
            if c.endswith('.ini'):
                self.config_file = c
        
        self.__verify_only = verify_only
        
        self.__broks = {}
        self.__is_master = False
        
        self.__sync_name = sync_name
        self.http_process_is_ready = Event()
        
        # Now tab for external_commands
        self.__external_commands = []
        
        self._interface_arbiter = IForArbiter(self)
        self._interface_stats = IStats(self)
        self._interface_internal_source = IForInternalSource(self)
        self._interface_internal_analyser = IForInternalAnalyser(self)
        self._interface_trusted_source = IForTrustedSource(self)
        self._interface_item = IForItemApi(self)
        
        ##
        # Accessed by IForArbiter interface
        # Use to know if we must still be alive or not
        self.must_run = True
        self.master_key = ''  # for CLI interaction
        
        self.http_start_time = int(time.time())
        self.http_daemon_ready = False
        
        self.conf = SynchronizerConfig()
        self.global_conf = Config()
        
        self.workers = {}  # dict of active workers
        
        self.datamgr = None
        self.helper = Helper(self)
        
        self.__http_process = None
        self.SERVICE_IMPORT_FLAG_PATH = u'/var/lib/shinken/'
        
        self.__source_entry = {}  # Number of objects of each type saved from each source
        
        self.source_controller = None
        
        self.datamanagerV2_creation_lock = threading.RLock()
        self.MASS_IMPORT_LOCK = multiprocessing.Lock()
        self.current_version = ToolsBoxString.convert_newlines_to_html_line_break(get_shinken_current_version_and_patch())
        self.current_short_version = get_shinken_current_short_version()
        self.documentation_links = DocumentationLinks(self.current_version)
        
        self._suuid = u''
        self.id_path = u'/etc/shinken/_default/shinken_id.cfg'
        self.user_id_path = u'/etc/shinken-user/configuration/daemons/shinken_id.cfg'
        self.id_skeleton_path = u'/etc/shinken-skeletons/_default/shinken_id.cfg.skeleton'
        
        self.img_tags = []
        self.arbiter_synchronizer_import_max_try = 90
        self.wait_time_http_process_is_ready = 180
        
        self.use_local_log = True
        
        self.sources = []  # type: List[Source]
        
        self.logger_auth = LoggerFactory.get_logger(u'AUTHENTICATION')
        self.logger_user_authentication = None  # type: Optional[LoggerAuthentication]
        self.logger_conf = LoggerFactory.get_logger(u'CONFIGURATION')
        
        self.source_module_instances = []
    
    
    @property
    def datamanagerV2(self):
        with self.datamanagerV2_creation_lock:
            datamanagerV2 = getattr(self, '_datamanagerV2', None)
            if datamanagerV2 is None:
                mongo_component = component_manager.get_mongo_component()
                
                dataprovider = DataProviderMetadata(DataProviderMongo(mongo_component, self.database_cipher), self)
                dataprovider.load_from_data_provider()
                datamanagerV2 = DataManagerV2(dataprovider, synchronizer=self)
                setattr(self, '_datamanagerV2', datamanagerV2)
        return datamanagerV2
    
    
    @property
    def database_cipher(self):
        database_cipher = getattr(self, '_database_cipher', None)
        if database_cipher is None:
            database_cipher = DatabaseCipher(self.conf.protect_fields__activate_encryption, self.conf.protect_fields__substrings_matching_fields, self.conf.protect_fields__encryption_keyfile)
            setattr(self, '_database_cipher', database_cipher)
        return database_cipher
    
    
    @property
    def frontend_cipher(self):
        frontend_cipher = getattr(self, '_frontend_cipher', None)
        if frontend_cipher is None:
            frontend_cipher = FrontendCipher(self.conf.protect_fields__activate_encryption, self.conf.protect_fields__substrings_matching_fields, self.protect_fields__are_viewable_by_admin_si)
            setattr(self, '_frontend_cipher', frontend_cipher)
        return frontend_cipher
    
    
    @property
    def state_controller(self):
        state_controller = getattr(self, '_state_controller', None)
        if state_controller is None:
            state_controller = StateController(self)
            setattr(self, '_state_controller', state_controller)
        return state_controller
    
    
    @property
    def massive_change_controller(self):
        massive_change_controller = getattr(self, '_massive_change_controller', None)
        if massive_change_controller is None:
            massive_change_controller = MassiveChangeController(self)
            setattr(self, '_massive_change_controller', massive_change_controller)
        return massive_change_controller
    
    
    @property
    def arbiter_controller(self):
        arbiter_controller = getattr(self, '_arbiter_controller', None)
        if arbiter_controller is None:
            arbiter_controller = ArbiterController(self)
            setattr(self, '_arbiter_controller', arbiter_controller)
        return arbiter_controller
    
    
    def check_private_http_key(self, query_key):
        return query_key == self._private_http_key
    
    
    def get_private_http_key(self):
        return self._private_http_key
    
    
    ############################################################################
    ## Daemon class overloaded methods BEGINS ##################################
    ############################################################################
    def do_loop_turn(self):
        if self.must_run:
            # Main loop
            self.run()
    
    
    # We are stopping the daemon. So stop sub process, and exit
    def do_stop(self):
        logger.info('Stopping synchronizer daemon')
        
        # And also kill our http_sub process
        if self.__http_process and self.__http_process.is_alive():
            try:
                conn = self.get_synchronizer_syncui_connection()
                conn.request('GET', '/api/terminate')
                conn.getresponse()
            except httplib.BadStatusLine:
                # SyncUI already stopped
                pass
        
        if hasattr(self, 'sources'):
            # Save for load at init.
            self.save_sources_names_list_retention_file()
        # Call the generic daemon part
        super(Synchronizer, self).do_stop()
    
    
    def load_global_config(self):
        cfg_p = '/etc/shinken/shinken.cfg'
        buf = self.global_conf.read_config([cfg_p])
        self.global_conf.read_config_buf(buf)
    
    
    def _add_internal_sources(self, raw_objects):
        discovery = {
            u'source_name'                                                                     : [u'discovery'],
            u'mongodb_uri'                                                                     : [u'mongodb://localhost/?safe=false'],
            u'rules_path'                                                                      : [u''],
            u'nmap_mac_prefixes_path'                                                          : [u''],
            u'enabled'                                                                         : [u'0'],
            u'internal'                                                                        : [u'1'],
            u'import_interval'                                                                 : [u'5'],
            u'module_type'                                                                     : [u'discovery-import'],
            u'mongodb_database'                                                                : [u'synchronizer'],
            u'mongodb_use_ssh_tunnel'                                                          : [u'0'],
            u'mongodb_use_ssh_retry_failure'                                                   : [u'1'],
            u'mongodb_ssh_user'                                                                : [u'shinken'],
            u'mongodb_ssh_keyfile'                                                             : [u'~shinken/.ssh/id_rsa'],
            u'mongodb_retry_timeout'                                                           : [u'60'],
            u'discovery-import__database__retry_connection_X_times_before_considering_an_error': [u'15'],
            u'discovery-import__database__wait_X_seconds_before_reconnect'                     : [u'5'],
            u'data_backend'                                                                    : [u'mongodb'],
            u'_SE_UUID'                                                                        : [u'core-source-cd318c925adc11e5b594080027f08538'],
            u'order'                                                                           : [u'96'],
            u'imported_from'                                                                   : [u'internal_source'],
            u'_SE_UUID_HASH'                                                                   : [u'651d84f003aac9af55d395ec67a6d9a8'],
            u'last_import_detailed'                                                            : [u'1'],
            u'put_in_staging'                                                                  : [u'0'],
            u'put_in_staging_user'                                                             : ['']
        }
        server_analyzer = {
            u'analyzer_name'       : [u'server-analyzer'],
            u'max_parallel_analyse': [u'16'],
            u'enabled'             : [u'1'],
            u'internal'            : [u'1'],
            u'import_interval'     : [u'0'],
            u'module_type'         : [u'server-analyzer'],
            u'order'               : [u'97'],
            u'imported_from'       : [u'internal_source'],
        }
        shinken_listener = {
            u'module_type'    : [u'listener-shinken'],
            u'enabled'        : [u'1'],
            u'import_interval': [u'0'],
            u'listener_name'  : [u'listener-shinken'],
            u'internal'       : [u'1'],
            u'always_enabled' : [u'1'],
            u'order'          : [u'98'],
            u'imported_from'  : [u'internal_source'],
            u'host'           : [u'0.0.0.0'],
            u'port'           : [u'7777'],
        }
        sync_ui = {
            u'module_type'     : [u'syncui-import'],
            u'enabled'         : [u'1'],
            u'import_interval' : [u'1'],
            u'source_name'     : [u'syncui'],
            u'internal'        : [u'1'],
            u'always_enabled'  : [u'1'],
            u'order'           : [u'10000000'],
            u'imported_from'   : [u'internal_source'],
            u'mongodb_uri'     : [u'mongodb://localhost/?safe=false'],
            u'mongodb_database': [u'synchronizer'],
            u'_SE_UUID'        : [u'core-source-dcc311805adc11e59956080027f08538'],
            u'_SE_UUID_HASH'   : [u'61572fa90ae477393e39bbe8b96b2d0d']
        }
        
        if u'source' not in raw_objects:
            raw_objects[u'source'] = []
        # find if another syncui is already present in the source (from cfg) and remove it
        existing_syncui = next((s for s in raw_objects[u'source'] if s.get(u'source_name', u'') == sync_ui[u'source_name']), None)
        if existing_syncui:
            if existing_syncui[u'order'] != [u'0']:
                sync_ui[u'order'] = existing_syncui[u'order']
            sync_ui[u'imported_from'] = existing_syncui[u'imported_from']
            raw_objects[u'source'].remove(existing_syncui)
        
        existing_syncui_import = next((s for s in raw_objects[u'source'] if s.get(u'module_type', u'') == sync_ui.get(u'module_type', u'')), None)
        if existing_syncui_import:
            logger.error(u'You can have only one source of type "%s" and it must be named "%s"' % (sync_ui[u'module_type'][0], sync_ui[u'source_name'][0]))
            sys.exit(2)
        
        # find if another discovery is already present in the source (from cfg) and remove it
        existing_discovery = next((s for s in raw_objects[u'source'] if s.get(u'source_name', u'') == discovery.get(u'source_name', u'')), None)
        if existing_discovery:
            # get the original discovery order and assign to the new discovery
            discovery[u'order'] = existing_discovery[u'order']
            properties_override_from_cfg = [
                u'enabled',
                u'rules_path',
                u'mongodb_uri',
                u'mongodb_database',
                u'mongodb_use_ssh_tunnel',
                u'mongodb_use_ssh_retry_failure',
                u'mongodb_ssh_user',
                u'mongodb_ssh_keyfile',
                u'mongodb_retry_timeout',
                u'discovery-import__database__retry_connection_X_times_before_considering_an_error',
                u'discovery-import__database__wait_X_seconds_before_reconnect',
                u'nmap_mac_prefixes_path',
                u'import_interval',
                u'put_in_staging',
                u'put_in_staging_user'
            ]
            for _property in properties_override_from_cfg:
                discovery[_property] = existing_discovery.get(_property, discovery[_property])
            discovery[u'imported_from'] = existing_discovery[u'imported_from']
            raw_objects[u'source'].remove(existing_discovery)
        
        raw_objects[u'source'].append(sync_ui)
        raw_objects[u'source'].append(discovery)
        
        if u'listener' not in raw_objects:
            raw_objects[u'listener'] = []
        raw_objects[u'listener'].append(shinken_listener)
        
        if u'analyzer' not in raw_objects:
            raw_objects[u'analyzer'] = []
        raw_objects[u'analyzer'].append(server_analyzer)
        
        for sync in raw_objects[u'synchronizer']:
            source_to_add = set((u'listener-shinken', u'syncui', u'discovery', u'server-analyzer'))
            source_name_list = set(sync[u'sources'][0].split(u',')) if sync.get(u'sources', False) else set()
            sync[u'sources'] = [u','.join((source_name_list | source_to_add))]
        return raw_objects
    
    
    def load_id(self):
        if not os.path.isfile(self.id_path):
            if not os.path.isfile(self.id_skeleton_path):
                sys.exit('%s doesn\'t exist, sorry I bail out.' % self.id_skeleton_path)
            suuid = uuid.uuid4().hex
            shutil.copyfile(self.id_skeleton_path, self.id_path)
            # If user file already exists, don't eras it ! Maybye the user erase the /etc/shinken/_default directory ....
            if not os.path.exists(self.user_id_path):
                shutil.copyfile(self.id_path, self.user_id_path)
            
            with open(self.id_path, 'r+') as f:
                filedata = f.read()
                filedata = filedata.replace('_ID_', suuid)
                filedata = filedata.replace('purpose', 'purpose | DO NOT EDIT')
                filedata += '#----------------------------------------------------------------------------------\n' \
                            '#=========================== User parameters customizations =======================\n' \
                            '                                                       # If you need to customise a value in this file, set it in this file\n'
                filedata += 'cfg_file=%s' % self.user_id_path
            
            with open(self.id_path, 'w') as f:
                f.write(filedata)
        
        else:
            data = {}
            with open(self.id_path, 'r') as f:
                for line in f.readlines():
                    if not line.strip().startswith('#') and len(line.strip().split('=')) == 2:
                        split_line = line.strip().split('=')
                        data[split_line[0]] = split_line[1]
            
            cfg_file = data.get('cfg_file', "")
            suuid = data.get('shinken_id', "")
            if not suuid:
                sys.exit('The file %s has been changed, critical error' % self.id_path)
            if cfg_file:
                if os.path.exists(cfg_file):
                    with open(cfg_file, 'r') as f:
                        for line in f.readlines():
                            if not line.strip().startswith('#') and len(line.strip().split('=')) == 2:
                                split_line = line.strip().split('=')
                                data[split_line[0]] = split_line[1]
                    
                    temp_suuid = data.get('shinken_id', suuid)
                    if temp_suuid != '_ID_':
                        if temp_suuid.isalnum():
                            suuid = temp_suuid
                        else:
                            logger.error('Your SUUID in %s has forbidden characters, default was used instead.' % cfg_file)
                else:
                    # the cfg_file for the user is specified in shinken_id.cfg but the file doesn't exist (Ex. : restore from 02.06.XX)
                    # We create the file from skeleton
                    shutil.copyfile(self.id_skeleton_path, self.user_id_path)
        
        self._suuid = suuid
        if not suuid.isalnum():
            logger.error('Your SUUID in %s has forbidden characters, fix it.' % cfg_file)
            sys.exit('Bad config, sorry I bail out')
    
    
    def __create_sync_history_lifespan_text(self):
        day, hour, minute = 0, 0, 0
        minutes = self.sync_history_lifespan
        hours, minutes = divmod(minutes, 60)
        days, hours = divmod(hours, 24)
        if minutes == 1:
            minute, minutes = 1, 0
        if hours == 1:
            hour, hours = 1, 0
        if days == 1:
            day, days = 1, 0
        periods = [(self._('timeperiods.days'), days), (self._('timeperiods.day'), day), (self._('timeperiods.hours'), hours), (self._('timeperiods.hour'), hour), (self._('timeperiods.minutes'), minutes), (self._('timeperiods.minute'), minute)]
        time_string = ', '.join('{0} {1}'.format(value, name) for name, value in periods if value)
        self.sync_history_lifespan_text = self._('source.no_previous_executions') % time_string
    
    
    def load_config_file(self):
        self.load_id()
        
        buf = self.conf.read_config(self.__config_files)
        raw_objects = self.conf.read_config_buf(buf)
        
        # add internal sources into the raw objects now
        raw_objects = self._add_internal_sources(raw_objects)
        
        self.conf.split_sources_and_modules(raw_objects)
        self.load_global_config()
        
        # Load default properties read from the configuration
        self.global_conf.fill_default()
        self.lang = getattr(self.conf, 'lang', 'en')
        self.langs_path = os.path.join(os.path.abspath(os.path.dirname(__file__)), 'htdocs', 'js', 'traductions')
        self.langs = {'en': None, 'fr': None}
        
        try:
            self.global_conf.max_plugins_output_length = int(self.global_conf.max_plugins_output_length)
        except (ValueError, TypeError):
            self.global_conf.max_plugins_output_length = 8192
        
        self.load_default_properties_values()
        self.conf.load_properties_display_configuration()
        
        # Setup logging
        self.local_log = getattr(self.conf, 'local_log', '/var/log/shinken/synchronizerd.log')
        self.log_level = getattr(self.conf, 'log_level', INFO)
        logger.setLevel(self.log_level)
        self.register_local_log()
        self.human_timestamp_log = getattr(self.conf, 'human_timestamp_log', 1)
        logger.set_human_format(on=self.human_timestamp_log)
        
        self.load_default_properties_values()
        
        # First we need to get arbiters and modules
        # so we can ask them for objects
        self.conf.create_objects_for_type(raw_objects, 'synchronizer')
        self.conf.create_objects_for_type(raw_objects, 'module')
        self.conf.create_objects_for_type(raw_objects, 'source')
        self.conf.create_objects_for_type(raw_objects, 'listener')
        self.conf.create_objects_for_type(raw_objects, 'analyzer')
        self.conf.create_objects_for_type(raw_objects, 'tagger')
        self.conf.create_objects_for_type(raw_objects, 'arbiter')
        self.conf.early_synchronizer_linking()
        
        self.me = None
        # Search which Synchronizer I am
        for sync in self.conf.synchronizers.enabled():
            if sync.is_me(self.__sync_name):
                self.me = sync
                self.__is_master = not self.me.spare
                if self.__is_master:
                    logger.info('I am a master Synchronizer: %s' % sync.get_name())
                else:
                    logger.info('I am a spare Synchronizer: %s' % sync.get_name())
                
                # Set myself as alive ;)
                self.me.alive = True
            else:  # not me
                pass
        
        if not self.me:
            sys.exit("Error: I cannot find my own Synchronizer object, I bail out. "
                     "To solve this, please make sure that the 'enabled' parameter is set to 1, "
                     "or change the 'host_name' parameter in "
                     "the object %s in the Synchronizer configuration file "
                     "with the value '%s'. "
                     "Thanks." % (self.__sync_name if self.__sync_name else "Synchronizer", socket.gethostname()))
        logger.set_name(self.me.get_name())
        self._check_config()
        
        for s in self.me.sources:
            for m in s.modules:
                if not m in self.me.modules:
                    self.me.modules.append(m)
        for s in self.me.taggers:
            for m in s.modules:
                if m not in self.me.modules:
                    self.me.modules.append(m)
        
        # We will need a real display name early in this daemon because we will launch external modules very soon
        self.compute_basic_display_name()
        # So I can update our name in the ini file
        self.save_daemon_name_into_configuration_file(self.me.get_name())
        
        logger.info("My own modules: " + ','.join([m.get_name() for m in self.me.modules]))
        
        self.modules_dir = getattr(self.conf, 'modules_dir', '')
        self.master_key = getattr(self.conf, 'master_key', '')  # For CLI queries
        
        component_manager._build(self.conf, self.__sync_name, self.http_start_time)
        component_manager.init()
        
        # Ok it's time to load the module manager now!
        self.load_modules_manager()
        # we request the instances without them being *started*
        # (for those that are concerned ("external" modules):
        # we will *start* these instances after we have been daemonized (if requested)
        self.modules_manager.set_modules(self.me.modules)
        self.do_load_modules()
        
        # Give to our modules a link to us
        for inst in self.modules_manager.get_all_alive_instances():
            f = getattr(inst, 'load', None)
            if f is not None and callable(f):
                try:
                    f(self)
                except Exception:
                    logger.error('Fail to load module %s' % inst.get_name())
                    logger.print_stack()
        
        # Call modules that manage this read configuration pass
        self.hook_point('read_configuration')
        
        # Resume standard operations
        self.conf.create_objects(raw_objects)
        
        # Linkify the modules instances
        self.conf.sources.linkify_modules_instances(self.modules_manager)
        self.conf.taggers.linkify_modules_instances(self.modules_manager)
        
        # Maybe conf is already invalid
        if not self.conf.conf_is_correct:
            self.conf.show_errors()
            sys.exit("***> One or more problems was encountered while processing the config files...")
        
        # Change Nagios2 names to Nagios3 ones
        self.conf.old_properties_names_to_new()
        
        # Manage all post-conf modules
        self.hook_point('early_configuration')
        
        # Create Template links
        self.conf.linkify_templates()
        
        # Link discovery runs with our commands
        self.conf.discoveryruns.linkify(self.conf.commands)
        
        # Create Name reversed list for searching list
        self.conf.create_reversed_list()
        
        # Cleaning Twins objects
        self.conf.remove_twins()
        
        # late conf for arbiters
        self.conf.arbiters.fill_default()
        self.conf.arbiters.pythonize()
        
        # Implicit inheritance for services
        # self.conf.apply_implicit_inheritance()
        
        self.conf.load_packs()
        
        # Fill default values
        super(SynchronizerConfig, self.conf).fill_default_common()
        
        # Remove templates from config
        # SAVE TEMPLATES
        self.packs = self.conf.packs
        self.sources = self.me.sources
        self.conf.sources = self.me.sources
        self.taggers = self.me.taggers
        
        # We removed templates, and so we must recompute the
        # search lists
        self.conf.create_reversed_list()
        
        # Pythonize values
        # super(SynchronizerConfig, self.conf).pythonize()
        self.conf.pythonize_common()
        
        # Manage all post-conf modules
        self.hook_point('late_configuration')
        
        # Fill realms for pollers and such things
        self.conf.realms.explode()
        # Link realms with other realms
        self.conf.realms.linkify()
        # and link poller/reactionner with realms
        self.conf.reactionners.linkify(self.conf.realms, self.conf.modules)
        self.conf.pollers.linkify(self.conf.realms, self.conf.modules)
        
        self.conf.realms.prepare_for_satellites_conf()
        
        self.conf.is_correct()
        
        # The conf can be incorrect here if the cut into parts see errors like
        # a realm with hosts and not schedulers for it
        if not self.conf.conf_is_correct:
            self.conf.show_errors()
            sys.exit("Configuration is incorrect, sorry, I bail out")
        else:
            logger.info('Things look okay - No serious problems were detected during the pre-flight check')
        
        # Now clean objects of temporary/unnecessary attributes for live work:
        self.conf.clean()
        
        # Exit if we are just here for config checking
        if self.__verify_only:
            sys.exit(0)
        
        # Ok, here we must check if we go on or not.
        # TODO: check OK or not
        self.use_local_log = self.conf.use_local_log
        self.pidfile = os.path.abspath(self.conf.lock_file)
        self.idontcareaboutsecurity = self.conf.idontcareaboutsecurity
        self.user = self.conf.shinken_user
        self.group = self.conf.shinken_group
        self.daemon_enabled = self.conf.daemon_enabled
        
        self.daemon_thread_pool_size = self.conf.daemon_thread_pool_size
        self.http_backend = self.conf.http_backend
        self.share_dir = self.conf.share_dir
        
        # Mongodb part
        self.data_backend = self.conf.data_backend
        self.mongodb_conf = component_manager.get_configuration_component().get_mongo_configuration()
        self.mongodb_uri = self.conf.mongodb_uri
        self.mongodb_database_name = self.conf.mongodb_database
        self.mongodb_retry_timeout = int(self.conf.mongodb_retry_timeout)
        self.mongodb_use_ssh_tunnel = getattr(self.conf, 'mongodb_use_ssh_tunnel', False) in ['1', True]
        self.mongodb_ssh_user = getattr(self.conf, 'mongodb_ssh_user', 'shinken')  # NOTE: cannot user os.getenv('USER') as we are root here
        self.mongodb_ssh_keyfile = getattr(self.conf, 'mongodb_ssh_keyfile', '~shinken/.ssh/id_rsa')
        self.mongodb_ssh_tunnel_timeout = int(getattr(self.conf, 'mongodb_ssh_tunnel_timeout', '2'))
        
        # Change history retention
        self.elements_nb_stored_days_of_history = int(getattr(self.conf, u'synchronizer_staging_elements_nb_stored_days_of_history', 365))
        
        # History to keep for import runs
        self.sync_history_lifespan = int(self.conf.sync_history_lifespan)
        self.__create_sync_history_lifespan_text()
        
        # If the user set a workdir, let use it. If not, use the pidfile directory
        if self.conf.workdir == '':
            self.workdir = os.path.abspath(os.path.dirname(self.pidfile))
        else:
            self.workdir = self.conf.workdir
        
        ##  We need to set self.host & self.port to be used by do_daemon_init_and_start
        # By default: listen to all unless set in the configuration
        self.host = '0.0.0.0'
        if hasattr(self.conf, 'bind_addr'):
            self.host = self.conf.bind_addr
        self.port = self.me.port
        self.use_ssl = getattr(self.conf, 'use_ssl', False)
        logger.debug('synchronier deamon uri [%s://%s:%s]' % ('https' if self.use_ssl else 'http', self.host, self.port))
        self.server_cert = getattr(self.conf, 'server_cert', '')
        self.ca_cert = getattr(self.conf, 'ca_cert', '')
        self.server_key = getattr(self.conf, 'server_key', '')
        self.hard_ssl_name_check = getattr(self.conf, 'hard_ssl_name_check', False)
        
        # Manage https part for the UIs
        self.http_use_ssl = getattr(self.conf, 'http_use_ssl', '0') == '1'
        self.http_ssl_cert = getattr(self.conf, 'http_ssl_cert', '')
        self.http_ssl_key = getattr(self.conf, 'http_ssl_key', '')
        
        self.http_remote_user_enable = getattr(self.conf, 'http_remote_user_enable', '0') == '1'
        self.http_remote_user_variable = getattr(self.conf, 'http_remote_user_variable', 'X-Remote-User')
        self.http_remote_user_case_sensitive = getattr(self.conf, 'http_remote_user_case_sensitive', '1') == '1'
        
        self.api_trusted_source_enabled = getattr(self.conf, 'api_trusted_source_enabled', '0') == '1'
        self.api_item_enabled = getattr(self.conf, 'api_item_enabled', '0') == '1'
        
        self.max_file_descriptor_limit = 8192  # this daemon do not need a lof of file descriptors
        
        cfg_has_protect_fields_duplicate_params = False
        for file_name, params in self.conf.configuration_duplicate_params.iteritems():
            for param in params:
                if "protect_fields__" in param:
                    cfg_has_protect_fields_duplicate_params = True
                    logger.error("[protected fields configuration] Parameter [%s] is defined multiple times in file %s." % (param, file_name))
        
        if cfg_has_protect_fields_duplicate_params:
            logger.error("[PROTECTED FIELDS] The synchronizer will not start to prevent possible data corruption.")
            logger.error("[PROTECTED FIELDS] Please fix the configuration file before restarting the synchronizer.")
            sys.exit(1)
        
        self.protect_fields__activate_database_encryption = BoolProp().pythonize(getattr(self.conf, 'protect_fields__activate_encryption', '0'))
        self.protect_fields__are_viewable_by_admin_si = BoolProp().pythonize(getattr(self.conf, 'protect_fields__are_viewable_by_admin_si', '0'))
        self.protect_fields__activate_interface_encryption = self.protect_fields__activate_database_encryption
        
        if self.protect_fields__activate_database_encryption:
            self.protect_fields__encryption_keyfile = getattr(self.conf, 'protect_fields__encryption_keyfile', None)
            
            try:
                keyfile_stats = os.stat(self.protect_fields__encryption_keyfile)
                if keyfile_stats.st_mode & 0777 != 0600:
                    logger.warn('[warning] keyfile %s permissions are too permissive' % self.protect_fields__encryption_keyfile)
            except OSError as e:
                logger.error("[PROTECTED FIELDS]  Make sure the file exists and that the user 'shinken' is allowed to read it")
                logger.error("[PROTECTED FIELDS]  The synchronizer will not start because it will not be able to process encrypted data.")
                logger.error("[PROTECTED FIELDS]  Cannot read the protected fields secret file %s : %s " % (self.protect_fields__encryption_keyfile, str(e)))
                sys.exit(1)
        
        self.protect_fields__substrings_matching_fields = getattr(self.conf, 'protect_fields__substrings_matching_fields')
        if isinstance(self.protect_fields__substrings_matching_fields, basestring):
            self.protect_fields__substrings_matching_fields = split_and_strip_list(self.protect_fields__substrings_matching_fields)
        
        self.wait_time_http_process_is_ready = read_int_in_configuration(self.conf, u'wait_time_http_process_is_ready', u'180')
        
        # get a list of tag images available for display
        self.img_tags = []
        image_template_folder = os.path.join(self.share_dir, u'images', u'sets')
        # Look at the share/images/sets/*/tag.png and save the *
        for folder in os.listdir(image_template_folder):
            folder_full_path = os.path.join(image_template_folder, folder)
            if os.path.isdir(folder_full_path) and os.path.exists(os.path.join(folder_full_path, u'tag.png')):
                self.img_tags.append(folder)
        
        for source in self.sources:
            source.app = self
        logger.info(u'Configuration Loaded')
        
        synchronizer_import_module = next((i for i in self.conf.modules if i.module_type == u'synchronizer-import'), None)
        if synchronizer_import_module:
            self.arbiter_synchronizer_import_max_try = read_int_in_configuration(synchronizer_import_module, u'max_try', u'90')
        
        self.logger_user_authentication = LoggerAuthentication(u'synchronizer', self.conf, LoggerFactory.get_logger(u'AUTHENTICATION LOG'))
        print u''
    
    
    def _check_config(self):
        self.conf.check_config(self.me.modules, u'module')
        self.conf.check_config(self.me.taggers, u'tagger')
        
        if not self.me.configuration_errors:
            self.me.configuration_errors = []
        
        sources_discovery_import = [getattr(s, u'module_type', u'') for s in self.me.sources if getattr(s, u'module_type', u'') == u'discovery-import']
        if len(sources_discovery_import) > 1:
            self.me.configuration_errors.insert(0, u'You can have only one source of type "%s" and it must be named "%s"' % (u'discovery-import', u'discovery'))
        
        for s in self.conf.sources:
            source_name = s.get_name()
            if not hasattr(s, u'module_type'):
                file_name = s.imported_from.split(u':')[0]
                self.me.configuration_errors.insert(0, u'The source imported from file "%s" does not have a "module_type"' % file_name)
            if not source_name:
                self.me.configuration_errors.insert(0, u'missing source name in source imported from file %s' % s.imported_from)
            if len(source_name) > SRC_NAME_MAX_LENGTH:
                self.me.configuration_errors.insert(0, u'source name [%s] is too long : more than %d characters' % (s.source_name, SRC_NAME_MAX_LENGTH))
        
        # Remove taggers not loaded by the daemon from the configuration
        loaded_tagger_names = [tagger.get_name() for tagger in self.me.taggers]
        to_del = []
        for tagger in self.conf.taggers:
            tagger_name = tagger.get_name()
            if not tagger_name:
                self.me.configuration_errors.insert(0, u'ERROR: missing tagger name in tagger imported from file %s' % s.imported_from)
            if tagger_name not in loaded_tagger_names:
                to_del.append(tagger.id)
        
        for tagger_id in to_del:
            del self.conf.taggers[tagger_id]
        
        for err in self.me.configuration_errors:
            if err:
                self.logger_conf.error(err)
        
        if self.me.configuration_errors:
            sys.exit(2)
    
    
    ############################################################################
    ## Daemon class overloaded methods END #####################################
    ############################################################################
    
    def _load_web_configuration(self):
        self.plugins = []
        
        self.http_port = int(getattr(self.conf, 'http_port', '7766'))
        self.http_host = getattr(self.conf, 'http_host', '0.0.0.0')
        self.auth_secret = getattr(self.conf, 'auth_secret', 'CHANGE_ME').encode('utf8', 'replace')
        self.http_backend = getattr(self.conf, 'http_backend', 'auto')
        self.remote_user_enable = getattr(self.conf, 'remote_user_enable', '0')
        self.remote_user_variable = getattr(self.conf, 'remote_user_variable', 'X-Remote-User')
        
        # Load the photo dir and make it a absolute path
        self.photo_dir = 'photos'  # getattr(modconf, 'photo_dir', 'photos')
        self.photo_dir = os.path.abspath(self.photo_dir)
    
    
    # Main loop function
    def main(self):
        try:
            # Log will be broks
            for line in self.get_header():
                logger.info(line)
            self.daily_log_version()
            
            self.load_config_file()
            self._load_data_backend()
            
            # now we have the conf and mongo setup, we can no setup the internal source
            self._setup_internal_sources()
            
            migrate_database(self)
            
            self._setup_database_format()
            
            add_core_items(self)
            
            # Look if we are enabled or not. If ok, start the daemon mode
            self.look_for_early_exit()
            
            self._load_web_configuration()
            
            self.do_daemon_init_and_start(close_logs_in_stdout_and_stderr=True)
            
            # By default the http lock is on, the daemon must release it explicitly
            if self.http_daemon:
                self.http_daemon.lock.release()
            
            # reload mongodb after fork because mongo cannot handle fork + socket
            self._load_data_backend()
            
            self.http_daemon.register(self._interface_arbiter)
            self.http_daemon.register(self._interface_stats)
            self.http_daemon.register(self._interface_internal_source)
            self.http_daemon.register(self._interface_internal_analyser)
            if self.api_trusted_source_enabled:
                self.http_daemon.register(self._interface_trusted_source)
            if self.api_item_enabled:
                self.http_daemon.register(self._interface_item)
            
            # For multiprocess things, we should not have
            # sockettimeouts. will be set explicitly in Pyro calls
            socket.setdefaulttimeout(None)
            
            # ok we are now fully daemon (if requested) now we can start our "external" modules (if any):
            self.modules_manager.start_external_instances()
            
            self.have_configuration = True
            ## And go for the main loop
            self.do_mainloop()
        except ShinkenMongoException:
            logger.error(u'The daemon must have access to mongoDb to start but it\'s unreachable, bail out')
            try:
                self.do_stop()
            except:
                pass
            
            sys.exit(2)
        
        except SystemExit as exp:
            # With a 2.4 interpreter the sys.exit() in load_config_file
            # ends up here and must be handled.
            sys.exit(exp.code)
        except Exception:
            logger.critical("The daemon did have an unrecoverable error. It must exit.")
            logger.critical("You can log a bug to your Shinken integrator with the error message:")
            logger.print_stack()
            try:
                self.do_stop()
            except:
                pass
            raise
    
    
    def _setup_new_conf(self):
        """ Setup a new conf received from a Master arbiter. """
        conf = self.new_conf
        self.new_conf = None
        self.cur_conf = conf
        self.conf = conf
    
    
    # Give try to load the mongodb and turn until we are killed or the mongo goes up
    def _load_data_backend(self):
        # type: () -> NoReturn
        start = time.time()
        while not self.interrupted:
            try:
                self._load_data_backend_try(start)
                # connect was ok? so instantiate API for modules and bail out :)
                self.datamgr = self._ContactDataManager(self)
                
                # when we make a mongo connection after fork we force a rebuild of the datamanagerV2
                setattr(self, '_datamanagerV2', None)
                return
            except Exception as e:
                now = time.time()
                if now > start + self.mongodb_retry_timeout:  # more than 1min without mongo? not good!
                    logger.error(
                        'Cannot contact the mongodb server for more than %ss, bailing out' % self.mongodb_retry_timeout)
                    sys.exit(2)
                
                # hummer the connection if need
                time.sleep(0.1)
    
    
    def _load_data_backend_try(self, start_time):
        # type:(float) -> NoReturn
        mongo_component = component_manager.get_mongo_component()
        mongo_component_without_auto_reconnect = component_manager.get_mongo_component_without_auto_reconnect()
        
        if mongo_component.connected:
            mongo_component.recreate_connection()  # already init? So we just forked, let's recreate the connection then
        else:
            mongo_component.init()
        
        if mongo_component_without_auto_reconnect.connected:
            mongo_component_without_auto_reconnect.recreate_connection()  # already init? So we just forked, let's recreate the connection then
        else:
            mongo_component_without_auto_reconnect.init()
        
        try:
            logger.info('Launching mongodb database tuning')
            # Auto-tune all collections, to not grow endlessly
            trash_manager.set_mongo(mongo_component)
            
            can_use_power_of_2_sizes = True
            
            # storageEngne information is not available on mongos,
            # fallback to version number as usePowerOf2Sizes is no more available on version >= 4.2.0
            stats = mongo_component.get_server_status()
            if u'storageEngine' in stats:
                engine = stats[u'storageEngine'][u'name']
                can_use_power_of_2_sizes = (engine == u'mmapv1')
            else:
                if u'version' in stats:
                    mongo_version = stats[u'version']
                else:
                    info = mongo_component.get_server_info()
                    mongo_version = info.get(u'version', u'0.0.0')
                
                mongo_version = mongo_version.split(u'.')
                map(int, mongo_version)
                if mongo_version[0] > 4 or (mongo_version[0] == 4 and mongo_version[1] >= 2):
                    can_use_power_of_2_sizes = False
            
            if can_use_power_of_2_sizes:
                collection_names = mongo_component.list_name_collections()
                for name in collection_names:
                    mongo_component.edit_coll_mod(name, usePowerOf2Sizes=True)
            
            for item_type, def_item in DEF_ITEMS.iteritems():
                for state in [ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA, ITEM_STATE.PRODUCTION]:
                    data_provider = DataProviderMongo(mongo_component, self.database_cipher)
                    col = data_provider.get_collection(item_type, state)
                    col.ensure_index([(def_item['key_name'], ASCENDING)], name=def_item['key_name'])
            
            # Manage history collection
            collection_name = u'history'
            history_index_name = u'history_idx'
            index_field = [(u'date', ASCENDING)]
            history_collection = mongo_component.get_collection(collection_name)
            expire_after_seconds = self.elements_nb_stored_days_of_history * 86400
            try:
                idx_info = history_collection.index_information()
            except OperationFailure as e:
                # No database -> no index
                if e.code == 26:
                    idx_info = {}
                else:
                    raise
            if history_index_name in idx_info and idx_info.get(history_index_name, {}).get(u'expireAfterSeconds', -42) != expire_after_seconds:
                logger.info(
                    u'The expire_after_seconds of the index [%s] of collection [%s] need to be update to the new value %ss (%s).' % (history_index_name, collection_name, expire_after_seconds, print_human_readable_period(expire_after_seconds)))
                return_update_index = mongo_component.edit_coll_mod(collection_name, index={u'keyPattern': {index_field[0][0]: index_field[0][1]}, u'expireAfterSeconds': expire_after_seconds})
                logger.debug(u'Result of update index:[%s]' % return_update_index)
            
            history_collection.ensure_index(index_field, name=history_index_name, expire_after_seconds=expire_after_seconds)
        except ConnectionFailure as e:
            now = time.time() - start_time
            if now < 0.1:
                logger.error(u'%s (Retry :  %s / %s seconds)' % (e, int(time.time() - start_time), self.mongodb_conf.ssh_tunnel_timeout))
            if now % 5 <= 0.11 and now > 0.11:
                logger.warning(u'%s (Retry :  %s / %s seconds)' % (e, int(time.time() - start_time), self.mongodb_conf.ssh_tunnel_timeout))
            raise
        except Exception, e:
            logger.error(u'Fail to connect to mongo with error : %s' % getattr(e, 'message', e))
            logger.print_stack()
            raise
        logger.debug(u'Connection OK for %s' % self.mongodb_conf.name_database)
    
    
    # Main function
    def run(self):
        self.source_controller = SourceController(self)
        
        # Prepare a sub-process that will manage the HTTP part
        # so we don't mess our bottle part
        self._spawn_http_sub_process()
        
        # Start a thread dedicated to look at valid http result
        # TODO: can be removed if we find or update bottle
        self._launch_http_daemon_health()
        
        self.source_controller.launch_merger_thread()
        
        self.analyzer_controller = AnalyzerController(self)
        
        # # TODO Find a way to manage the datamagerv2 in synchronizer process (not the syncui)
        self._datamanagerV2 = "Please do not use datamanger inside the daemon process"
        
        # ensure the configuration db exists before starting the sources
        self.source_controller.check_or_create_sources_db_config()
        self.source_controller.start_sources()
        # And analyzers too
        self.analyzer_controller.start_analyzers()
        
        while not self.interrupted:
            self.daily_log_version()
            start_snap = cpu_stats_helper.get_thread_cpu_snapshot()
            # Look up at my http son, if dead, I stop too
            if self.__http_process is None or not self.__http_process.is_alive():
                logger.error('ERROR: The http_process %s is down! Please look at your logs for more information. I bail out' % self.__http_process)
                self._spawn_http_sub_process()
            elif self.http_daemon_ready:
                t0 = time.time()
                for s in self.sources:
                    # Do not schedule syncui, we will be only after each merge
                    if s.get_name() == 'syncui':
                        continue
                    if s.need_import():
                        logger.debug('NEED IMPORT: True for source %s' % s.get_name())
                        self.source_controller.launch_source_import(s)
                if time.time() - t0 >= 1:
                    logger.warning("Loop time [%s]s" % (time.time() - t0))
            
            # Look up at my module if any was dead i restart them
            self.check_and_del_zombie_modules()
            
            self.source_controller.update_last_synchronization()
            
            # logger.debug('[PERFS] %s' % start_snap.get_diff())
            self.sleep(1)
    
    
    def save_sources_names_list_retention_file(self):
        if self.is_daemonized:
            p_sources_dat = os.path.join(self.SERVICE_IMPORT_FLAG_PATH, 'sources_names.dat')
            with open(p_sources_dat, 'w') as f:
                f.write(json.dumps([s.source_name for s in self.sources if s.enabled]))
    
    
    @staticmethod
    def _setup_database_format():
        mongo_component = component_manager.get_mongo_component()
        col = mongo_component.col_synchronizer_info
        database_info = col.find_one({'_id': 'database_info'})
        if not database_info:
            database_info = {'_id': 'database_info'}
        
        database_info['new_link_format'] = 1
        database_info['overrides_encrypted'] = 1
        col.save(database_info)
    
    
    ############################################################################
    ## HTTP API (interface) THINGS END #########################################
    ############################################################################
    # Called from HTTP API
    def get_daemons(self, daemon_type):
        """ Returns the daemons list defined in our conf for the given type """
        # shouldn't the 'daemon_types' (whatever it is above) be always present?
        return getattr(self.conf, daemon_type + 's', None)
    
    
    # set the order and enabled properties that are stored in mongo
    def _setup_internal_sources(self):
        mongo_component = component_manager.get_mongo_component()
        
        internal_sources = [s for s in self.sources if s.internal]
        for source in internal_sources:
            if u'internal_source' not in source.imported_from:
                continue
            source_name = source.get_name()
            source_conf = mongo_component.col_internal_source_confs.find_one({'_id': source_name})
            # the mongo configuration haven't been setup for now, keep the hard coded valued until next run
            if not source_conf:
                continue
            conf_order = source_conf.get('order', None)
            if conf_order != source.order:
                source.order = conf_order
            conf_enabled = source_conf.get('enabled', None)
            if conf_enabled != source.enabled:
                source.enabled = True
    
    
    # Someone did change a source enabled bool from the API, so save this in the cfg file too if present
    # Called from HTTP API
    @staticmethod
    def save_source_enabled_into_file(source):
        cfg_pth, define_linenb = source.imported_from.split(':', 1)
        define_linenb = int(define_linenb)
        value = '1' if source.enabled else '0'
        return Synchronizer.__change_property_into_cfg(cfg_pth, define_linenb, 'enabled', value)
    
    
    def save_source_enabled_into_mongo(self, source):
        mongo_component = component_manager.get_mongo_component()
        
        source_name = source.get_name()
        source_conf = mongo_component.col_internal_source_confs.find_one({'_id': source_name})
        if not source_conf:
            source_conf = {
                '_id'    : source_name,
                'order'  : source.order,
                'enabled': source.enabled
            }
        else:
            source_conf['enabled'] = source.enabled
        mongo_component.col_internal_source_confs.save(source_conf)
    
    
    # Someone did change a source enabled bool from the API, so save this in the cfg file too
    # if present
    # Called from HTTP API
    @staticmethod
    def save_source_order_into_file(source):
        cfg_pth, define_linenb = source.imported_from.split(':', 1)
        define_linenb = int(define_linenb)
        value = source.order
        return Synchronizer.__change_property_into_cfg(cfg_pth, define_linenb, 'order', value)
    
    
    def save_source_order_for_internal_source(self, source):
        try:
            self.save_source_order_into_mongo(source)
        except ShinkenMongoException:
            return u'Cannot save source order into MongoDb'
        if source.imported_from != u'internal_source':
            return self.save_source_order_into_file(source)
    
    
    def save_source_order_into_mongo(self, source):
        mongo_component = component_manager.get_mongo_component()
        
        source_name = source.get_name()
        source_conf = mongo_component.col_internal_source_confs.find_one({'_id': source_name})
        if not source_conf:
            source_conf = {'_id': source_name}
            source_conf['order'] = source.order
            source_conf['enabled'] = source.enabled
        else:
            source_conf['order'] = source.order
        mongo_component.col_internal_source_confs.save(source_conf)
    
    
    ############################################################################
    ## HTTP API (interface) THINGS END #########################################
    ############################################################################
    
    ############################################################################
    ## HTTP CONFIGURATION THINGS BEGIN ##########################################
    ############################################################################
    def _spawn_http_sub_process(self):
        # Prepare a sub-process that will manage the HTTP part
        # so we don't mess our bottle part
        father_pid = os.getpid()
        p = Process(target=self.__launch_http_sub_process, name="syncui-http", args=(father_pid,))
        p.start()
        self.__http_process = p
    
    
    # Look every second the http access, and if the favico is no more available, must be a bad
    # start from bottle, so kill the http_daemon
    def __http_daemon_health_check(self):
        while not self.interrupted:
            time.sleep(1)
            # Note: we must look at both plugin static and global static part, each can fail
            # without the other one to fail. Fuck you bottle
            uris = ["/static/%d/ping/js/ping.js" % self.http_start_time, "/static/%d/js/ping.js" % self.http_start_time]
            for uri in uris:
                try:
                    args = {}
                    f_ = httplib.HTTPConnection
                    if self.http_use_ssl:
                        f_ = httplib.HTTPSConnection
                        # If we are in SSL mode, do not look at certificate too much
                        # NOTE: ssl.SSLContext is only available on last python 2.7 versions
                        if hasattr(ssl, 'SSLContext'):
                            ssl_context = ssl.SSLContext(ssl.PROTOCOL_TLSv1)
                            ssl_context.check_hostname = False
                            ssl_context.verify_mode = ssl.CERT_NONE
                        else:
                            ssl_context = None
                        if ssl_context:
                            args['context'] = ssl_context
                    conn = f_('%s:%s' % (self.http_host, self.http_port), **args)
                    conn.request("GET", uri)
                    r1 = conn.getresponse()
                    is_alive = True
                    if r1.status != 200:  # not a good run
                        is_alive = False
                        logger.error('[HTTP Health check] BAD RESPONSE   the check for the HTTP server did fail with an error: %s' % r1.status)
                    else:  # 200: check if valid
                        buf = r1.read()
                        if 'check for availability' not in buf:
                            is_alive = False
                            logger.error('[HTTP Health check] BAD PING CONTENT   the check for the HTTP server did fail, the file do not have the accepted content.')
                    
                    self.http_daemon_ready = is_alive
                    if not is_alive:
                        logger.error('[HTTP Health check] HTTP PROCESS KILL    HTTP Server health check did fail, killing it for restart')
                        self.http_process.terminate()
                        self.http_process.kill()
                except Exception as exp:
                    # logger.error('[HTTP Health check] Unmanaged error in the interface hearthbeat: %s' % exp)
                    pass
    
    
    def _launch_http_daemon_health(self):
        # Launch the daemon_health_check
        t = threading.Thread(None, target=self.__http_daemon_health_check, name='http-daemon-health-check')
        t.daemon = True
        t.start()
    
    
    def __launch_http_sub_process(self, father_pid):
        try:
            self.__do_launch_http_sub_process(father_pid)
        except Exception, exp:
            logger.error('The HTTP daemon failed with the error %s, exiting' % str(exp))
            logger.error('Back trace of this error')
            logger.print_stack()
            # Hard mode exit from a thread
            os._exit(2)
    
    
    ############################################################################
    ## HTTP CONFIGURATION THINGS END ##########################################
    ############################################################################
    
    ############################################################################
    ## BOTTLE CONFIGURATION THINGS BEGIN ########################################
    ############################################################################
    def __do_launch_http_sub_process(self, father_pid):
        import os
        try:
            import shinken.synchronizer.bottle as bottle
            import shinken.synchronizer as SYNC
            import shinken.synchronizer.plugins
        except ImportError:
            import synchronizer.bottle as bottle
            import synchronizer as SYNC
            import synchronizer.plugins
        
        # We do not need to hold the main process http socket, we can shutdown it
        # Note: this is done in a thread, so won't lock us
        self.http_daemon.shutdown(quiet=True)
        
        logger.set_name('%s-%s' % (self.me.get_name(), 'sync-ui'))
        
        # As we have a new process, we must start our cleaning thread
        start_malloc_trim_thread()
        
        # remember my father pid, and if missing, kill myself
        self.father_pid = father_pid
        
        try:
            from setproctitle import setproctitle
            setproctitle("%s [ - Configuration UI subprocess ] " % self.daemon_display_name)
        except:
            pass
        
        # This is to avoid double module (on synchronizer and synchronizer-ui)
        # But we need source module with custom tabs to be instantiated on synchronizer-ui so we can talk to them
        self.source_module_instances = [inst for inst in self.modules_manager.get_all_instances() if inst.properties.get('tabs', False)]
        # Then we remove every instance from the modules manager
        self.modules_manager.clear_instances_by_module_on_type('synchronizer-ui')
        self.source_controller.do_after_fork()
        
        
        def look_at_father():
            import os, time
            while True:
                try:
                    os.kill(self.father_pid, 0)  # fake kill
                except:  # no more father? fuck!
                    from shinken.log import logger
                    # Maybe we have a regular stop with transaction protection
                    # BUT the transaction manager do not have already time to load the signal, so let it half a second to load it
                    time.sleep(0.5)
                    # If we are not in a transaction protection, means that it's not a normal signal handling, we must protect the system
                    if not transaction_protecter.is_drain_stop_in_progress():
                        logger.error('[%s:%s] My master process %s is dead, I exit.' % ('Synchronizer:httpthread', os.getpid(), self.father_pid))
                        os._exit(0)  # in a thread, raw kill
                
                # Look up at my module if any was dead i restart them
                # Here we set the skip external because we are in a child process and there is no possible to set an external module to a subprocess/submodule
                self.check_and_del_zombie_modules(skip_external=True)
                time.sleep(1)
        
        
        fatherlookup_thr = threading.Thread(target=look_at_father, name='look_at_father')
        fatherlookup_thr.daemon = True
        fatherlookup_thr.start()
        
        # self.__init_datamanager()
        self._load_data_backend()
        
        self.bottle = bottle
        
        self.request = self.bottle.request
        self.abort = self._abort
        self.redirect = self.bottle.redirect
        self.response = self.bottle.response
        self.__hook_error()
        
        bottle.debug(True)
        
        # Import bottle lib to make bottle happy
        bottle_dir = os.path.abspath(os.path.dirname(bottle.__file__))
        sys.path.insert(0, bottle_dir)
        
        synclib = SYNC.__path__[0]
        self.synclib = synclib
        bottle.TEMPLATE_PATH.append(os.path.join(synclib, 'views'))
        bottle.TEMPLATE_PATH.append(bottle_dir)
        bottle.TEMPLATE_PATH.append(COMMON_UI_VIEWS_DIRECTORY)
        
        try:
            self.__load_plugins(shinken.synchronizer.plugins.__path__[0])
        except Exception, exp:
            logger.error(u'The HTTP daemon failed with the error %s, exiting' % str(exp))
            logger.error(u'Back trace of this error')
            logger.print_stack()
        
        self._init_sources_routes()
        
        # We must be sure the check plugins styles css are compiled before
        try:
            self.compiled_css_path, self.compiled_css_hash = external_resources.load_and_combine_css()
        except Exception:  # there was a problem in the css compilation, already log
            raise
        
        # Declare the whole app static files AFTER the plugin ones
        self.__declare_common_static()
        
        # Manage signal so we will be able to exit
        for sig in (signal.SIGTERM, signal.SIGINT, signal.SIGUSR1, signal.SIGUSR2):
            signal.signal(sig, self.__http_process_manage_signal)
        
        syncuicommon.set_app(self)
        syncuicommon.compute_all_diff_staging_production()
        self.http_process_is_ready.set()
        try:
            bottle.run(host=self.http_host, port=self.http_port, quiet=True, server=CherryPyServerHTTP, use_ssl=self.http_use_ssl, ssl_key=self.http_ssl_key, ssl_cert=self.http_ssl_cert, daemon_thread_pool_size=64)
            logger.error("bottle stop")
        except Exception, exp:
            logger.error('The HTTP daemon failed with the error %s, exiting' % str(exp))
            logger.error('Back trace of this error:')
            logger.print_stack()
            self.do_stop()
            # Hard mode exit from a subprocess for kill all my thread
            os._exit(2)
    
    
    def _abort(self, code=500, text='Unknown Error: Application stopped.', warning=False):
        from .bottle import HTTPError
        
        http_error = HTTPError(code, text)
        http_error.warning = warning
        http_error.text = text
        raise http_error
    
    
    # Allow to manage an error:
    # * if json: return as json
    # * else:
    #   * if not allow redirect (default): show the error page with the error message
    #   * else: redirect to the page with the error as 'error' argument
    def __error_handler(self, error, redirect=None, origin_url=''):
        return_value = None
        error_msg = getattr(error, 'text', error._status_line)
        request_url = repr(self.request.url)
        logger.error("ERROR at [%s] msg : %s" % (request_url, error_msg))
        logger.debug("FULL Error: %s" % traceback.format_exc())
        if error.traceback:
            first_line = True
            for line_stack in error.traceback.splitlines():
                if first_line:
                    logger.error("ERROR stack : %s" % line_stack)
                    first_line = False
                else:
                    logger.error("%s" % line_stack)
        # is it a json?
        if self.response and self.response.content_type == 'application/json':
            return_value = json.dumps(error_msg)
        else:  # not a json? show a page. can be the redirect one, or the "error" template
            if redirect is not None:
                params = {
                    u'error'     : error_msg.encode('utf-8', 'ignore'),
                    u'origin_url': base64.b64encode(origin_url) if origin_url else u'',
                }
                redirect_to = u'%s?%s' % (redirect, urllib.urlencode(params))
                return_value = self.bottle.template("redirect", redirect_to=redirect_to)
                return return_value
            # No redirection, go to the error template page to show a real traceback and co
            user_p = self.get_user_auth()
            if not user_p:
                user_p = {}
            return_value = self.bottle.template("error", e=error, app=self, user=user_p)
        return return_value
    
    
    def __hook_error(self):
        try:
            import shinken.synchronizer.bottle as bottle
        except ImportError:
            import synchronizer.bottle as bottle
        
        
        @bottle.error(500)
        def custom500(error):
            return self.__error_handler(error)
        
        
        @bottle.error(404)
        def custom404(error):
            return self.__error_handler(error)
        
        
        # On not authorized error, redirect to the login page
        @bottle.error(401)
        def custom401(error):
            return self.__error_handler(error, redirect='/user/login', origin_url=self.request.environ.get('REQUEST_URI'))
        
        
        # Same on "should connect" error, propose to login
        @bottle.error(403)
        def custom403(error):
            return self.__error_handler(error, redirect='/user/login')
        
        
        @bottle.error(409)
        def custom409(error):
            return self.__error_handler(error)
    
    
    def __http_process_manage_signal(self, sig, frame):
        logger.debug("I'm sub process %d and I received signal %s" % (os.getpid(), str(sig)))
        if sig == 10:  # if USR1, ask a memory dump
            if sys.version_info[1] == 6:  # python 2.6
                try:
                    from guppy import hpy
                    hp = hpy()
                    logger.error('(support-only) MEMORY DUMP (to be sent to the support):\n%s' % hp.heap())
                    return
                except ImportError:
                    logger.error('(support-only) MEMORY DUMP: FAIL check if guppy lib is installed')
            if sys.version_info[1] == 7:  # python 2.7
                try:
                    import meliae.scanner
                    import meliae.loader
                    _f = "/tmp/memorydump-%s.json" % self.name
                    meliae.scanner.dump_all_objects(_f)
                    logger.error('(support-only) Memory information dumped to file %s (to be sent to the support)' % _f)
                except ImportError:
                    logger.error('(support-only) MEMORY DUMP: FAIL check if meliae lib is installed')
        elif sig == 12:  # if USR2, ask objects dump
            self.need_objects_dump = True
    
    
    def _init_source_routes(self, source):
        try:
            module_type = source.get_module().module_type
            logger.debug('init source routes %s' % module_type)
            
            route_container = source.get_route_container()
            
            for route_info in route_container:
                route = route_info.get_route()
                controller = route_info.controller
                template = route_info.get_template()
                tab_info_folder = os.path.dirname(inspect.getfile(route_info.__class__))
                
                if template:
                    controller = self.bottle.view(template)(controller)
                
                logger.debug('add new [%s] tab with route [%s] in folder [%s]' % (route_info.name, route_info.get_route(), tab_info_folder))
                
                controller = self.__nocache_wrapper(controller)
                controller = self.__stats_wrapper(controller, route)
                self.bottle.route(route, callback=controller, method='GET')
                
                static_folder_path = os.path.join(tab_info_folder, 'static')
                static_route = '/source/static/%d%s/:path#.+#' % (self.http_start_time, route)
                self._add_static_route(static_folder_path, static_route)
                
                # And we add the views dir of this plugin in our TEMPLATE_PATH
                self.bottle.TEMPLATE_PATH.append(os.path.join(tab_info_folder, 'http_template'))
        
        
        
        except Exception, exp:
            module_type = source.get_module().module_type
            logger.error('init source routes[%s] fail : [%s]' % (module_type, exp))
            logger.print_stack()
    
    
    def _add_static_route(self, static_folder_path, static_route):
        if os.path.isdir(static_folder_path):
            
            def plugin_static(path):
                if path.endswith('.py'):
                    return self.bottle.abort(400, 'Python files are filter')
                return self.bottle.static_file(path, root=static_folder_path)
            
            
            self.bottle.route(static_route, callback=plugin_static)
    
    
    def __load_plugin(self, fdir, plugin_dir, module_path=None):
        logger.debug("Try to load [%s] from [%s]" % (fdir, plugin_dir))
        try:
            # Put the full qualified path of the module we want to load
            # for example we will give  webui/plugins/eltdetail/
            mod_path = os.path.join(plugin_dir, fdir)
            
            # Then we load the eltdetail.py inside this directory
            sys.path.append(mod_path)
            moddir = module_path
            if module_path is None:
                moddir = fdir
            
            imp.load_module('shinken.synchronizer.plugins.%s' % (fdir), *imp.find_module(fdir, [plugin_dir]))
            module = imp.load_module('shinken.synchronizer.plugins.%s.%s' % (fdir, moddir), *imp.find_module(moddir, [mod_path]))
            m_dir = os.path.abspath(os.path.dirname(module.__file__))
            
            # logger.debug("Loaded module %s" % module)
            pages = getattr(module, 'pages', {})
            module_name = module.__name__
            route_wrappers.app = self
            exceptions.app = self
            # logger.debug("Try to load pages %s" % pages)
            for (f, entry) in pages.items():
                routes = entry.get('routes', None)
                view = entry.get('view', None)
                wrappers = entry.get('wrappers', ['auth_admin'])
                function_name = f.__name__
                for wrapper in wrappers:
                    wrap = {
                        'auth'                                      : abort_if_not_admin_nor_admin_si,
                        'auth_admin'                                : abort_if_not_admin,
                        'transaction'                               : protect_transaction,
                        'db_transaction'                            : protect_db_transaction,
                        'protect_internal_call'                     : protect_internal_call,
                        'format_validator_message_if_not_admin'     : format_validator_message_if_not_admin,
                        'format_validator_message_if_base_exception': format_validator_message_if_base_exception
                    }.get(wrapper, None)
                    if wrap:
                        f = wrap(f)
                
                # IMPORTANT: apply VIEW BEFORE route!
                if view:
                    f = self.bottle.view(view)(f)
                
                # Maybe there is no route to link, so pass
                if routes:
                    for r in routes:
                        method = entry.get('method', 'GET')
                        logger.debug("link function [%s-%s] and route [%s-%s] method" % (module_name, function_name, method, r))
                        
                        f = self.__nocache_wrapper(f)
                        f = self.__stats_wrapper(f, r)
                        f = self.bottle.route(r, callback=f, method=method)
                
                # If the plugin declare a static entry, register it
                # and remember: really static! because there is no lock for them!
                if os.path.isdir(os.path.join(m_dir, 'htdocs')):
                    self.__add_static(fdir, m_dir)
            
            # And we add the views dir of this plugin in our TEMPLATE_PATH
            self.bottle.TEMPLATE_PATH.append(os.path.join(m_dir, 'views'))
            
            # And finally register me so the pages can get data and other
            # useful stuff
            if hasattr(module, 'set_app'):
                module.set_app(self)
            else:
                module.app = self
            
            return
        except Exception, exp:
            logger.error("Loading plugins [%s-%s] fail : [%s]" % (plugin_dir, fdir, exp))
            logger.print_stack()
    
    
    # For this request, ask for no cache at all. for example static that have always the
    # same address across version
    def __set_no_cache_headers(self):
        # The Cache-Control is per the HTTP 1.1 spec for clients and proxies
        # (and implicitly required by some clients next to Expires).
        # The Pragma is per the HTTP 1.0 spec for prehistoric clients.
        # The Expires is per the HTTP 1.0 and 1.1 spec for clients and proxies.
        # In HTTP 1.1, the Cache-Control takes precedence over Expires, so it's after all for HTTP 1.0 proxies only.
        self.bottle.response.headers['Cache-Control'] = 'no-cache,no-store,must-revalidate'
        self.bottle.response.headers['Pragma'] = 'no-cache'
        self.bottle.response.headers['Expires'] = '0'
    
    
    def __set_allow_cache_headers(self):
        # Ask for no tuning from cache or in the middle proxy
        self.bottle.response.headers['Cache-Control'] = 'no-transform,public,max-age=86400,s-maxage=86400'
        self.bottle.response.headers['Expires'] = '86400'  # old http 1.0 way
    
    
    def _init_sources_routes(self):
        with_tab_sources = []
        for source in self.sources:
            if source.get_tab_container():
                with_tab_sources.append(source)
        
        logger.info('init source routes for source:[%s] ' % ','.join([i.get_name() for i in with_tab_sources]))
        
        for source in with_tab_sources:
            self._init_source_routes(source)
        
        static_folder_path = os.path.dirname(inspect.getfile(http_lib_external))
        static_route = '/http_lib/static/%d/:path#.+#' % self.http_start_time
        self._add_static_route(static_folder_path, static_route)
        
        static_folder_path = os.path.dirname(inspect.getfile(source_translation))
        static_route = '/source/translation/static/%d/:path#.+#' % self.http_start_time
        self._add_static_route(static_folder_path, static_route)
    
    
    # Here we will load all plugins (pages) under the webui/plugins directory.
    # Each one can have a page, views and htdocs dir that we must route correctly
    def __load_plugins(self, plugin_dir):
        # Load module directories
        modules_to_load = set([source.get_module().module_type for source in self.sources if source.get_tabs()])
        modules_to_load = list(modules_to_load)
        # Load plugin directories
        if not os.path.exists(plugin_dir):
            return
        plugin_dirs = [fname for fname in os.listdir(plugin_dir) if os.path.isdir(os.path.join(plugin_dir, fname))]
        
        logger.debug("Load plugin dirs %s" % plugin_dirs)
        sys.path.append(plugin_dir)
        # We try to import them, but we keep only the one of our type
        for fdir in plugin_dirs:
            self.__load_plugin(fdir, plugin_dir)
        for fdir in modules_to_load:
            self.__load_plugin(fdir, "/var/lib/shinken/modules", "module")
    
    
    def __add_static(self, fdir, m_dir):
        def plugin_static(path):
            return self.bottle.static_file(path, root=os.path.join(m_dir, 'htdocs'))
        
        
        # logger.debug('HTTP: declaring static root: %s=> %s' % (fdir, static_route))
        self.bottle.route('/static/%d/' % self.http_start_time + fdir + '/:path#.+#', callback=plugin_static)
    
    
    def __nocache_wrapper(self, f):
        def nocache_version(**args):
            # set no cache as explicit
            self.__set_no_cache_headers()
            return f(**args)
        
        
        return nocache_version
    
    
    def __stats_wrapper(self, f, path):
        
        def stats_wrapper(**args):
            snap = cpu_stats_helper.get_thread_cpu_snapshot()
            start_time = time.time()
            
            # Do the real call
            return_value = f(**args)
            
            # And let the stats be updated for this call (for healthcheck and checks)
            call_time = time.time() - start_time
            syncuicommon.update_call_stats(path, call_time)
            if call_time > 0.05:  # if more than 50ms, debug print the real call consumption
                logger.debug('[HTTP CALL] [PERFS] [ %s ] %s' % (path, snap.get_diff()))
            return return_value
        
        
        return stats_wrapper
    
    
    def __declare_common_static(self):
        @self.bottle.route('/static/photos/:path#.+#')
        def give_photo(path):
            # If the file really exist, give it. If not, give a dummy image.
            if os.path.exists(os.path.join(self.photo_dir, path + '.jpg')):
                return self.bottle.static_file(path + '.jpg', root=self.photo_dir)
            else:
                return self.bottle.static_file('images/user.png', root=os.path.join(self.synclib, 'htdocs'))
        
        
        # Give the combined css file
        # IMPORTANT: defined BEFORE the static/*
        # NOTE: no cache as we can maybe regenerate the file
        @self.bottle.route('/static/%d/css/check_plugin_styles_css.css' % self.http_start_time)
        def give_check_plugin_styles_css():
            self.__set_no_cache_headers()
            return self.bottle.static_file(os.path.basename(self.compiled_css_path), root=os.path.dirname(self.compiled_css_path))
        
        
        @self.bottle.route('/static/%s/common_ui/:path#.+#' % self.http_start_time)
        def common_ui_static(path):
            root = COMMON_UI_HTDOCS_DIRECTORY
            self.__set_allow_cache_headers()
            return self.bottle.static_file(path, root=root)
        
        
        # Route static files css files
        @self.bottle.route('/static/%d/' % self.http_start_time + ':path#.+#')
        def server_static(path):
            # By default give from the root in bottle_dir/htdocs. If the file is missing,
            # search in the share dir
            root = os.path.join(self.synclib, 'htdocs')
            p = os.path.join(root, path)
            
            if not os.path.exists(p):
                root = self.share_dir
            self.__set_allow_cache_headers()
            return self.bottle.static_file(path, root=root)
        
        
        # And add the favicon ico too
        @self.bottle.route('/favicon.ico')
        def give_favicon():
            return self.bottle.static_file('favicon.ico', root=os.path.join(self.synclib, 'htdocs', 'images'))
    
    
    def call_internal_put_source_in_staging(self, user_name, source_name):
        conn = self.get_synchronizer_syncui_connection()
        params = urllib.urlencode({
            'private_key': self.get_private_http_key(),
            'user_name'  : user_name,
            'filter'     : 'sources:%s' % source_name,
        })
        conn.request("GET", "/internal/force_trusted_source_behaviour?%s" % params)
        response = conn.getresponse()
        buf = response.read()
        result = json.loads(buf)
        conn.close()
        return result
    
    
    ############################################################################
    ## BOTTLE CONFIGURATIO THINGS END ##########################################
    ############################################################################
    
    ############################################################################
    ## WEB UI THINGS API BEGIN #################################################
    ############################################################################
    
    # Called from UI plugins
    def get_api_sources_from_backend(self):
        conn = self.get_synchronizer_deamon_connection()
        conn.request("GET", "/get_api_sources")
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached.')
        buf = r1.read()
        diffs_stats = json.loads(buf)
        conn.close()
        return diffs_stats
    
    
    # Called from UI plugins
    def get_api_source_controller_importing_from_backend(self):
        conn = self.get_synchronizer_deamon_connection()
        conn.request("GET", "/get_api_source_controller_importing")
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached.')
        buf = r1.read()
        diffs_stats = json.loads(buf)
        conn.close()
        return diffs_stats
    
    
    # Called from UI plugins
    def get_api_source_run_number(self, source_name):
        conn = self.get_synchronizer_deamon_connection()
        conn.request("GET", "/get_api_source_run_number?source_name=%s" % source_name)
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached : %s' % err)
        buf = r1.read()
        source_run_number = json.loads(buf)
        conn.close()
        return source_run_number
    
    
    # Called from UI plugins
    def set_source_enabled_to_backend(self, sname, state):
        conn = self.get_synchronizer_deamon_connection()
        conn.request("GET", "/set_api_sources?sname=%s&enabled=%s" % (sname, state))
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached: %s' % err)
        buf = r1.read()
        diffs_stats = json.loads(buf)
        conn.close()
        return diffs_stats
    
    
    # Called from UI plugins
    def set_sources_order_to_backend(self, order):
        conn = self.get_synchronizer_deamon_connection()
        conn.request("GET", "/set_api_sources_order?order=%s" % (order))
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached: %s' % err)
        buf = r1.read()
        diffs_stats = json.loads(buf)
        conn.close()
        return diffs_stats
    
    
    # Called from UI plugins
    def set_source_conf_enabled_to_backend(self, _source_name, _conf_id, _enabled):
        if _enabled not in ('0', '1'):
            _enabled = '1' if _enabled else '0'
        
        params = urllib.urlencode({
            'source_name': _source_name,
            'conf_id'    : _conf_id,
            'enabled'    : _enabled,
        })
        conn = self.get_synchronizer_deamon_connection()
        conn.request("GET", "/set_api_sources_conf?%s" % params)
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached: %s' % err)
        buf = r1.read()
        diffs_stats = json.loads(buf)
        conn.close()
        return diffs_stats
    
    
    # Called from UI plugins
    def api_remove_source_item_to_backend(self, source_name, info_items):
        
        params = urllib.urlencode({
            'info_items' : base64.urlsafe_b64encode(json.dumps(info_items)),
            'source_name': source_name,
        })
        conn = self.get_synchronizer_deamon_connection()
        conn.request("GET", "/api_remove_source_item?%s" % params)
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached: %s' % err)
        buf = r1.read()
        conn.close()
        return ''
    
    
    # Called from UI plugins
    def delete_source_conf_to_backend(self, _source_name, _conf_id):
        params = urllib.urlencode({
            'source_name': _source_name,
            'conf_id'    : _conf_id,
        })
        
        conn = self.get_synchronizer_deamon_connection()
        conn.request("GET", "/delete_sources_conf?%s" % params)
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached: %s' % err)
        conn.close()
        return
    
    
    # Called from UI plugins
    def save_source_conf_to_backend(self, _source_name, _conf_id, _new_conf):
        params = urllib.urlencode({
            'source_name': _source_name,
            'conf_id'    : _conf_id,
            'new_conf'   : base64.b64encode(json.dumps(_new_conf)),
        })
        
        conn = self.get_synchronizer_deamon_connection()
        conn.request("GET", "/save_sources_conf?%s" % params)
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached: %s' % err)
        buf = r1.read()
        _to_return = json.loads(buf)
        conn.close()
        return _to_return
    
    
    # Called from UI plugins
    def api_force_source(self, sname):
        conn = self.get_synchronizer_deamon_connection()
        conn.request("GET", "/api_force_source?sname=%s" % sname)
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached: %s' % err)
        buf = r1.read()
        diffs_stats = json.loads(buf)
        conn.close()
        return diffs_stats
    
    
    # Called from UI plugins
    def api_clean_source(self, sname):
        conn = self.get_synchronizer_deamon_connection()
        conn.request("GET", "/api_clean_source?sname=%s" % sname)
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached: %s' % err)
        buf = r1.read()
        diffs_stats = json.loads(buf)
        conn.close()
        return diffs_stats
    
    
    # Called from UI plugins
    def api_restart_source(self, sname):
        conn = self.get_synchronizer_deamon_connection()
        conn.request("GET", "/api_restart_source?sname=%s" % sname)
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached: %s' % err)
        buf = r1.read()
        diffs_stats = json.loads(buf)
        conn.close()
        return diffs_stats
    
    
    # Called from UI plugins
    def add_host_to_analyzer(self, analyzer_name, state, uuid):
        conn = self.get_synchronizer_deamon_connection()
        conn.request("GET", "/add_host_to_analyzer?analyzer_name=%s&state=%s&uuid=%s" % (analyzer_name, state, uuid))
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached: %s' % err)
        buf = r1.read()
        r = json.loads(buf)
        conn.close()
        return r
    
    
    # Called from UI plugins
    def verify_access_element(self, element, elt_class, user):
        # spec : http://151.80.162.119:8080/browse/SEF-1111
        
        is_shinken_admin = user.is_admin()
        is_si_admin = user.is_expert()
        elt_visibilty = (element.get('for_all_users', '1') == '1') or elt_class == 'hosts' or elt_class == 'contacts'
        
        if is_shinken_admin:
            return True
        if is_si_admin and elt_visibilty:
            return True
        
        return False
    
    
    # Called from UI plugins
    def is_tag_exist(self, tag):
        return tag in self.img_tags
    
    
    # Called from UI plugins (testapi)
    def get_api_key(self):
        return str(self.api_key)
    
    
    # Called from UI plugins
    # Called from self
    def get_user_auth(self, warn_no_bottle_thread=True):
        # start_time = time.time()
        try:
            user_id = self.request.get_cookie('user', secret=self.auth_secret)
        except RuntimeError:
            if warn_no_bottle_thread:
                logger.warning('The request is not ready . You are not in a bottle thread')
            return None
        
        # If the http_remote_user_enable is allowed, we check header
        if user_id:
            user_auth = self.datamanagerV2.find_item_by_id(user_id, ITEM_TYPE.CONTACTS, ITEM_STATE.STAGGING)
        elif not self.http_remote_user_enable:
            return None
        elif self.http_remote_user_variable not in self.request.headers:
            return None
        else:
            # try to look in headers
            # Maybe the user did want to look at Header auth, so look for this case, but only if there is no cookie here
            # logger.info('[Auth] Searching for auth in header [%s]' % self.http_remote_user_variable)
            
            user_name = self.request.headers[self.http_remote_user_variable].strip()
            if self.http_remote_user_case_sensitive:
                user_auth = self.datamanagerV2.find_item_by_name_with_case_sensitive(user_name, ITEM_TYPE.CONTACTS, ITEM_STATE.STAGGING)
            else:
                user_auth = self.datamanagerV2.find_item_by_name(user_name, ITEM_TYPE.CONTACTS, ITEM_STATE.STAGGING)
            
            # logger.info('[Auth] search user: [%s] / case_sensitive [%s]' % (user_name, case_sensitive))
        
        # If the connected user is delete or disabled we disconnect him
        if not user_auth or user_auth.get('enabled', '1') == '0':
            return None
        # logger.debug('[Auth] get_user_auth time [%0.5f]ms' % ((time.time() - start_time) * 1000))
        return user_auth
    
    
    # Called from UI plugins (login)
    def authenticate_user_by_modules(self, username, password, requester, authentication_phase_id=None):
        # type: (unicode, unicode, unicode, Union[unicode, None]) -> Union[bool, ContactItem]
        
        start_time = time.time()
        
        # Need to get an uuid to trace anonymously this request ( do not trace user -> RGPD Safe )
        if not authentication_phase_id:
            authentication_phase_id = u'id-%s' % uuid.uuid1().get_hex()
        
        authentication_logger = self.logger_auth.get_sub_part(requester)
        
        authentication_logger.info(u'Need to check a user authentication ( authentication phase %s )' % authentication_phase_id)
        
        if not username:
            authentication_logger.info(u'No username was received, can\'t continue')
            self.logger_user_authentication.login_failed(
                user_uuid=u'(uuid unavailable)',
                user_name=u'(name unavailable)',
                requester_ip=self.request.remote_addr,
                run_time=time.time() - start_time,
                requester_module=requester,
                error=u'No username was received'
            )
            return False
        
        authentication_logger.debug(u'A user try to connect from address "%s" with username "%s" ( authentication phase %s )' % (self.request.environ.get(u'REMOTE_ADDR'), username, authentication_phase_id))
        authenticated_contact = None
        
        for mod in self.modules_manager.get_internal_instances():
            check_auth_function = getattr(mod, u'check_auth', None)
            if check_auth_function and callable(check_auth_function):
                authentication_logger.info(u'Use module "%s" to authenticate user ( authentication phase %s )' % (mod.get_name(), authentication_phase_id))
                try:
                    authenticated_contact = check_auth_function(username, password, requester, authentication_phase_id)
                except Exception as exp:
                    logger.warning(u'[%s] The mod "%s" raise an exception during authentication phase %s : %s, I\'m tagging it to restart later' % (self.name, mod.get_name(), authentication_phase_id, str(exp)))
                    logger.debug(u'Exception type: %s' % type(exp))
                    logger.debug(u'Back trace of this kill: %s' % (traceback.format_exc()))
                    self.modules_manager.did_crash(mod)
                
                if authenticated_contact:
                    contact_name = authenticated_contact.get_name()
                    authentication_logger.info(u'The user has been authenticated by the module %s ( authentication phase %s )' % (mod.get_name(), authentication_phase_id))
                    authentication_logger.debug(u'User "%s" is authenticated ( authentication phase %s )' % (contact_name, authentication_phase_id))
                    
                    self.logger_user_authentication.login(
                        user_uuid=authenticated_contact.get(u'_id', u'(uuid unavailable)'),
                        user_name=contact_name if contact_name else u'(uuid unavailable)',
                        requester_ip=self.request.remote_addr,
                        run_time=time.time() - start_time,
                        requester_module=requester,
                        auth_module=mod.get_name(),
                    )
                    # No need for other modules
                    break
                else:
                    authentication_logger.info(u'Module "%s" didn\'t authenticate user ( authentication phase %s )' % (mod.get_name(), authentication_phase_id))
        
        if authenticated_contact is False:
            authentication_logger.info(u'No more authentication module, user is not authenticated ( authentication phase %s )' % authentication_phase_id)
            authentication_logger.debug(u'User "%s" can\'t be authenticate by modules ( authentication phase %s )' % (username, authentication_phase_id))
            self.logger_user_authentication.login_failed(
                user_uuid=u'(uuid unavailable)',
                user_name=username if username else u'(name unavailable)',
                requester_ip=self.request.remote_addr,
                run_time=time.time() - start_time,
                requester_module=requester,
            )
        
        return authenticated_contact
    
    
    # Called from UI plugins
    def launch_analyze_batch(self, analyzer_name, hosts_to_analyze, overload_parameters, launch_batch_uuid):
        conn = self.get_synchronizer_deamon_connection()
        
        hosts_uuid = hosts_to_analyze.keys()
        hosts_state = hosts_to_analyze.values()
        
        # Protect parameters against base characters
        # NOTE: encode to transform unicode/utf8 into raw str to please b64encode
        overload_parameters_64 = urllib.quote_plus(base64.b64encode(overload_parameters.encode('utf-8', 'ignore')))
        conn.request("GET",
                     "/add_hosts_to_analyzer?analyzer_name=%s&hosts_uuids=%s&hosts_states=%s&overload_parameters=%s&launch_batch_uuid=%s" % (analyzer_name, ",".join(hosts_uuid), ",".join(hosts_state), overload_parameters_64, launch_batch_uuid))
        
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached: %s' % err)
        buf = r1.read()
        launch_analyze_batch = json.loads(buf)
        conn.close()
        return launch_analyze_batch
    
    
    # Called from UI plugins
    def stop_analyze_batch(self, launch_batch_uuid):
        conn = self.get_synchronizer_deamon_connection()
        conn.request('GET', '/stop_analyze_batch?launch_batch_uuid=%s' % launch_batch_uuid)
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached: %s' % err)
        buf = r1.read()
        res = json.loads(buf)
        conn.close()
        return res
    
    
    # Called from UI plugins
    def get_analyze_jobs_result(self, launch_batch_uuid):
        conn = self.get_synchronizer_deamon_connection()
        conn.request("GET", "/get_analyze_jobs_result?launch_batch_uuid=%s" % (launch_batch_uuid))
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            self.abort(500, 'Backend cannot be reached: %s' % err)
        buf = r1.read()
        res = json.loads(buf)
        conn.close()
        return res
    
    
    ############################################################################
    ## WEB UI THINGS API END ###################################################
    ############################################################################
    
    @staticmethod
    def change_property_into_cfg(cfg_pth, define_linenb, prop, value, new_property_comments_lines=[]):
        return Synchronizer.__change_property_into_cfg(cfg_pth, define_linenb, prop, value, new_property_comments_lines=new_property_comments_lines)
    
    
    # Look for the file cfg_pth:define_linenb object, and if founded the prop key, change
    # the value
    @staticmethod
    def __change_property_into_cfg(cfg_pth, define_linenb, prop, value, new_property_comments_lines=[]):
        if not os.path.exists(cfg_pth):
            msg_error = 'Cannot change %s:%s into the file %s: as the file do not exists' % (prop, value, cfg_pth)
            logger.error(msg_error)
            return msg_error
        try:
            f = codecs.open(cfg_pth, 'r', 'utf8')
            lines = f.read().splitlines()
            f.close()
        except Exception as exp:
            msg_error = 'Cannot change %s:%s into the file %s: %s' % (prop, value, cfg_pth, exp)
            logger.error(msg_error)
            if hasattr(exp, 'errno'):  # only IOError have this
                if (exp.errno == errno.EPERM) or (exp.errno == errno.EACCES):
                    msg_error = 'Cannot change %s:%s into the file %s : Permission denied' % (prop, value, cfg_pth)
                if exp.errno == errno.ENOENT:
                    msg_error = 'Cannot change %s:%s into the file %s : File not found' % (prop, value, cfg_pth)
            return msg_error
        new_lines = []
        line_nb = 0  # lines start at 1 so set 0 for the begining
        in_object = False  # flag to know if we are in the valid object or not
        prop_found = False
        for line in lines:
            line_nb += 1
            strip_line = line.strip()
            if strip_line.startswith('#'):  # comments directly go to new_file
                new_lines.append(line)
                continue
            if line_nb == define_linenb and strip_line.startswith('define '):
                in_object = True  # congrats, that's the valid object
                new_lines.append(line)
                continue
            if strip_line.startswith('}') and in_object:  # we exit from the object
                # If we didn't find our property, just append it to item.
                if not prop_found:
                    # if we are adding the property, then preadd our comment if there are somes
                    new_lines.extend(new_property_comments_lines)
                    new_line = '    %s      %s' % (prop, value)
                    new_lines.append(new_line)
                in_object = False
                new_lines.append(line)
                continue
            # if we are not in the object, just add it
            if not in_object:
                new_lines.append(line)
                continue
            
            elts = re.split("[" + string.whitespace + "]+", strip_line, 1)
            
            # skip lines we don't care in the object
            if elts[0] != prop:
                new_lines.append(line)
                continue
            # ok, we have the property here, set it!
            new_line = '    %s      %s' % (prop, value)
            new_lines.append(new_line)
            prop_found = True
        
        # If do not finish with a new line, add it
        if new_lines[-1].strip() != '':
            new_lines.append('\n')
        
        tmp_path = cfg_pth + '.tmp'
        try:
            f = codecs.open(tmp_path, 'w', 'utf8')
            f.write('\n'.join(new_lines))
            f.close()
        except Exception as exp:
            msg_error = 'Cannot change %s:%s into the file %s: %s' % (prop, value, cfg_pth, exp)
            logger.error(msg_error)
            if hasattr(exp, 'errno'):  # only IOError have this
                if (exp.errno == errno.EPERM) or (exp.errno == errno.EACCES):
                    msg_error = 'Cannot change %s:%s into the file %s : Permission denied' % (prop, value, cfg_pth)
                if exp.errno == errno.ENOENT:
                    msg_error = 'Cannot change %s:%s into the file %s : File not found' % (prop, value, cfg_pth)
            return msg_error
        
        # now the new file is ok, write the original
        try:
            shutil.move(tmp_path, cfg_pth)
        except Exception as exp:
            msg_error = 'Cannot change %s:%s into the file %s: %s' % (prop, value, cfg_pth, exp)
            logger.error(msg_error)
            if hasattr(exp, 'errno'):  # only IOError have this
                if (exp.errno == errno.EPERM) or (exp.errno == errno.EACCES):
                    msg_error = 'Cannot change %s:%s into the file %s : Permission denied' % (prop, value, cfg_pth)
                if exp.errno == errno.ENOENT:
                    msg_error = 'Cannot change %s:%s into the file %s : File not found' % (prop, value, cfg_pth)
            return msg_error
    
    
    def load_default_properties_values(self):
        PROPERTIES_MAPPING = {
            'low_host_flap_threshold'    : ('low_flap_threshold', ITEM_TYPE.HOSTS),
            'high_host_flap_threshold'   : ('high_flap_threshold', ITEM_TYPE.HOSTS),
            'low_service_flap_threshold' : ('low_flap_threshold', ITEM_TYPE.SERVICESHOSTS),
            'high_service_flap_threshold': ('high_flap_threshold', ITEM_TYPE.SERVICESHOSTS),
        }
        
        for config_property, (object_property, item_type) in PROPERTIES_MAPPING.iteritems():
            _new_value = getattr(self.global_conf, config_property, None)
            if _new_value:
                DEF_ITEMS[item_type]['class'].properties[object_property].set_default(_new_value)
        
        default_values = {}
        for (k, v) in self.conf.__dict__.iteritems():
            # print "Checking for Default key? %s => %s" % (k, v)
            # example: [DEFAULT:host] view_contacts = nobody
            if k.startswith('[DEFAULT:'):
                _elts = k.split(']', 1)
                _type = _elts[0].replace('[DEFAULT:', '')
                _type = 'service' if _type == 'check' else _type
                _key = _elts[1].strip()
                _value = v.strip()
                if _type not in NAGIOS_TABLES:
                    logger.warning('[default] Type [%s] unknown. Default value [%s]-[%s] : [%s] is ignore' % (_type, _type, _key, _value))
                    continue
                _class = NAGIOS_TABLES[_type][0]
                if _key not in _class.properties:
                    logger.warning('[default] Property [%s] for class [%s] is unknown. Default value [%s]-[%s] : [%s] is ignore' % (_key, _type, _type, _key, _value))
                    continue
                set_item_property(default_values, '%s.%s' % (_type, _key), _value)
        
        validator = Validator(app=self, datamanagerV2=None)
        for _type in ('service', 'host'):
            item_type = _type + 's'
            default_entries = default_values.get(_type, None)
            if default_entries:
                validation = validator._rule_valide_sla_threshold(default_entries, item_type)
                if validation:
                    logger.warning('[default] default values sla_warning_threshold/sla_critical_threshold will be ignore because : %s' % '. '.join([i['message'] for i in validation]))
                    default_entries.pop('sla_warning_threshold', None)
                    default_entries.pop('sla_critical_threshold', None)
                default_view_contacts = default_entries.get('view_contacts', 'nobody').strip()
                if not VIEW_CONTACTS_DEFAULT_VALUE.is_valid_default_value(default_view_contacts):
                    msg = '[CONFIGURATION] default value for [DEFAULT:host] view_contacts in file "/etc/shinken-user/configuration/default_element_properties/default_host_properties.cfg" is incorrect [%s], sorry I bail out.' % default_view_contacts
                    logger.error(msg)
                    sys.exit(2)
        
        for _type, default_entries in default_values.iteritems():
            for _key, _value in default_entries.iteritems():
                _class = NAGIOS_TABLES[_type][0]
                item_property = _class.properties[_key]
                item_property.set_default(_value)
                logger.debug('[default] new default value for [%s]-[%s] : [%s]' % (_type, _key, _value))
    
    
    # ----------------------------
    # Translation call
    # ----------------------------
    
    def _flat_dict(self, node, flat_trad_dico, key_name=''):
        _key_name = ''
        for k, v in node.iteritems():
            if key_name:
                _key_name = '.'.join([key_name, k])
            else:
                _key_name = k
            if isinstance(v, dict):
                self._flat_dict(v, flat_trad_dico, _key_name)
            else:
                flat_trad_dico[_key_name] = v
    
    
    def _get_lang(self):
        # First find lang of the current call, and by default or if it fail take the global lang parameter
        try:
            lang = self.request.query.get('lang', self.lang)
        except (AttributeError, RuntimeError):
            lang = self.lang
        
        flat_trad = self.langs.get(lang, None)
        # If missing, load the file
        if flat_trad is None:
            path_dico_file = os.path.join(self.langs_path, lang + '.js')
            if not os.path.exists(path_dico_file):
                logger.error('Cannot load the lang file %s: no such file' % path_dico_file)
                return ''
            f = open(path_dico_file, 'r')
            buf = f.read()
            f.close()
            lines = buf.splitlines()
            new_lines = []
            for line in lines:
                line = line.strip()
                if line.startswith('//'):
                    continue
                if line.startswith('var lang'):
                    line = '{'
                if line.startswith('};'):
                    line = '}'
                new_lines.append(line)
            buf = '\n'.join(new_lines)
            raw_trad = json.loads(buf)
            
            flat_trad = {}
            self._flat_dict(raw_trad, flat_trad)
            self.langs[lang] = flat_trad
        return flat_trad, lang
    
    
    def t(self, translate_key):
        # type: (unicode) -> unicode
        return self._(translate_key)
    
    
    def _(self, s):
        # First find lang of the current call, and by default or if it fail take the global lang parameter
        flat_trad, lang = self._get_lang()
        
        o = flat_trad.get(s, None)
        if o is None:
            logger.error('Traduction: cannot find %s in the lang %s' % (s, lang))
            return 'TO_TRAD(%s)' % s
        return o
    
    
    def is_translate_existing(self, s):
        flat_trad, _ = self._get_lang()
        return s in flat_trad
    
    
    def get_synchronizer_deamon_connection(self):
        if self.use_ssl:
            # If we are in SSL mode, do not look at certificate too much
            # NOTE: ssl.SSLContext is only availabe on last python 2.7 versions
            if hasattr(ssl, 'SSLContext'):
                ssl_context = ssl.SSLContext(ssl.PROTOCOL_TLSv1)
                ssl_context.check_hostname = False
                ssl_context.verify_mode = ssl.CERT_NONE
            else:
                ssl_context = None
            
            args = {}
            if ssl_context:
                args['context'] = ssl_context
            conn = httplib.HTTPSConnection('%s:%d' % (self.host, self.port), **args)
        else:
            conn = httplib.HTTPConnection('%s:%d' % (self.host, self.port))
        return conn
    
    
    def get_synchronizer_syncui_connection(self):
        if self.http_use_ssl:
            # If we are in SSL mode, do not look at certificate too much
            # NOTE: ssl.SSLContext is only availabe on last python 2.7 versions
            if hasattr(ssl, 'SSLContext'):
                ssl_context = ssl.SSLContext(ssl.PROTOCOL_TLSv1)
                ssl_context.check_hostname = False
                ssl_context.verify_mode = ssl.CERT_NONE
            else:
                ssl_context = None
            
            args = {}
            if ssl_context:
                args['context'] = ssl_context
            conn = httplib.HTTPSConnection('%s:%d' % (self.http_host, self.http_port), **args)
        else:
            conn = httplib.HTTPConnection('%s:%d' % (self.http_host, self.http_port))
        return conn
