#!/usr/bin/env python-shinken
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.


import logging.handlers
import optparse
import os
import shutil
import subprocess
import sys
import traceback
from datetime import datetime

from lib_sanatize import sanatize_decorator
from lib_sanatize.config_file_sanatize import revert_configuration_files_from_new_format
from lib_sanatize.sanatize_common import rename_graphite_checks_metrics_files
from lib_sanatize.sanatize_decorator import ALL_FIX, add_doc, add_context_daemons, add_version, auto_launch, need_shinken_stop, add_fix
from lib_sanatize.sanatize_log import DEFAULT_LOG_FILENAME, sanitize_log, log
from lib_sanatize.sanatize_utils import VERSION, get_output_and_color
from lib_sanatize.sanatize_utils import __load_full_configuration
from lib_sanatize.sanatize_utils import is_shinken_running, natural_version, do_match_daemons, get_synchronizer_db, connect_to_mongo
from shinkensolutions.cfg_formatter.cfg_formatter import FORMATTER_FILES

try:
    import shinken
except ImportError:
    print 'Cannot import shinken lib, please install it before launching this tool'
    sys.exit(2)

from shinkensolutions.localinstall import do_import_data_history_sanatize
from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Dict

try:
    from shinken.synchronizer.dao.helpers import get_name_from_type, safe_add_to_dict
    from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
    from shinken.synchronizer.dao.def_items import DEF_ITEMS, ITEM_STATE, ITEM_TYPE, SERVICE_OVERRIDE, METADATA
    from shinken.synchronizer.dao.items import get_item_instance
    from shinken.synchronizer.synchronizerdaemon import Synchronizer
    from shinken.synchronizer.dao.validators.validator import Validator
except ImportError:
    from synchronizer.dao.helpers import get_name_from_type, safe_add_to_dict
    from synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
    from synchronizer.dao.def_items import DEF_ITEMS, ITEM_STATE, ITEM_TYPE, SERVICE_OVERRIDE, METADATA
    from synchronizer.dao.items import get_item_instance
    from synchronizer.synchronizerdaemon import Synchronizer
    from synchronizer.dao.validators.validator import Validator

if not os.getuid() == 0:
    print "ERROR: this script must be run as root"
    sys.exit(2)

REVERT_SANATIZE_SUMMARY = {}


# ########################## FIX Functions
@add_fix
@add_doc('Change the format of work_area_info so that multiple users can own the changes.')
@add_context_daemons(['synchronizer'])
@add_version('02.07.06')
@auto_launch(True)
@need_shinken_stop(True)
def change_work_area_info_for_multiple_users():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    try:
        from shinken.synchronizer.business.item_controller.work_area_helper import WORK_AREA_INFO_KEY
    except ImportError:
        logger.error(u'You cannot run this fix on your shinken version please upgrade it.')
        return g_did_run
    
    data_provider_mongo = DataProviderMongo(db)
    for item_state in [ITEM_STATE.NEW, ITEM_STATE.MERGE_SOURCES, ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION]:
        for item_type in DEF_ITEMS.iterkeys():
            items = data_provider_mongo.find_items(item_type, item_state)
            for item in items:
                _item_modified = False
                _work_area_info = item.get(WORK_AREA_INFO_KEY, {})
                get_by_user = _work_area_info.get(u'get_by_user', [])
                get_by_user_name = _work_area_info.get(u'get_by_user_name', [])
                if isinstance(get_by_user, list) and get_by_user:
                    _work_area_info[u'get_by_user'] = get_by_user[0]
                    _item_modified = True
                if isinstance(get_by_user_name, list) and get_by_user_name:
                    _work_area_info[u'get_by_user_name'] = get_by_user_name[0]
                    _item_modified = True
                if _item_modified:
                    data_provider_mongo.save_item(item, item_type, item_state)
                    g_did_run = True
    return g_did_run


@add_fix
@add_doc("Will add the spare_daemon option to the brokers cfg")
@add_context_daemons(['arbiter'])
@add_version('02.07.06')
@auto_launch(True)
@need_shinken_stop(False)
def fix_new_spare_daemon_broker_option():
    g_did_run = False
    return g_did_run


@add_fix
@add_doc('This fix delete service override unlink to checks and the one that cannot be relink to any checks.')
@add_context_daemons(['synchronizer'])
@add_version('02.07.06')
@auto_launch(True)
def deleted_service_override_useless():
    g_did_run = False
    return g_did_run


@add_fix
@add_doc('This sanatize rename the graphite perfdata metric files.')
@add_context_daemons([])  # To make the function run in all daemons
@add_version('02.07.06')
@auto_launch(True)
@need_shinken_stop(True)
def rename_graphite_scheduler_checks_metrics_files():
    folders_to_rename = {
        u'nb_pollers.wsp'                                  : u'nb_poller.wsp',
        u'nb_pollers_in_overload.wsp'                      : u'nb_poller_in_overload.wsp',
        u'nb_reactionners.wsp'                             : u'nb_reactionner.wsp',
        u'nb_reactionners_in_overload.wsp'                 : u'nb_reactionner_in_overload.wsp',
        u'notifications_and_event_handlers_done_by_sec.wsp': u'notifications_done_by_sec.wsp',
        u'nb_late_event_handlers.wsp'                      : u'nb_late_event.wsp'
        
    }
    return rename_graphite_checks_metrics_files(folders_to_rename)


@add_fix
@add_doc('This sanatize replace the duplicates uuid in each dashboard widgets by new ones.')
@add_context_daemons(['broker'])
@add_version('02.07.06')
@auto_launch(True)
@need_shinken_stop(False)
def replace_duplicate_widgets_uuids():
    g_did_run = False
    return g_did_run


@add_fix
@add_doc('This sanatize decode data encoded before save.')
@add_context_daemons([u'synchronizer'])
@add_version('02.07.06')
@auto_launch(True)
@need_shinken_stop(False)
def safety_replacement_encoded_character_in_data():
    g_did_run = False
    return g_did_run


@add_fix
@add_doc('This sanatize replace the 10 000 value of daily_clean_batch_size into 10000.')
@add_context_daemons(['arbiter'])
@add_version('02.07.06')
@auto_launch(True)
@need_shinken_stop(False)
def replace_bad_formatted_sla_option_daily_clean_batch_size():
    g_did_run = False
    return g_did_run


@add_fix
@add_doc('This sanatize decode data encoded before save.')
@add_context_daemons([])
@add_version('02.07.06')
@auto_launch(True)
@need_shinken_stop(False)
def replace_wmic_symlink_depending_on_os_version():
    link_path = u'/var/lib/shinken/libexec/wmic'
    if not os.path.islink(link_path) and os.path.exists(link_path):
        return False
    if os.path.islink(link_path):
        try:
            os.unlink(link_path)
        except:
            pass
    g_did_run = False
    original_file = u'%s_centos6' % link_path
    if os.path.exists(original_file):
        shutil.copy(original_file, link_path)
        g_did_run = True
    return g_did_run


def convert_broker_module_livedata_configuration_files_to_new_format(modules):
    # type:(List[unicode]) -> Dict[unicode, bool]
    
    broker_module_livedata_cfg_params_mapping = {
        u'lang'                                                                                       : [u'lang', u'broker__module_livedata__lang'],
        u'host'                                                                                       : [u'host', u'broker__module_livedata__listening_address'],
        u'port'                                                                                       : [u'port', u'broker__module_livedata__listening_port'],
        u'use_ssl'                                                                                    : [u'use_ssl', u'broker__module_livedata__use_ssl'],
        u'ssl_cert'                                                                                   : [u'ssl_cert', u'broker__module_livedata__ssl_cert'],
        u'ssl_key'                                                                                    : [u'ssl_key', u'broker__module_livedata__ssl_key'],
        u'broker_module_livedata__broks_getter__activate_late_set_catchup'                            : [u'broker_module_livedata__broks_getter__activate_late_set_catchup', u'broker__module_livedata__broks_getter__activate_late_set_catchup'],
        u'broker_module_livedata__broks_getter__nb_late_set_allowed_before_catchup'                   : [u'broker_module_livedata__broks_getter__nb_late_set_allowed_before_catchup',
                                                                                                         u'broker__module_livedata__broks_getter__nb_late_set_allowed_before_catchup'],
        u'broker_module_livedata__broks_getter__catchup_broks_managed_by_module_in_a_catchup_loop'    : [u'broker_module_livedata__broks_getter__catchup_broks_managed_by_module_in_a_catchup_loop',
                                                                                                         u'broker__module_livedata__broks_getter__catchup_broks_managed_by_module_in_a_catchup_loop'],
        u'broker_module_livedata__broks_getter__catchup_run_endless_until_nb_late_set_allowed_reached': [u'broker_module_livedata__broks_getter__catchup_run_endless_until_nb_late_set_allowed_reached',
                                                                                                         u'broker__module_livedata__broks_getter__catchup_run_endless_until_nb_late_set_allowed_reached'],
        u'broker_module_livedata__broks_getter__include_deserialisation_and_catchup_in_lock'          : [u'broker_module_livedata__broks_getter__include_deserialisation_and_catchup_in_lock',
                                                                                                         u'broker__module_livedata__broks_getter__include_deserialisation_and_catchup_in_lock'],
        u'token'                                                                                      : [u'token', u'broker__module_livedata__token'],
    }
    
    return revert_configuration_files_from_new_format(modules, u'broker_module_livedata', broker_module_livedata_cfg_params_mapping, cfg_file_name=FORMATTER_FILES.BROKER_MODULE_LIVEDATA)


def convert_livedata_module_sla_provider_configuration_files_to_new_format(modules):
    # type:(List[unicode]) -> Dict[unicode, bool]
    
    livedata_module_sla_provider_cfg_params_mapping = {
        u'livedata_module_sla_provider__database__uri'                                                 : [u'livedata_module_sla_provider__database__uri', u'broker__module_livedata__module_sla_provider__database__uri'],
        u'livedata_module_sla_provider__database__name'                                                : [u'livedata_module_sla_provider__database__name', u'broker__module_livedata__module_sla_provider__database__name'],
        u'livedata_module_sla_provider__use_ssh_tunnel'                                                : [u'livedata_module_sla_provider__use_ssh_tunnel', u'broker__module_livedata__module_sla_provider__use_ssh_tunnel'],
        u'livedata_module_sla_provider__ssh_user'                                                      : [u'livedata_module_sla_provider__ssh_user', u'broker__module_livedata__module_sla_provider__ssh_user'],
        u'livedata_module_sla_provider__ssh_keyfile'                                                   : [u'livedata_module_sla_provider__ssh_keyfile', u'broker__module_livedata__module_sla_provider__ssh_keyfile'],
        u'livedata_module_sla_provider__ssh_tunnel_timeout'                                            : [u'livedata_module_sla_provider__ssh_tunnel_timeout', u'broker__module_livedata__module_sla_provider__ssh_tunnel_timeout'],
        u'livedata_module_sla_provider__database__retry_connection_X_times_before_considering_an_error': [u'livedata_module_sla_provider__database__retry_connection_X_times_before_considering_an_error',
                                                                                                          u'broker__module_livedata__module_sla_provider__database__retry_connection_X_times_before_considering_an_error'],
        u'livedata_module_sla_provider__database__wait_X_seconds_before_reconnect'                     : [u'livedata_module_sla_provider__database__wait_X_seconds_before_reconnect',
                                                                                                          u'broker__module_livedata__module_sla_provider__database__wait_X_seconds_before_reconnect'],
        u'livedata_module_sla_provider__no_data_period'                                                : [u'livedata_module_sla_provider__no_data_period', u'broker__module_livedata__module_sla_provider__no_data_period'],
    }
    
    return revert_configuration_files_from_new_format(modules, u'livedata_module_sla_provider', livedata_module_sla_provider_cfg_params_mapping, cfg_file_name=FORMATTER_FILES.LIVEDATA_MODULE_SLA_PROVIDER)


def convert_mongo_module_configuration_file_to_old_format(modules):
    # type:(List[unicode]) -> Dict[unicode, bool]
    # This revert has no sanatize because the new format is only applied to cfg unmodified (patchnew and rpmnew)
    
    mongodb_module_cfg_params_mapping = {
        u'uri'                  : [u'mongodb__database__uri', u'database__uri', u'uri'],
        u'database'             : [u'mongodb__database__name', u'database__name', u'database'],
        u'use_ssh_tunnel'       : [u'mongodb__database__use_ssh_tunnel', u'database__use_ssh_tunnel', u'use_ssh_tunnel'],
        u'use_ssh_retry_failure': [u'mongodb__database__use_ssh_retry_failure', u'database__use_ssh_retry_failure', u'use_ssh_retry_failure'],
        u'ssh_user'             : [u'mongodb__database__ssh_user', u'database__ssh_user', u'ssh_user'],
        u'ssh_keyfile'          : [u'mongodb__database__ssh_keyfile', u'database__ssh_keyfile', u'ssh_keyfile'],
        u'mongo_timeout'        : [u'mongodb__database__ssh_tunnel_timeout', u'database__ssh_tunnel_timeout', u'mongo_timeout'],
    }
    return revert_configuration_files_from_new_format(modules, u'mongodb', mongodb_module_cfg_params_mapping, cfg_file_name=FORMATTER_FILES.MODULE_MONGO)


@add_fix
@add_doc(u'This sanatize convert livedata-module-sla-provider configuration file to the new format')
@add_context_daemons([u'arbiter'])
@add_version(u'02.08.01')
@auto_launch(True)
@need_shinken_stop(True)
def update_configuration_file_format():
    g_did_run = False
    
    try:
        raw_objects = __load_full_configuration()
    except Exception as err:
        logger.error(str(err))
        return g_did_run
    
    format_functions = {
        u'broker_module_livedata'      : convert_broker_module_livedata_configuration_files_to_new_format,
        u'livedata_module_sla_provider': convert_livedata_module_sla_provider_configuration_files_to_new_format,
        u'mongodb'                     : convert_mongo_module_configuration_file_to_old_format,
    }
    
    modules = raw_objects.get(u'module', [])
    _max_fix_name_length = 0
    result = {}
    # TODO Change value of separator (=> sanatize_decorator.NUMBER_INDENT_SPACES + 20) to indent text with sanatize (Only one sanatize actually)
    _message_template_module = u'   \033[%%dm - Reverting %%-%ds' % (sanatize_decorator.NUMBER_INDENT_SPACES + 50 - len(u'- Reverting   '))
    _message_template_file_modified = u'         \033[%%dm %%-%ds \033[0m:  \033[%%dm %%s \033[0m' % (sanatize_decorator.NUMBER_INDENT_SPACES + 36)
    
    summary = []
    for module_type, func in format_functions.iteritems():
        updated_files = func(modules)
        result[module_type] = updated_files
        
        for file_path_name in updated_files.keys():
            name_length = len(file_path_name)
            if name_length > _max_fix_name_length:
                _max_fix_name_length = name_length
    
    for module_type, _updated_files in result.iteritems():
        
        if True in _updated_files.values():
            
            summary.append(_message_template_module % (35, module_type))
            
            for file_path, g_did_run_file in _updated_files.iteritems():
                log_color, to_log = get_output_and_color(g_did_run_file)
                summary.append(_message_template_file_modified % (35, file_path, log_color, to_log))
            
            g_did_run = True
    
    REVERT_SANATIZE_SUMMARY[u'update_configuration_file_format'] = summary
    
    return g_did_run


# We are adding the comment for the parameter broker__manage_brok__sub_process_broks_pusher_queue_batch_size
# on all broker.cfg files
@add_fix
@add_doc(u"Will add the broker__manage_brok__sub_process_broks_pusher_queue_batch_size option to the brokers cfg")
@add_context_daemons([u'arbiter'])
@add_version(u'02.07.06')
@auto_launch(True)
def fix_new_broker_queue_batch_size_parameter():
    # Nothing to do, we only add a comment
    return False


if __name__ == '__main__':
    logger.handlers = []
    logger_handler = logging.StreamHandler()
    logger.addHandler(logger_handler)
    logger_handler.setFormatter(logging.Formatter('%(levelname)s: %(message)s'))
    logger.setLevel('INFO')
    
    parser = optparse.OptionParser("%prog ", version="%prog: " + VERSION, description='This tool  is used to check the state of your Shinken Enterprise installation and configuration')
    parser.add_option('-l', '--list', dest='list_only', action='store_true', help="List available sanatize to revert")
    parser.add_option('-r', '--run', dest='run_one', help="Run only one specific revert")
    parser.add_option('-a', '--auto', dest='run_auto', action='store_true', help="Run automatic reverts at once")
    parser.add_option('', '--all', dest='run_all', action='store_true', help="Run ALL reverts at once")
    parser.add_option('-v', '--verbose', dest='verbose', action='store_true', help="Show verbose output")
    parser.add_option('-q', '--quiet', dest='quiet', action='store_true', help="Do not output when no revert has been done")
    parser.add_option('-n', '--no-hint', dest='no_hint', action='store_true', help="Do not display hints for users")
    parser.add_option('-L', '--log_file', dest='log_file', default=DEFAULT_LOG_FILENAME, help="The log file path")
    parser.add_option('-i', '--info-if-not-exists', dest='info_if_not_exists', action='store_true', help="If trying to revert a sanatize that can't be reverted, abort and log info")
    parser.add_option('', '--old', dest='old', default='', help="Key to rename")
    parser.add_option('', '--new', dest='new', default='', help="New name of the key")
    parser.add_option('', '--mongo-host', dest='mongo_host', default='localhost', help="MongoDB server hostname")
    parser.add_option('', '--mongo-port', dest='mongo_port', default='27017', help="MongoDB server port")
    parser.add_option('', '--mongo-use-ssh', dest='mongo_use_ssh', default=False, action='store_true', help="use SSH to connect to MongoDB server")
    parser.add_option('', '--mongo-ssh-key', dest='mongo_ssh_key', default='/var/lib/shinken/.ssh/id_rsa', help="SSH private key used to connect to MongoDB server")
    parser.add_option('', '--mongo-ssh-user', dest='mongo_ssh_user', default='shinken', help="user on MongoDB server to connect to with SSH")
    
    opts, args = parser.parse_args()
    
    # Look if the user ask for local or global, and if not, guess
    list_only = opts.list_only
    run_all = opts.run_all
    run_one = opts.run_one
    run_auto = opts.run_auto
    
    sanitize_log.set_console_level(logging.INFO)
    if opts.verbose or list_only:
        logger.setLevel('DEBUG')
        sanitize_log.set_console_level(logging.DEBUG)
    sanitize_log.set_file_handler(os.path.expanduser(opts.log_file))
    
    log.debug("------------------------------------------------------------------------------")
    log.debug("------------------------------------------------------------------------------")
    log.debug("Reverting sanatize at %s" % datetime.now().strftime('%H:%M:%S %d-%m-%Y'))
    log.debug("------------------------------------------------------------------------------")
    log.debug("------------------------------------------------------------------------------")
    
    # First sort ALL_FIX based on their versions
    ALL_FIX = sorted(ALL_FIX, key=natural_version)
    
    if list_only:
        cur_version = ''
        for f in ALL_FIX:
            if not hasattr(f, 'auto_launch'):
                raise Exception('Please specify auto launch parameter on revert [ %s ].' % f.__name__)
            v = f.version
            if v != cur_version:
                cur_version = v
                log.info("\n################ Version \033[35m%s\033[0m ##########" % v)
            log.info("-------------------")
            log.info(" \033[32m%-40s\033[0m:\n\tDoc: %s" % (f.__name__, f.doc))
        sys.exit(0)
    
    if [run_all, run_one, run_auto].count(True) > 1:
        sys.exit("Error: You cannot ask for a run one and run all and run auto option. Please choose")
    if not run_all and not run_one and not run_auto:
        parser.print_help()
        sys.exit(0)
    
    _need_shinken_stop = True
    # TODO Change value of separator (=> sanatize_decorator.NUMBER_INDENT_SPACES + 50) to indent text with sanatize (Only one sanatize actually)
    message_template = "   \033[%%dm%%-%ds\033[0m:  \033[%%dm %%s \033[0m" % (sanatize_decorator.NUMBER_INDENT_SPACES + 30)
    if run_one:
        
        selected_fix = next((f for f in ALL_FIX if f.__name__ == run_one), None)
        if not selected_fix:
            if opts.info_if_not_exists:
                log.info(message_template % (35, run_one, 36, u'skip (cannot be reverted)'))
                sys.exit(0)
            else:
                log.error(u'\n\033[31m error: the revert "%s" does not exists\033[0m\n' % run_one)
                sys.exit(1)
        
        _need_shinken_stop = getattr(selected_fix, u'need_shinken_stop', True)
    
    if _need_shinken_stop or run_all or run_auto:
        if is_shinken_running():
            log.warning("\033[33m###################### Warning ############################\033[0m")
            log.warning("\033[33m# Shinken is currently running. We will stop it now.      #\033[0m")
            log.warning("\033[33m# (We cannot revert sanatized data if shinken is running) #\033[0m")
            log.warning("\033[33m###########################################################\033[0m\n")
            subprocess.call(["/etc/init.d/shinken stop > /dev/null 2>&1"], shell=True)
        # Ensure mongod is started before run sanatize
        subprocess.call(["/etc/init.d/mongod start > /dev/null 2>&1"], shell=True)
    connect_to_mongo(mongo_host=opts.mongo_host, mongo_port=opts.mongo_port, use_ssh=opts.mongo_use_ssh, ssh_key=opts.mongo_ssh_key, ssh_user=opts.mongo_ssh_user)
    
    cur_version = ''
    errors_count = 0
    
    for f in ALL_FIX:
        fname = f.__name__
        if run_all or fname == run_one or (run_auto and auto_launch):
            # If the function is not launchable on this type of server, skip it
            
            v = f.version
            if v != cur_version and not opts.quiet:
                cur_version = v
                log.debug('')
                log.debug("################ Version \033[35m%s\033[0m ##########" % v)
            
            log.debug('-----')
            log.debug('Executing: %s' % fname)
            
            error_message = None
            
            if do_match_daemons(f):
                try:
                    has_error = False
                    if fname == 'fix_rename_key':
                        r = f(opts.old, opts.new)
                    else:
                        r = f()
                        if not isinstance(r, bool):
                            error_message = r
                            has_error = True
                            errors_count += 1
                except Exception, exp:
                    has_error = True
                    errors_count += 1
                    print '\n\033[31m ERROR: %s => %s\033[0m\n' % (fname, traceback.format_exc())
                    r = None
                
                if has_error:
                    state = "reverted [Failure]"
                    color = 31  # red
                elif r:
                    state = "reverted [OK]"
                    color = 32  # green
                    data_type = getattr(f, "data_type", 'configuration')
                    # We did execute a sanatize, save it in the context
                    do_import_data_history_sanatize(fname, data_type=data_type)
                else:
                    state = "skip (unecessary)"
                    color = 36  # ligh blue
            else:
                state = u'skip (this sanatize is only needed for this daemons : %s)' % u', '.join(f.context_daemons)
                color = 36  # ligh blue
                
                r = True
            
            if r or not opts.quiet:
                log.info(message_template % (35, fname, color, state))
                if fname in REVERT_SANATIZE_SUMMARY:
                    for action in REVERT_SANATIZE_SUMMARY[fname]:
                        log.info(action)
                if error_message:
                    print '\n\033[31m ERROR: %s => %s\033[0m\n' % (fname, error_message)
                    for message_line in error_message.split('\n'):
                        log.error("        \033[%dm %s" % (color, message_line))
    
    if errors_count:
        if not opts.no_hint:
            log_file = opts.log_file if opts.log_file else DEFAULT_LOG_FILENAME
            log.error("\n\n\033[31mSome errors occurred while reverting the fixes."
                      " Please check the log file for more information (\033[36m%s\033[31m)."
                      "Send this file to your Shinken support if needed.\033[0m" % log_file)
        sys.exit(1)
