#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import cPickle

from shinken.misc.type_hint import TYPE_CHECKING, cast
from shinken.toolbox.pickledb import ShinkenUnpickler, ShinkenSecurityUnpicklingError
from shinkensolutions.data_hub.data_hub_driver.abstract_data_hub_driver_file import AbstractDataHubDriverFile, AbstractDataHubDriverConfigFile
from shinkensolutions.data_hub.data_hub_factory.data_hub_factory import DataHubFactory

if TYPE_CHECKING:
    from shinken.misc.type_hint import Any
    from shinken.log import PartLogger
    from shinkensolutions.data_hub.data_hub_driver.abstract_data_hub_driver import AbstractDataHubDriverConfig
    from shinkensolutions.data_hub.data_hub import DataHubConfig


class DataHubDriverConfigPickle(AbstractDataHubDriverConfigFile):
    def __init__(self,
                 base_directory,
                 namespace=u'',
                 data_location_name=u'',
                 daemon_name=u'',
                 module_name=u'',
                 submodule_name=u'',
                 file_prefix=u'shinken_',
                 ):
        # type: ( unicode, unicode, unicode, unicode, unicode, unicode, unicode ) -> None
        super(DataHubDriverConfigPickle, self).__init__(
            name=u'PICKLE',
            namespace=namespace,
            base_directory=base_directory,
            data_location_name=data_location_name,
            daemon_name=daemon_name,
            module_name=module_name,
            submodule_name=submodule_name,
            file_ext=u'pickle',
            file_prefix=file_prefix
        )


def data_hub_driver_pickle_factory(logger, driver_config, _data_hub_config):
    # type: (PartLogger, AbstractDataHubDriverConfig, DataHubConfig) -> DataHubDriverPickle
    return DataHubDriverPickle(logger, cast(DataHubDriverConfigPickle, driver_config))


DataHubFactory.register_driver_factory(DataHubDriverConfigPickle, data_hub_driver_pickle_factory)


class DataHubDriverPickle(AbstractDataHubDriverFile):
    def __init__(self, logger, driver_config):
        # type: (PartLogger, DataHubDriverConfigPickle)-> None
        super(DataHubDriverPickle, self).__init__(logger, driver_config)
    
    
    def _pickle_data(self, data):  # noqa: overridable
        # type: (Any) -> str
        return cPickle.dumps(data, cPickle.HIGHEST_PROTOCOL)
    
    
    def _unpickle_data(self, serialized_data):  # noqa: overridable
        # type: (str) -> Any
        return ShinkenUnpickler.loads(serialized_data)
    
    
    def _serialized_data(self, data):
        # type: (Any) -> str
        try:
            return self._pickle_data(data)
        except cPickle.PickleError as err:
            try:
                error_msg = u'Error when serializing [ %r ]: %s' % (data, err)
            except Exception:  # noqa: Do not need to propagate this exception
                error_msg = u'Error when serializing data: %s' % err
            self._logger_writer.error(error_msg)
            raise
    
    
    def _deserialized_data(self, serialized_data):
        # type: (str) -> Any
        try:
            return self._unpickle_data(serialized_data)
        except ShinkenSecurityUnpicklingError as err:
            self._logger_reader.error(u'SECURITY ERROR BLOCKED: %s' % err)
            raise
        except cPickle.PickleError as err:
            self._logger_reader.error(u'Error when deserializing received data: %s' % err)
            raise
