#!/bin/bash

function display_help() {
  printf "install-patch [options]\n"
  printf "This script will install the patch PATCH${PATCH_VERSION} for the shinken version v${SHINKEN_VERSION}-release.${PATCH_LANG,,}\n"
  printf "Options:\n"
  printf "  --help          -h                     : will display this help\n"
  printf "  --force                                : removes the need to hit a key when required during installation. All daemons will be restarted (Only if --disable-daemons-restart-after-update is not used)\n"
  printf "  --disable-daemons-restart-after-update : Skip the daemons restart step at the end of the installation. (Note: This will work even with --force used)\n"
  printf "  --ignore-pre-setup-non-blocking-errors : PLEASE USE THIS OPTION ONLY WITH YOUR DEDICATED SUPPORT\n"
  printf "                                           Ignore errors of the shinken-backup before the patch installation\n"
  printf "\n"
}

force=""
DO_NOT_START_DAEMON=0

while [ $# -ge 1 ]; do
  case "$1" in
  -h | --help)
    display_help # Call your function
    # no shifting needed here, we're done.
    exit 0
    ;;
  --force)
    #  It's better to assign a string, than a number like "verbose=1"
    #  because if you're debugging script with "bash -x" code like this:
    #    if [ "$verbose" ] ...
    #  You will see:
    #    if [ "verbose" ] ...
    force="--force"
    shift
    ;;
  --disable-daemons-restart-after-update)
    DO_NOT_START_DAEMON=1
    shift
    ;;
  --ignore-pre-setup-non-blocking-errors)
    BYPASS_NON_BLOCKING_ERRORS="bypass"
    shift
    ;;
  --) # End of all options
    shift
    break
    ;;
  -*)
    echo "Error: Unknown option: $1" >&2
    display_help
    exit 1
    ;;
  *) # No more options
    break
    ;;
  esac
done

if [[ ! -f lib/install_lib/common.sh ]]; then
  printf "\033[31m Error: please launch the patch from its directory\033[0m\n"
  exit 2
fi

. lib/install_lib/common.sh

if [[ "$force" != "" ]]; then
  printf -- "\n${YELLOW}NOTE: FORCE MODE ENABLED${RESET}\n"
fi
if [[ ${DO_NOT_START_DAEMON} -eq 1 ]]; then
  printf -- "\n${YELLOW}NOTE: DAEMONS RESTART DISABLED${RESET}\n"
fi

run_python --check-version

restore_all_patch_valid_rights

############
# STEP 1
############
print_step "Patch information" "1" "11"
init_patch_name_and_location "$PATCH_ID"
display_acceptance_banner "$force"
run_python --patch-is-not-installed

############
# STEP 2
############
print_step "Shinken make a configuration backup" "2" "11"
run_python_non_blocking --backup-config --backup-path "${OUTPUT_FOLDER}/BACKUP_CONFIGURATION_PRE_PATCH---${PATCH_ID}---/"

############
# STEP 3
############
print_step "Looking if rpms need to be updated and check system parameters" "3" "11"
run_python --update-rpms-and-system --rpms-path "rpms" --log-path "${LOG_PATH}"

############
# STEP 4
############
print_step "Apply new selinux rules" "4" "11"
run_python --remove-selinux-rules --selinux-rules-path "selinux/updated-rules"
run_python --apply-selinux-rules --selinux-rules-path "selinux/all-rules"

############
# STEP 5
############

print_step "Stopping Shinken daemons" "5" "11"
run_python --stop "${force}"

############
# STEP 6
############
print_step "Revert old patch(s)" "6" "11"
run_python --revert-patch

############
# STEP 7
############
print_step "Backup files" "7" "11"
run_python --backup-before-patch

############
# STEP 8
############
print_step "Apply patch" "8" "11"
run_python --apply-patch-files --log-path "${LOG_PATH}"

if [[ $(which shinken-context-manage) ]]; then
  shinken-context-manage --install-patch --data-type "installation" --patch-name "$PATCH_NAME"
fi

############
# STEP 9
############
print_step "Executing sanatize" "9" "11"
run_python_and_return_code --sanatize --log-path "${LOG_PATH}"
ERROR_ON_SANATIZE=$?

############
# STEP 10
############
print_step "Restart tools used by Shinken : ${TOOLS_USED_BY_SHINKEN}" "10" "11"
# Checking if the command "systemctl" is in the system path, we cannot reload daemons without it
if command -v systemctl 2 &>/dev/null; then
  systemctl daemon-reload
fi

if [[ ${TOOLS_USED_BY_SHINKEN} == "" ]]; then
  printf -- "\n${YELLOW}There is no tools used by Shinken to restart${RESET}\n\n"
fi

for tool_used_by_shinken in ${TOOLS_USED_BY_SHINKEN}; do
  run_python --restart --service-list "${tool_used_by_shinken}" "${force}"
done

############
# STEP 11
############
print_step "Restart Shinken daemons" "11" "11"
if [[ ${DO_NOT_START_DAEMON} -eq 1 ]]; then
  printf -- "\n${YELLOW}SKIPPED: --disable-daemons-restart-after-update activated. You need to restart your shinken daemons yourself.${RESET}\n"
else
  run_python --start "${force}"
fi

############
# END
############
# the ,, in the variable means "to lower"
if [[ ${ERROR_ON_SANATIZE} -eq 1 ]]; then
  printf "\n\033[31mSome errors occurred while running the fixes.\n"
  printf "Please check the log file for more information (\033[32m${SANATIZE_LOGFILE}\033[31m).\n"
  printf "Send this file to your Shinken support if needed.\033[0m\n"
fi

printf -- "\n$YELLOW Your shinken installation was patched from version v${SHINKEN_VERSION}-release.${PATCH_LANG,,} to PATCH${PATCH_VERSION}${RESET}\n\n"

if [[ -f "${LOG_PATH}/files_patchnew_created.log" ]]; then
  cat "${LOG_PATH}/files_patchnew_created.log"
fi
