#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2017:
# This file is part of Shinken Enterprise, all rights reserved.

import ConfigParser
import json
import os
import shutil
import socket
import subprocess
import sys
import time
import traceback
import uuid
from StringIO import StringIO
from hashlib import md5

if os.name != 'nt':
    CONTEXT_PATH = '/var/lib/shinken/context.json'
else:  # specific path for windows
    CONTEXT_PATH = r'c:\shinken\var\context.json'
VERSION = '2.03.02'  # DEFAULT VALUE before have the context file, so very old

WARNING_COLOR = 35
ERROR_COLOR = 31
OK_COLOR = 32

# What to print in a shell string to have colors.
# IMPORTANT: always set the RESET at the end of your string
WARNING_SHELL_TAG = '\033[%dm' % WARNING_COLOR
OK_SHELL_TAG = '\033[%dm' % OK_COLOR
ERROR_SHELL_TAG = '\033[%dm' % ERROR_COLOR
RESET_SHELL_TAG = '\033[0m'

if os.path.exists(CONTEXT_PATH):
    _context = json.loads(open(CONTEXT_PATH, 'r').read())
    VERSION = str(_context.get('current_version', VERSION).split(u'-')[0])
    VERSION = VERSION.replace('.fr', '').replace('.us', '')

# ENV variable to force unset at all cost
unset_variables = ['http_proxy', 'https_proxy', 'ftp_proxy']
for v in os.environ:
    if v.lower() in unset_variables:
        os.environ[v] = ''

stdout = sys.stdout

devnull = open(os.devnull, 'w')


def protect_stdout():
    sys.stdout = devnull


def unprotect_stdout():
    sys.stdout = stdout


ADDITIONNAL_DAEMONS = ['gatherer']
POSSIBLE_DAEMONS = ['poller', 'synchronizer', 'scheduler', 'reactionner', 'receiver', 'broker', 'arbiter']
# linker and provider are not ready yet
POSSIBLE_DAEMONS.sort()

POSSIBLE_ADDONS = ['nagvis', 'nagvis-shinken-architecture']
POSSIBLE_ADDONS.sort()

POSSIBLE_DATA = ['configuration', 'sla', 'metrology', 'user', 'events', 'modules']

INSTALLATION_TYPES = ['INSTALLATION', 'UPDATE', 'FIX', 'RESTORE', 'SANATIZE', 'PATCH', 'ADDON', 'UNPATCH', 'REMOVE ADDON']


def get_local_daemons():
    if not os.path.exists(CONTEXT_PATH):
        return POSSIBLE_DAEMONS
    with open(CONTEXT_PATH, 'r') as f:
        context = json.loads(f.read())
    local_daemons = [k for (k, installed) in context['daemons'].iteritems() if installed]
    return local_daemons


def get_local_addons():
    ctx = __get_context()
    return ctx['addons']


def get_local_instances_for_type(daemon_type):
    context = __get_context()
    return list(context['local_instances'][daemon_type].iteritems())


def get_shinken_current_version():
    # type: () -> unicode
    context = __get_context()
    return context[u'current_version'] if context[u'current_version'].startswith(u'V') else u'V%s' % context[u'current_version']


def get_shinken_current_patch():
    context = __get_context()
    last_install = context[u'installation_history'][-1]
    if last_install.get(u'type') == u'PATCH':
        return last_install.get(u'patch_name') if last_install.get(u'patch_name').startswith(u'V') else u'V%s' % last_install.get(u'patch_name')
    # There is no patch currently installed, returning None
    return None


def get_shinken_current_version_and_patch():
    current_version = get_shinken_current_version()
    current_patch = get_shinken_current_patch()
    if current_patch is not None:
        return u'Version : %s\nPatch   : %s' % (current_version, current_patch)
    return u'Version : %s' % current_version


def get_shinken_raw_current_version():
    context = __get_context()
    current_version = context['current_version']
    last_install = context['installation_history'][-1]
    return current_version, last_install.get('patch_name', current_version)


def get_shinken_current_short_version():
    # type: () -> unicode
    context = __get_context()
    version = context[u'current_version'] if context[u'current_version'].startswith(u'V') else u'V%s' % context[u'current_version']
    version_suffix = version[10:12]
    return version[0:13] if version_suffix.isdigit() else version[0:9]


# Return the next id for the daemon typ, but as a STRING because fucking json do not manage
# keys as integers
def get_next_local_instance_number_available(daemon_type):
    context = __get_context()
    # Look if there is a hole in the ids currently set
    ids = context['local_instances'][daemon_type].keys()
    ids.sort()
    for i in xrange(len(ids)):
        if str(i) not in ids:
            return str(i)
    # All is filled? get a new id so
    return str(len(ids))


# AT INSTALL: create the context file is not exiting already
# will set the install version inside, and only enabled daemons (maybe all)
# and all addons will be enabled (cannot choose at install)
def create_context_file(code_version, installation_path, enabled_daemons, installer_version):
    # If not root, get away moron
    check_root()
    
    if os.path.exists(CONTEXT_PATH):
        raise Exception('The context file %s already exists' % CONTEXT_PATH)
    now = int(time.time())
    context = {'creation_time'            : now,
               'install_path'             : installation_path,
               'daemons'                  : {},  # OLD structure entry, to clean when all the U01 versions are out of scope
               'local_instances'          : {},
               'addons'                   : {},
               'addons_history'           : {},
               'installed_version'        : code_version,  # VERY FIRST install version
               'current_version'          : code_version,  # CURRENT running version
               'current_installer_version': installer_version,  # LAST installer version used, can be "" if standard release
               'installation_history'     : [],
               'data_history'             : {}
               }
    for data_type in POSSIBLE_DATA:
        context['data_history'][data_type] = []
    
    # Enable daemons only if ask at creation (central role= all daemons are ON)
    for daemon_type in POSSIBLE_DAEMONS:
        is_enabled = (daemon_type in enabled_daemons) or 'central' in enabled_daemons
        # enabled (or not) the global type, AND the instance 0
        context['daemons'][daemon_type] = is_enabled
        context['local_instances'][daemon_type] = {'0': is_enabled}
    
    __add_installation_version(context, code_version, 'INSTALLATION', now, installer_version=installer_version)
    # At creation all addons are enabled, no choice
    for addon_name in POSSIBLE_ADDONS:
        context['addons'][addon_name] = True
    
    for data_type in POSSIBLE_DATA:
        __add_data_version(data_type, context, code_version, 'INSTALLATION', now)
    
    __write_context(context)


def _get_server_identity():
    return __get_server_identity()


# returns the server identify so we know on which server it was installed before:
# server_name = hostname of the server
# server_uuid = bios uuid in lower, same as the arbiter one in the init.d script
#               if the bios uuid is not available, hash of MAC addresses and uuid
def __get_server_identity():
    try:
        server_name = socket.gethostname()
    except Exception:
        server_name = u'unknown'
    try:
        product_uuid_path = u'/sys/class/dmi/id/product_uuid'
        if os.path.exists(product_uuid_path):
            with open(product_uuid_path) as fp:
                server_uuid = fp.readline().strip().lower()
        else:
            server_uuid_path = u'/var/lib/shinken/server.uuid'
            if os.path.exists(server_uuid_path):
                with open(server_uuid_path) as fp:
                    server_uuid = fp.readline().strip().lower()
            else:
                server_uuid = u''
            macs = []
            for iface in os.listdir(u'/sys/class/net/'):
                try:
                    with open(u'/sys/class/net/%s/address' % iface) as fp:
                        macs.append(fp.readline().strip().lower())
                except IOError:
                    # This interface doesn't have any HW address, just skip it
                    pass
            mac_hash = md5(u''.join(sorted(macs))).hexdigest()
            if server_uuid.count(u'-') == 0 or mac_hash != server_uuid.split(u'-')[1]:
                server_uuid = u'sk-%s-%s' % (mac_hash, uuid.uuid4())
    except Exception:
        traceback.print_exc()
        server_uuid = u'unknown'
    return server_name, server_uuid


# Add an installation entry in the context['installation_history']
# An entry looks like:
#  {'type': 'INSTALLATION', 'date':EPOCH, 'version':VERSION, 'server_name': NOM_SERVER, 'server_uuid': UUID_SERVER, 'installer_version': STR}
def __add_installation_version(_context, installation_version, installation_type, installation_epoch, installer_version):
    if installation_type not in INSTALLATION_TYPES:
        raise ValueError('The installation type %s is not in the allowed list' % installation_type)
    
    installation_histories = _context['installation_history']
    server_name, server_uuid = __get_server_identity()
    installation_entry = {'type'             : installation_type,
                          'date'             : int(installation_epoch),
                          'version'          : installation_version,
                          'server_name'      : server_name,
                          'server_uuid'      : server_uuid,
                          'installer_version': installer_version
                          }
    installation_histories.append(installation_entry)


# Add an installation entry in the context['installation_history']
# An entry looks like:
#  {'type': 'INSTALLATION', 'date':EPOCH, 'version':VERSION, 'server_name': NOM_SERVER, 'server_uuid': UUID_SERVER}
def __add_data_version(data_type, _context, data_version, installation_type, installation_epoch, backup_path='', sanatize_name='', comment='', installer_version=''):
    if installation_type not in INSTALLATION_TYPES:
        raise ValueError('The installation type %s is not in the allowed list' % installation_type)
    
    histories = _context['data_history'].get(data_type, [])
    if not histories:
        _context['data_history'][data_type] = histories
        installation_type = 'INSTALLATION'
    server_name, server_uuid = __get_server_identity()
    installation_entry = {'type'             : installation_type,
                          'date'             : int(installation_epoch),
                          'version'          : data_version,
                          'server_name'      : server_name,
                          'server_uuid'      : server_uuid,
                          'backup_path'      : backup_path,
                          'sanatize_name'    : sanatize_name,
                          'comment'          : comment,
                          'installer_version': installer_version,
                          }
    histories.append(installation_entry)


# Add an installation entry in the context['installation_history']
# An entry looks like:
#  {'type': 'INSTALLATION', 'date':EPOCH, 'version':VERSION, 'server_name': NOM_SERVER, 'server_uuid': UUID_SERVER}
def __add_patch_install(data_type, _context, data_version, installation_type, installation_epoch, backup_path='', patch_name=''):
    if installation_type not in INSTALLATION_TYPES:
        raise ValueError('The Patch type %s is not in the allowed list' % installation_type)
    
    if data_type == 'installation':
        histories = _context['installation_history']
    else:
        histories = _context['data_history'].get(data_type, [])
    server_name, server_uuid = __get_server_identity()
    installation_entry = {'type'       : installation_type,
                          'date'       : int(installation_epoch),
                          'version'    : data_version,
                          'server_name': server_name,
                          'server_uuid': server_uuid,
                          'backup_path': backup_path,
                          'patch_name' : patch_name
                          }
    histories.append(installation_entry)


def __add_addons_version(addon_type, _context, installation_type, installation_epoch, old_map_uuid, new_map_uuid):
    if installation_type not in INSTALLATION_TYPES:
        raise ValueError('The installation type %s is not in the allowed list' % installation_type)
    
    histories = _context['addons_history'][addon_type]
    server_name, server_uuid = __get_server_identity()
    installation_entry = {'type'        : installation_type,
                          'date'        : int(installation_epoch),
                          'server_name' : server_name,
                          'server_uuid' : server_uuid,
                          'old_map_uuid': old_map_uuid,
                          'new_map_uuid': new_map_uuid
                          }
    histories.append(installation_entry)


def __sanatize_context_data(_context):
    # Fix missing daemon types (like after an update), by default disable them
    for dtype in POSSIBLE_DAEMONS:
        if dtype not in _context['daemons']:
            _context['daemons'][dtype] = False
    
    # For missing addons, enable them by default
    if _context.get("addons") is None:
        _context['addons'] = {}
    for addontype in POSSIBLE_ADDONS:
        if addontype not in _context['addons']:
            _context['addons'][addontype] = True
    
    if 'addons_history' not in _context:
        _context['addons_history'] = {}
    for addon_type in POSSIBLE_ADDONS:
        if addon_type not in _context['addons_history']:
            _context['addons_history'][addon_type] = []
    
    # Maybe there was central = True, if so, fix it
    if _context['daemons'].get('central', False):
        for dtype in _context['daemons']:
            _context['daemons'][dtype] = True
        _context['daemons']['central'] = False
    
    # set local_instances dict if missing
    if 'local_instances' not in _context:
        _context['local_instances'] = {}
    for (dtype, enabled) in _context['daemons'].iteritems():
        if dtype not in _context['local_instances']:
            _context['local_instances'][dtype] = {'0': enabled}
    
    if 'installation_history' not in _context:
        _context['installation_history'] = []
        # Stack the installed version, at the creation_date
        installed_version = _context['installed_version']
        __add_installation_version(_context, installed_version, 'INSTALLATION', int(_context['creation_time']), installer_version='')
        # Stack the current version if different
        current_version = _context['current_version']
        if current_version != installed_version:  # if there was an update in the past, set it, we just don't know when so fake the date
            __add_installation_version(_context, current_version, 'UPDATE', 0, installer_version='')
    
    installed_version = _context['installed_version']
    current_version = _context['current_version']
    if 'data_history' not in _context:
        _context['data_history'] = {}
    
    # This parts that already exist in old version but not in context
    _02_03_03_format_data_type = ['configuration', 'sla', 'metrology', 'user']
    for data_type in _02_03_03_format_data_type:
        if data_type not in _context['data_history']:
            _context['data_history'][data_type] = []
            # Stack the installed version, at the creation_date
            __add_data_version(data_type, _context, installed_version, 'INSTALLATION', int(_context['creation_time']))
            # Stack the current version if different
            if current_version != installed_version:  # if there was an update in the past, set it, we just don't know when so display UNKNOWN DATE
                __add_data_version(data_type, _context, current_version, 'UPDATE', 0)


def get_installation_history():
    __assert_context()
    context = __get_context()
    return context['installation_history']


def get_data_history(data_type):
    __assert_context()
    if data_type not in POSSIBLE_DATA:
        raise ValueError("Invalid data type")
    context = __get_context()
    return context['data_history'].get(data_type, [])


def sanatize_context_file():
    check_root()
    __assert_context()
    context = __get_context()  # already sanatize it
    # so just rewrite it, and we are clean
    __write_context(context)


def __get_context():
    with open(CONTEXT_PATH, 'r') as f:
        context = json.loads(f.read())
    
    # When we load the context file, always be sure that it was sanatized from old versions structure
    __sanatize_context_data(context)
    
    return context


def __write_context(context):
    with open('/var/lib/shinken/context.json.tmp', 'w') as f:
        f.write(json.dumps(context, indent=4))
    shutil.move('/var/lib/shinken/context.json.tmp', '/var/lib/shinken/context.json')


# If context file is missing, exit
def __assert_context():
    if not os.path.exists('/var/lib/shinken/context.json'):
        print("ERROR: Missing context file, you cannot use this command currently")
        sys.exit(2)


# For some modules operation and cache, we need to have the full file hash, so if something did change
# like an install, or even a patch install, to know if the cache need to be reset (like for webui)
def get_context_hash():
    with open(CONTEXT_PATH, 'r') as f:
        context_hash = md5(f.read()).hexdigest()
        return context_hash


def set_local_daemon(daemon, b):
    __assert_context()
    context = __get_context()
    # Now really set the value for our daemon type
    context['daemons'][daemon] = b
    __write_context(context)


def set_local_addon(addon, b):
    __assert_context()
    context = __get_context()
    context['addons'][addon] = b
    __write_context(context)


def set_local_daemon_instance(daemon_type, daemon_id, enabled):
    __assert_context()
    context = __get_context()
    # Now really set the value for our daemon type
    context['local_instances'][daemon_type][str(daemon_id)] = enabled
    
    __write_context(context)


def remove_local_daemon_instance(daemon_type, daemon_id):
    __assert_context()
    context = __get_context()
    # Now really set the value for our daemon type
    try:
        del context['local_instances'][daemon_type][str(daemon_id)]
    except Exception:
        print("WARNING: the instance %d was already removed from the context file")
        return
    
    __write_context(context)


def check_root():
    if os.getuid() != 0:
        print("You must be logged as root to launch this tool")
        sys.exit(2)


def is_local_daemon_instance_exists(daemon_type, daemon_id):
    __assert_context()
    context = __get_context()
    return str(daemon_id) in context['local_instances'][daemon_type]


def get_local_daemon_configuration_file_path(daemon_type, daemon_id):
    ini_suffix = ''
    if daemon_id != '0':
        ini_suffix = '-%s' % daemon_id
    cfg_p = '/etc/shinken/daemons/%sd%s.ini' % (daemon_type, ini_suffix)
    # now exceptions...
    if daemon_type == 'arbiter':
        cfg_p = '/etc/shinken/shinken.cfg'
    if daemon_type == 'synchronizer':
        cfg_p = '/etc/shinken/synchronizer.cfg'
    
    return cfg_p


def parse_ini_file(ini_path):
    config = ConfigParser.ConfigParser()
    
    # Beware: ini file do not like space before comments and such things
    data = StringIO('\n'.join(line.strip() for line in open(ini_path)))
    
    config.readfp(data)
    if config._sections == {}:
        return None
    
    r = {}
    try:
        for (key, value) in config.items('daemon'):
            r[key] = value
    except ConfigParser.InterpolationMissingOptionError:
        return None
    return r


def get_instance_name(daemon_type, daemon_id):
    if daemon_type in ADDITIONNAL_DAEMONS:
        return daemon_type
    
    # Grok the pid file path inside the configuration file
    cfg_p = get_local_daemon_configuration_file_path(daemon_type, daemon_id)
    # HOOK: for arbiter and synchronizer, it's somewhere else
    if daemon_type == 'arbiter':
        cfg_p = '/etc/shinken/daemons/arbiterd.ini'
    if daemon_type == 'synchronizer':
        cfg_p = '/etc/shinken/daemons/synchronizerd.ini'
    
    if cfg_p.endswith('.ini'):  # look at the name parameter
        if not os.path.exists(cfg_p):
            return 'unnamed-%s' % daemon_type
        r = parse_ini_file(cfg_p)
        if r is None or 'name' not in r:
            return 'unnamed-%s' % daemon_type
        return r['name']
    if cfg_p.endswith('.cfg'):  # cfg files are not managed currently, and arbiter+synchronizer are manually managed, so how did you get there??
        return daemon_type
    
    return '(unknown-type)'


def _actions_on_daemon(action, daemon_type, daemon_id=None):
    id_param = ""
    if daemon_id:
        id_param = " --id %s" % daemon_id
    cmd = "/etc/init.d/shinken-%s %s %s" % (daemon_type, id_param, action)
    p = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
    out = p.stdout.read()
    err = p.stderr.read()
    if err:
        print(' \033[%dm ERROR (%s): %s\033[0m' % (ERROR_COLOR, daemon_type, err))


def stop_daemon(daemon_type, daemon_id=None):
    _actions_on_daemon('stop', daemon_type, daemon_id)


def start_daemon(daemon_type, daemon_id=None):
    _actions_on_daemon('start', daemon_type, daemon_id)


def add_update_version_to_context(update_version, installer_version):
    check_root()
    __assert_context()
    context = __get_context()
    # Now really set the value for our daemon type
    context['current_version'] = update_version
    context['current_installer_version'] = installer_version
    __add_installation_version(context, update_version, 'UPDATE', int(time.time()), installer_version)
    for data_type in POSSIBLE_DATA:
        __add_data_version(data_type, context, update_version, 'UPDATE', int(time.time()), installer_version=installer_version)
    __write_context(context)


def add_patch_to_context(installation_type, patch_types, patch_name):
    check_root()
    __assert_context()
    context = __get_context()
    update_version = context.get('current_version')
    
    list_of_patch_types = [patch_type.strip() for patch_type in patch_types.split(',')]
    
    for patch_type in list_of_patch_types:
        __add_patch_install(patch_type, context, update_version, installation_type, int(time.time()), patch_name=patch_name)
    
    __write_context(context)


# SHINKEN-RESTORE --configuration: we are importing a full new configuration, so we must also load it's data_history->configuration
# in this function, we have a context file in the backup, so >= 2.5.0
# * Backup from >= 2.5.0 => data_history->configuration is used, just erase our own entry with it
def do_import_data_history_from_file(data_type, import_data_history_from_file, backup_path='', comment=''):
    if not os.path.exists(import_data_history_from_file):
        raise OSError("The context file %s do not exist." % import_data_history_from_file)
    
    if not data_type in POSSIBLE_DATA:
        raise ValueError("The available data types are %s" % ", ".join(POSSIBLE_DATA))
    
    # Load the distant context file
    with open(import_data_history_from_file, 'r') as f:
        imported_context = json.loads(f.read())
    
    __assert_context()
    context = __get_context()
    
    # Erase our own, data are FULL new, so drop the old values
    context['data_history'][data_type] = imported_context['data_history'].get(data_type, [])
    
    # Add a entry so we did RESTORE it in our version
    __add_data_version(data_type, context, context['current_version'], 'RESTORE', int(time.time()), backup_path=backup_path, comment=comment)
    
    # We did update, write it down
    __write_context(context)


# SHINKEN-RESTORE --configuration: we are importing a full new configuration, so we must also load it's data_history->configuration
# in this function, we have DO NOT have a context file in the backup, so < 2.5.0 but we have a version in the backup (so > 2.03.03-U01-RC240)
# we must fake our data with this version
def do_import_data_history_from_version(import_data_history_from_version, backup_path=''):
    __assert_context()
    context = __get_context()
    
    # Erase our own, data are FULL new, so drop the old values
    for data_type in POSSIBLE_DATA:
        context['data_history'][data_type] = []
        # Add a entry from the version we have
        __add_data_version(data_type, context, import_data_history_from_version, 'INSTALLATION', int(_context['creation_time']))
        # and the current RESTORE action
        __add_data_version(data_type, context, context['current_version'], 'RESTORE', int(time.time()), backup_path=backup_path)
    
    # We did update, write it down
    __write_context(context)


# SHINKEN-RESTORE --configuration: we are importing a full new configuration, so we must also load it's data_history->configuration
# ok this time we are in a deep shit, we do not even have a backup version AT ALL (2.03.03-U01). Do not try to guess, sorry
def do_import_data_history_no_version(backup_path=''):
    __assert_context()
    context = __get_context()
    # Erase our own, data are FULL new, so drop the old values
    for data_type in POSSIBLE_DATA:
        context['data_history'][data_type] = []
        # just add the current RESTORE action
        __add_data_version(data_type, context, context['current_version'], 'RESTORE', int(time.time()), backup_path=backup_path)
    # We did update, write it down
    __write_context(context)


# SHINKEN-RESTORE & satatize-data  we save the executed sanatize pass
def do_import_data_history_sanatize(sanatize_name, data_type='configuration'):
    __assert_context()
    context = __get_context()
    
    # add the entry with the sanatize name
    __add_data_version(data_type, context, context['current_version'], 'SANATIZE', int(time.time()), sanatize_name=sanatize_name)
    
    # We did update, write it down
    __write_context(context)


def merge_from_key(key, file):
    __assert_context()
    context = __get_context()
    if key not in context:
        print("ERROR: while merging %s into context.json, the key %s does not exist in context.json" % (file, key))
        sys.exit(2)
    
    if not os.path.exists(file):
        print("ERROR: while merging %s into context.json, the file %s does not exist in context.json" % (file, file))
        sys.exit(2)
    try:
        data_to_merge = json.load(open(file))
        context[key].update(data_to_merge)
        __write_context(context)
    except ValueError:
        print("ERROR: while merging %s into context.json, the file %s does not contain valid JSON" % (file, file))
        sys.exit(2)


def disable_addons():
    __assert_context()
    context = __get_context()
    __sanatize_context_data(context)
    for addon in context['addons']:
        context['addons'][addon] = False
    __write_context(context)


def nagvis_update_uuid(old_uuid, new_uuid):
    __assert_context()
    context = __get_context()
    __sanatize_context_data(context)
    
    if old_uuid == new_uuid:
        return
    
    try:
        basedir = "/etc/shinken/external/nagvis/etc/conf.d/"
        os.unlink("%s/rotation_shinken_%s.ini.php" % (basedir, old_uuid))
        for file in os.listdir(basedir):
            if old_uuid in file:
                os.unlink("%s/%s" % (basedir, file))
        os.unlink("%s/shinken_architecture-%s.cfg" % (basedir, old_uuid))
        os.unlink("%s/shinken_global-%s.cfg" % (basedir, old_uuid))
    except OSError:
        pass
    
    try:
        with open("/var/lib/shinken/architecture_export_received.json") as fp:
            exports = json.load(fp)
    except:
        exports = {}
    if old_uuid in exports:
        with open("/var/lib/shinken/architecture_export_received.json", "w") as fp:
            del exports[old_uuid]
            json.dump(exports, fp)
    
    for addon in ['nagvis', 'nagvis-shinken-architecture']:
        __add_addons_version(addon, context, 'UPDATE', int(time.time()), old_uuid, new_uuid)
    __write_context(context)
