#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

import os
import socket

from shinken.ipc.shinken_queue.shinken_queue_lib import ShinkenThreadedSocket, socket_pair
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Number, Optional


class ShinkenQueue(object):
    def __init__(self, in_father_name=None, in_child_name=None):
        # noinspection SpellCheckingInspection
        if hasattr(socket, u'socketpair'):
            s1, s2 = socket.socketpair()
        else:
            s1, s2 = socket_pair()
        self.socket_for_parent_process = ShinkenThreadedSocket(s1, in_father_name)
        self.socket_for_child_process = ShinkenThreadedSocket(s2, in_child_name)
        self.parent_pid = os.getpid()
    
    
    def get_queues_size(self):
        # just for stats, to be reviewed  # type: () -> Tuple[int,int,int,int]
        threaded_socket = self.socket_for_parent_process if os.getpid() == self.parent_pid else self.socket_for_child_process
        return threaded_socket.get_queues_size()
    
    
    def recv(self):
        # type: () -> bytes
        return self.socket_for_parent_process.recv() if os.getpid() == self.parent_pid else self.socket_for_child_process.recv()
    
    
    def poll(self, timeout=0.0):
        # type: (Optional[Number]) -> bool
        return self.socket_for_parent_process.poll(timeout) if os.getpid() == self.parent_pid else self.socket_for_child_process.poll(timeout)
    
    
    def write_poll(self, timeout=0.0):
        # type: (Optional[Number]) -> bool
        return self.socket_for_parent_process.write_poll(timeout) if os.getpid() == self.parent_pid else self.socket_for_child_process.write_poll(timeout)
    
    
    def send(self, data):
        # type: (bytes) -> int
        return self.socket_for_parent_process.send(data) if os.getpid() == self.parent_pid else self.socket_for_child_process.send(data)
    
    
    def close(self):
        # type: () -> None
        self.socket_for_parent_process.close()
        self.socket_for_child_process.close()
