#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import codecs
import os
from ctypes import c_int
from logging.handlers import TimedRotatingFileHandler
from multiprocessing import RLock, Value

import itertools

import six

# Typing
from shinken.ipc.share_item import ShareItem
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, List, NoReturn, Dict, Union

try:
    from pwd import getpwnam  # noqa => all usages are only for unix
    from grp import getgrnam  # noqa
except ImportError:
    pass
    # This import is needed only to change owner on linux, not used on windows

# Windows is special, as usual, because we will have to set filehandler as non inheritable
if os.name == 'nt':
    import msvcrt
    import win32api  # noqa => no need for requirements, only windows
    import win32con

####################################
# END MONKEY PATCH
####################################

import os
import time
import logging
import pprint
import sys
import traceback
from difflib import Differ
from io import StringIO
from logging import Formatter, StreamHandler, DEBUG, WARNING, INFO, CRITICAL, ERROR

from datetime import datetime
from shinken.misc.type_hint import NoReturn, List, Tuple

try:
    from shinken.misc.termcolor import cprint
except (SyntaxError, ImportError) as exp:
    # Outch can't import a cprint, do a simple print
    def cprint(s, color='', end=''):  # noqa => we know about unused parameters
        print(s)

human_timestamp_log = True  # by default, have human format date

human_date_format = '%Y-%m-%d %H:%M:%S'  # [2017-03-02 13:38:15]

# noinspection PyTypeChecker
defaultFormatter_u = Formatter(u'[%(created)i] %(levelname)-7s: %(message)s')
# noinspection PyTypeChecker
defaultFormatter_named_u = Formatter(u'[%(created)i] %(levelname)-7s: [ %(name)s ] %(message)s')
# noinspection PyTypeChecker
humanFormatter_u = Formatter(u'[%(asctime)s] %(levelname)-7s: %(message)s', human_date_format)
# noinspection PyTypeChecker
humanFormatter_named_u = Formatter(u'[%(asctime)s] %(levelname)-7s: [ %(name)s ] %(message)s', human_date_format)

defaultFormatter = Formatter('[%(created)i] %(levelname)-7s: %(message)s')
defaultFormatter_named = Formatter('[%(created)i] %(levelname)-7s: [ %(name)s ] %(message)s')
humanFormatter = Formatter('[%(asctime)s] %(levelname)-7s: %(message)s', human_date_format)
humanFormatter_named = Formatter('[%(asctime)s] %(levelname)-7s: [ %(name)s ] %(message)s', human_date_format)

COLORS = {'DEBUG': 'cyan', 'INFO': 'magenta', 'WARNING': 'yellow', 'CRITICAL': 'red', 'ERROR': 'red'}

PERF_LOG_LEVEL = DEBUG
PERF_LOG_MIN_TIME = 0.1
PERF_LOG_WARN_TIME = 1

LOG_CHAPTER_SIZE = 16
LOG_SECTION_SIZE = 22

LONG_FLUSH_LIMIT = 1  # if flush is longer than 1s, warn about it
LONG_FLUSH_KEEP_DURATION = 60  # keep the long flush during a 1min delay
DEFAULT_LONG_FLUSH_STATS = {'is_too_long': False, 'write_duration': 0.0, 'log_path': ''}


def get_chapter_string(chapter):
    return format_part(chapter, LOG_CHAPTER_SIZE)


def get_section_string(chapter):
    return format_part(chapter, LOG_SECTION_SIZE)


def format_part(section_name, section_size=None):
    if section_size is None:
        s_format = '[ %s ]'
    else:
        s_format = '[ %%-%ds ]' % section_size
    return s_format % section_name


class FixedTimedRotatingFileHandler(TimedRotatingFileHandler):
    baseFilename = ''
    suffix = ''
    backupCount = 0
    mode = ''
    encoding = ''
    interval = 0
    utc = None  # type: Optional[bool]
    when = ''
    
    
    def __init__(self, *args, **kwargs):
        # NOTE: cannot user super() as python 2.6 Handler is not new-style class
        TimedRotatingFileHandler.__init__(self, *args, **kwargs)
        self._current_pid = os.getpid()
        ####################################
        # WARNING !! MONKEY PATCH for TimedRotatingFileHandler
        # TimedRotatingFileHandler doesn't support multiprocessing
        # SEE SEF-3518
        ####################################
        self.roll_over_version = Value(c_int, 0)
        self._rollover_lock = RLock()
        self.delay = 0
        self.my_roll_over_version = 0
        
        self._last_big_flush_date = 0
        self._last_big_flush = 0.0
    
    
    def get_long_flush_stats(self):
        now = int(time.time())
        if self._last_big_flush_date < now - LONG_FLUSH_KEEP_DURATION:
            return DEFAULT_LONG_FLUSH_STATS
        return {'is_too_long': True, 'write_duration': self._last_big_flush, 'log_path': self._get_current_file_name()}
    
    
    def _get_current_file_name(self):
        if os.name == 'nt':
            return "%s.%s" % (self.baseFilename, time.strftime("%Y-%m-%d", time.localtime(time.time())))
        else:
            return self.baseFilename
    
    
    def flush(self):
        before = time.time()
        # NOTE: cannot user super() as python 2.6 Handler is not new-style class
        TimedRotatingFileHandler.flush(self)
        flush_time = time.time() - before
        if flush_time < LONG_FLUSH_LIMIT:
            return
        now = int(time.time())
        
        # Ok a big one, is the current still ok?
        if self._last_big_flush_date < now - LONG_FLUSH_KEEP_DURATION:
            self._last_big_flush_date = 0
            self._last_big_flush = 0.0
        
        if flush_time > self._last_big_flush:
            # older or lower, update it
            self._last_big_flush_date = now
            self._last_big_flush = flush_time
            _prefix = '[%s] WARNING : [ LOGGER ]' % time.strftime(human_date_format)
            
            log_msg = '[ WRITING ] The log writes time is very high (%.2fs). Please look at your log disk performance.' % flush_time
            for msg in ('', '-' * 100, log_msg, '-' * 100, ''):
                log_line = '%s %s\n' % (_prefix, msg)
                self.stream.write(log_line)
                cprint(log_line)  # also log in debug
    
    
    def _open(self):
        """
        Open the current base file with the (original) mode and encoding.
        Return the resulting stream.
        """
        current_file_name = self._get_current_file_name()
        if getattr(self, 'encoding', None) is None:  # NOTE: if use inside apache, self.encoding is missing
            stream = open(current_file_name, self.mode)
        else:
            stream = codecs.open(current_file_name, self.mode, self.encoding)
        
        if os.name != 'nt':
            # We need to change the current_file owner :
            os.chown(current_file_name, getpwnam('shinken').pw_uid, getgrnam('shinken').gr_gid)
        
        if os.name == 'nt':
            # NOTE: on windows we do not want such file to be inherit if we are fork() because the logger object
            # won't be available
            fd = stream.fileno()  # The log handler file descriptor
            fh = msvcrt.get_osfhandle(fd)  # The actual windows handler
            win32api.SetHandleInformation(fh, win32con.HANDLE_FLAG_INHERIT, 0)  # Disable inheritance
        return stream
    
    
    def doRollover(self):
        with self._rollover_lock:
            if self.stream:
                self.stream.close()
                self.stream = None
            # get the time that this sequence started at and make it a TimeTuple
            currentTime = int(time.time())  # noqa => ok with upercase
            dstNow = time.localtime(currentTime)[-1]  # noqa => ok with upercase
            t = self.rolloverAt - self.interval
            if self.utc:
                timeTuple = time.gmtime(t)  # noqa => ok with upercase
            else:
                timeTuple = time.localtime(t)  # noqa => ok with upercase
                dstThen = timeTuple[-1]  # noqa => ok with upercase
                if dstNow != dstThen:
                    if dstNow:
                        addend = 3600
                    else:
                        addend = -3600
                    timeTuple = time.localtime(t + addend)  # noqa => ok with upercase
            if self.my_roll_over_version >= self.roll_over_version.value:
                if os.name != 'nt':
                    dfn = self.baseFilename + "." + time.strftime(self.suffix, timeTuple)
                    if os.path.exists(dfn):
                        os.remove(dfn)
                    # Issue 18940: A file may not have been created if delay is True.
                    current_file_name = self._get_current_file_name()
                    if os.path.exists(current_file_name):
                        os.rename(current_file_name, dfn)
                if self.backupCount > 0:
                    for s in self.getFilesToDelete():  # noqa => ok with unknown method
                        os.remove(s)
                self.roll_over_version.value += 1
            if not self.delay:
                self.stream = self._open()  # noqa => ok with define outside __init__
            newRolloverAt = self.computeRollover(currentTime)  # noqa => ok with upercase
            while newRolloverAt <= currentTime:
                newRolloverAt = newRolloverAt + self.interval  # noqa => ok with upercase
            # If DST changes and midnight or weekly rollover, adjust for this.
            if (self.when == 'MIDNIGHT' or self.when.startswith('W')) and not self.utc:
                dstAtRollover = time.localtime(newRolloverAt)[-1]  # noqa => ok with upercase
                if dstNow != dstAtRollover:
                    if not dstNow:  # DST kicks in before next rollover, so we need to deduct an hour
                        addend = -3600
                    else:  # DST bows out before next rollover, so we need to add an hour
                        addend = 3600
                    newRolloverAt += addend
            self.rolloverAt = newRolloverAt  # noqa => ok with define outside init
            self.my_roll_over_version += 1
    
    
    # Important: we do manage the case that the handler is called from another PID after a fork()
    # and then recreate the lock because if lock, won't be ever release from
    # the other pid-thread
    def handle(self, record):
        cur_pid = os.getpid()
        if cur_pid != self._current_pid:
            self._current_pid = cur_pid  # before createLock, to reduce the risk that another thread will create lock too
            self.createLock()
        
        # NOTE: we MUST do the emit ourselve because we should catch the .release() if another thread did
        #       createLock(), so catch RuntimeError on the release
        # NOTE2: I prefer NOT try/except RuntimeError the whole super(emit) but only what is need
        rv = self.filter(record)
        if rv:
            self.acquire()
            try:
                self.emit(record)
            finally:
                try:
                    self.release()
                except RuntimeError:  # oups, another thread did createLock
                    pass
        return rv


class ColorStreamHandler(StreamHandler):
    def __init__(self, *args, **kwargs):
        # NOTE: cannot user super() as python 2.6 Handler is not new-style class
        StreamHandler.__init__(self, *args, **kwargs)
        self._current_pid = os.getpid()
    
    
    def _set_local_formatter(self, encoding='unicode'):
        if encoding == 'unicode':
            if getattr(self, 'name', None) is not None:
                self.setFormatter(human_timestamp_log and humanFormatter_named_u or defaultFormatter_named_u)
            else:
                self.setFormatter(human_timestamp_log and humanFormatter_u or defaultFormatter_u)
        else:
            if getattr(self, 'name', None) is not None:
                self.setFormatter(human_timestamp_log and humanFormatter_named or defaultFormatter_named)
            else:
                self.setFormatter(human_timestamp_log and humanFormatter or defaultFormatter)
    
    
    # Important: we do manage the case that the handler is called from another PID after a fork()
    # and then recreate the lock because if lock, won't be ever release from
    # the other pid-thread
    def handle(self, record):
        cur_pid = os.getpid()
        if cur_pid != self._current_pid:
            self._current_pid = cur_pid  # before createLock, to reduce the risk that another thread will create lock too
            self.createLock()
        
        # NOTE: we MUST do the emit ourselve because we should catch the .release() if another thread did
        #       createLock(), so catch RuntimeError on the release
        # NOTE2: I prefer NOT try/except RuntimeError the whole super(emit) but only what is need
        rv = self.filter(record)
        if rv:
            self.acquire()
            try:
                self.emit(record)
            finally:
                try:
                    self.release()
                except RuntimeError:  # oups, another thread did createLock
                    pass
        return rv
    
    
    def emit(self, record):
        msg = ''
        try:
            msg = self.format(record)
        except UnicodeDecodeError:
            self._set_local_formatter(encoding='str')
            msg = self.format(record)
        except UnicodeEncodeError:
            self._set_local_formatter(encoding='unicode')
        
        encoded_msg = u''
        try:
            if six.PY2 and isinstance(msg, unicode):
                encoded_msg = msg.encode('utf8', 'ignore')
            else:
                encoded_msg = msg
        except UnicodeEncodeError:
            encoded_msg = msg
        except Exception:  # noqa => we need ALL exceptions here
            self.handleError(record)
        finally:
            try:
                cprint(encoded_msg, COLORS.get(record.levelname, 'white'))
            except UnicodeDecodeError:  # Match cases where print is not happy, last try
                encoded_msg = encoded_msg.decode('utf8', 'ignore')
                cprint(encoded_msg, COLORS.get(record.levelname, 'white'))


class Log(logging.Logger):
    """
    Shinken logger class, wrapping access to Python logging standard library.
    See : https://docs.python.org/2/howto/logging.html#logging-flow for more detail about
    how log are handled
    """
    
    
    def __init__(self, name="Shinken", level=DEBUG, do_log_parameters=False):
        name = self._get_name_formated(name)  # NOTE: do not self.set_name() because it will raise a No handlers could be found for logger "Shinken        "
        logging.Logger.__init__(self, name, level)
        self.do_log_parameters = do_log_parameters
        self.set_human_format(on=True)  # activate by default human format
        self.log_file_path = None
        self.loggers_info = None
        self._my_daemon = None
    
    
    def is_debug(self):
        return self.level == DEBUG
    
    
    def setLevel(self, level):
        """
        Set level of logger and handlers.
        The logger need the lowest level (see link above)
        """
        if not isinstance(level, int):
            level = getattr(logging, level, None)
            if not level or not isinstance(level, int):
                raise TypeError('log level must be an integer')
        
        self.level = level
        # Only set level to file and/or console handler
        for handler in self.handlers:
            handler.setLevel(level)
    
    
    @staticmethod
    def _get_name_formated(name):
        return '%-15s' % name
    
    
    def set_name(self, name, log_this_change=True):
        name = self._get_name_formated(name)
        old_name = self.name
        self.name = name
        if old_name != self.name and log_this_change:
            self.info('Changing logger name: %s => %s' % (old_name, self.name))
    
    
    def load_obj(self, _object, _name=None):
        """
        We load the object where we will put log broks with the 'add' method
        """
        self._my_daemon = _object
        
        if _name is not None:
            self.set_name(_name)
        if self.name is not None:
            # We need to se the name format to all other handlers
            for handler in self.handlers:
                handler.setFormatter(humanFormatter_named)
        
        # Be sure to keep human format choice
        logger.set_human_format(on=human_timestamp_log)
    
    
    def register_local_log(self, path, level=None):
        """
        The shinken logging wrapper can write to a local file if needed
        and return the file descriptor so we can avoid to close it.

        Add logging to a local log-file.

        The file will be rotated once a day
        """
        
        # Todo : Create a config var for backup count
        self.log_file_path = path
        self.loggers_info = LoggersInfo(key_name=self.log_file_path, reinit=True)
        handler = FixedTimedRotatingFileHandler(path, 'midnight', backupCount=5)
        
        if level is not None:
            handler.setLevel(level)
        if self.name is not None:
            handler.setFormatter(humanFormatter_named)
        else:
            handler.setFormatter(humanFormatter)
        self.addHandler(handler)
        
        # Todo : Do we need this now we use logging?
        return handler.stream.fileno()
    
    
    def get_stats(self):
        for handler in self.handlers:
            if hasattr(handler, 'get_long_flush_stats'):
                return handler.get_long_flush_stats()
        # Maybe no handler was present, if so return default with no errors
        return DEFAULT_LONG_FLUSH_STATS
    
    
    def log_nagios_message(self, level, message):
        # NOTE lazy load the BRok to avoid recursive import
        from brok import Brok
        if self._my_daemon is None:
            return
        msg = '[%d] %7s: [%s] %s\n' % (time.time(), level.upper(), logger.name, message)
        brok = Brok('log_monitoring', {'log': msg})
        self._my_daemon.add_Brok(brok)
        # Also log into the scheduler log as classic log
        self.info(message)
    
    
    def get_log_file_path(self):
        return self.log_file_path
    
    
    def set_human_format(self, on=True):
        """
        Set the output as human format.

        If the optional parameter `on` is False, the timestamps format
        will be reset to the default format.
        """
        global human_timestamp_log
        human_timestamp_log = bool(on)
        
        # Apply/Remove the human format to all handlers
        for handler in self.handlers:
            if self.name is not None:
                handler.setFormatter(human_timestamp_log and humanFormatter_named or defaultFormatter_named)
            else:
                handler.setFormatter(human_timestamp_log and humanFormatter or defaultFormatter)
    
    
    def print_stack(self, prefix='', level=logging.ERROR):
        formatted_lines = traceback.format_exc().splitlines()
        first_line = formatted_lines[0]
        if first_line == 'None':
            for line in traceback.format_stack():
                self.log(level, "%s%s" % (prefix, line))
        else:
            self.log(level, "%sERROR stack : %s" % (prefix, first_line))
            for line in formatted_lines[1:]:
                self.log(level, "%s%s" % (prefix, line))
    
    
    def log_perf(self, start_time, tag_name, msg, min_time=PERF_LOG_MIN_TIME, warn_time=PERF_LOG_WARN_TIME, info_time=-1, prefix=u''):
        time_spend = time.time() - start_time
        if time_spend > min_time:
            from_str = tag_name if isinstance(tag_name, basestring) else tag_name.__class__.__name__
            log_level = DEBUG
            if info_time != -1 and time_spend > info_time:
                log_level = INFO
            if time_spend > warn_time:
                log_level = WARNING
            if from_str:
                from_str = u'[ %s ] ' % from_str
            else:
                from_str = u''
            if prefix:
                prefix = u'%s ' % prefix
            else:
                prefix = u''
            self.log(log_level, '%s%s[ PERF ] [ %.3fs ] %s' % (prefix, from_str, time_spend, msg))
    
    
    def log_parameters(self, func):
        """ Use this decorator on any function or method to log its parameters """
        
        if self.do_log_parameters:
            def wrapper(*args, **kwargs):
                self.debug("Calling %s with the following parameters :" % func.__name__)
                
                if args is not None:
                    self.debug("| args:")
                    for index, arg in enumerate(args):
                        self.debug("|    arg %s => $%s$" % (index, pprint.pformat(arg)))
                
                if kwargs is not None and kwargs != {}:
                    self.debug("| kwargs:")
                    for (k, v) in kwargs:
                        self.debug("|    %s => $%s$" % (k, pprint.pformat(v)))
                
                return func(*args, **kwargs)
            
            
            return wrapper
        else:
            return func
    
    
    def diff(self, item1, item2):
        """ Returns diff between 2 items in the standard UNIX diff format
        :param item1: First item to compare
        :param item2: Second item to compare
        :return:
        """
        
        
        def colorize(s):
            if s.startswith('-'):
                return "\033[31m%s\033[0m" % s
            elif s.startswith('+'):
                return "\033[32m%s\033[0m" % s
            else:
                return s
        
        
        return "".join(map(colorize, list(Differ().compare(
            StringIO(unicode(pprint.pformat(item1) + "\n")).readlines(),
            StringIO(unicode(pprint.pformat(item2) + "\n")).readlines()))))
    
    
    def unregister_all(self):
        hdls = [hdl for hdl in self.handlers]
        for hld in hdls:
            self.removeHandler(hld)


# --- create the main logger ---
logging.setLoggerClass(Log)
# noinspection PyTypeChecker
logger = logging.getLogger('Shinken')  # type: Log

if hasattr(sys.stdout, 'isatty'):
    csh = ColorStreamHandler(sys.stdout)
    if logger.name is not None:
        csh.setFormatter(humanFormatter_named)
    else:
        csh.setFormatter(humanFormatter)
    logger.addHandler(csh)


def log_perf(start_time, tag_name, msg, log_level=PERF_LOG_LEVEL, min_time=PERF_LOG_MIN_TIME):
    time_spend = time.time() - start_time
    from_str = tag_name if isinstance(tag_name, basestring) else '%s-%s' % (tag_name.__class__.__name__, id(tag_name))
    if time_spend > min_time:
        logger.log(log_level, "[%s][perf][%.3f]s %s" % (from_str, time_spend, msg))


# Function use for old Nag compatibility. We need to send a different king of brok for this king of log
# We have another logger and use one for Shinken logs and another for monitoring data
def naglog_result(level, message, *args):  # noqa => we know aout *args
    logger.log_nagios_message(level, message)


class LoggersInfo(ShareItem):
    FULL_NAME_PREFIX = 'full_name_____'
    ENABLE_PREFIX = 'enable_____'
    
    
    def set_logger_full_name(self, loggers_info_id, full_name):
        # type: (str, unicode) -> NoReturn
        setattr(self, '%s%s' % (LoggersInfo.FULL_NAME_PREFIX, loggers_info_id), full_name)
    
    
    def set_logger_enable(self, loggers_info_id, enable):
        # type: (str, bool) -> None
        for key_name in self.get_all_attr_read_only().iterkeys():
            if key_name.startswith('%s%s' % (LoggersInfo.ENABLE_PREFIX, loggers_info_id)):
                setattr(self, key_name, enable)
        setattr(self, '%s%s' % (LoggersInfo.ENABLE_PREFIX, loggers_info_id), enable)
    
    
    def get_logger_enable(self, loggers_info_id):
        # type: (str) -> bool
        attr_name = u'%s%s' % (LoggersInfo.ENABLE_PREFIX, loggers_info_id)
        return getattr(self, attr_name.encode(u'ascii', u'ignore'), True)
    
    
    def get_all_logger_info(self):
        # type: () -> Dict[unicode, Union[str,bool]]
        info = {}
        for key_name, value in self.get_all_attr_read_only().iteritems():
            if key_name.startswith(LoggersInfo.FULL_NAME_PREFIX):
                loggers_info_id = key_name[len(LoggersInfo.FULL_NAME_PREFIX):]
                info[loggers_info_id] = info.get(loggers_info_id, {})
                info[loggers_info_id][u'name'] = value
                info[loggers_info_id][u'display_name'] = ' '.join(('[%s]' % j for j in value if j))
                info[loggers_info_id][u'id'] = loggers_info_id
            if key_name.startswith(LoggersInfo.ENABLE_PREFIX):
                loggers_info_id = key_name[len(LoggersInfo.ENABLE_PREFIX):]
                info[loggers_info_id] = info.get(loggers_info_id, {})
                info[loggers_info_id][u'enable'] = value
                info[loggers_info_id][u'id'] = loggers_info_id
        return info
    
    
    def clear(self):
        # type: () -> NoReturn
        self.clear_all_attr()


# loggers_info = LoggersInfo()


class PartLogger(object):
    
    def __init__(self, name, level=0, display_parts=None, full_name=None, part_name_size=LOG_CHAPTER_SIZE, register=True):
        # type: (unicode, int, List[Tuple[unicode, int]], Optional[List[unicode]], int,bool) -> NoReturn
        self.name = name
        self.full_name = [logger.name.strip(), name] if full_name is None else full_name[:] + [name]
        self.loggers_info_id = '__'.join([i.replace(' ', '_') for i in self.full_name])
        self.register = register
        self._global_register()
        self.level = level
        self.part_name_size = part_name_size
        self.sub_part_logger = {}
        
        self.display_name = format_part(self.name, part_name_size) if self.name else u''
        
        if display_parts:
            self.display_parts = display_parts[:]
            self.display_parts.append((self.display_name, part_name_size))
        else:
            self.display_parts = [(self.display_name, part_name_size)]
        
        self._update_part_string()
    
    
    def __getstate__(self):
        return self.__dict__.copy()
    
    
    def __setstate__(self, state):
        # Warning : if you add a property it will be missing in state dict. So you must add your property if missing in state
        self.__dict__.update(state)
        if not hasattr(self, 'register'):
            self.register = False
        self._global_register()
    
    
    def _can_global_register(self):
        if not self.register:
            return False
        # Only on linux we must be log as root or shinken to register the log
        if sys.platform.startswith('win'):
            return True
        
        user_uid = os.geteuid()
        if user_uid == 0:
            return True
        group_gid = os.getegid()
        
        if user_uid == getpwnam('shinken').pw_uid:
            return True
        if group_gid == getgrnam('shinken').gr_gid:
            return True
        return False
    
    
    def _global_register(self):
        loggers_info = logger.loggers_info
        if not self._can_global_register() or not loggers_info:
            return
        
        loggers_info.set_logger_full_name(self.loggers_info_id, self.full_name)
        loggers_info.set_logger_enable(self.loggers_info_id, True)
    
    
    def _update_part_string(self):
        self._parts_string = u' '.join([i[0] for i in self.display_parts if i[0]])
    
    
    def get_sub_part(self, sub_part, part_name_size=None, register=True):
        if sub_part in self.sub_part_logger:
            return self.sub_part_logger[sub_part]
        if part_name_size is None:
            part_name_size = len(sub_part)
        
        if part_name_size is None:
            part_name_size = len(sub_part)
        
        sub_part_log = PartLogger(sub_part, self.level + 1, self.display_parts, self.full_name, part_name_size, self.register and register)
        self.sub_part_logger[sub_part] = sub_part_log
        return sub_part_log
    
    
    # DEPRECATED : DO NOT USE ANYMORE
    def set_display_name(self, display_name):
        self.set_default_part(display_name)
    
    
    def set_default_part(self, default_part):
        def_part = self.display_parts[0]
        self.display_parts[0] = (format_part(default_part, def_part[1]), def_part[1])
        for sub_part_logger in self.sub_part_logger.itervalues():
            sub_part_logger.set_default_part(default_part)
        self._update_part_string()
    
    
    def debug(self, *args):
        self.log(DEBUG, *args)
    
    
    def info(self, *args):
        self.log(INFO, *args)
    
    
    def warning(self, *args):
        self.log(WARNING, *args)
    
    
    def error(self, *args):
        self.log(ERROR, *args)
    
    
    def critical(self, *args):
        self.log(CRITICAL, *args)
    
    
    def log(self, level, *args):
        loggers_info = logger.loggers_info
        if not logger.isEnabledFor(level) or (loggers_info and not loggers_info.get_logger_enable(self.loggers_info_id)):
            return
        
        message_index = len(args) - 1
        message = args[message_index]
        extra_part = args[:message_index]
        if extra_part:
            message_format = u' '.join(itertools.chain([self._parts_string], [format_part(sub_part) for sub_part in extra_part], [u'%s' % message]))
        else:
            message_format = u'%s %s' % (self._parts_string, message)
        logger.log(level, message_format)
    
    
    def print_stack(self, prefix='', level=logging.ERROR):
        logger.print_stack(prefix='%s %s' % (self._parts_string, prefix), level=level)
    
    
    @staticmethod
    def format_sla_date(sla_date):
        try:
            return datetime.strptime('%s %s 00:00:00' % sla_date, '%j %Y %H:%M:%S').strftime('%d-%m-%Y')
        except:
            return '%s' % sla_date
    
    
    @staticmethod
    def format_time(_time):
        try:
            return datetime.fromtimestamp(_time).strftime('%d-%m-%Y %H:%M:%S')
        except:
            return '%s' % _time
    
    
    @staticmethod
    def format_hours(_time):
        try:
            return datetime.fromtimestamp(_time).strftime('%H:%M:%S')
        except:
            return '%s' % _time
    
    
    @staticmethod
    def format_time_with_micro(_time):
        try:
            return datetime.fromtimestamp(_time).strftime('%d-%m-%Y %H:%M:%S:%f')
        except:
            return '%s' % _time
    
    
    @staticmethod
    def format_time_as_sla(_time):
        try:
            return datetime.fromtimestamp(_time).strftime('%j_%Y %H:%M:%S')
        except:
            return '%s' % _time
    
    
    @staticmethod
    def format_duration_in_sec(raw_time):
        return '%.3fs' % raw_time
    
    
    @staticmethod
    def format_duration(time_period, time_format=u's'):
        from shinkensolutions.shinken_time_helper import print_human_readable_period  # lazy loading to avoid circular import
        return print_human_readable_period(time_period, time_format)
    
    
    @staticmethod
    def format_chrono(raw_time):
        raw_time = time.time() - raw_time
        return PartLogger.format_duration(raw_time)
    
    
    @staticmethod
    def format_datetime(_datetime):
        # type: (datetime) -> unicode
        try:
            return _datetime.strftime(u'%H:%M:%S %d-%m-%Y (%Z)').decode(u'utf8', u'ignore')
        except:
            return u'%s' % _datetime
    
    
    @staticmethod
    def get_level():
        return logger.getEffectiveLevel()
    
    
    @staticmethod
    def set_level(level):
        return logger.setLevel(level)
    
    
    @staticmethod
    def set_human_format():
        return logger.set_human_format(True)
    
    
    @staticmethod
    def is_debug():
        return logger.getEffectiveLevel() == DEBUG
    
    
    def log_perf(self, start_time, tag_name, msg, min_time=PERF_LOG_MIN_TIME, warn_time=PERF_LOG_WARN_TIME, info_time=-1):
        logger.log_perf(start_time, tag_name, msg, min_time, warn_time, info_time, self._parts_string)
    
    
    def set_enable(self, _enable):
        # type: (bool) -> None
        loggers_info = logger.loggers_info
        if not loggers_info:
            return
        loggers_info.set_logger_enable(self.loggers_info_id, _enable)
    
    
    def is_enable(self):
        # type: () -> bool
        loggers_info = logger.loggers_info
        if not loggers_info:
            return True
        return loggers_info.get_logger_enable(self.loggers_info_id)


PART_INITIALISATION = u'INITIALISATION'
DEFAULT_LOG = u''
loggers = {DEFAULT_LOG: PartLogger(DEFAULT_LOG)}


# Create logger for a specific part if not already exists
class LoggerFactory(object):
    @classmethod
    def get_logger(cls, name=DEFAULT_LOG, part_name_size=LOG_CHAPTER_SIZE):
        # type: (unicode, int) -> PartLogger
        if not name:
            return loggers[DEFAULT_LOG]
        
        # find by name
        log = loggers.get(name, None)
        
        # dit not exist we make it
        if not log:
            log = PartLogger(name, part_name_size=part_name_size)
            loggers[name] = log
        
        return log
