# !/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.
import os

from shinken.misc.type_hint import TYPE_CHECKING

try:
    import pwd
    import grp
    from pwd import getpwnam
    from grp import getgrnam
    
    
    def get_cur_user():
        return pwd.getpwuid(os.getuid()).pw_name
    
    
    def get_cur_group():
        return grp.getgrgid(os.getgid()).gr_name
except ImportError as exp:  # Like in nt system or Android
    pwd = grp = None
    
    
    def get_cur_user():
        return "shinken"
    
    
    def get_cur_group():
        return "shinken"

if TYPE_CHECKING:
    from shinken.log import PartLogger


def verify_if_shinken_own_file_or_directory(path, logger=None):
    # type: (unicode, PartLogger) -> bool
    if pwd and os.getuid() != 0:
        # Non-root user cannot create file with different owner, no need to crash if write operation is possible
        return True
    
    user_owner = pwd.getpwuid(os.stat(path).st_uid).pw_name if pwd else u'shinken'
    group_owner = grp.getgrgid(os.stat(path).st_gid).gr_name if pwd else u'shinken'
    
    result = True
    if user_owner != u'shinken':
        if logger:
            logger.error(u'[ %s ] user owner is not [ shinken ] but [ %s ]' % (path, user_owner))
        result = False
    
    if group_owner != u'shinken':
        if logger:
            logger.error(u'[ %s ] group owner is not [ shinken ] but [ %s ]' % (path, group_owner))
        result = False
    
    return result


def set_shinken_owner_on_file_or_directory(path, logger=None):
    # type: (unicode, PartLogger) -> None
    try:
        if pwd and grp:
            change_owner(path, u'shinken', u'shinken')
    except Exception as e:
        if logger:
            logger.error(u'Can\'t give ownership to shinken:shinken on [ %s ]' % path)
        raise e


def change_owner(path, user, group):
    if pwd and grp and os.getuid() == 0:
        os.chown(path, pwd.getpwnam(user).pw_uid, grp.getgrnam(group).gr_gid)
