#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import codecs
import os
import shutil
import time

from shinken.misc.file_owner_utils import set_shinken_owner_on_file_or_directory
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.locking.shinken_locking.shinken_interprocess_rlock import ShinkenInterProcessRLock

try:
    import pwd
    import grp
    from pwd import getpwnam
    from grp import getgrnam
    
    
    def get_cur_user():
        return pwd.getpwuid(os.getuid()).pw_name
    
    
    def get_cur_group():
        return grp.getgrgid(os.getgid()).gr_name
except ImportError as exp:  # Like in nt system or Android
    pwd = grp = None
    
    
    def get_cur_user():
        return "shinken"
    
    
    def get_cur_group():
        return "shinken"

if TYPE_CHECKING:
    from shinken.log import PartLogger


class ShinkenFileError(IOError):
    def __init__(self, message):
        self.message = message


class ShinkenAccessRightsError(ShinkenFileError):
    pass


class ShinkenFileNotFoundError(ShinkenFileError):
    pass


def make_file_hidden(file_path):
    # type: (unicode) -> unicode
    file_dir = os.path.dirname(file_path)
    # The « file_path » contains a complete path with at least ont directory
    if file_dir:
        return os.path.join(file_dir, u'.%s' % os.path.basename(file_path))
    # The « file_path » is a simple file name
    return u'.%s' % file_path


def safe_write_file(file_path, to_write):
    # type: (unicode, unicode) -> None
    tmp_path = make_file_hidden(u'%s.tmp' % file_path)
    
    # To write binary files, encoding should be « None »
    with codecs.open(tmp_path, 'w', 'utf8') as encoded_file:
        encoded_file.write(to_write)
        encoded_file.flush()
        os.fsync(encoded_file.fileno())
    
    shutil.move(tmp_path, file_path)


def safe_write_binary_file_and_force_mtime(file_path, to_write):
    # type: (unicode, str) -> None
    tmp_path = make_file_hidden(u'%s.tmp' % file_path)
    
    # To write binary files, encoding should be « None »
    with open(tmp_path, 'wb') as binary_file:
        binary_file.write(to_write)
        binary_file.flush()
        os.fsync(binary_file.fileno())
    
    shutil.move(tmp_path, file_path)
    now = time.time()
    # Forcing the modification time to now, this was benched and we lost ~4% process time for 1M executions
    # The loss is acceptable because without this, the share_item would have update issues when really fast accesses
    os.utime(file_path, (now, now))


def write_file_with_lock(file_path, to_write):
    # type: (unicode, unicode) -> None
    # This will overwrite file content
    with ShinkenInterProcessRLock(file_path) as locked_file:
        locked_file.write(u'%s' % to_write)
        locked_file.flush()
        os.fsync(locked_file.fileno())


def get_lock_file_name(file_path):
    # type: (unicode) -> unicode
    return os.path.join(os.path.dirname(file_path), u'.%s.lock' % os.path.basename(file_path))


def locked_safe_write_file(file_path, to_write, force_shinken_rights=False):
    # type: (unicode, unicode, bool) -> None
    lock_file = get_lock_file_name(file_path)
    with ShinkenInterProcessRLock(lock_file):
        write_file_set_owner(file_path, to_write, force_shinken_rights)


def write_file_set_owner(file_path, to_write, set_owner=False):
    tmp_file_path = u'%s.tmp' % file_path
    with open(tmp_file_path, u'w+b') as tmp_file:
        tmp_file.write(to_write)
        tmp_file.flush()
        os.fsync(tmp_file.fileno())
    shutil.move(tmp_file_path, file_path)
    
    if set_owner:
        set_shinken_owner_on_file_or_directory(file_path)


def check_file_or_directory_access(target_path, logger=None, check_read=True, check_write=True, check_exists=True):
    # type: (unicode, PartLogger, bool, bool, bool) -> None
    if check_exists and not os.path.exists(target_path):
        if logger:
            logger.debug(u'The path does not exist [ %s ] was it move ?' % target_path)
        raise ShinkenFileNotFoundError(u'The path does not exist : %s' % target_path)
    
    has_access_issue = False
    if check_read and not os.access(target_path, os.R_OK):
        if logger:
            logger.error(u'No read access : %s' % target_path)
        has_access_issue = True
    
    if check_write and not os.access(target_path, os.W_OK):
        if logger:
            logger.error(u'No write access : %s' % target_path)
        has_access_issue = True
    
    if has_access_issue:
        raise ShinkenAccessRightsError(u'Got access issues for path : %s' % target_path)


def is_run_by_root():
    # type: () -> bool
    return get_cur_user() == u'root'
