#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

""" This Class is the service one, s it manage all service specific thing.
If you look at the scheduling part, look at the scheduling item class"""
import hashlib
import re
import time
from collections import deque
from collections import namedtuple

from shinken.autoslots import AutoSlots
from shinken.eventhandler import EventHandler
from shinken.log import logger, naglog_result
from shinken.macroresolver import MacroResolver
from shinken.misc.html_formater import prepend_explaination_about_set_impact_to_check_output
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.item import InheritableItems
from shinken.objects.proxyitem import proxyitemsmgr
from shinken.objects.schedulingitem import SchedulingItem, PROPERTIES_RETRIEVABLE_FROM_COMMAND_TO_HOST
from shinken.objects.state_id import STATE_ID
from shinken.property import BoolProp, IntegerProp, CharProp, StringProp, ListProp, LinkToOthersObjectsProp, RawProp, VirtualProp
from shinken.rawdata import TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL
from shinken.util import (
    strip_and_uniq,
    format_t_into_dhms_format,
    to_list_string_of_names,
    get_service_realm,
    get_in_scheduled_downtime,
    get_in_inherited_downtime,
    get_problem_has_been_acknowledged,
    get_in_inherited_acknowledged,
    get_brok_instance_uuid,
    DuplicateForEachParser,
    DuplicateForEachStatus,
)

if TYPE_CHECKING:
    from shinken.objects.host import Host, Hosts
    from shinken.objects.command import Commands
    from shinken.misc.type_hint import List

SECONDS_MAX_DATE_VALUE = 157852800
MINUTES_MAX_DATE_VALUE = 2630880
MINUTE_PROPERTY = set((u'check_interval', u'retry_interval', u'notification_interval', u'first_notification_delay'))
SECOND_PROPERTY = set((u'warning_threshold_cpu_usage', u'freshness_threshold', u'check_running_timeout'))

Priority = namedtuple(u'Priority', [u'priority', u'twin_id'])

PROPERTIES_RETRIEVABLE_FROM_HOST = (
    u'contacts',
    u'contact_groups',
    u'notification_interval',
    u'notifications_enabled',
    u'first_notification_delay',
    u'notification_options',
    u'notification_period',
    u'resultmodulations',
    u'business_impact_modulations',
    u'escalations',
    u'poller_tag',
    u'reactionner_tag',
    u'check_period',
    u'business_impact',
    u'maintenance_period',
    u'sla_warning_threshold',
    u'sla_critical_threshold',
)

LINK_SEP = u','
RAW_LINK_SEP = u'!'
PREFIX_LINK_UUID = u'uuid!'
PREFIX_LINK_DFE_KEY = u'!dfe_key!'

CODE_MAP_STATE_FROM_RETURN_CODE = {0: STATE_ID.OK, 1: STATE_ID.WARN, 2: STATE_ID.CRIT, 3: STATE_ID.UNKNOWN}


class IncorrectRawLinkException(Exception):
    pass


class Service(SchedulingItem):
    # AutoSlots create the __slots__ with properties and
    # running_properties names
    __metaclass__ = AutoSlots
    
    # Every service have a unique ID, and 0 is always special in
    # database and co...
    id = 1
    # The host and service do not have the same 0 value, now yes :)
    ok_up = 'OK'
    # used by item class for format specific value like for Broks
    my_type = 'service'
    # name of the type when display log for end users
    my_type_log_display = 'check'
    
    # properties defined by configuration
    # required: is required in conf
    # default: default value if no set in conf
    # pythonize: function to call when transforming string to python object
    # fill_brok: if set, send to broker. there are two categories:
    #  full_status for initial and update status, check_result for check results
    # no_slots: do not take this property for __slots__
    properties = SchedulingItem.properties.copy()
    properties.update({
        # Simple properties, can be set into structure
        'is_volatile'            : BoolProp(default='0', fill_brok=['full_status']),
        'max_check_attempts'     : IntegerProp(default='1', fill_brok=['full_status']),
        'check_interval'         : IntegerProp(fill_brok=['full_status', 'check_result']),
        'retry_interval'         : IntegerProp(fill_brok=['full_status', 'check_result']),
        'obsess_over_service'    : BoolProp(default='0', fill_brok=['full_status']),
        
        'low_flap_threshold'     : IntegerProp(default='-1', fill_brok=['full_status']),
        'high_flap_threshold'    : IntegerProp(default='-1', fill_brok=['full_status']),
        'flap_detection_enabled' : BoolProp(default='1', fill_brok=['full_status']),
        
        # Put in the passthrough list
        'merge_host_contacts'    : BoolProp(default='0', fill_brok=['full_status']),
        'host_dependency_enabled': BoolProp(default='1', fill_brok=['full_status']),
        
        # Configuration Structure, like links to others elements
        'host_name'              : StringProp(fill_brok=['full_status', 'check_result', 'next_schedule']),
        'hostgroup_name'         : StringProp(default='', fill_brok=['full_status'], merging='join'),
        'service_description'    : StringProp(fill_brok=['full_status', 'check_result', 'next_schedule']),
        'display_name'           : StringProp(default='', fill_brok=['full_status']),
        'servicegroups'          : LinkToOthersObjectsProp(default='', fill_brok=['full_status'], brok_transformation=to_list_string_of_names, merging='join'),
        
        'check_command'          : StringProp(fill_brok=['full_status']),
        
        # Theses one can be size bound
        'flap_detection_options' : ListProp(default='o,w,c,u', fill_brok=['full_status']),
        'notification_options'   : ListProp(default='d,u,r,f', fill_brok=['full_status'], merging='join'),  # same default than host
        
        # Easy Service dep definition
        'service_dependencies'   : ListProp(default='', merging='join'),  # TODO: find a way to brok it?
        
        # service generator
        'duplicate_foreach'      : StringProp(default=''),
        'default_value'          : StringProp(default=''),
        
        # from template is used to compute check priority
        'from_template'          : ListProp(default=''),
        
    })
    
    # properties keep but we don't show differences
    passthrough = SchedulingItem.passthrough.copy()
    passthrough.update({
        ### Remove from the properties list.
        'initial_state'                             : CharProp(default='o', fill_brok=['full_status']),
        
        'retain_status_information'                 : BoolProp(default='1', fill_brok=['full_status']),
        'retain_nonstatus_information'              : BoolProp(default='1', fill_brok=['full_status']),
        
        'notes'                                     : StringProp(default='', fill_brok=['full_status']),
        'action_url'                                : StringProp(default='', fill_brok=['full_status']),
        'icon_image'                                : StringProp(default='', fill_brok=['full_status']),
        'icon_image_alt'                            : StringProp(default='', fill_brok=['full_status']),
        'icon_set'                                  : StringProp(default='', fill_brok=['full_status']),
        'failure_prediction_enabled'                : BoolProp(default='0', fill_brok=['full_status']),
        'parallelize_check'                         : BoolProp(default='1', fill_brok=['full_status']),
        
        'labels'                                    : ListProp(default='', fill_brok=['full_status'], merging='join'),
        
        # BUSINESS CORRELATOR PART
        # Business rules output format template
        'business_rule_output_template'             : StringProp(default='', fill_brok=['full_status']),
        # Business rules notifications mode
        'business_rule_smart_notifications'         : BoolProp(default='0', fill_brok=['full_status']),
        # Treat downtimes as acknowledgements in smart notifications
        'business_rule_downtime_as_ack'             : BoolProp(default='0', fill_brok=['full_status']),
        # Enforces child nodes notification options
        'business_rule_host_notification_options'   : ListProp(default='', fill_brok=['full_status']),
        'business_rule_service_notification_options': ListProp(default='', fill_brok=['full_status']),
        
        'apply_on_type'                             : StringProp(default='', fill_brok=['full_status']),
        
        ### still exists in the properties list.
        'display_name'                              : StringProp(default='', fill_brok=['full_status']),
        'servicegroups'                             : StringProp(default='', fill_brok=['full_status'], brok_transformation=to_list_string_of_names, merging='join'),
        
        'time_to_orphanage'                         : IntegerProp(default="300", fill_brok=['full_status']),
        'merge_host_contacts'                       : BoolProp(default='0', fill_brok=['full_status']),
        'host_dependency_enabled'                   : BoolProp(default='1', fill_brok=['full_status']),
        
        'stalking_options'                          : ListProp(default='', fill_brok=['full_status']),
        'aggregation'                               : StringProp(default='', fill_brok=['full_status']),
    })
    
    # properties prohibited
    blacklist = {
        'service_dependencies': ListProp(default='', merging='join'),
        'checkmodulations'    : ListProp(default='', fill_brok=['full_status'], merging='ordered'),
    }
    
    # properties used in the running state
    running_properties = SchedulingItem.running_properties.copy()
    running_properties.update({
        ##################### Purely scheduler or arbiter internal
        'host'                                 : StringProp(default=None),
        'state_before_hard_unknown_reach_phase': StringProp(default='OK', retention=True),  # Manage the unknown/unreach during hard state
        'last_cluster_state_id'                : IntegerProp(default=-1, retention=True, useless_in_configuration=True),
        
        #################### Need brok sending
        'host_uuid'                            : StringProp(default='', fill_brok=['full_status', 'check_result', 'next_schedule']),
        'instance_uuid'                        : StringProp(default='', fill_brok=['full_status'], brok_transformation=get_brok_instance_uuid, fill_raw_data=[TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL]),
        
        'state_id'                             : IntegerProp(default=3, fill_brok=['full_status', 'check_result'], fill_raw_data=[TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL], retention=True),
        
        'last_time_ok'                         : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'last_time_warning'                    : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'last_time_critical'                   : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'last_time_unknown'                    : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        
        'problem_has_been_acknowledged'        : BoolProp(default=False, fill_brok=['full_status', 'check_result'], brok_transformation=get_problem_has_been_acknowledged, retention=True, useless_in_configuration=True),
        'in_inherited_acknowledged'            : BoolProp(default=False, fill_brok=['full_status', 'check_result'], brok_transformation=get_in_inherited_acknowledged, useless_in_configuration=True),
        
        'customs'                              : RawProp(default={}),
        
        'in_partial_downtime'                  : BoolProp(default=False, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'in_scheduled_downtime'                : BoolProp(default=False, fill_brok=['full_status', 'check_result'], brok_transformation=get_in_scheduled_downtime, retention=True, useless_in_configuration=True),
        'in_inherited_downtime'                : BoolProp(default=False, fill_brok=['full_status', 'check_result'], brok_transformation=get_in_inherited_downtime, useless_in_configuration=True),
        
        # Virtual because it will look at host value
        'realm'                                : VirtualProp(default='', fill_brok=['full_status', 'check_result', 'next_schedule'], brok_transformation=get_service_realm),
        
    })
    
    # Mapping between Macros and properties (can be prop or a function)
    macros = {
        'SERVICEUUID'                  : 'get_instance_uuid',
        'SERVICEDESC'                  : 'service_description',
        'SERVICEDISPLAYNAME'           : 'display_name',
        'SERVICESTATE'                 : 'state',
        'SERVICESTATEID'               : 'state_id',
        'LASTSERVICESTATE'             : 'last_state',
        'LASTSERVICESTATEID'           : 'last_state_id',
        'SERVICESTATETYPE'             : 'state_type',
        'SERVICEATTEMPT'               : 'attempt',
        'MAXSERVICEATTEMPTS'           : 'max_check_attempts',
        'SERVICEISVOLATILE'            : 'is_volatile',
        'SERVICELATENCY'               : 'latency',
        'SERVICEEXECUTIONTIME'         : 'execution_time',
        'SERVICEDURATION'              : 'get_duration',
        'SERVICEDURATIONSEC'           : 'get_duration_sec',
        'SERVICEDOWNTIMECOMMENT'       : 'get_downtime_comment',
        'SERVICEDOWNTIMEAUTHOR'        : 'get_downtime_author',
        'SERVICEPERCENTCHANGE'         : 'percent_state_change',
        'SERVICEGROUPNAME'             : 'get_groupname',
        'SERVICEGROUPNAMES'            : 'get_groupnames',
        'LASTSERVICECHECK'             : 'last_chk',
        'LASTSERVICESTATECHANGE'       : 'last_state_change',
        'LASTSERVICEOK'                : 'last_time_ok',
        'LASTSERVICEWARNING'           : 'last_time_warning',
        'LASTSERVICEUNKNOWN'           : 'last_time_unknown',
        'LASTSERVICECRITICAL'          : 'last_time_critical',
        'SERVICEOUTPUT'                : 'output',
        'LONGSERVICEOUTPUT'            : 'long_output',
        'SERVICEPERFDATA'              : 'perf_data',
        'LASTSERVICEPERFDATA'          : 'last_perf_data',
        'SERVICECHECKCOMMAND'          : 'get_check_command',
        'SERVICEACKAUTHOR'             : 'get_ack_author_name',
        'SERVICEACKAUTHORNAME'         : 'get_ack_author_name',
        'SERVICEACKAUTHORALIAS'        : 'get_ack_author_name',
        'SERVICEACKCOMMENT'            : 'get_ack_comment',
        'SERVICEACTIONURL'             : 'action_url',
        'SERVICENOTESURL'              : 'notes_url',
        'SERVICENOTESMULTIURL'         : 'notes_multi_url',
        'SERVICENOTES'                 : 'notes',
        'SERVICEBUSINESSIMPACT'        : 'business_impact',
        'SERVICEFIRSTNOTIFICATIONDELAY': 'first_notification_delay',
        'SERVICENOTIFICATIONNUMBER'    : 'current_notification_number',
    }
    
    # Some macros can be cached, and should be computed only once as a cache
    static_macros = set(['SERVICEUUID', 'SERVICEDESC', 'SERVICEDISPLAYNAME', 'HOSTADDRESS', 'SERVICENOTESURL', 'SERVICENOTESMULTIURL', 'SERVICEFIRSTNOTIFICATIONDELAY'])
    
    # This tab is used to transform old parameters name into new ones
    # so from Nagios2 format, to Nagios3 ones.
    # Or Shinken deprecated names like criticity
    old_properties = {
        'normal_check_interval': 'check_interval',
        'retry_check_interval' : 'retry_interval',
        'criticity'            : 'business_impact',
        'hostgroup'            : 'hostgroup_name',
        'hostgroups'           : 'hostgroup_name',
        ## 'criticitymodulations':    'business_impact_modulations',
    }
    
    
    def __init__(self, params={}, skip_useless_in_configuration=False):
        super(Service, self).__init__(params, skip_useless_in_configuration)
        # Private key use only in arbiter
        self._dfe_key = ''
        self._origin_uuid = ''  # uuid of the service use to make the duplicate for each
        self._dfe_original_name_without_key = ''
    
    
    #######
    #                   __ _                       _   _
    #                  / _(_)                     | | (_)
    #   ___ ___  _ __ | |_ _  __ _ _   _ _ __ __ _| |_ _  ___  _ __
    #  / __/ _ \| '_ \|  _| |/ _` | | | | '__/ _` | __| |/ _ \| '_ \
    # | (_| (_) | | | | | | | (_| | |_| | | | (_| | |_| | (_) | | | |
    #  \___\___/|_| |_|_| |_|\__, |\__,_|_|  \__,_|\__|_|\___/|_| |_|
    #                         __/ |
    #                        |___/
    ######
    
    # Give a nice name output
    def get_name(self):
        if hasattr(self, 'service_description'):
            return self.service_description
        if hasattr(self, 'name'):
            return self.name
        return 'SERVICE-DESCRIPTION-MISSING'
    
    
    def get_name_without_dfe(self):
        return self._dfe_original_name_without_key or self.get_name()
    
    
    # Get the servicegroups names
    def get_groupnames(self):
        return ','.join([sg.get_name() for sg in self.servicegroups])
    
    
    # Need the whole name for debugging purpose
    def get_dbg_name(self):
        return "%s/%s" % (getattr(self.host, 'host_name', '(no name'), getattr(self, 'service_description', '(no name)'))
    
    
    def get_full_name(self):
        if self.host and hasattr(self.host, 'host_name') and hasattr(self, 'service_description'):
            return "%s/%s" % (self.host.host_name, self.service_description)
        return 'UNKNOWN-SERVICE'
    
    
    def get_instance_uuid(self):
        if self.host and hasattr(self.host, 'uuid') and hasattr(self, 'uuid'):
            return "%s-%s" % (self.host.uuid, self.uuid)
        return 'UNKNOWN-SERVICE'
    
    
    # Get our realm, so in fact our host one
    def get_realm(self):
        if self.host is None:
            return None
        return self.host.get_realm()
    
    
    def get_hostgroups(self):
        return self.host.hostgroups
    
    
    def get_host_tags(self):
        return self.host.tags
    
    
    # we can inherit or have default value from host, but we cannot match the 'd' (DOWN)
    # so we are switching to known value for the service: WARNING & CRITICAL
    def fix_service_notification_options_inheritance(self):
        # Convert notification options to match d (DOWN on host) to w,c
        self.notification_options = self.notification_options.replace("d", "w,c")
    
    
    # Check is required prop are set:
    # template are always correct
    # contacts OR contactgroups is need
    def is_correct(self):
        state = True
        cls = self.__class__
        
        source = getattr(self, 'imported_from', 'unknown')
        
        desc = getattr(self, 'service_description', 'unnamed')
        hname = getattr(self, 'host_name', 'unnamed')
        
        special_properties = ('check_period', 'notification_interval', 'host_name',
                              'hostgroup_name', 'notification_period', 'check_command', 'retry_interval', 'check_interval')
        
        if getattr(self, 'service_description', None) and getattr(self, 'name', None):
            logger.error("[item::%s-%s] Service cannot have a service_description and a name." % (self.name, self.service_description))
            state = False
        
        for property_name in MINUTE_PROPERTY:
            value = getattr(self, property_name, None)
            if value and value > MINUTES_MAX_DATE_VALUE:
                logger.error('[item::%s] %s property is too long. Value : %s . The maximum is %s' % (self.get_name(), property_name, value, MINUTES_MAX_DATE_VALUE))
                state = False
        
        for property_name in SECOND_PROPERTY:
            value = getattr(self, property_name, None)
            if value and value > SECONDS_MAX_DATE_VALUE:
                logger.error('[item::%s] %s property is too long. Value : %s . The maximum is %s' % (self.get_name(), property_name, value, SECONDS_MAX_DATE_VALUE))
                state = False
        
        for prop, entry in cls.properties.iteritems():
            if prop not in special_properties:
                if not hasattr(self, prop) and entry.required:
                    logger.error("The service %s on host '%s' does not have %s" % (desc, hname, prop))
                    state = False  # Bad boy...
        
        low_flap = self.low_flap_threshold if self.low_flap_threshold != -1 else self.__class__.global_low_flap_threshold
        high_flap = self.high_flap_threshold if self.high_flap_threshold != -1 else self.__class__.global_high_flap_threshold
        if high_flap < low_flap:
            logger.error("The service %s on host '%s' have incoherent flap thresholds (respectively low %s%% and high %s%%)" % (desc, hname, low_flap, high_flap))
            state = False
        
        if self.sla_warning_threshold < self.sla_critical_threshold:
            logger.error("The service %s on host '%s' have incoherent SLA thresholds (respectively critical %s%% and warning %s%%)" % (desc, hname, self.sla_warning_threshold, self.sla_critical_threshold))
            state = False
        
        for custom in self.customs.iterkeys():
            if self.custom_data_format.search(custom):
                logger.error("[item::%s] custom property %s is malformed. Special characters are forbidden" % (self.get_name(), custom))
                state = False
        
        # Then look if we have some errors in the conf
        # Juts print warnings, but raise errors
        for err in self.configuration_warnings:
            logger.warning("[service::%s] %s" % (desc, err))
        
        # Raised all previously saw errors like unknown contacts and co
        if self.configuration_errors != []:
            state = False
            for err in self.configuration_errors:
                logger.error("[service::%s] %s" % (self.get_full_name(), err))
        
        # If no notif period, set it to None, mean 24x7
        if not hasattr(self, 'notification_period'):
            self.notification_period = None
        
        # Set display_name if need
        if getattr(self, 'display_name', '') == '':
            self.display_name = getattr(self, 'service_description', '')
        
        # If we got an event handler, it should be valid
        if getattr(self, 'event_handler', None) and not self.event_handler.is_valid():
            logger.error("%s: my event_handler %s is invalid" % (self.get_name(), self.event_handler.original_command))
            state = False
        
        # See SEF-5670 and SEF-4496
        if not hasattr(self, 'check_command'):
            if getattr(self, 'active_checks_enabled'):
                logger.error("%s: I've got no check_command" % self.get_name())
                state = False
            else:
                self.check_command = None
                self.retry_interval = 1
                self.check_interval = 1
        else:
            # Ok got a command, but maybe it's invalid
            if not self.check_command.is_valid():
                logger.error("%s: my check_command %s is invalid" % (self.get_name(), self.check_command.original_command))
                state = False
            if self.got_business_rule:
                if self.business_rule is None:
                    logger.error("%s: my business rule is invalid" % (self.get_name(),))
                    state = False
                elif self.business_rule.configuration_errors:
                    logger.error("%s: my business rule is invalid" % (self.get_name(),))
                    for bperror in self.business_rule.configuration_errors:
                        logger.error("%s: %s" % (self.get_name(), bperror))
                    state = False
        
        active_mandatory_props = ('retry_interval', 'check_interval')
        if getattr(self, 'active_checks_enabled') and not self.got_business_rule:
            for prop in active_mandatory_props:
                if not hasattr(self, prop):
                    logger.error("The service %s on host '%s' does not have %s" % (desc, hname, prop))
                    state = False  # Bad boy...
        else:
            for prop in active_mandatory_props:
                if not hasattr(self, prop):
                    setattr(self, prop, 1)
        
        if self.host is None:
            logger.warning("The host_name '%s' for the service '%s' is unknown or disabled." % (self.host_name, desc,))
            # do not set tis a a true error, only we will delete this after
        if hasattr(self, 'service_description'):
            if '$KEY$' in self.service_description:
                logger.error("%s: Missing duplicate_foreach parameter. Failed to substitute name.")
                return False
            for c in cls.illegal_object_name_chars:
                if c in self.service_description:
                    logger.error("%s: My service_description got the character %s that is not allowed." % (self.get_name(), c))
                    state = False
        return state
    
    
    # The service is dependent of his father dep
    # Must be AFTER linkify
    # TODO: implement "not host dependent" feature.
    def fill_daddy_dependency(self):
        #  Depend of host, all status, is a networkdep
        # and do not have timeperiod, and follow parents dep
        if self.host is not None and self.host_dependency_enabled:
            # I add the dep in MY list
            self.act_depend_of.append((self.host,
                                       ('d', 'u', 's', 'f'),
                                       'network_dep',
                                       self.host.check_period,
                                       True)
                                      )
            # I add the dep in Daddy list
            self.host.act_depend_of_me.append((self,
                                               ('d', 'u', 's', 'f'),
                                               'network_dep',
                                               self.host.check_period,
                                               True)
                                              )
            
            # And the parent/child dep lists too
            self.host.register_son_in_parent_child_dependencies(self)
    
    
    # Register the dependency between 2 service for action (notification etc)
    def add_service_act_dependency(self, srv, status, timeperiod, inherits_parent):
        # first I add the other the I depend on in MY list
        self.act_depend_of.append((srv, status, 'logic_dep',
                                   timeperiod, inherits_parent))
        # then I register myself in the other service dep list
        srv.act_depend_of_me.append((self, status, 'logic_dep',
                                     timeperiod, inherits_parent))
        
        # And the parent/child dep lists too
        srv.register_son_in_parent_child_dependencies(self)
    
    
    # Register the dependency between 2 service for action (notification etc)
    # but based on a BUSINESS rule, so on fact:
    # ERP depend on database, so we fill just database.act_depend_of_me
    # because we will want ERP mails to go on! So call this
    # on the database service with the srv=ERP service
    def add_business_rule_act_dependency(self, srv, status, timeperiod, inherits_parent):
        # I only register so he know that I WILL be a impact
        self.act_depend_of_me.append((srv, status, 'business_dep',
                                      timeperiod, inherits_parent))
        
        # And the parent/child dep lists too
        self.register_son_in_parent_child_dependencies(srv)
    
    
    # Register the dependency between 2 service for checks
    def add_service_chk_dependency(self, srv, status, timeperiod, inherits_parent):
        # first I add the other the I depend on in MY list
        self.chk_depend_of.append((srv, status, 'logic_dep',
                                   timeperiod, inherits_parent))
        # then I register myself in the other service dep list
        srv.chk_depend_of_me.append((self, status, 'logic_dep',
                                     timeperiod, inherits_parent))
        
        # And the parent/child dep lists too
        srv.register_son_in_parent_child_dependencies(self)
    
    
    # For a given host, look for all copy we must
    # create for for_each property
    def duplicate(self, host):
        duplicates = []
        
        # In macro, it's all in UPPER case
        prop = self.duplicate_foreach.strip().upper()
        
        # If I do not have the property, we bail out
        if prop in host.customs:
            # Get the list entry, and the not one if there is one
            entry = host.customs[prop]
            # Look at the list of the key we do NOT want maybe,
            # for _disks it will be _!disks
            not_entry = host.customs.get('_' + '!' + prop[1:], '').split(',')
            not_keys = strip_and_uniq(not_entry)
            
            default_value = getattr(self, 'default_value', '')
            # Transform the generator string to a list
            # Missing values are filled with the default value
            (key_values, errcode) = DuplicateForEachParser.parse(entry, default_value)
            
            if key_values:
                for key_value in key_values:
                    key = key_value['KEY']
                    # Maybe this key is in the NOT list, if so, skip it
                    if key in not_keys:
                        continue
                    new_s = self.copy(skip_useless_in_configuration=True)
                    new_s.host_name = host.get_name()
                    new_s.host_uuid = host.uuid
                    if self.is_tpl():  # if template, the new one is not
                        new_s.register = 1
                    for key in key_value:
                        dollar_key_string = '$%s$' % key
                        if key == 'KEY':
                            if hasattr(self, 'service_description'):
                                # We want to change all illegal chars to a _ sign. We can't use class.illegal_obj_char
                                # because in the "explode" phase, we do not have access to this data! :(
                                new_s._dfe_key = key_value[key]
                                new_s._origin_uuid = self.uuid
                                safe_key_value = re.sub(r'[' + "`~!$%^&*\"|'<>?,()=" + ']+', '_', key_value[key])
                                new_s._dfe_original_name_without_key = self.service_description.replace(dollar_key_string, '')
                                new_s.service_description = self.service_description.replace(dollar_key_string, safe_key_value)
                        # Here is a list of property where we will expand the $KEY$ by the value
                        _the_expandables = ['check_command', 'service_dependencies', 'event_handler']
                        for prop in _the_expandables:
                            if hasattr(self, prop):
                                # here we can replace VALUE, VALUE1, VALUE2,...
                                setattr(new_s, prop, getattr(new_s, prop).replace(dollar_key_string, key_value[key]))
                        # Also loop over service CUSTOM DATA
                        for (custom_key, custom_value) in new_s.customs.iteritems():
                            if dollar_key_string in custom_value:
                                new_s.customs[custom_key] = custom_value.replace(dollar_key_string, key_value[key])
                    # s got the same uuid than the source service, it's a problem, so set it's uuid as
                    # a hex about it's service description, the only part quite stable
                    if hasattr(new_s, 'service_description'):
                        new_s.uuid = hashlib.md5(new_s.service_description.encode('utf-8')).hexdigest()
                    # And then add in our list this new service
                    duplicates.append(new_s)
            else:
                # If error, we should link the error to the host, because self is a template, and so won't be checked not print!
                if errcode == DuplicateForEachStatus.SYNTAX_ERROR:
                    err = "The custom property '%s' of the host '%s' is not a valid entry %s for a service generator" % (self.duplicate_foreach.strip(), host.get_name(), entry)
                    logger.warning(err)
                    host.configuration_errors.append(err)
                elif errcode == DuplicateForEachStatus.NODE_ERROR:
                    err = "The custom property '%s' of the host '%s' has an invalid node range %s" % (self.duplicate_foreach.strip(), host.get_name(), entry)
                    logger.warning(err)
                    host.configuration_errors.append(err)
        return duplicates
    
    
    #####
    #                         _
    #                        (_)
    #  _ __ _   _ _ __  _ __  _ _ __   __ _
    # | '__| | | | '_ \| '_ \| | '_ \ / _` |
    # | |  | |_| | | | | | | | | | | | (_| |
    # |_|   \__,_|_| |_|_| |_|_|_| |_|\__, |
    #                                  __/ |
    #                                 |___/
    ####
    
    # Set unreachable: our host is DOWN, but it mean nothing for a service
    def set_unreachable(self):
        pass
    
    
    # We just go an impact, so we go UNKNOWN but only if it's enable in the configuration
    def set_impact_state(self):
        cls = self.__class__
        if cls.enable_problem_impacts_states_change and self.state != 'PENDING':
            # Keep a trace of the old state (problem came back before a new checks)
            self.state_before_impact = self.state
            self.state_id_before_impact = self.state_id
            self.output_before_impact = self.output
            # this flag will know if we override the impact state
            self.state_changed_since_impact = False
            self.state = 'UNKNOWN'  # exit code UNDETERMINED
            self.state_id = 3
            self.output = prepend_explaination_about_set_impact_to_check_output(self.output)
    
    
    # Ok, we are no more an impact, if no news checks
    # override the impact state, we came back to old
    # states
    # And only if we enable the state change for impacts
    def unset_impact_state(self):
        cls = self.__class__
        if cls.enable_problem_impacts_states_change and not self.state_changed_since_impact:
            self.state = self.state_before_impact
            self.state_id = self.state_id_before_impact
            self.output = self.output_before_impact
            proxyitemsmgr.update_state(self.get_instance_uuid(), self.state_id)
        # In all cases, clean the output that can be big
        self.output_before_impact = ''
    
    
    # Set state with exit code return by the check. Also update last_state.
    def set_state_from_exit_status(self, return_code):
        now = time.time()
        self.last_state_update = now
        
        # we should put in last_state the good last state:
        # if not just change the state by an problem/impact
        # we can take current state. But if it's the case, the
        # real old state is self.state_before_impact (it's the TRUE
        # state in fact)
        # but only if the global conf have enable the impact state change
        cls = self.__class__
        if cls.enable_problem_impacts_states_change \
                and self.is_impact \
                and not self.state_changed_since_impact:
            self.last_state = self.state_before_impact
        else:  # standard case
            self.last_state = self.state
        
        if return_code == 0:
            self.state = 'OK'
            self.last_time_ok = int(self.last_state_update)
        elif return_code == 1:
            self.state = 'WARNING'
            self.last_time_warning = int(self.last_state_update)
        elif return_code == 2:
            self.state = 'CRITICAL'
            self.last_time_critical = int(self.last_state_update)
        elif return_code == 3:
            self.state = 'UNKNOWN'
            self.last_time_unknown = int(self.last_state_update)
        else:
            self.state = 'UNKNOWN'  # exit code UNDETERMINED
            self.last_time_unknown = int(self.last_state_update)
        
        self.output = Service.get_unknown_return_code_output(return_code, self.output)
        self.state_id = Service.map_state_from_return_code(return_code)
        # If business rule for host, save the raw result from bp_rule computation somewhere else
        if self.got_business_rule:
            self.bp_state = return_code
        
        # Update the proxy objects with the new value
        proxyitemsmgr.update_state(self.get_instance_uuid(), self.state_id)
    
    
    @staticmethod
    def map_state_from_return_code(return_code):
        return CODE_MAP_STATE_FROM_RETURN_CODE.get(return_code, STATE_ID.UNKNOWN)
    
    
    @staticmethod
    def get_unknown_return_code_output(return_code, current_output):
        if return_code not in CODE_MAP_STATE_FROM_RETURN_CODE:
            return u'''Shinken ERROR: The exit code returned by the command [ %s ] is not valid, we set the check to UNKNOWN. Valid values are : %s.<br><br>The output of the command is :<br>----------------------------<br>%s''' % (
                return_code, u','.join(map(str, CODE_MAP_STATE_FROM_RETURN_CODE.iterkeys())), current_output)
        else:
            return current_output
    
    
    # Return True if status is the state (like OK) or small form like 'o'
    def is_state(self, status):
        if status == self.state:
            return True
        # Now low status
        elif status == 'o' and self.state == 'OK':
            return True
        elif status == 'c' and self.state == 'CRITICAL':
            return True
        elif status == 'w' and self.state == 'WARNING':
            return True
        elif status == 'u' and self.state == 'UNKNOWN':
            return True
        return False
    
    
    def is_dfe(self):
        return getattr(self, 'duplicate_foreach', False)
    
    
    # The last time when the state was not OK
    def last_time_non_ok_or_up(self):
        non_ok_times = filter(lambda x: x > self.last_time_ok, [self.last_time_warning,
                                                                self.last_time_critical,
                                                                self.last_time_unknown])
        if len(non_ok_times) == 0:
            last_time_non_ok = 0  # program_start would be better
        else:
            last_time_non_ok = min(non_ok_times)
        return last_time_non_ok
    
    
    # Add a log entry with a SERVICE ALERT like:
    # SERVICE ALERT: server;Load;UNKNOWN;HARD;1;I don't know what to say...
    def raise_alert_log_entry(self):
        # if self.attempt == 0:
        #     raise Exception('Attempts cannot be 0 here! %s %s  state=%s  last_state=%s  state_type=%s  max_check_attempts=%s' % (self, self.get_dbg_name(), self.state, self.last_state, self.state_type, self.max_check_attempts))
        naglog_result('critical', 'SERVICE ALERT: %s;%s;%s;%s;%d;%s'
                      % (self.host.get_name(), self.get_name(),
                         self.state, self.state_type,
                         self.attempt, self.output))
    
    
    # If the configuration allow it, raise an initial log like
    # CURRENT SERVICE STATE: server;Load;UNKNOWN;HARD;1;I don't know what to say...
    def raise_initial_state(self):
        if self.__class__.log_initial_states:
            naglog_result('info', 'CURRENT SERVICE STATE: %s;%s;%s;%s;%d;%s'
                          % (self.host.get_name(), self.get_name(),
                             self.state, self.state_type,
                             self.attempt, self.output))
    
    
    # Add a log entry with a Freshness alert like:
    # Warning: The results of host 'Server' are stale by 0d 0h 0m 58s (threshold=0d 1h 0m 0s).
    # I'm forcing an immediate check of the host.
    def raise_freshness_log_entry(self, t_stale_by, t_threshold):
        logger.warning("The results of service '%s' on host '%s' are stale "
                       "by %s (threshold=%s).  I'm forcing an immediate check "
                       "of the service."
                       % (self.get_name(), self.host.get_name(),
                          format_t_into_dhms_format(t_stale_by),
                          format_t_into_dhms_format(t_threshold)))
    
    
    # Raise a log entry with a Notification alert like
    # SERVICE NOTIFICATION: superadmin;server;Load;OK;notify-by-rss;no output
    def raise_notification_log_entry(self, n):
        contact = n.contact
        command = n.command_call
        if n.type in ('DOWNTIMESTART', 'DOWNTIMEEND', 'DOWNTIMECANCELLED',
                      'CUSTOM', 'ACKNOWLEDGEMENT', 'FLAPPINGSTART',
                      'FLAPPINGSTOP', 'FLAPPINGDISABLED'):
            state = '%s (%s)' % (n.type, self.state)
        else:
            state = self.state
        if self.__class__.log_notifications:
            naglog_result('critical', "SERVICE NOTIFICATION: %s;%s;%s;%s;%s;%s"
                          % (contact.get_name(),
                             self.host.get_name(), self.get_name(), state,
                             command.get_name(), self.output))
    
    
    # Raise a log entry with a Eventhandler alert like
    # SERVICE EVENT HANDLER: test_host_0;test_ok_0;OK;SOFT;4;eventhandler
    def raise_event_handler_log_entry(self, command):
        if self.__class__.log_event_handlers:
            naglog_result('critical', "SERVICE EVENT HANDLER: %s;%s;%s;%s;%s;%s"
                          % (self.host.get_name(), self.get_name(),
                             self.state, self.state_type,
                             self.attempt, command.get_name()))
    
    
    # Raise a log entry with FLAPPING START alert like
    # SERVICE FLAPPING ALERT: server;LOAD;STARTED; Service appears to have started flapping (50.6% change >= 50.0% threshold)
    def raise_flapping_start_log_entry(self, change_ratio, threshold):
        naglog_result('critical', "SERVICE FLAPPING ALERT: %s;%s;STARTED; "
                                  "Service appears to have started flapping "
                                  "(%.1f%% change >= %.1f%% threshold)"
                      % (self.host.get_name(), self.get_name(),
                         change_ratio, threshold))
    
    
    # Raise a log entry with FLAPPING STOP alert like
    # SERVICE FLAPPING ALERT: server;LOAD;STOPPED; Service appears to have stopped flapping (23.0% change < 25.0% threshold)
    def raise_flapping_stop_log_entry(self, change_ratio, threshold):
        naglog_result('critical', "SERVICE FLAPPING ALERT: %s;%s;STOPPED; "
                                  "Service appears to have stopped flapping "
                                  "(%.1f%% change < %.1f%% threshold)"
                      % (self.host.get_name(), self.get_name(),
                         change_ratio, threshold))
    
    
    # If there is no valid time for next check, raise a log entry
    def raise_no_next_check_log_entry(self):
        logger.warning("I cannot schedule the check for the service '%s' on "
                       "host '%s' because there is not future valid time"
                       % (self.get_name(), self.host.get_name()))
    
    
    # Raise a log entry when a downtime begins
    # SERVICE DOWNTIME ALERT: test_host_0;test_ok_0;STARTED; Service has entered a period of scheduled downtime
    def raise_enter_downtime_log_entry(self):
        naglog_result('critical', "SERVICE DOWNTIME ALERT: %s;%s;STARTED; "
                                  "Service has entered a period of scheduled "
                                  "downtime"
                      % (self.host.get_name(), self.get_name()))
    
    
    # Raise a log entry when a downtime has finished
    # SERVICE DOWNTIME ALERT: test_host_0;test_ok_0;STOPPED; Service has exited from a period of scheduled downtime
    def raise_exit_downtime_log_entry(self):
        naglog_result('critical', "SERVICE DOWNTIME ALERT: %s;%s;STOPPED; Service "
                                  "has exited from a period of scheduled downtime"
                      % (self.host.get_name(), self.get_name()))
    
    
    # Raise a log entry when a downtime prematurely ends
    # SERVICE DOWNTIME ALERT: test_host_0;test_ok_0;CANCELLED; Service has entered a period of scheduled downtime
    def raise_cancel_downtime_log_entry(self):
        naglog_result('critical', "SERVICE DOWNTIME ALERT: %s;%s;CANCELLED; "
                                  "Scheduled downtime for service has been cancelled."
                      % (self.host.get_name(), self.get_name()))
    
    
    # Is stalking?
    # Launch if check is waitconsume==first time
    # and if c.status is in self.stalking_options
    def manage_stalking(self, c):
        need_stalk = False
        if c.status == 'waitconsume':
            if c.exit_status == 0 and 'o' in self.stalking_options:
                need_stalk = True
            elif c.exit_status == 1 and 'w' in self.stalking_options:
                need_stalk = True
            elif c.exit_status == 2 and 'c' in self.stalking_options:
                need_stalk = True
            elif c.exit_status == 3 and 'u' in self.stalking_options:
                need_stalk = True
            
            if c.output == self.output:
                need_stalk = False
        if need_stalk:
            logger.info("Stalking %s: %s" % (self.get_name(), c.output))
    
    
    # A service is in downtime if:
    # * it is itself in downtime
    # * its host is in downtime
    def is_in_downtime(self):
        return self.in_scheduled_downtime or self.host.in_scheduled_downtime
    
    
    def is_in_inherited_downtime(self):
        return self.host.in_scheduled_downtime
    
    
    def get_downtime_comment(self):
        return next((downtime.comment for downtime in self.downtimes if downtime.is_in_effect), '')
    
    
    def get_downtime_author(self):
        return next((downtime.author for downtime in self.downtimes if downtime.is_in_effect), '')
    
    
    def is_acknowledged(self):
        return self.problem_has_been_acknowledged or self.host.problem_has_been_acknowledged
    
    
    def is_in_inherited_acknowledged(self):
        return self.acknowledgement.automatic if self.acknowledgement else (self.host.problem_has_been_acknowledged and not self.host.acknowledgement.automatic)
    
    
    def get_acknowkledge_id(self):
        if self.acknowledgement:
            return self.acknowledgement.id
        elif self.is_in_inherited_acknowledged():
            return self.host.get_acknowkledge_id()
        elif self.partial_acknowledge:
            return self.partial_acknowledge.id
        else:
            return None
    
    
    # Service number of downtimes is the number of active on the host + number of the service itself
    def get_number_of_active_downtimes(self):
        return self.scheduled_downtime_depth + self.host.scheduled_downtime_depth
    
    
    # Give data for checks's macros
    def get_data_for_checks(self):
        return [self.host, self]
    
    
    # Give data for event handlers's macros
    def get_data_for_event_handler(self):
        return [self.host, self]
    
    
    # Give data for notifications'n macros
    def get_data_for_notifications(self, contact, n):
        return [self.host, self, contact, n]
    
    
    # See if the notification is launchable (time is OK and contact is OK too)
    def notification_is_blocked_by_contact(self, n, contact):
        return not contact.want_service_notification(self.last_chk, self.state, n.type, self.business_impact, n.command_call)
    
    
    def get_duration_sec(self):
        return str(int(self.duration_sec))
    
    
    def get_duration(self):
        m, s = divmod(self.duration_sec, 60)
        h, m = divmod(m, 60)
        return "%02dh %02dm %02ds" % (h, m, s)
    
    
    def get_ack_author_name(self):
        if self.acknowledgement is None:
            return ''
        return self.acknowledgement.author
    
    
    def get_ack_comment(self):
        if self.acknowledgement is None:
            return ''
        return self.acknowledgement.comment
    
    
    def get_check_command(self):
        return self.check_command.get_name()
    
    
    # Check if a notification for this service is suppressed at this time
    def notification_is_blocked_by_item(self, notification_type):
        # type: (unicode) -> bool
        # TODO
        # forced notification
        # pass if this is a custom notification
        
        # Block if notifications are program-wide disabled
        if not self.enable_notifications:
            return True
        
        # Block if notifications are disabled for this service
        if not self.notifications_enabled:
            return True
        
        # Block if the current status is in the notification_options w,u,c,r,f,s
        if u'n' in self.notification_options:
            return True
        
        if notification_type in (u'PROBLEM', u'RECOVERY'):
            if self.state == u'UNKNOWN' and u'u' not in self.notification_options:
                return True
            if self.state == u'WARNING' and u'w' not in self.notification_options:
                return True
            if self.state == u'CRITICAL' and u'c' not in self.notification_options:
                return True
            if self.state == u'OK' and u'r' not in self.notification_options:
                return True
        
        if notification_type in (u'FLAPPINGSTART', u'FLAPPINGSTOP', u'FLAPPINGDISABLED') and u'f' not in self.notification_options:
            return True
        if notification_type in (u'DOWNTIMESTART', u'DOWNTIMEEND', u'DOWNTIMECANCELLED') and u's' not in self.notification_options:
            return True
        
        # Acknowledgements make no sense when the status is ok/up
        if notification_type == u'ACKNOWLEDGEMENT' and self.state == self.ok_up:
            return True
        
        # We need to know how much active downtimes we have
        nb_active_downtimes = self.get_number_of_active_downtimes()
        
        # When in downtime, only allow end-of-downtime notifications
        if nb_active_downtimes > 1 and notification_type not in (u'DOWNTIMEEND', u'DOWNTIMECANCELLED'):
            return True
        
        # Block if host is in a scheduled downtime
        if self.host.in_scheduled_downtime:
            return True
        
        # Block if in a scheduled downtime and a problem arises, or flapping event
        if nb_active_downtimes > 0 and notification_type in (u'PROBLEM', u'RECOVERY', u'FLAPPINGSTART', u'FLAPPINGSTOP', u'FLAPPINGDISABLED'):
            return True
        
        # Block if the status is SOFT
        if self.state_type == u'SOFT' and notification_type == u'PROBLEM':
            return True
        
        # Block if the problem has already been acknowledged
        if self.problem_has_been_acknowledged and notification_type != u'ACKNOWLEDGEMENT':
            return True
        
        # Block if flapping
        if self.is_flapping and notification_type not in (u'FLAPPINGSTART', u'FLAPPINGSTOP', u'FLAPPINGDISABLED'):
            return True
        
        # Block if host is down
        if self.host.state != self.host.ok_up:
            return True
        
        return False
    
    
    # Get a oc*p command if item has obsess_over_*
    # command. It must be enabled locally and globally
    def get_obsessive_compulsive_processor_command(self):
        cls = self.__class__
        if not cls.obsess_over or not self.obsess_over_service:
            return
        
        m = MacroResolver()
        data = self.get_data_for_event_handler()
        cmd = m.resolve_command(cls.ocsp_command, data)
        command_name = getattr(getattr(self.ocsp_command, 'command', {}), 'command_name', 'MISSING_NAME')
        e = EventHandler(cmd, timeout=cls.ocsp_timeout, command_name=command_name)
        
        # ok we can put it in our temp action queue
        self.actions.append(e)
    
    
    # We want to know the uuid of the definition that did create this check
    # so can be:
    # * genuine check => own uuid
    # * duplicate for each based check => give back the uuid of the definition (object with $KEY$)
    def get_uuid_from_definition_object(self):
        if self._origin_uuid:
            return self._origin_uuid
        return self.uuid
    
    
    def get_active_downtime_uuids(self):
        active_downtime_uuids = super(Service, self).get_active_downtime_uuids()
        active_downtime_uuids.extend(self.host.get_active_downtime_uuids())
        return active_downtime_uuids


def flatten_tpls(host, lst):
    lst.append(Services._compute_prefix_from_template(host.get_name(), host))
    for t in host.templates:
        flatten_tpls(t, lst)


# Class for list of services. It's mainly, mainly for configuration part
class Services(InheritableItems):
    inner_class = Service  # use for know what is in items
    
    # cache for the service override split pattern
    _service_override_pattern = re.compile(ur'^([^,]+),\s*([^\s]+)\s+(.*)$', re.UNICODE | re.DOTALL)
    
    
    def __init__(self, items):
        super(Services, self).__init__(items)
        self.__host_templates_expression_to_hosts_cache = {}
        self.__host_templates_expression_to_templates_cache = {}
    
    
    def load_value_from_global_conf(self):
        # type:() -> None
        for service in self:
            service.load_value_from_global_conf()
    
    
    # Create the reversed list for speedup search by host_name/name
    # We also tag service already in list: they are twins. It's a a bad things.
    # Hostgroups service have an ID higher than host service. So it we tag
    # an id that already are in the list, this service is already
    # exist, and is a host,
    # or a previous hostgroup, but define before.
    def create_reversed_list(self):
        self.reversed_list = {}
        self.twins = []
        
        all_services = [s for s in self]
        for s in all_services:
            if hasattr(s, 'service_description') and hasattr(s, 'host_name'):
                s_desc = getattr(s, 'service_description')
                s_host_name = getattr(s, 'host_name')
                key = (s_host_name, s_desc)
                if key not in self.reversed_list:
                    self.reversed_list[key] = s.id
                else:
                    old_id = self.reversed_list[key]
                    # If the user defined a definition order, we take it for comparison.
                    old_def_order = getattr(self[old_id], 'definition_order', 100)
                    new_def_order = getattr(self[s.id], 'definition_order', 100)
                    if old_def_order < new_def_order:
                        del self[s.id]
                    elif old_def_order > new_def_order:
                        del self[old_id]
                        self.reversed_list[key] = s.id
                    # Else it's a twin. We will solve it by using inheritance in cascade (SEF-1156)
                    elif old_def_order == new_def_order:
                        if old_id not in self.twins:
                            self.twins.append(old_id)
                        self.twins.append(s.id)
        
        # For service, the reversed_list is not used for search, so we del it
        del self.reversed_list
    
    
    def remove_twins(self, hosts):
        reversed_list = {}
        for twin_id in self.twins:
            host = hosts.find_by_name(self[twin_id].host_name)
            if host is None:
                # The service will actually be processed by remove_orphan_services()
                continue
            
            # Add the twin_id and his priority in a list sorted by key "host,check"
            key = host.get_name(), self[twin_id].get_name()
            if key in reversed_list:
                reversed_list[key].append(self._get_service_priority(twin_id, host))
            else:
                reversed_list[key] = [self._get_service_priority(twin_id, host)]
        
        # the we sort the list each key with the first and keep only the first if the priority is not the same as the second (or more)
        for key, twins_services in reversed_list.iteritems():
            twins_services = sorted(twins_services, key=lambda k: k[0])
            if twins_services[0].priority == twins_services[1].priority:
                self.configuration_errors.append("Impossible to guess which of the %d check named '%s' to apply to host '%s'." % (len(twins_services), key[1], key[0]))
            for _, twin_id in twins_services[1:]:
                # here we delete all twin, EVEN the errors otherwise the twin list will be recompute and the logs will be ugly
                del self[twin_id]
        
        self.twins = []
    
    
    def _get_service_priority(self, twin_id, host):
        # We set the template level to max +1 because if the service not match any template, it must be attach to hostgroup
        templates = []
        flatten_tpls(host, templates)
        template_level = len(templates) + 1
        for i, tpl in enumerate(templates):
            if tpl in self[twin_id].from_template:
                template_level = i
                break
        
        # The we take in account if a service is a DFE
        is_dfe = 1 if self[twin_id].is_dfe() else 0
        
        # Finally we set the priority with :
        # - the template level * 1000
        # - We add 1 if the check is a dfe
        # The lowest priority win
        # We return this priority with the twin_id
        return Priority(template_level * 1000 + is_dfe, twin_id)
    
    
    def _show_twin_error(self, item):
        if item.duplicate_foreach:
            logger.error("[items] The check name generated from the duplicate foreach '%s' is duplicated in host named '%s'." % (item.get_name(), item.host_name))
        else:
            logger.error("[items] 'check.%s' is duplicated from %s" % (item.get_name(), getattr(item, 'imported_from', getattr(item, 'sources', "unknown source"))))
    
    
    # Search a service id by its name and host_name
    def find_srv_id_by_name_and_hostname(self, host_name, name):
        for s in self:
            # Runtime first, available only after linkify
            if hasattr(s, 'service_description') and hasattr(s, 'host'):
                if s.service_description == name and s.host == host_name:
                    return s.id
            # At config part, available before linkify
            if hasattr(s, 'service_description') and hasattr(s, 'host_name'):
                if s.service_description == name and s.host_name == host_name:
                    return s.id
        return None
    
    
    def remove_by_name_and_hostname(self, hname, sdesc):
        s = self.find_srv_by_name_and_hostname(hname, sdesc)
        if s is None:
            return
        id_ = s.id
        # We can remove from items
        if id_ in self.items:
            del self.items[id_]
        # And also from  reversed list
        if hasattr(self, 'reversed_list'):
            key = (hname, sdesc)
            if key in self.reversed_list:
                del self.reversed_list[key]
    
    
    # Search a service by its name and host_name
    def find_srv_by_name_and_hostname(self, host_name, name):
        if hasattr(self, 'hosts'):
            h = self.hosts.find_by_name(host_name)
            if h is None:
                return None
            return h.find_service_by_name(name)
        
        id = self.find_srv_id_by_name_and_hostname(host_name, name)
        if id is not None:
            return self.items[id]
        else:
            return None
    
    
    # Search a service by it's uuid and host_uuid
    def find_srv_by_host_uuid_and_check_uuid(self, host_uuid, check_uuid):
        if hasattr(self, 'hosts'):
            h = self.hosts.find_by_uuid(host_uuid)
            if h is not None:
                return h.find_service_by_uuid(check_uuid)
        return None
    
    
    # Removes service exceptions based on host configuration
    def remove_exclusions(self, hosts):
        # lazy import, to avoid recursive import
        from .host import EXCLUDE_FROM_TEMPLATES_RESULTS
        
        to_remove = set()
        
        # NOTE: loop over services is costly, so we are indexing host->services only once here
        _host_to_services_index = {}
        for service in self:
            if service.is_tpl():
                continue
            service_host_name = service.host_name
            if service_host_name not in _host_to_services_index:
                _host_to_services_index[service_host_name] = []
            _host_to_services_index[service_host_name].append(service)
        
        for host in hosts:
            if host.is_tpl():
                continue
            
            in_host_services = _host_to_services_index.get(host.host_name, [])  # beware: maybe this host does not have any service, so is not in the dict
            
            for in_host_service in in_host_services:
                # step 1 - Exclude service excluded by the host
                if host.is_check_exclude(in_host_service):
                    logger.debug(u'Removing check %s/%s due to exclusion on the host' % (in_host_service.host_name, in_host_service.get_name()))
                    to_remove.add(in_host_service.id)
                    continue
                
                # step 2 - Exclude service excluded by the host template
                # If all templates which have the service excluded the service it will be remove
                # So if one template do not excluded the service it will be keep
                remove_by_template = False
                for host_template in host.templates:
                    template_result = host_template.is_service_excluded_by_template(in_host_service)
                    # logger.debug('EXCLUSION: %s/%s is from this template %s?  => %s' % (in_host_service.host_name, in_host_service.get_name(), host_template.get_name(), template_result))
                    # if the template do not have it, we don't stop here
                    if template_result == EXCLUDE_FROM_TEMPLATES_RESULTS.DONT_HAVE:
                        continue
                    # so now have it, must have a exclude or not
                    # if it excludes, mark it, but all templates that have it must exclude it
                    elif template_result == EXCLUDE_FROM_TEMPLATES_RESULTS.EXCLUDE:
                        remove_by_template = True
                    # DO NOT EXCLUDE => so not all templates exclude it, so we allow it
                    elif template_result == EXCLUDE_FROM_TEMPLATES_RESULTS.ALLOW:
                        remove_by_template = False
                        break
                
                if remove_by_template:
                    logger.debug(u'Removing check %s/%s due to exclusion on a template' % (in_host_service.host_name, in_host_service.get_name()))
                    to_remove.add(in_host_service.id)
        
        for sid in to_remove:
            del self[sid]
        
        return len(to_remove)
    
    
    def remove_orphan_services(self):
        to_remove = [service.id for service in self if service.host is None]
        
        for service_id in to_remove:
            del self[service_id]
    
    
    # Make link between elements:
    # service -> host
    # service -> command
    # service -> timeperiods
    # service -> contacts
    def linkify(self, hosts, commands, timeperiods, contacts,
                resultmodulations, businessimpactmodulations, escalations,
                servicegroups, checkmodulations, macromodulations):
        self.linkify_with_timeperiods(timeperiods, 'notification_period')
        self.linkify_with_timeperiods(timeperiods, 'check_period')
        self.linkify_with_timeperiods(timeperiods, 'maintenance_period')
        self.linkify_s_by_hst(hosts)
        self.linkify_s_by_sg(servicegroups)
        self.linkify_one_command_with_commands(commands, 'check_command')
        self.linkify_one_command_with_commands(commands, 'event_handler')
        self.linkify_with_contacts(contacts)
        self.linkify_with_resultmodulations(resultmodulations)
        self.linkify_with_business_impact_modulations(businessimpactmodulations)
        # WARNING: all escalations will not be linked here
        # (just the escalation here, not serviceesca or hostesca).
        # This last one will be link in escalations linkify.
        self.linkify_with_escalations(escalations)
        self.linkify_with_checkmodulations(checkmodulations)
        self.linkify_with_macromodulations(macromodulations)
    
    
    def _search_service_for_host_link(self, host, name_expr, cache, cache_index):
        # To speed up search if several properties are overridden on the same service, we keep them in temporary cache
        host_name = getattr(host, 'host_name', getattr(host, 'name'))
        key = "%s/%s" % (host_name, name_expr)
        if key in cache:
            return cache[key]
        
        # Looks for corresponding service
        # As hosts and service are not yet linked, we have to walk
        # through our cache to look at all services associated to this host.
        host_services = cache_index.get(host_name, {}).values()
        service = None
        for s in host_services:
            if not hasattr(s, "host_name") or not hasattr(s, "service_description"):
                # this is a template
                continue
            
            # NOTE: the synchronizer can give us the name in 2 ways:
            # UUID => it found the real check, so we are sure that we must take this one
            #           We check for service duplicate for each if the dfe_key match
            # NAME => it only have its name, like in duplicate foreach, so we fallback to this
            if name_expr.startswith(PREFIX_LINK_UUID):
                uuid_to_search, dfe_key = host.parse_service_exclusion_expr(name_expr)
                link_match_service_without_dfe = (not s._dfe_key and s.uuid == uuid_to_search)
                link_match_service_with_dfe = (s._dfe_key and s._dfe_key == dfe_key and uuid_to_search == s._origin_uuid)
                if s.host_name == host_name and (link_match_service_without_dfe or link_match_service_with_dfe):
                    service = s
                    break
            else:  # search by name
                if s.host_name == host_name and s.service_description == name_expr:
                    service = s
                    break
        cache[key] = service
        
        # Returning None is not a problem
        return service
    
    
    def fix_service_notification_options_inheritance(self):
        for service in self:
            service.fix_service_notification_options_inheritance()
    
    
    def __create_host_to_service_temporary_cache_index(self):
        cache_index = {}
        for service in self:
            if not hasattr(service, 'host_name') or not hasattr(service, 'service_description'):
                continue
            host_name = service.host_name
            service_description = service.service_description
            try:
                cache_index[host_name][service_description] = service
            except KeyError:  # first time the host is lookup
                cache_index[host_name] = {service_description: service}
        return cache_index
    
    
    def override_properties(self, hosts):
        from shinken.objects.host import Host
        
        _host_to_service_temporary_index = self.__create_host_to_service_temporary_cache_index()
        
        for host in hosts:
            # We're only looking for hosts having service overrides defined
            if not hasattr(host, 'service_overrides') or not host.service_overrides:
                continue
            
            # NOTE: we use a cache to not compute the exp every time, and cache can be local to host
            # because we cannot have a service across 2 hosts
            cache = {}
            if isinstance(host.service_overrides, list):
                service_overrides = host.service_overrides
            else:
                service_overrides = [so.strip() for so in host.service_overrides.split(Host.properties['service_overrides'].separator)]
            for ovr in service_overrides:
                # Checks service override syntax
                match = self._service_override_pattern.match(ovr)
                if match is None:
                    err = u'Error: invalid service override syntax: %s' % ovr
                    host.configuration_errors.append(err)
                    continue
                name_expr, prop, value = match.groups()
                
                service = self._search_service_for_host_link(host, name_expr, cache, _host_to_service_temporary_index)
                
                # If the service is missing, just means that the synchronizer did send us a useless information about
                # a disabled check or such thing. not a problem, UI will display the warning
                if service is None:
                    logger.debug('NOTE: we cannot find the service for the override %s for the host %s' % (ovr, host.get_name()))
                    continue
                
                # Checks if override is allowed
                excludes = ['host_name', 'service_description', 'use', 'servicegroups', 'trigger', 'trigger_name']
                if prop in excludes:
                    err = "Error: trying to override '%s', a forbidden property for service '%s'" % (prop, name_expr)
                    host.configuration_errors.append(err)
                    continue
                
                # Only check_command args can be overridden (if not a bp_rule)
                if prop == 'check_command_args':
                    check_command = getattr(service, 'check_command')
                    if check_command:
                        command_name = check_command.split('!', 1)[0]
                        if command_name == 'bp_rule':
                            err = "Error: cannot override a bp_rule for '%s' in service '%s'" % (prop, name_expr)
                            host.configuration_errors.append(err)
                            continue
                        else:
                            value = "%s!%s" % (command_name, value)
                            prop = "check_command"
                
                if prop.startswith("_"):
                    custom_name = prop.upper()
                    service.customs[custom_name] = value
                else:
                    if value.startswith('+'):
                        separator = Service.properties[prop].separator
                        value = value[1:]
                        value = separator.join(list(getattr(service, prop, '').split(separator)) + list(value.split(separator)))
                    
                    if value == 'null':
                        if prop in PROPERTIES_RETRIEVABLE_FROM_HOST:
                            if hasattr(host, prop):
                                setattr(service, prop, getattr(host, prop))
                            elif hasattr(service, prop):
                                # What ever the service value was, we want the default value one (we delete it and set the default value)
                                delattr(service, prop)
                                service.fill_default_one_property(prop)
                        else:
                            # The value cannot be take from host, and what ever the service value was,
                            # we want the default value one (we delete it and set the default value)
                            delattr(service, prop)
                            service.fill_default_one_property(prop)
                    else:
                        setattr(service, prop, value)
    
    
    # We can link services with hosts so
    # We can search in O(hosts) instead
    # of O(services) for common cases
    def optimize_service_search(self, hosts, build_service_on_host_index=True):
        self.hosts = hosts
        if build_service_on_host_index:
            for host in self.hosts:  # type: Host
                host.build_service_index()
    
    
    # We just search for each host the id of the host
    # and replace the name by the id
    # + inform the host we are a service of him
    def linkify_s_by_hst(self, hosts):
        for service in self:
            # If we do not have a host_name, we set it as a template element to delete. (like Nagios)
            if not hasattr(service, 'host_name'):
                service.host = None
                continue
            try:
                host_name = service.host_name
                # The new member list, in id
                host = hosts.find_by_name(host_name)
                service.host = host
                service.host_uuid = host.uuid
                # Let the host know we are his service
                if service.host is not None:
                    host.add_service_link(service)
                else:  # Ok, the host do not exist!
                    err = u'Warning: the service \'%s\' got an invalid host_name \'%s\'' % (service.get_name(), host_name)
                    service.configuration_warnings.append(err)
                    continue
            except AttributeError as exp:
                pass  # Will be catch at the is_correct moment
    
    
    # We look for service groups property in services and link them
    def linkify_s_by_sg(self, service_groups):
        for s in self:
            if not s.is_tpl():
                new_service_groups = deque()
                if hasattr(s, 'servicegroups') and len(s.servicegroups) != 0:
                    sgs = s.servicegroups  # already split & strip & unique
                    for sg_name in sgs:
                        sg_name = sg_name.strip()
                        sg = service_groups.find_by_name(sg_name)
                        if sg is not None:
                            new_service_groups.append(sg)
                        else:
                            err = u'Error: the service group \'%s\' of the service \'%s\' is disabled or does not exist' % (sg_name, s.get_dbg_name())
                            s.configuration_errors.append(err)
                s.servicegroups = tuple(new_service_groups)
    
    
    # In the scheduler we need to relink the commandCall with the real commands
    def late_linkify_s_by_commands(self, commands):
        props = ('check_command', 'event_handler')
        for service in self:
            for prop in props:
                cc = getattr(service, prop, None)
                if cc:
                    cc.late_linkify_with_command(commands)
    
    
    # Delete services by ids
    def delete_services_by_id(self, ids):
        # type: (List[int]) -> None
        for service_id in ids:
            del self[service_id]
    
    
    # The configuration UI is not able currently to give check/service poller_tag as None when forcing unset
    # (not take ost one), and give "null" instead. so for this change poller_tag "null"  => "None"
    def fix_null_poller_tag_service_into_real_none(self):
        for s in self:
            if not s.is_tpl():
                if getattr(s, 'poller_tag', 'None') == 'null':  # null here means: take default value
                    s.poller_tag = 'None'
    
    
    @staticmethod
    def _apply_inheritance_from_host(service, hosts):
        # type: (Service, Hosts) -> None
        for prop in PROPERTIES_RETRIEVABLE_FROM_HOST:
            if not hasattr(service, prop) and hasattr(service, u'host_name'):
                h = hosts.find_by_name(service.host_name)
                if h is not None and hasattr(h, prop):
                    setattr(service, prop, getattr(h, prop))
    
    
    @staticmethod
    def _apply_inheritance_from_command_to_host(service, commands, hosts):
        # type: (Service, Commands, Hosts) -> None
        
        for property_name in PROPERTIES_RETRIEVABLE_FROM_COMMAND_TO_HOST:
            if service.apply_inheritance_from_command(service, commands, property_name):
                continue
            
            # If check have a value, we don't want to check parents
            if hasattr(service, property_name):
                setattr(service, u'%s_from' % property_name, u'check')
                continue
            
            if hasattr(service, u'host_name'):
                host = hosts.find_by_name(service.host_name)
                if host is not None and hasattr(host, property_name):
                    setattr(service, property_name, getattr(host, property_name))
                    setattr(service, u'%s_from' % property_name, u'host')
                    continue
            
            setattr(service, u'%s_from' % property_name, u'global')
    
    
    # Apply implicit inheritance for special properties:
    # contact_groups, notification_interval , notification_period
    # So service will take info from host if necessary
    def apply_implicit_inheritance(self, hosts, commands):
        # type:(Hosts, Commands) -> None
        for service in self:
            if not service.is_tpl():
                self._apply_inheritance_from_host(service, hosts)
                self._apply_inheritance_from_command_to_host(service, commands, hosts)
            # Convert notification options
            if hasattr(service, u'notification_options'):
                service.notification_options = service.notification_options.replace(u'd', u'w,c')
    
    
    # Create dependencies for services (daddy ones)
    def apply_dependencies(self):
        for s in self:
            s.fill_daddy_dependency()
    
    
    # For services the main clean is about service with bad hosts
    def clean(self):
        to_remove = [service.id for service in self if not service.host]
        
        for service_id in to_remove:
            del self[service_id]
    
    
    def __create_service_from_another_basic(self, s, hname):
        new_s = s.copy(skip_useless_in_configuration=True)
        new_s.host_name = hname
        if s.is_tpl():  # if template, the new one is not
            new_s.register = 1
        self.items[new_s.id] = new_s
    
    
    def __create_service_from_another_duplicate_for_each(self, s, hname, all_hosts):
        # the generator case, we must create several new services
        # we must find our host, and get all key:value we need
        h = all_hosts.find_by_name(hname.strip())
        
        if h is not None:
            for new_s in s.duplicate(h):
                self.items[new_s.id] = new_s
        else:  # TODO: raise an error?
            err = 'Error: The hostname %s  for the service %s is disabled or does not exist !' % (hname, s.get_name())
            s.configuration_errors.append(err)
    
    
    # Add in our queue a service create from another. Special case:
    # is a template: so hname is a name of template, so need to get all
    # hosts that inherit from it.
    def copy_create_service_from_another(self, all_hosts, s, host_set):
        if getattr(s, 'duplicate_foreach', '') == '':
            # Now really create the services
            for hname in host_set:
                self.__create_service_from_another_basic(s, hname)
        else:  # duplicate for each case
            # Now really create the services
            for hname in host_set:
                self.__create_service_from_another_duplicate_for_each(s, hname, all_hosts)
    
    
    # From a complex template expression, we grok real hosts that match it
    # Her we are managing the cache level
    def _get_hosts_from_hosttemplates_expression(self, htemplates_expression, hosts):
        r = self.__host_templates_expression_to_hosts_cache.get(htemplates_expression, None)
        if r is not None:  # cache hit
            return r
        r = self.evaluate_hostgroup_expression(htemplates_expression, hosts, hosts.templates, look_in='templates')
        self.__host_templates_expression_to_hosts_cache[htemplates_expression] = r
        return r
    
    
    def _get_templates_from_hosttemplates_expression(self, htemplates_expression, hosts):
        r = self.__host_templates_expression_to_templates_cache.get(htemplates_expression, None)
        if r is not None:  # cache hit
            return r
        r = self.evaluate_hostgroup_expression(htemplates_expression, hosts, hosts.templates, look_in='onlytemplates')
        self.__host_templates_expression_to_templates_cache[htemplates_expression] = r
        return r
    
    
    # From a complex template expression, we grok templates that match it, so we know in the next step that
    # the checks are from this templates, and if we must keep them from a exclude point of view (exclude is
    # computed BY TEMPLATE so we must know which template came with a specific check)
    def _tag_templates_with_service_definition_from_hosttemplates_expression(self, htemplates_expression, hosts, service_uuid):
        host_templates_names = self._get_templates_from_hosttemplates_expression(htemplates_expression, hosts)
        for host_templates_name in host_templates_names:
            template = hosts.find_template(host_templates_name)
            if template is None:
                continue
            template.set_templates_with_service_uuid_recursive(hosts, service_uuid)
    
    
    @staticmethod
    def _compute_prefix_from_template(hname, item):
        return 'tpl!' + hname if item.is_tpl() else 'host!' + hname
    
    
    @staticmethod
    def extract_uuid_from_raw_link_string(link_string):
        # type: (unicode) -> List[unicode]
        try:
            if PREFIX_LINK_UUID not in link_string:
                raise IncorrectRawLinkException(u'The link [ %s ] is not a correct link list' % link_string)
            
            # We are supposed to get a "list" of links separated by comas
            links = link_string.split(LINK_SEP, 2)
            # Possible links format are:
            # - uuid!LINK_UUID
            # - uuid!LINK_UUID!DFE_KEY_NAME!DFE_KEY
            return [link.split(RAW_LINK_SEP)[1].strip() for link in links]
        except IndexError:
            raise IncorrectRawLinkException(u'The link [ %s ] is not a correct link list' % link_string)
    
    
    # We create new service if necessary (host groups and co)
    def explode(self, hosts, hostgroups, servicegroups, servicedependencies):
        # The "old" services will be removed. All services with
        # more than one host or a host group will be in it
        srv_to_remove = []
        
        t0 = time.time()
        # items::explode_host_groups_into_hosts
        # take all hosts from our hostgroup_name into our host_name property
        _split_patern = re.compile("[,&|()]+")
        for service in self:
            service.from_template = [Services._compute_prefix_from_template(hname, service) for hname in _split_patern.split(getattr(service, 'host_name', '')) if hname]
            if service.is_tpl():
                for (prop, entry) in service.properties.iteritems():
                    if not entry.handle_additive_inheritance:
                        continue
                    if not hasattr(service, prop):
                        continue
                    value = getattr(service, prop)
                    if value.startswith('+'):
                        setattr(service, prop, value[1:])
        
        logger.info('[performance] - (detail for support only) explode::services::from templates [%.2f]' % (time.time() - t0))
        
        t0 = time.time()
        self.explode_host_groups_into_hosts(hosts, hostgroups)
        logger.info('[performance] - (detail for support only) explode::services::explode_host_groups_into_hosts [%.2f]' % (time.time() - t0))
        
        t0 = time.time()
        # Then for every host create a copy of the service with just the host
        # because we are adding services, we can't just loop in it
        service_to_check = self.items.keys()
        for id in service_to_check:
            service = self.items[id]
            duplicate_for_hosts = []  # get the list of our host_names if more than 1
            not_hosts = []  # the list of !host_name so we remove them after
            
            # Services must have a name or service description
            if not getattr(service, 'service_description', getattr(service, 'name', '')):
                self.configuration_errors.append("A service is missing description or name. Cannot apply configuration.")
            
            # If do not have an host_name, just delete it
            if not getattr(service, 'host_name', ''):
                srv_to_remove.append(service.id)
                continue
            
            # if not s.is_tpl(): # Exploding template is useless
            # Explode for real service or template with a host_name
            hnames = service.host_name.split(',')
            hnames = strip_and_uniq(hnames)
            # We will duplicate if we have multiple host_name
            # or if we are a template (so a clean service)
            if len(hnames) >= 2 or service.is_tpl() or (getattr(service, 'duplicate_foreach', '') != ''):
                for hname in hnames:
                    # If the name begin with a !, we put it in
                    # the not list
                    if hname.startswith('!'):
                        not_hosts.append(hname[1:])
                    else:  # the standard list
                        duplicate_for_hosts.append(hname)
                
                # remove duplicate items from duplicate_for_hosts:
                duplicate_for_hosts = list(set(duplicate_for_hosts))
                
                # Ok now we clean the duplicate_for_hosts with all hosts
                # of the not
                for hname in not_hosts:
                    if hname in duplicate_for_hosts:
                        duplicate_for_hosts.remove(hname)
                
                # Now we duplicate the service for all hosts
                hosts_set = set()
                for hname in duplicate_for_hosts:
                    if not service.is_tpl():
                        hosts_set.add(hname)
                    else:
                        # For template it's more tricky: it's a template name
                        # we've got, not a real host_name/ So we must get a list of host_name
                        # that use this template
                        # Use the complex expression manager for it, it will call find_hosts_that_use_template
                        # for the templates it think it's useful
                        htemplates_expression = hname
                        hosts_for_this_expression = self._get_hosts_from_hosttemplates_expression(htemplates_expression, hosts)
                        hosts_set = hosts_set.union(hosts_for_this_expression)
                        self._tag_templates_with_service_definition_from_hosttemplates_expression(htemplates_expression, hosts, service.uuid)
                
                self.copy_create_service_from_another(hosts, service, hosts_set)
                
                # Since we exploded the service, original service must be deleted.
                # Keep ones that have configuration errors for error handling
                if not service.configuration_errors:
                    srv_to_remove.append(id)
            
            else:  # Maybe the hnames was full of same host, so we must reset the name
                for hname in hnames:  # So even if len == 0, we are protected
                    service.host_name = hname
        logger.info('[performance] - (detail for support only) explode::services::copy_create_service_from_another   %.2f' % (time.time() - t0))
        
        t0 = time.time()
        # We clean all service that was for multiple hosts.
        self.delete_services_by_id(srv_to_remove)
        logger.info('[performance] - (detail for support only) explode::services::delete_services_by_id   %.2f' % (time.time() - t0))
        
        t0 = time.time()
        # Servicegroups property need to be fullfill for got the informations
        # And then just register to this service_group
        for service in self:
            if not service.is_tpl() and hasattr(service, 'service_description'):
                sname = service.service_description
                shname = getattr(service, 'host_name', '')
                if hasattr(service, 'servicegroups'):
                    sgs = service.servicegroups.split(',')
                    for sg in sgs:
                        servicegroups.add_member(shname + ',' + sname, sg)
        logger.info('[performance] - (detail for support only) explode::services::service groups add members   %.2f' % (time.time() - t0))
        
        t0 = time.time()
        # Now we explode service_dependencies into Servicedependency
        # We just create serviceDep with goods values (as STRING!),
        # the link pass will be done after
        for service in self:
            # Templates are useless here
            if not service.is_tpl():
                if hasattr(service, 'service_dependencies'):
                    if service.service_dependencies != '':
                        sdeps = service.service_dependencies.split(',')
                        # %2=0 are for hosts, !=0 are for service_description
                        i = 0
                        hname = ''
                        for elt in sdeps:
                            if i % 2 == 0:  # host
                                hname = elt.strip()
                            else:  # description
                                desc = elt.strip()
                                # we can register it (s) (depend on) -> (hname, desc)
                                # If we do not have enough data for s, it's no use
                                if hasattr(service, 'service_description') and hasattr(service, 'host_name'):
                                    if hname == '':
                                        hname = service.host_name
                                    servicedependencies.add_service_dependency(service.host_name, service.service_description, hname, desc)
                            i += 1
        logger.info('[performance] - (detail for support only) explode::services::service dependencies   %.2f' % (time.time() - t0))
    
    
    # The contactgroups -> contacts explode must be done only AFTER
    # we did inherit from the hosts (same as host).
    def post_inheritance_explode(self, contactgroups):
        t0 = time.time()
        # items::explode_contact_groups_into_contacts
        # take all contacts from our contact_groups into our contact property
        self.explode_contact_groups_into_contacts(contactgroups)
        logger.info('[performance] - (detail for support only) explode::services::explode_contact_groups_into_contacts   %.2f' % (time.time() - t0))
    
    
    # Will create all business tree for the services
    def create_business_rules(self, proxy_items):
        for s in self:
            s.create_business_rules(proxy_items)
    
    
    # Will link all business service/host with theirs
    # dep for problem/impact link
    def create_business_rules_dependencies(self, hosts, services):
        for s in self:
            s.create_business_rules_dependencies(hosts, services)
