import threading
import traceback

from .log import logger
from .daemon import Interface
from .satellite import BaseSatellite


# Interface for method from the arbiter to the daemon with the inventory
class IArbiterToInventorySatellite(Interface):
    # Used by the arbiter to push us the realm inventory
    def push_realm_inventory(self, realm_name, realm_inventory):
        return self.app.load_realm_inventory(realm_name, realm_inventory)
    
    
    push_realm_inventory.doc = '(internal) push a realm inventory to'
    push_realm_inventory.method = 'POST'
    push_realm_inventory.need_lock = False  # the daemon function manage with a lock
    push_realm_inventory.display_name = u'Arbiter server send element inventory to the daemon'
    
    
    # Used by the arbiter to know if we already have a inventory for a realm, and if so avoid
    # pushing a new inventory
    def already_have_realm_inventory(self, realm_name, inventory_hash):
        return self.app.already_have_realm_inventory(realm_name, inventory_hash)
    
    
    already_have_realm_inventory.doc = '(internal) returns if the daemon already have the inventory based on its hash'
    already_have_realm_inventory.need_lock = False  # the daemon function manage with a lock


# Class for daemons with inventories, currently Broker & Receiver
class WithInventorySatellite(BaseSatellite):
    properties = BaseSatellite.properties.copy()
    
    
    def __init__(self, name, config_file, is_daemon, do_replace, debug, debug_file, daemon_id=0):
        super(WithInventorySatellite, self).__init__(name, config_file, is_daemon, do_replace, debug, debug_file, daemon_id)
        # Will have one inventory for each realm
        self._realms_inventory = {}
        # each time we receive a inventory, compute differences with previous inventory (is any so modules can faster load it)
        self._realms_inventory_differences = {}
        self._realms_inventory_lock = threading.RLock()
        # The arbiter let us know about the realms that are allowed to talk to us
        # it let us know also if a realm that was present before did disapear and so need to be deleted
        self.known_realms = []
    
    
    # We will give all receiver modules a way to have the inventory if they want
    def assert_module_inventory_are_updated(self):
        with self._realms_inventory_lock:
            for instance in self.modules_manager.get_all_alive_instances():
                f = getattr(instance, 'update_inventory_from_daemon', None)
                if f is None:
                    continue
                try:
                    f(self._realms_inventory, self._realms_inventory_differences)
                except Exception as exp:
                    logger.warning("The mod %s raise an exception: %s, I'm tagging it to restart later" % (instance.get_name(), str(exp)))
                    logger.warning("Exception type: %s" % type(exp))
                    logger.warning("Back trace of this kill: %s" % (traceback.format_exc()))
                    self.modules_manager.did_crash(instance, "The mod %s raise an exception: %s" % (instance.get_name(), str(exp)))
    
    
    # When a realm is renamed/deleted on the arbiter side we need to
    # clean it on our inventory list so modules will be updated too
    # NOTE: you WILL have realms that are not already in the inventory
    #       because realms are pushing it after schedulers, so it take time
    def _clean_old_realms_in_inventory(self):
        with self._realms_inventory_lock:
            in_inventory = set(self._realms_inventory.keys())
            known_realms = set(self.known_realms)
            for realm_name in in_inventory - known_realms:
                logger.info('Removing the realm %s from the inventory' % realm_name)
                del self._realms_inventory[realm_name]
    
    
    def already_have_realm_inventory(self, realm_name, checked_inventory_hash):
        with self._realms_inventory_lock:
            realm_inventory = self._realms_inventory.get(realm_name, None)
            if realm_inventory is None:
                return False
            current_realm_inventory_hash = realm_inventory.get_hash()
            return current_realm_inventory_hash == checked_inventory_hash
    
    
    def load_realm_inventory(self, realm_name, realm_inventory):
        with self._realms_inventory_lock:
            old_inventory = self._realms_inventory.get(realm_name, None)
            self._realms_inventory[realm_name] = realm_inventory
            logger.info('The realm %s inventory is now updated with %s elements and the hash %s' % (realm_name, realm_inventory.get_len(), realm_inventory.get_hash()))
            # Now compute differences for this realm from the old one
            # * new elements
            # * updated elements
            # * removed elements
            inventory_differences = None
            if old_inventory is not None:
                inventory_differences = old_inventory.compute_differences_from_new_inventory(realm_inventory)
            # set or reset differences
            self._realms_inventory_differences[realm_name] = inventory_differences
            logger.debug('DID COMPUTED DIFFERENCES: %s' % inventory_differences)
