#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import time
from collections import namedtuple

from shinken.misc.os_utils import ShinkenAccessRightsError
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.data_hub.data_hub_exception.data_hub_exception import DataHubNotReady, DataHubFatalException, DataHubItemNotFound, DataHubCorruptedData

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Tuple, Any, Dict
    from shinken.log import PartLogger
    from shinkensolutions.data_hub.data_hub_driver.abstract_data_hub_driver import AbstractDataHubDriver, AbstractDataHubDriverConfig


def handle_data_hub_fatal_error(func):
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except DataHubFatalException as err:
            self = args[0]
            self._logger_base_hub.error(u'The Data Hub stopped because of this fatal error : %s' % err)
            raise DataHubFatalException(err)
    
    
    return wrapper


DataHubItem = namedtuple(u'DataHubItem', [u'item_data', u'last_modification'])


class DataHubConfig(object):
    
    def __init__(self, data_hub_id, data_hub_category, data_type, data_id_key_name, driver_config, must_save_configuration=False):
        # type: (unicode, unicode, unicode, unicode, AbstractDataHubDriverConfig, bool) -> None
        self.data_hub_id = data_hub_id
        self.data_hub_category = data_hub_category
        self.data_type = data_type
        self.data_id_key_name = data_id_key_name
        self.driver_config = driver_config
        self.must_save_configuration = must_save_configuration


class DataHub(object):
    def __init__(self, logger, data_hub_id, data_type, data_id_key_name, data_hub_driver):
        # type: (PartLogger, unicode, unicode, unicode, AbstractDataHubDriver) -> None
        self._data_hub_id = data_hub_id
        self._data_type = data_type
        self._data_hub_driver = data_hub_driver
        self._data_id_key_name = data_id_key_name
        
        self._is_initialized = False
        
        self._logger = logger
        self._logger_base_hub = self._logger.get_sub_part(u'DATA HUB : %s' % data_hub_id)
        self._logger_hub_init = self._logger_base_hub.get_sub_part(u'INIT')
        self._logger_hub_save = self._logger_base_hub.get_sub_part(u'SAVE')
        self._logger_hub_load = self._logger_base_hub.get_sub_part(u'GET')
        self._logger_hub_retention = self._logger_base_hub.get_sub_part(u'SAVE RETENTION')
        self._logger_hub_remove = self._logger_base_hub.get_sub_part(u'REMOVE')
        self._logger_hub_last_mod = self._logger_base_hub.get_sub_part(u'GET LAST MODIF')
        self._logger_backup = self._logger_base_hub.get_sub_part(u'BACKUP')
        self._logger_restore = self._logger_base_hub.get_sub_part(u'RESTORE')
    
    
    def init(self):
        self._data_hub_driver.init()
        self._verify_right_access_on_all_data()
        self._is_initialized = True
    
    
    def get_id(self):
        return self._data_hub_id
    
    
    def _verify_right_access_on_all_data(self):
        self._data_hub_driver.check_right_access_issues_then_fix_them()
    
    
    def is_initialized(self):
        return self._is_initialized
    
    
    def check_data_in_hub(self, all_data_id):
        # type: (List[unicode]) -> Tuple[List[unicode],List[unicode]]
        self._logger_hub_init.info(u'Starting check %s %s' % (len(all_data_id), self._data_type))
        
        if not self.is_initialized():
            self._logger_hub_init.error(u'The data hub is not initialized')
            raise DataHubNotReady()
        
        start = time.time()
        
        valid_data_id = []
        invalid_data_id = []
        for data_id in all_data_id:
            is_data_correct = self._data_hub_driver.is_data_correct(data_id)
            if is_data_correct:
                self._logger_hub_init.info(u'%s with %s [ %s ] successfully check' % (self._data_type, self._data_id_key_name, data_id))
                valid_data_id.append(data_id)
            else:
                self._logger_hub_init.info(u'%s with %s [ %s ] is invalid' % (self._data_type, self._data_id_key_name, data_id))
                invalid_data_id.append(data_id)
        
        elapsed = time.time() - start
        self._logger_hub_init.info(u'Check data in hub ended in [ %0.3f ] seconds' % elapsed)
        return valid_data_id, invalid_data_id
    
    
    def get_lock(self, data_id):
        return self._data_hub_driver.get_lock(data_id)
    
    
    @handle_data_hub_fatal_error
    def get_data(self, data_id):
        # type: (unicode) -> Any
        if not self.is_initialized():
            self._logger_hub_load.error(u'Cannot get the %s with %s [ %s ] because the Data Hub is not initialized, please retry later' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubNotReady()
        
        try:
            data = self._data_hub_driver.read(data_id, log_error=True)
        except DataHubItemNotFound as err:
            self._logger_hub_load.info(u'The %s with %s [ %s ] was not found in the Data Hub' % (self._data_type, self._data_id_key_name, data_id))
            raise err
        except ShinkenAccessRightsError as err:
            raise DataHubFatalException(err)
        except ValueError as err:
            self._logger_hub_load.error(u'The %s with %s [ %s ] could not be retrieved because its data is corrupted' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubCorruptedData(err.message)
        
        return data
    
    
    @handle_data_hub_fatal_error
    def get_data_and_last_modification(self, data_id):
        # type: (unicode) -> DataHubItem
        if not self.is_initialized():
            self._logger_hub_load.error(u'Cannot get the %s with %s [ %s ] because the Data Hub is not initialized, please retry later' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubNotReady()
        
        try:
            data, last_modification = self._data_hub_driver.read_and_get_last_modification_date(data_id, log_error=True)
        except DataHubItemNotFound as err:
            self._logger_hub_load.info(u'The %s with %s [ %s ] was not found in the Data Hub' % (self._data_type, self._data_id_key_name, data_id))
            raise err
        except ShinkenAccessRightsError as err:
            raise DataHubFatalException(err)
        except ValueError as err:
            self._logger_hub_load.error(u'The %s with %s [ %s ] could not be retrieved because its data is corrupted' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubCorruptedData(err.message)
        
        return DataHubItem(data, last_modification)
    
    
    @handle_data_hub_fatal_error
    def get_last_modification_date(self, data_id):
        # type: (unicode) -> int
        if not self.is_initialized():
            self._logger_hub_last_mod.error(u'Cannot get the last modification time of %s with %s [ %s ] because the Data Hub is not initialized, please retry later' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubNotReady()
        
        try:
            last_modification_date = self._data_hub_driver.get_last_modification_date(data_id)
        except DataHubItemNotFound as err:
            self._logger_hub_last_mod.info(u'The %s with %s [ %s ] was not found in the Data Hub' % (self._data_type, self._data_id_key_name, data_id))
            raise err
        except ShinkenAccessRightsError as err:
            raise DataHubFatalException(err)
        except ValueError as err:
            self._logger_hub_last_mod.error(u'The %s with %s [ %s ] could not be retrieved because its data is corrupted' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubCorruptedData(err.message)
        
        return last_modification_date
    
    
    @handle_data_hub_fatal_error
    def save_data(self, data_id, data):
        # type: (unicode, Any) -> None
        if not self.is_initialized():
            self._logger_hub_save.error(u'Cannot save the %s with %s [ %s ] because the Data Hub is not initialized, please retry later' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubNotReady()
        
        try:
            self._data_hub_driver.write(data_id, data)
        except Exception as e:
            raise DataHubFatalException(e.message)
    
    
    @handle_data_hub_fatal_error
    def remove_data(self, data_id):
        # type: (unicode) -> None
        if not self.is_initialized():
            self._logger_hub_remove.error(u'Cannot remove the %s with %s [ %s ] because the Data Hub is not initialized, please retry later' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubNotReady()
        
        try:
            self._data_hub_driver.remove(data_id)
        except Exception as e:
            raise DataHubFatalException(e.message)
    
    
    @handle_data_hub_fatal_error
    def remove_data_and_lock(self, data_id):
        # type: (unicode) -> None
        if not self.is_initialized():
            self._logger_hub_remove.error(u'Cannot remove the %s with %s [ %s ] because the Data Hub is not initialized, please retry later' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubNotReady()
        
        try:
            self._data_hub_driver.hard_remove(data_id)
        except Exception as e:
            raise DataHubFatalException(e.message)
    
    
    @handle_data_hub_fatal_error
    def get_number_of_stored_data(self):
        # type: () -> int
        if not self.is_initialized():
            self._logger_hub_init.error(u'The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return self._data_hub_driver.get_number_of_stored_data()
        except Exception as e:
            raise DataHubFatalException(e.message)
    
    
    @handle_data_hub_fatal_error
    def get_all_data_id(self):
        # type: () -> List[unicode]
        if not self.is_initialized():
            self._logger_hub_init.error(u'The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return self._data_hub_driver.get_all_data_id()
        except Exception as e:
            raise DataHubFatalException(e.message)
    
    
    @handle_data_hub_fatal_error
    def get_all_data_with_id(self):
        # type: () -> List[Tuple[unicode, Any]]
        if not self.is_initialized():
            self._logger_hub_init.error(u'The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return [(data_id, self._data_hub_driver.read(data_id)) for data_id in self._data_hub_driver.get_all_data_id()]
        except Exception as e:
            raise DataHubFatalException(e.message)
    
    
    @handle_data_hub_fatal_error
    def get_all_data(self):
        # type: () -> List[Any]
        if not self.is_initialized():
            self._logger_hub_init.error(u'The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return [(data_id, self._data_hub_driver.read(data_id)) for data_id in self._data_hub_driver.get_all_data_id()]
        except Exception as e:
            raise DataHubFatalException(e.message)
    
    
    @handle_data_hub_fatal_error
    def destroy(self):
        # type: () -> None
        if not self.is_initialized():
            self._logger_hub_init.error(u'The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            self._data_hub_driver.destroy()
        except Exception as e:
            raise DataHubFatalException(e.message)
    
    
    @handle_data_hub_fatal_error
    def get_total_size(self):
        # type: () -> int
        if not self.is_initialized():
            self._logger_hub_init.error(u'The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return self._data_hub_driver.get_total_size()
        except Exception as e:
            self._logger_hub_init.error(e)
            raise DataHubFatalException(e.message)
    
    
    def backup(self):
        # type: () -> Dict[unicode, Any]
        if not self.is_initialized():
            self._logger_hub_init.error(u'The data hub is not initialized')
            raise DataHubNotReady()
        
        backup = {}
        for data_id in self._data_hub_driver.get_all_data_id():
            data = self._data_hub_driver.read(data_id)
            backup[data_id] = data
        return backup
    
    
    def restore(self, backup_data):
        # type: (Dict[unicode, Any]) -> None
        if not self.is_initialized():
            self._logger_hub_init.error(u'The data hub is not initialized')
            raise DataHubNotReady()
        
        for item_id, item_data in backup_data.iteritems():
            self._data_hub_driver.restore(item_id, item_data)
    
    
    def sanitize_data(self, item_id, item_data):
        # type: (unicode, Any) -> None
        if not self.is_initialized():
            self._logger_hub_init.error(u'The data hub is not initialized')
            raise DataHubNotReady()
        
        self._data_hub_driver.sanitize(item_id, item_data)
