#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

import json
import os
import random
import shutil
import time
from subprocess import Popen, PIPE

try:
    import pwd
    import grp
except ImportError:  # windows
    pwd = grp = None

WHISPER_DIR = u'/opt/graphite/storage/whisper'
NB_METRICS_COUNT_FILE = u'/opt/graphite/storage/whisper/.nb_metrics'


class GRAPHITE_STATS_KEY(object):
    TIME_READ = u'graphite_stats_time'
    LOCAL_TIME = u'graphite_local_time'


class GraphiteMetricsCounter(object):
    UNWANTED_METRIC_DIRECTORIES = (u'carbon', u'.cacheinvalidation', u'.nb_metrics')
    
    
    def __init__(self):
        self.metrics = 0
        self.level_0 = 0
    
    
    def update_count(self):
        self._update_level_0()
        self._do_compute_number_of_metrics_file()
        graphite_stats_time = int(time.time())
        self._write_metric_file(graphite_stats_time)
    
    
    def _update_level_0(self):
        self.level_0 = len([folder for folder in os.listdir(WHISPER_DIR) if folder not in self.UNWANTED_METRIC_DIRECTORIES])
    
    
    # This will be computed by the iostats_collector script
    def _do_compute_number_of_metrics_file(self):
        # type: () -> None
        self.metrics = 0
        
        try:
            # It's fastest to go with a find command instead of doing it ourselves
            cmd = u'''find %s -type f  -name '*.wsp'  | grep -v '%s/carbon' | wc -l''' % (WHISPER_DIR, WHISPER_DIR)
            p = Popen(cmd, stdout=PIPE, stderr=PIPE, shell=True)
            output, stderr = p.communicate()
            self.metrics = int(output.strip())
            return
        except Exception:
            pass
        # oops, something was wrong, switch to a manual mode (maybe we are no mor allowed to launch command, like under apache)
        for root, dirs, files in os.walk(WHISPER_DIR):
            root = root.replace(WHISPER_DIR, '')
            if root.startswith(self.UNWANTED_METRIC_DIRECTORIES):
                continue
            for basename in files:
                if basename.endswith(u'.wsp'):
                    self.metrics += 1
    
    
    def _get_count_export(self, graphite_stats_time):
        return {
            u'metrics'                  : self.metrics,
            u'level_0'                  : self.level_0,
            GRAPHITE_STATS_KEY.TIME_READ: graphite_stats_time
        }
    
    
    def _write_metric_file(self, graphite_stats_time):
        # type: (int) -> None
        tmp_file = u'%s.tmp.%d' % (NB_METRICS_COUNT_FILE, random.randint(1, 100000))
        count = self._get_count_export(graphite_stats_time)
        with open(tmp_file, u'w') as f:
            f.write(json.dumps(count))
        os.chown(tmp_file, pwd.getpwnam('apache').pw_uid, grp.getgrnam('apache').gr_gid)
        shutil.move(tmp_file, NB_METRICS_COUNT_FILE)  # atomic move
