#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import errno
import os
import socket
import struct
import subprocess
import sys
from stat import S_ISREG, S_IRUSR, S_IWUSR, S_IXUSR, S_IRGRP, S_IWGRP, S_IXGRP, S_IROTH, S_IWOTH, S_IXOTH

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.util import make_unicode

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Dict

try:
    import fcntl
except ImportError:
    fcntl = None

try:
    from pwd import getpwnam
    from grp import getgrnam
except ImportError as e:
    # temporary workaround:
    logger.debug('pwd and grp modules not available on this platform: %s => using mocks instead', e)
    from collections import namedtuple
    
    PasswordEntry = namedtuple('PasswordEntry', 'pw_name pw_passwd pw_uid pw_gid pw_gecos pw_dir pw_shell')
    
    
    def getpwnam(user):
        return PasswordEntry(pw_name='shinken',
                             pw_passwd='',
                             pw_uid=0,
                             pw_gid=0,
                             pw_gecos='Shinken User',
                             pw_dir='C:/Users/shinken',
                             pw_shell='cmd.exe')
    
    
    GroupEntry = namedtuple('GroupEntry', 'gr_name gr_passwd gr_gid gr_mem')
    
    
    def getgrnam(group):
        return GroupEntry(gr_name='shinken',
                          gr_passwd='',
                          gr_gid=0,
                          gr_mem='shinken')

# This dict use the uniw mask to compute the rights
unix_rights = {
    'user' : {
        'read'   : S_IRUSR,
        'write'  : S_IWUSR,
        'execute': S_IXUSR
    },
    'group': {
        'read'   : S_IRGRP,
        'write'  : S_IWGRP,
        'execute': S_IXGRP
        
    },
    'other': {
        'read'   : S_IROTH,
        'write'  : S_IWOTH,
        'execute': S_IXOTH
    }
}


def check_right_on_path(path, mode='read', path_type='directory', user=None, group=None):
    ''' Use this function to check the rights on folder and file.
    :param path: the path to check
    :param mode: which mode you want for your path : 'read', 'write', 'execute'
    :param path_type: is your path is a directory or a file
    :param user: on which user check right ? if None, it will use the current process’s effective user id
    :param group: on which group check right ? if None, it will use the current process’s effective group id
    :return: True if rights are OK. Raise if you can access to your file
    '''
    
    path = os.path.join(path)
    # First check if the path exists
    if not os.path.exists(path):
        raise OSError(errno.ENOENT, 'The path doesn\'t exists', path)
    
    # Is the target is the good one ?
    if (path_type == 'directory' and not os.path.isdir(path)) or \
            (path_type == 'file' and not os.path.isfile(path)):
        raise OSError(errno.ENOENT, 'This is not a %s  : ' % (path_type), path)
    
    # Will check if we use the user, the group or other
    user_uid = getpwnam(user).pw_uid if user else os.geteuid()
    
    if user_uid == 0:
        # You're root ? ... ok you can go
        return True
    
    group_gid = getgrnam(group).gr_gid if group else os.getegid()
    file_stat = os.stat(path)
    
    if user_uid == file_stat.st_uid:
        key_used = 'user'
    elif group_gid == file_stat.st_gid:
        key_used = 'group'
    else:
        key_used = 'other'
    
    # Now will check the rights with the good mask
    if bool(file_stat.st_mode & unix_rights[key_used][mode]):
        return True
    else:
        raise OSError(errno.EACCES, 'User %s can not access to the %s with the %s mode ' % (user, path_type, mode), path)


def check_right_on_file(path, mode='read', user=None, group=None):
    ''' Use this function to check the rights on file.
    :param path: the path to check
    :param mode: which mode you want for your path : 'read', 'write', 'execute'
    :param user: on which user check right ? if None, it will use the current process’s effective user id
    :param group: on which group check right ? if None, it will use the current process’s effective group id
    :return: True if rights are OK. Raise if you can access to your file
    '''
    return check_right_on_path(path, mode, 'file', user, group)


def check_right_on_directory(path, mode='read', user=None, group=None):
    ''' Use this function to check the rights on folder.
    :param path: the path to check
    :param mode: which mode you want for your path : 'read', 'write', 'execute'
    :param user: on which user check right ? if None, it will use the current process’s effective user id
    :param group: on which group check right ? if None, it will use the current process’s effective group id
    :return: True if rights are OK. Raise if you can access to your folder
    '''
    return check_right_on_path(path, mode, 'directory', user, group)


def create_tree(path, user='shinken', group='shinken', mode=0o755):
    # Warning! the mode will be overwrite by os.umask if set
    full_path = [p for p in path.split(os.sep) if p]
    # to manage relatif path or not
    parents = '/' if os.path.isabs(path) else ''
    for dir in full_path:
        
        current_path = os.path.join(parents, dir)
        
        if not os.path.exists(current_path):
            try:
                check_right_on_directory(parents, 'write')
            except OSError:
                raise OSError(errno.EACCES, 'Cannot create the full tree. Permission denied ', parents)
            os.mkdir(current_path, mode)
            set_ownership(current_path, user, group)
        
        try:
            check_right_on_directory(current_path, 'read', user, group)
            check_right_on_directory(current_path, 'execute', user, group)
        except OSError:
            raise OSError(errno.EACCES, 'The path already exists but cannot jump in. Permission denied ', current_path)
        
        # Now, next level
        parents = current_path


def run_command(command):
    # type: (unicode) -> unicode
    _, stdout, stderr = run_command_with_return_code(command)
    return u'%s%s' % (stdout, stderr)


def run_command_with_return_code(command, shell=True, env=None):
    # type: (unicode, bool, Optional[Dict]) -> (int, unicode, unicode)
    if env is None:
        _process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=shell)
    else:
        _process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=shell, env=env)
    stdout, stderr = _process.communicate()
    return _process.returncode, make_unicode(stdout), make_unicode(stderr)


def is_regular_file(path):
    if os.path.exists(path):
        mode = os.stat(path).st_mode
        return S_ISREG(mode)
    return False


if sys.platform.startswith('win'):
    import tempfile
    
    SHARE_ITEM_SYNC_FILE_PATH = os.path.join(tempfile.gettempdir(), u'share_item')
    
    
    def set_ownership(path, user=u'shinken', group=u'shinken', is_link=False):
        # type: (unicode, unicode, unicode, bool) -> None
        pass
    
    
    def rename_file(source, target):
        if os.path.exists(target):
            os.remove(target)
        os.rename(source, target)
else:
    from pwd import getpwnam
    from grp import getgrnam
    
    SHARE_ITEM_SYNC_FILE_PATH = '/dev/shm/share_item'
    
    
    def set_ownership(path, user=u'shinken', group=u'shinken', is_link=False):
        # type: (unicode, unicode, unicode, bool) -> None
        try:
            uid = getpwnam(user).pw_uid
        except KeyError:
            raise OSError(errno.EINVAL, u'This user doesn\'t exists', user)
        try:
            gid = getgrnam(group).gr_gid
        except KeyError:
            raise OSError(errno.EINVAL, u'This group doesn\'t exists', group)
        os.chown(path, uid, gid)
        if is_link:
            os.lchown(path, uid, gid)
    
    
    def rename_file(source, target):
        os.rename(source, target)


def get_file_info(path):
    file_stat = os.stat(path)
    # thanks to https://github.com/naparuba/opsbro/blob/master/opsbro/compliancemgr.py#L69
    file_permissions = int(oct(file_stat.st_mode & 0o777)[1:])  # => to have something like 644
    file_owner = getpwuid(file_stat.st_uid).pw_name
    file_group = getgrgid(file_stat.st_gid).gr_name
    return file_permissions, file_owner, file_group


def get_linux_local_addresses(_logger):
    try:
        rc, stdout, stderr = run_command_with_return_code(u'hostname -I')
    except Exception as exp:
        _logger.info(u'Cannot use the hostname -I call for linux (%s), trying to guess local addresses' % exp)
        stdout = ''
    buf = stdout.decode(u'utf8', u'ignore').strip()
    res = [s.strip() for s in buf.split(' ') if s.strip()]
    
    # Some system like in alpine linux that don't have hostname -I call so try to guess
    if len(res) == 0:
        _logger.debug(u'Cannot use the hostname -I call for linux, trying to guess local addresses')
        for prefi in (u'bond', u'eth', u'venet', u'wlan'):
            for i in range(0, 10):
                ifname = u'%s%d' % (prefi, i)
                try:
                    addr = _get_ip_address(ifname)
                    res.append(addr)
                except IOError:  # no such interface
                    pass
    res = sorted(res, cmp=_sort_local_addresses)
    return res


def _get_ip_address(ifname):
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    return socket.inet_ntoa(fcntl.ioctl(
        s.fileno(),
        0x8915,  # SIOCGIFADDR
        struct.pack(u'256s', ifname[:15])
    )[20:24])


def _sort_local_addresses(addr1, addr2):
    addr1_is_192 = addr1.startswith(u'192.')
    addr2_is_192 = addr2.startswith(u'192.')
    addr1_is_10 = addr1.startswith(u'10.')
    addr2_is_10 = addr2.startswith(u'10.')
    addr1_is_172 = addr1.startswith(u'172.')
    addr2_is_172 = addr2.startswith(u'172.')
    addr1_is_127 = addr1.startswith(u'127.')
    addr2_is_127 = addr2.startswith(u'127.')
    
    # lower is better
    addr1_order = 4
    if addr1_is_192:
        addr1_order = 1
    elif addr1_is_172:
        addr1_order = 2
    elif addr1_is_10:
        addr1_order = 3
    if addr1_is_127:
        addr1_order = 5
    addr2_order = 4
    if addr2_is_192:
        addr2_order = 1
    elif addr2_is_172:
        addr2_order = 2
    elif addr2_is_10:
        addr2_order = 3
    if addr2_is_127:
        addr2_order = 5
    
    if addr1_order > addr2_order:
        return 1
    elif addr1_order < addr2_order:
        return -1
    return 0


def create_symlink_from_os_version(symlink_path, os_version, user=u'shinken'):
    # type: (unicode, unicode, unicode) -> bool
    target_path = u'%s_%s' % (symlink_path, os_version)
    # The link target dos not exists
    if not os.path.exists(target_path):
        raise OSError(u'The link target does not exists')
    
    if os.path.islink(symlink_path):
        old_target_name = os.path.basename(os.readlink(symlink_path))
        target_name = os.path.realpath(target_path)
        # There is already a link pointing the the wnated desitination
        if target_name == old_target_name:
            return False
    
    # We want to override the existing file / link
    if os.path.exists(symlink_path):
        # Remove the symlink_path, but handle the symlinks. Will NOT remove the target if the symlink_path is a symbolic link
        os.unlink(symlink_path)
    
    os.symlink(target_path, symlink_path)
    set_ownership(symlink_path, user, user, is_link=True)
    return True
