import zlib
import os
import time

from .log import logger
from .misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, NoReturn, Bool

# 500Mo= 524288000
SHINKEN_COMPRESSOR_MAX_CHUNK_SIZE = int(os.environ.get('SHINKEN_COMPRESSOR_MAX_CHUNK_SIZE', '524288000'))
SHINKEN_LOG_SHINKEN_COMPRESSOR_FLAG_NAME = 'SHINKEN_LOG_SHINKEN_COMPRESSOR_FLAG'
SHINKEN_LOG_SHINKEN_COMPRESSOR_FLAG = os.environ.get(SHINKEN_LOG_SHINKEN_COMPRESSOR_FLAG_NAME, '0') == '1'

# Chunk of 500Mo
MAX_CHUNK_SIZE = SHINKEN_COMPRESSOR_MAX_CHUNK_SIZE

COMPRESSION_LEVEL = 1  # Fast compression


class Kompressor(object):
    def __init__(self):
        pass
    
    
    def log_debug(self, s):
        # type (unicode) -> NoReturn
        if SHINKEN_LOG_SHINKEN_COMPRESSOR_FLAG:
            logger.debug(s)
    
    
    @staticmethod
    def _cut_into_chunks(s):
        return (s[i:i + MAX_CHUNK_SIZE] for i in range(0, len(s), MAX_CHUNK_SIZE))
    
    
    def compress(self, string_to_compress):
        start = time.time()
        len_string_to_compress = len(string_to_compress)
        if len_string_to_compress < MAX_CHUNK_SIZE:
            _compressed = zlib.compress(string_to_compress, COMPRESSION_LEVEL)
            self.log_debug('[COMPRESSOR] [PERF] [%.3fs]s Compress %s bytes into %s (was done without chunks)' % (time.time() - start, len_string_to_compress, len(_compressed)))
            return _compressed
        
        nb_chunks = 0
        _all_compress = []
        for chunk in self._cut_into_chunks(string_to_compress):
            nb_chunks += 1
            _compressed_chunk = zlib.compress(chunk)
            _all_compress.append(_compressed_chunk)
        _compressed = b''.join(_all_compress)
        self.log_debug('[COMPRESSOR] [PERF] [%.3fs]s Compress %s bytes into %s (was done in %d chunks)' % (time.time() - start, len_string_to_compress, len(_compressed), nb_chunks))
        return _compressed
    
    
    def decompress(self, compressed_string):
        start = time.time()
        zd = zlib.decompressobj()
        total_uncompressed_strings = []
        nb_chunks = 1
        uncompressed = zd.decompress(compressed_string)
        total_uncompressed_strings.append(uncompressed)
        while zd.unused_data != b'':
            nb_chunks += 1
            still_data = zd.unused_data
            zd = zlib.decompressobj()
            uncompressed = zd.decompress(still_data)
            total_uncompressed_strings.append(uncompressed)
        
        res = b''.join(total_uncompressed_strings)
        self.log_debug('[COMPRESSOR] [PERF] [%.3fs]s Uncompress %s bytes into %s (was done in %d chunks)' % (time.time() - start, len(compressed_string), len(res), nb_chunks))
        return res


compresser = Kompressor()
