import uuid
import time

from .misc.type_hint import Optional, Dict

# Purely random uuid4 for regenerator that do not have currently a configuration to manage
DEFAULT_CONFIGURATION_UUID = '6dbc00f706ca4a3b9a60a843a370a634'


class SCHEDULER_MANAGED_CONFIGURATION_KEYS(object):
    SHARD_ID = 'shard_id'
    CONFIG_INCARN_DUMP = 'configuration_incarnation_dump'
    SCHEDULER_NAME = 'scheduler_name'
    IS_ACTIVE = 'is_active'


class ConfigurationIncarnation(object):
    def __init__(self):
        self._uuid = uuid.uuid4().hex
        self._creation_date = int(time.time())
        self._author = '(unknown)'
        self._architecture_name = '(unknown)'
    
    
    def set_from_json_obj(self, dict_obj):
        _uuid = dict_obj.get('uuid', None)
        creation_date = dict_obj.get('creation_date', None)
        author = dict_obj.get('author', None)
        architecture_name = dict_obj.get('architecture_name', None)
        
        # NOTE: Architecture name is for display only currently, but will be mandatory in the future (after 02.07.06++)
        if _uuid is None or creation_date is None or author is None:
            raise ValueError('Cannot create a ConfigurationIncarnation because value is missing')
        
        self._uuid = _uuid
        self._creation_date = creation_date
        self._author = author
        self._architecture_name = architecture_name
        if self._architecture_name is None:
            self._architecture_name = '(unknown)'
    
    
    def set_author(self, author):
        self._author = author
    
    
    def get_author(self):
        return self._author
    
    
    def set_architecture_name(self, architecture_name):
        self._architecture_name = architecture_name
    
    
    def get_architecture_name(self):
        # Maybe we are given an old architecture name
        # TODO: remove after 02.08.02 (included)
        return getattr(self, '_architecture_name', '(unknown)')
    
    
    
    def get_uuid(self):
        return self._uuid
    
    
    def get_creation_date(self):
        return self._creation_date
    
    
    def reset_creation_date(self):
        self._creation_date = int(time.time())
    
    
    def __str__(self):
        return '[configuration_uuid=%s, arbiter=%s, architecture=%s, date=%s]' % (self._uuid, self._author, self.get_architecture_name(), time.strftime('%d-%m-%Y %H:%M:%S', time.localtime(self._creation_date)))
    
    
    def __repr__(self):
        return self.__str__()
    
    
    # Is a configuration_incarnation is the same than another?
    def is_equal(self, other_configuration_incarnation):
        # type: (Optional[ConfigurationIncarnation]) -> bool
        if other_configuration_incarnation is None:
            return False
        
        # uuid should be the same, if so, other are the same too
        return self._uuid == other_configuration_incarnation.get_uuid()
    
    
    def dump_as_json(self):
        dict_obj = {'uuid'             : self._uuid,
                    'creation_date'    : self._creation_date,
                    'author'           : self._author,
                    'architecture_name': self.get_architecture_name(),
                    }
        return dict_obj
    
    
    @staticmethod
    def create_from_json(dict_obj):
        # type: (Dict) -> PartConfigurationIncarnation
        configuration_incarnation = ConfigurationIncarnation()
        try:
            configuration_incarnation.set_from_json_obj(dict_obj)
        except ValueError:  # missing value
            return None
        return configuration_incarnation


class PartConfigurationIncarnation(object):
    def __init__(self, configuration_incarnation, shard_id, scheduler_name, is_active=True):
        # type: (ConfigurationIncarnation, int, str, bool) -> None
        self._configuration_incarnation = configuration_incarnation
        self._shard_id = shard_id
        self.scheduler_name = scheduler_name
        self._is_active = is_active  # know if the scheduler manage a real shard and is not a sleeping spare
    
    
    def __str__(self):
        # Shard with id = 0 means idle spare
        shard_str = '-none-' if self._shard_id == 0 else self._shard_id
        return '[shard_id=%s, scheduler=%s, configuration_uuid=%s, arbiter=%s, architecture=%s, date=%s, active=%s]' % (
            shard_str, self.scheduler_name, self._configuration_incarnation.get_uuid(), self._configuration_incarnation.get_author(),
            self._configuration_incarnation.get_architecture_name(), time.strftime('%d-%m-%Y %H:%M:%S', time.localtime(self._configuration_incarnation.get_creation_date())), self._is_active)
    
    
    def __repr__(self):
        return self.__str__()
    
    
    def get_uuid(self):
        return self._configuration_incarnation.get_uuid()
    
    
    def get_creation_date(self):
        return self._configuration_incarnation.get_creation_date()
    
    
    def get_author(self):
        return self._configuration_incarnation.get_author()
    
    
    def get_part_id(self):
        return self._shard_id
    
    
    def get_architecture_name(self):
        return self._configuration_incarnation.get_architecture_name()
    
    
    def dump_as_json(self):
        dict_obj = {
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.SHARD_ID          : self._shard_id,
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.SCHEDULER_NAME    : self.scheduler_name,
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.IS_ACTIVE         : self._is_active,
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.CONFIG_INCARN_DUMP: self._configuration_incarnation.dump_as_json(),
        }
        return dict_obj
    
    
    @staticmethod
    def create_from_json(dict_obj):
        # type: (Dict) -> PartConfigurationIncarnation
        configuration_incarnation_dump = dict_obj.get(SCHEDULER_MANAGED_CONFIGURATION_KEYS.CONFIG_INCARN_DUMP, {})
        try:
            configuration_incarnation = ConfigurationIncarnation()
            configuration_incarnation.set_from_json_obj(configuration_incarnation_dump)
        except ValueError:  # missing value
            return None
        shard_id = dict_obj.get(SCHEDULER_MANAGED_CONFIGURATION_KEYS.SHARD_ID, None)
        scheduler_name = dict_obj.get(SCHEDULER_MANAGED_CONFIGURATION_KEYS.SCHEDULER_NAME, None)
        is_active = dict_obj.get(SCHEDULER_MANAGED_CONFIGURATION_KEYS.IS_ACTIVE, None)
        if shard_id is None or scheduler_name is None or is_active is None:
            return None
        partial_configuration_incarnation = PartConfigurationIncarnation(configuration_incarnation, shard_id, scheduler_name, is_active=is_active)
        return partial_configuration_incarnation
    
    
    def is_newer_or_equal_than(self, _date):
        # type: (int) -> bool
        return self._configuration_incarnation.get_creation_date() >= _date
    
    
    def is_newer_than(self, _date):
        # type: (int) -> bool
        return self._configuration_incarnation.get_creation_date() > _date


class PartConfigurationIncarnationContainer(object):
    
    def __init__(self):
        self._data = {}
        self._configuration_handle_uuid = DEFAULT_CONFIGURATION_UUID
        self._configuration_handle_date = 0
        self.last_part_configuration_incarnation = None
    
    
    def add(self, part_configuration_incarnation):
        # type: (PartConfigurationIncarnation) -> None
        self._data[part_configuration_incarnation.get_part_id()] = part_configuration_incarnation
        self.last_part_configuration_incarnation = part_configuration_incarnation
        self._configuration_handle_uuid = part_configuration_incarnation.get_uuid()
        self._configuration_handle_date = part_configuration_incarnation.get_creation_date()
    
    
    def check(self, part_configuration_incarnation):
        # type: (PartConfigurationIncarnation) -> (bool, str)
        current_part_configuration_incarnation = self._data.get(part_configuration_incarnation.get_part_id(), None)
        
        i_have_this_part = current_part_configuration_incarnation is not None
        not_already_handle = not i_have_this_part or (i_have_this_part and part_configuration_incarnation.get_uuid() != current_part_configuration_incarnation.get_uuid())
        if_already_have_this_part__new_conf_must_be_newer_than_the_one_i_have = i_have_this_part and part_configuration_incarnation.is_newer_than(current_part_configuration_incarnation.get_creation_date())
        if_do_not_have_this_part__new_conf_must_be_newer_or_same_date = not i_have_this_part and part_configuration_incarnation.is_newer_or_equal_than(self._configuration_handle_date)
        
        if not not_already_handle:
            cause = 'I already handle it %s' % part_configuration_incarnation
        elif i_have_this_part:
            cause = 'The configuration send is older than the one we have. Current handle configuration %s. Send configuration : %s' % (current_part_configuration_incarnation, part_configuration_incarnation)
        else:
            cause = 'The configuration send is too old. Current handle configuration %s. Send configuration : %s' % (time.strftime('%d-%m-%Y %H:%M:%S', time.localtime(self._configuration_handle_date)), part_configuration_incarnation)
        
        return not_already_handle and (if_already_have_this_part__new_conf_must_be_newer_than_the_one_i_have or if_do_not_have_this_part__new_conf_must_be_newer_or_same_date), cause
    
    
    def get_configuration_handle_uuid(self):
        return self._configuration_handle_uuid
    
    
    def get_part_configuration_incarnation(self, part_id):
        return self._data.get(part_id, None)
    
    
    def get_last_part_configuration_incarnation(self):
        return self.last_part_configuration_incarnation
