#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.


import types

try:
    from collections import OrderedDict
except ImportError:
    from ordereddict import OrderedDict

from shinken.log import logger
from datetime import datetime

_dispatcher = {}
NO_COPY = 0


def _copy_list(list_to_cp, memo, dispatcher, ignore_warnings):
    y = memo.get(id(list_to_cp), None)
    if y is not None:
        return y
    
    new_list = list_to_cp[:]
    memo[id(list_to_cp)] = new_list
    for idx, item in enumerate(new_list):
        cp = get_copier(item, dispatcher)
        if cp == NO_COPY:
            continue
        elif cp:
            new_list[idx] = cp(item, memo, dispatcher, ignore_warnings)
        elif not ignore_warnings:
            logger.warning('no dispatcher found for type [%s]. no copy done. ' % type(item))
    return new_list


def _copy_dict(dict_to_cp, memo, dispatcher, ignore_warnings):
    y = memo.get(id(dict_to_cp), None)
    if y is not None:
        return y
    
    new_dict = dict_to_cp.copy()
    memo[id(dict_to_cp)] = new_dict
    for key, value in new_dict.iteritems():
        cp = get_copier(value, dispatcher)
        if cp == NO_COPY:
            continue
        elif cp:
            new_dict[key] = cp(value, memo, dispatcher, ignore_warnings)
        elif not ignore_warnings:
            logger.warning('no dispatcher found for type [%s]. no copy done. ' % type(value))
    return new_dict


def _copy_set(set_to_cp, memo, dispatcher, ignore_warnings):
    y = memo.get(id(set_to_cp), None)
    if y is not None:
        return y
    
    copy = set_to_cp.copy()
    memo[id(set_to_cp)] = copy
    return copy


_dispatcher[list] = _copy_list
_dispatcher[dict] = _copy_dict
_dispatcher[OrderedDict] = _copy_dict
_dispatcher[set] = _copy_set
_dispatcher[unicode] = NO_COPY
_dispatcher[str] = NO_COPY
_dispatcher[bool] = NO_COPY
_dispatcher[float] = NO_COPY
_dispatcher[int] = NO_COPY
_dispatcher[tuple] = NO_COPY
_dispatcher[datetime] = NO_COPY
_dispatcher[types.NoneType] = NO_COPY


def get_copier(item, dispatcher):
    item_type = type(item)
    if item_type == types.InstanceType:
        cp = dispatcher.get(item.__class__)
    else:
        cp = dispatcher.get(type(item))
    
    return cp


# This is a deep_copy optimisation :
# Immutable item will put in result but not copy
# list and dict ar copy but with fastest copy
# In special case, it's possible to ignore warning (you know there is a BaseItem in your dict)
def fast_deepcopy(item, additional_dispatcher=None, ignore_warnings=False):
    dispatcher = _dispatcher
    if additional_dispatcher:
        dispatcher = _dispatcher.copy()
        dispatcher.update(additional_dispatcher)
    
    cp = get_copier(item, dispatcher)
    if cp == NO_COPY:
        return item
    elif cp:
        return cp(item, {}, dispatcher, ignore_warnings)
    elif not ignore_warnings:
        logger.warning('no dispatcher found for type [%s]. no copy done. ' % type(item))
    
    return item
