#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import os

from shinken.daemon import DEFAULT_LIB_DIR
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modulesctx import modulesctx
from shinken.modulesmanager import ModulesManager
from shinkensolutions.lib_modules.configuration_reader import read_string_in_configuration, read_list_in_configuration

if TYPE_CHECKING:
    from shinken.objects.module import Module as ShinkenModuleDefinition
    from shinken.modules.base_sub_module.base_sub_module import BaseSubModule
    from shinken.misc.type_hint import Optional, Dict, List
    from shinken.log import PartLogger


class SubModuleHandler(object):
    logger = None  # type: Optional[PartLogger]
    module_type = None  # type: Optional[unicode]
    
    
    def __init__(self, module_configuration):
        # type: (ShinkenModuleDefinition) -> None
        self.sub_logger = self.logger.get_sub_part(u'MODULES-MANAGER')
        log_init = self.sub_logger.get_sub_part(u'INITIALISATION')
        self.modules_manager = None  # type: Optional[ModulesManager]
        self.modules = read_list_in_configuration(module_configuration, u'modules', [], log_fct=log_init)
        self.modules_dir = read_string_in_configuration(module_configuration, u'modules_dir', os.path.join(DEFAULT_LIB_DIR, u'modules'), log_fct=log_init)
        self._find_modules_path(log_init)
    
    
    def init_sub_modules(self):
        # type: () -> None
        self.modules_manager = ModulesManager(self.module_type, self.modules_dir, [])
        self.modules_manager.set_modules(self.modules)
        self._do_load_modules()
        
        for sub_module in self.modules_manager.get_all_alive_instances():
            self.update_sub_module(sub_module)
    
    
    def update_sub_module(self, sub_module):
        # type: (BaseSubModule) -> None
        # Use this methode to init data from module to sub module
        raise NotImplementedError()
    
    
    def get_sub_module_routes(self):
        # type: () -> List[Dict]
        _logger = self.sub_logger
        routes = []
        for sub_module in self.modules_manager.get_all_alive_instances():
            sub_module_routes = sub_module.get_routes()
            routes.append(sub_module_routes)
            _logger = self.logger.get_sub_part(u'MODULE:%s' % sub_module.get_name())
            _logger.info(u'end point added by sub modules')
            for route_definition in sub_module_routes.itervalues():
                for r in route_definition[u'routes']:
                    _logger.info(u'  * %s' % r)
        return routes
    
    
    def _do_load_modules(self):
        # type: () -> bool
        _logger = self.sub_logger
        all_was_started = self.modules_manager.load_and_init()
        loaded_modules = self.modules_manager.get_all_instances_name()
        if loaded_modules:
            _logger.info(u'Modules that are loaded successfully: %s' % (u''.join(loaded_modules)))
        if not all_was_started:
            _logger.error(u'Some modules did failed to start')
        return all_was_started
    
    
    # Find the absolute path of the shinken module directory and returns it.
    # If the directory do not exist, it raise a Exception
    def _find_modules_path(self, log_init):
        # type: (PartLogger) -> unicode
        self.modules_dir = os.path.abspath(self.modules_dir)
        log_init.info(u'Modules directory: %s' % self.modules_dir)
        if not os.path.exists(self.modules_dir):
            non_existing_module_dir_msg = u'The modules directory "%s" is missing! Bailing out. Please fix your configuration' % self.modules_dir
            log_init.error(non_existing_module_dir_msg)
            raise Exception(non_existing_module_dir_msg)
        
        # Ok remember to populate the modulesctx object
        modulesctx.set_modulesdir(self.modules_dir)
        
        return self.modules_dir
