#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

""" Config is the class to read, load and manipulate the user
 configuration. It read a main cfg (nagios.cfg) and get all informations
 from it. It create objects, make link between them, clean them, and cut
 them into independent parts. The main user of this is Arbiter, but schedulers
 use it too (but far less)"""

import itertools
import json
import os
import random
import re
import shutil
import string
import sys
import time
import traceback
from multiprocessing import Process, Manager

import shinken.misc.configuration_error_log as arbiter_configuration_messages
from businessimpactmodulation import Businessimpactmodulation, Businessimpactmodulations
from checkmodulation import CheckModulation, CheckModulations
from command import Command, Commands
from contact import Contact, Contacts
from contactgroup import Contactgroup, Contactgroups
from discoveryrule import Discoveryrule, Discoveryrules
from discoveryrun import Discoveryrun, Discoveryruns
from escalation import Escalation, Escalations
from host import Host, Hosts
from hostdependency import Hostdependency, Hostdependencies
from hostescalation import Hostescalation, Hostescalations
from hostextinfo import HostExtInfo, HostsExtInfo
from hostgroup import Hostgroup, Hostgroups
from item import Item
from macromodulation import MacroModulation, MacroModulations
from module import Module, Modules
from notificationway import NotificationWay, NotificationWays
from pack import Packs
from proxyitem import ProxyItem, ProxyItems, proxyitemsgraph
from realm import Realm, Realms
from resultmodulation import Resultmodulation, Resultmodulations
from service import Service, Services
from servicedependency import Servicedependency, Servicedependencies
from serviceescalation import Serviceescalation, Serviceescalations
from serviceextinfo import ServiceExtInfo, ServicesExtInfo
from servicegroup import Servicegroup, Servicegroups
from shinken.arbiterlink import ArbiterLink, ArbiterLinks
from shinken.brokerlink import BrokerLink, BrokerLinks
from shinken.compat import cPickle, StringIO
from shinken.graph import Graph
from shinken.log import logger, LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.host import VIEW_CONTACTS_DEFAULT_VALUE
from shinken.pollerlink import PollerLink, PollerLinks
from shinken.property import UnusedProp, BoolProp, IntegerProp, CharProp, StringProp, LogLevelProp, ListProp
from shinken.reactionnerlink import ReactionnerLink, ReactionnerLinks
from shinken.receiverlink import ReceiverLink, ReceiverLinks
from shinken.runtime_stats.threads_dumper import WatchDogThreadDumper
from shinken.schedulerlink import SchedulerLink, SchedulerLinks
from shinken.synchronizerlink import SynchronizerLink, SynchronizerLinks
from shinken.util import split_semicolon, to_mb_size, safe_add_to_dict
from shinkensolutions.os_helper import get_cur_user_name, get_cur_group_name
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString
from timeperiod import Timeperiod, Timeperiods

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Dict

no_longer_used_txt = 'This parameter is not longer take from the main file, but must be defined in the status_dat broker module instead. But Shinken will create you one if there are no present and use this parameter in it, so no worry.'
not_interresting_txt = 'We do not think such an option is interesting to manage.'
OLD_ARBITER_RETENTION_JSON = '/var/lib/shinken/arbiter-retention.json'
REALM_RETENTION_JSON = '/var/lib/shinken/arbiter-retention-%s.json'

# Elements names should not have such characters
ILLEGAL_NAME_CHARACTERS = """`~!$%^&*"|'<>?,()="""

random.seed(time.time())

raw_logger = LoggerFactory.get_logger()
logger_configuration = raw_logger.get_sub_part('CONFIGURATION')
logger_lastest_monitoring_configuration = logger_configuration.get_sub_part('LATEST-MONITORING-CONFIGURATION')
PROPERTIES_DISPLAY_ALLOWED_PROPERTIES = (u'notes_url',)


class ArbiterConfigException(Exception):
    pass


class Config(Item):
    cache_path = "objects.cache"
    my_type = "config"
    
    # Properties:
    # *required: if True, there is not default, and the config must put them
    # *default: if not set, take this value
    # *pythonize: function call to
    # *class_inherit: (Service, 'blabla'): must set this property to the
    #  Service class with name blabla
    #  if (Service, None): must set this property to the Service class with
    #  same name
    # *unused: just to warn the user that the option he use is no more used
    #  in Shinken
    # *usage_text: if present, will print it to explain why it's no more useful
    properties = {
        'master_arbiter_uuid'                                          : StringProp(default=''),
        'prefix'                                                       : StringProp(default='/usr/local/shinken/'),
        'workdir'                                                      : StringProp(default='/var/run/shinken/'),
        'config_base_dir'                                              : StringProp(default=''),  # will be set when we will load a file
        'modules_dir'                                                  : StringProp(default='/var/lib/shinken/modules'),
        'use_local_log'                                                : BoolProp(default='1'),
        'log_level'                                                    : LogLevelProp(default='WARNING'),
        'local_log'                                                    : StringProp(default='/var/log/shinken/arbiterd.log'),
        'log_file'                                                     : UnusedProp(text=no_longer_used_txt),
        'object_cache_file'                                            : UnusedProp(text=no_longer_used_txt),
        'precached_object_file'                                        : UnusedProp(text='Shinken does not use precached_object_files. Skipping.'),
        'resource_file'                                                : StringProp(default='/tmp/resources.txt'),
        'temp_file'                                                    : UnusedProp(text='Temporary files are not used in the shinken architecture. Skipping'),
        'status_file'                                                  : UnusedProp(text=no_longer_used_txt),
        'status_update_interval'                                       : UnusedProp(text=no_longer_used_txt),
        'shinken_user'                                                 : StringProp(default=get_cur_user_name()),
        'shinken_group'                                                : StringProp(default=get_cur_group_name()),
        'enable_notifications'                                         : BoolProp(default='1', class_inherit=[(Host, None), (Service, None), (Contact, None)]),
        'execute_service_checks'                                       : BoolProp(default='1', class_inherit=[(Service, 'execute_checks')]),
        'accept_passive_service_checks'                                : BoolProp(default='1', class_inherit=[(Service, 'accept_passive_checks')]),
        'execute_host_checks'                                          : BoolProp(default='1', class_inherit=[(Host, 'execute_checks')]),
        'accept_passive_host_checks'                                   : BoolProp(default='1', class_inherit=[(Host, 'accept_passive_checks')]),
        'enable_event_handlers'                                        : BoolProp(default='1', class_inherit=[(Host, None), (Service, None)]),
        'log_rotation_method'                                          : CharProp(default='d'),
        'log_archive_path'                                             : StringProp(default='/usr/local/shinken/var/archives'),
        'check_external_commands'                                      : BoolProp(default='1'),
        'command_check_interval'                                       : UnusedProp(text='another value than look always the file is useless, so we fix it.'),
        'command_file'                                                 : StringProp(default=''),
        'external_command_buffer_slots'                                : UnusedProp(text='We do not limit the external command slot.'),
        'check_for_updates'                                            : UnusedProp(
            text='network administrators will never allow such communication between server and the external world. Use your distribution packet manager to know if updates are available or go to the http://www.shinken-monitoring.org website instead.'),
        'bare_update_checks'                                           : UnusedProp(text=None),
        'lock_file'                                                    : StringProp(default='/var/run/shinken/arbiterd.pid'),
        'retain_state_information'                                     : UnusedProp(text='sorry, retain state information will not be implemented because it is useless.'),
        'state_retention_file'                                         : StringProp(default=''),
        'retention_update_interval'                                    : IntegerProp(default='60'),
        'use_retained_program_state'                                   : UnusedProp(text=not_interresting_txt),
        'use_retained_scheduling_info'                                 : UnusedProp(text=not_interresting_txt),
        'retained_host_attribute_mask'                                 : UnusedProp(text=not_interresting_txt),
        'retained_service_attribute_mask'                              : UnusedProp(text=not_interresting_txt),
        'retained_process_host_attribute_mask'                         : UnusedProp(text=not_interresting_txt),
        'retained_process_service_attribute_mask'                      : UnusedProp(text=not_interresting_txt),
        'retained_contact_host_attribute_mask'                         : UnusedProp(text=not_interresting_txt),
        'retained_contact_service_attribute_mask'                      : UnusedProp(text=not_interresting_txt),
        'use_syslog'                                                   : BoolProp(default='0'),
        'minimal_time_before_an_element_become_missing_data'           : IntegerProp(default='60', class_inherit=[(Host, 'missing_data_add_delay'), (Service, 'missing_data_add_delay')]),
        'minimal_time_before_an_element_become_missing_data_at_startup': IntegerProp(default='600'),
        'log_notifications'                                            : BoolProp(default='1', class_inherit=[(Host, None), (Service, None)]),
        'log_service_retries'                                          : BoolProp(default='1', class_inherit=[(Service, 'log_retries')]),
        'log_host_retries'                                             : BoolProp(default='1', class_inherit=[(Host, 'log_retries')]),
        'log_event_handlers'                                           : BoolProp(default='1', class_inherit=[(Host, None), (Service, None)]),
        'log_initial_states'                                           : BoolProp(default='1', class_inherit=[(Host, None), (Service, None)]),
        'log_external_commands'                                        : BoolProp(default='1'),
        'log_passive_checks'                                           : BoolProp(default='1'),
        'global_host_event_handler'                                    : StringProp(default='', class_inherit=[(Host, 'global_event_handler')]),
        'global_service_event_handler'                                 : StringProp(default='', class_inherit=[(Service, 'global_event_handler')]),
        'sleep_time'                                                   : UnusedProp(text='this deprecated option is useless in the shinken way of doing.'),
        'service_inter_check_delay_method'                             : UnusedProp(text='This option is useless in the Shinken scheduling. The only way is the smart way.'),
        'max_service_check_spread'                                     : IntegerProp(default='30', class_inherit=[(Service, 'max_check_spread')]),
        'service_interleave_factor'                                    : UnusedProp(text='This option is useless in the Shinken scheduling because it use a random distribution for initial checks.'),
        'max_concurrent_checks'                                        : UnusedProp(text='Limiting the max concurrent checks is not helpful to got a good running monitoring server.'),
        'check_result_reaper_frequency'                                : UnusedProp(text='Shinken do not use reaper process.'),
        'max_check_result_reaper_time'                                 : UnusedProp(text='Shinken do not use reaper process.'),
        'check_result_path'                                            : UnusedProp(text='Shinken use in memory returns, not check results on flat file.'),
        'max_check_result_file_age'                                    : UnusedProp(text='Shinken do not use flat file check resultfiles.'),
        'host_inter_check_delay_method'                                : UnusedProp(text='This option is unused in the Shinken scheduling because distribution of the initial check is a random one.'),
        'max_host_check_spread'                                        : IntegerProp(default='30', class_inherit=[(Host, 'max_check_spread')]),
        'interval_length'                                              : IntegerProp(default='60', class_inherit=[(Host, None), (Service, None)]),
        'auto_reschedule_checks'                                       : BoolProp(managed=False, default='1'),
        'auto_rescheduling_interval'                                   : IntegerProp(managed=False, default='1'),
        'auto_rescheduling_window'                                     : IntegerProp(managed=False, default='180'),
        'use_aggressive_host_checking'                                 : BoolProp(default='0', class_inherit=[(Host, None)]),
        'translate_passive_host_checks'                                : BoolProp(managed=False, default='1'),
        'passive_host_checks_are_soft'                                 : BoolProp(managed=False, default='1'),
        'enable_predictive_host_dependency_checks'                     : BoolProp(managed=False, default='1', class_inherit=[(Host, 'enable_predictive_dependency_checks')]),
        'enable_predictive_service_dependency_checks'                  : StringProp(managed=False, default='1'),
        'cached_host_check_horizon'                                    : IntegerProp(default='0', class_inherit=[(Host, 'cached_check_horizon')]),
        'cached_service_check_horizon'                                 : IntegerProp(default='0', class_inherit=[(Service, 'cached_check_horizon')]),
        'use_large_installation_tweaks'                                : UnusedProp(text='this option is deprecated because in shinken it is just an alias for enable_environment_macros=0'),
        'free_child_process_memory'                                    : UnusedProp(text='this option is automatic in Python processes'),
        'child_processes_fork_twice'                                   : UnusedProp(text='fork twice is not use.'),
        'enable_environment_macros'                                    : BoolProp(default='1', class_inherit=[(Host, None), (Service, None)]),
        'enable_flap_detection'                                        : BoolProp(default='1', class_inherit=[(Host, None), (Service, None)]),
        'low_service_flap_threshold'                                   : IntegerProp(default='25', class_inherit=[(Service, 'global_low_flap_threshold')]),
        'high_service_flap_threshold'                                  : IntegerProp(default='50', class_inherit=[(Service, 'global_high_flap_threshold')]),
        'low_host_flap_threshold'                                      : IntegerProp(default='25', class_inherit=[(Host, 'global_low_flap_threshold')]),
        'high_host_flap_threshold'                                     : IntegerProp(default='50', class_inherit=[(Host, 'global_high_flap_threshold')]),
        'soft_state_dependencies'                                      : BoolProp(managed=False, default='0'),
        'check_running_timeout'                                        : IntegerProp(default=u'60', class_inherit=[(Service, u'check_timeout'), (Host, u'check_timeout')]),
        'warning_threshold_cpu_usage'                                  : IntegerProp(default='30', class_inherit=[(Host, 'warning_threshold_cpu_usage_default'), (Service, 'warning_threshold_cpu_usage_default')]),
        'timeout_exit_status'                                          : IntegerProp(default='3'),
        'event_handler_timeout'                                        : IntegerProp(default='30', class_inherit=[(Host, None), (Service, None)]),
        'notification_timeout'                                         : IntegerProp(default='30', class_inherit=[(Host, None), (Service, None)]),
        'ocsp_timeout'                                                 : IntegerProp(default='15', class_inherit=[(Service, None)]),
        'ochp_timeout'                                                 : IntegerProp(default='15', class_inherit=[(Host, None)]),
        'perfdata_timeout'                                             : IntegerProp(default='5', class_inherit=[(Host, None), (Service, None)]),
        'obsess_over_services'                                         : BoolProp(default='0', class_inherit=[(Service, 'obsess_over')]),
        'ocsp_command'                                                 : StringProp(default='', class_inherit=[(Service, None)]),
        'obsess_over_hosts'                                            : BoolProp(default='0', class_inherit=[(Host, 'obsess_over')]),
        'ochp_command'                                                 : StringProp(default='', class_inherit=[(Host, None)]),
        'process_performance_data'                                     : BoolProp(default='1', class_inherit=[(Host, None), (Service, None)]),
        'host_perfdata_command'                                        : StringProp(default='', class_inherit=[(Host, 'perfdata_command')]),
        'service_perfdata_command'                                     : StringProp(default='', class_inherit=[(Service, 'perfdata_command')]),
        'host_perfdata_file'                                           : StringProp(default='', class_inherit=[(Host, 'perfdata_file')]),
        'service_perfdata_file'                                        : StringProp(default='', class_inherit=[(Service, 'perfdata_file')]),
        'host_perfdata_file_template'                                  : StringProp(default='/tmp/host.perf', class_inherit=[(Host, 'perfdata_file_template')]),
        'service_perfdata_file_template'                               : StringProp(default='/tmp/host.perf', class_inherit=[(Service, 'perfdata_file_template')]),
        'host_perfdata_file_mode'                                      : CharProp(default='a', class_inherit=[(Host, 'perfdata_file_mode')]),
        'service_perfdata_file_mode'                                   : CharProp(default='a', class_inherit=[(Service, 'perfdata_file_mode')]),
        'host_perfdata_file_processing_interval'                       : IntegerProp(managed=False, default='15'),
        'service_perfdata_file_processing_interval'                    : IntegerProp(managed=False, default='15'),
        'host_perfdata_file_processing_command'                        : StringProp(managed=False, default='', class_inherit=[(Host, 'perfdata_file_processing_command')]),
        'service_perfdata_file_processing_command'                     : StringProp(managed=False, default=None),
        'check_for_orphaned_services'                                  : BoolProp(default='1', class_inherit=[(Service, 'check_for_orphaned')]),
        'check_for_orphaned_hosts'                                     : BoolProp(default='1', class_inherit=[(Host, 'check_for_orphaned')]),
        'check_service_freshness'                                      : BoolProp(default='1', class_inherit=[(Service, 'global_check_freshness')]),
        'service_freshness_check_interval'                             : IntegerProp(default='60'),
        'check_host_freshness'                                         : BoolProp(default='1', class_inherit=[(Host, 'global_check_freshness')]),
        'host_freshness_check_interval'                                : IntegerProp(default='60'),
        'additional_freshness_latency'                                 : IntegerProp(default='15', class_inherit=[(Host, None), (Service, None)]),
        'enable_embedded_perl'                                         : BoolProp(managed=False, default='1', help='It will surely never be managed, but it should not be useful with poller performances.'),
        'use_embedded_perl_implicitly'                                 : BoolProp(managed=False, default='0'),
        'date_format'                                                  : StringProp(managed=False, default=None),
        'use_timezone'                                                 : StringProp(default='', class_inherit=[(Host, None), (Service, None), (Contact, None)]),
        'language'                                                     : StringProp(default='en', class_inherit=[(Host, None), (Service, None)]),
        'illegal_object_name_chars'                                    : StringProp(default=ILLEGAL_NAME_CHARACTERS, class_inherit=[(Host, None), (Service, None), (Contact, None), (HostExtInfo, None)]),
        'illegal_macro_output_chars'                                   : StringProp(default='', class_inherit=[(Host, None), (Service, None), (Contact, None)]),
        'use_regexp_matching'                                          : BoolProp(managed=False, default='0', help=' if you go some host or service definition like prod*, it will surely failed from now, sorry.'),
        'use_true_regexp_matching'                                     : BoolProp(managed=False, default=None),
        'admin_email'                                                  : UnusedProp(text='sorry, not yet implemented.'),
        'admin_pager'                                                  : UnusedProp(text='sorry, not yet implemented.'),
        'event_broker_options'                                         : UnusedProp(text='event broker are replaced by modules with a real configuration template.'),
        'broker_module'                                                : StringProp(default=''),
        'debug_file'                                                   : UnusedProp(text=None),
        'debug_level'                                                  : UnusedProp(text=None),
        'debug_verbosity'                                              : UnusedProp(text=None),
        'max_debug_file_size'                                          : UnusedProp(text=None),
        'modified_attributes'                                          : IntegerProp(default=0),
        # '$USERn$: {'required':False, 'default':''} # Add at run in __init__
        
        # SHINKEN SPECIFIC
        'idontcareaboutsecurity'                                       : BoolProp(default='0'),
        'daemon_enabled'                                               : BoolProp(default='1'),  # Put to 0 to disable the arbiter to run
        'daemon_thread_pool_size'                                      : IntegerProp(default='8'),
        'flap_history'                                                 : IntegerProp(default='20', class_inherit=[(Host, None), (Service, None)]),
        'max_plugins_output_length'                                    : IntegerProp(default='8192', class_inherit=[(Host, None), (Service, None)]),
        'no_event_handlers_during_downtimes'                           : BoolProp(default='0', class_inherit=[(Host, None), (Service, None)]),
        
        # Enable or not the notice about old Nagios parameters
        'disable_old_nagios_parameters_whining'                        : BoolProp(default='0'),
        
        # Now for problem/impact states changes
        'enable_problem_impacts_states_change'                         : BoolProp(default='0', class_inherit=[(Host, None), (Service, None)]),
        
        # More a running value in fact
        'resource_macros_names'                                        : ListProp(default=[]),
        
        # SSL PART
        # global boolean for know if we use ssl or not
        'use_ssl'                                                      : BoolProp(default='0',
                                                                                  class_inherit=[(SchedulerLink, None), (ReactionnerLink, None), (BrokerLink, None), (PollerLink, None), (ReceiverLink, None), (ArbiterLink, None),
                                                                                                 (SynchronizerLink, None)]),
        'ca_cert'                                                      : StringProp(default='etc/certs/ca.pem'),
        'server_cert'                                                  : StringProp(default='etc/certs/server.cert'),
        'server_key'                                                   : StringProp(default='etc/certs/server.key'),
        'hard_ssl_name_check'                                          : BoolProp(default='0'),
        
        # Log format
        'human_timestamp_log'                                          : BoolProp(default='1'),
        
        ## Discovery part
        'strip_idname_fqdn'                                            : BoolProp(default='1'),
        'runners_timeout'                                              : IntegerProp(default='3600'),
        
        # pack_distribution_file is for keeping a distribution history
        # of the host distribution in the several "packs" so a same
        # scheduler will have more change of getting the same host
        'pack_distribution_file'                                       : StringProp(default='pack_distribution.dat'),
        
        ## WEBUI part
        'webui_lock_file'                                              : StringProp(default='webui.pid'),
        'webui_port'                                                   : IntegerProp(default='8080'),
        'webui_host'                                                   : StringProp(default='0.0.0.0'),
        
        # Large env tweacks
        'use_multiprocesses_serializer'                                : BoolProp(default='0'),
        
        # Dispatch parameters
        'configuration_dispatch__initial_daemons_check__max_duration'  : IntegerProp(default='30'),
        
        # Password to allow daemon to answer to data export
        'daemon__export_data__password'                              : StringProp(default=''),
    }
    
    macros = {
        'PREFIX'             : 'prefix',
        'MAINCONFIGFILE'     : '',
        'STATUSDATAFILE'     : '',
        'COMMENTDATAFILE'    : '',
        'DOWNTIMEDATAFILE'   : '',
        'RETENTIONDATAFILE'  : '',
        'OBJECTCACHEFILE'    : '',
        'TEMPFILE'           : '',
        'TEMPPATH'           : '',
        'LOGFILE'            : '',
        'RESOURCEFILE'       : '',
        'COMMANDFILE'        : 'command_file',
        'HOSTPERFDATAFILE'   : '',
        'SERVICEPERFDATAFILE': '',
        'ADMINEMAIL'         : '',
        'ADMINPAGER'         : ''
        # 'USERn': '$USERn$' # Add at run time
    }
    
    static_macros = set()
    
    # We create dict of objects
    # Type: 'name in objects': {Class of object, Class of objects,
    # 'property for self for the objects(config)'
    types_creations = {
        'timeperiod'              : (Timeperiod, Timeperiods, 'timeperiods'),
        'service'                 : (Service, Services, 'services'),
        'servicegroup'            : (Servicegroup, Servicegroups, 'servicegroups'),
        'command'                 : (Command, Commands, 'commands'),
        'host'                    : (Host, Hosts, 'hosts'),
        'hostgroup'               : (Hostgroup, Hostgroups, 'hostgroups'),
        'contact'                 : (Contact, Contacts, 'contacts'),
        'contactgroup'            : (Contactgroup, Contactgroups, 'contactgroups'),
        'notificationway'         : (NotificationWay, NotificationWays, 'notificationways'),
        'checkmodulation'         : (CheckModulation, CheckModulations, 'checkmodulations'),
        'macromodulation'         : (MacroModulation, MacroModulations, 'macromodulations'),
        'servicedependency'       : (Servicedependency, Servicedependencies, 'servicedependencies'),
        'hostdependency'          : (Hostdependency, Hostdependencies, 'hostdependencies'),
        'arbiter'                 : (ArbiterLink, ArbiterLinks, 'arbiters'),
        'synchronizer'            : (SynchronizerLink, SynchronizerLinks, 'synchronizers'),
        'scheduler'               : (SchedulerLink, SchedulerLinks, 'schedulers'),
        'reactionner'             : (ReactionnerLink, ReactionnerLinks, 'reactionners'),
        'broker'                  : (BrokerLink, BrokerLinks, 'brokers'),
        'receiver'                : (ReceiverLink, ReceiverLinks, 'receivers'),
        'poller'                  : (PollerLink, PollerLinks, 'pollers'),
        'realm'                   : (Realm, Realms, 'realms'),
        'module'                  : (Module, Modules, 'modules'),
        'resultmodulation'        : (Resultmodulation, Resultmodulations, 'resultmodulations'),
        'businessimpactmodulation': (Businessimpactmodulation, Businessimpactmodulations, 'businessimpactmodulations'),
        'escalation'              : (Escalation, Escalations, 'escalations'),
        'serviceescalation'       : (Serviceescalation, Serviceescalations, 'serviceescalations'),
        'hostescalation'          : (Hostescalation, Hostescalations, 'hostescalations'),
        'discoveryrule'           : (Discoveryrule, Discoveryrules, 'discoveryrules'),
        'discoveryrun'            : (Discoveryrun, Discoveryruns, 'discoveryruns'),
        'hostextinfo'             : (HostExtInfo, HostsExtInfo, 'hostsextinfo'),
        'serviceextinfo'          : (ServiceExtInfo, ServicesExtInfo, 'servicesextinfo'),
    }
    
    # This tab is used to transform old parameters name into new ones
    # so from Nagios2 format, to Nagios3 ones
    old_properties = {
        'nagios_user' : 'shinken_user',
        'nagios_group': 'shinken_group',
        'modulesdir'  : 'modules_dir',
    }
    
    read_config_silent = 0
    
    early_created_types = ['arbiter', 'module']
    
    configuration_types = ['void', 'timeperiod', 'command', 'contactgroup', 'hostgroup',
                           'contact', 'notificationway', 'checkmodulation', 'macromodulation', 'host', 'service', 'servicegroup',
                           'servicedependency', 'hostdependency', 'arbiter', 'scheduler', 'synchronizer',
                           'reactionner', 'broker', 'receiver', 'poller', 'realm', 'module',
                           'resultmodulation', 'escalation', 'serviceescalation', 'hostescalation',
                           'discoveryrun', 'discoveryrule', 'businessimpactmodulation',
                           'hostextinfo', 'serviceextinfo']
    
    
    def __init__(self):
        self.params = {}
        self.resource_macros_names = []
        # By default the conf is correct
        self.conf_is_correct = True
        # We tag the conf with a magic_hash, a random value to
        # idify this conf
        random.seed(time.time())
        self.magic_hash = random.randint(1, 100000)
        self.configuration_errors = []
        self.configuration_duplicate_params = {}
        self.source_configuration_params = {}
        self.packs_dirs = []
        self.packs = Packs({})
        # default properties values can be overridden by the user for some properties
        self.default_properties_values = {}
        self.properties_display_text = self._init_properties_display_text()
        self.conf_is_empty = True
        
        # We create properties in types_creations at init.
        # They will be replaced with configuration value in self.create_objects
        types_creations = self.__class__.types_creations
        for _type in types_creations:
            (cls, clss, prop) = types_creations[_type]
            setattr(self, prop, clss([]))
        
        # The configuration will be cut into number of smaller configuration
        # they are the sharded_configurations (as much as active scheduler)
        self.sharded_configurations = {}
        self.ignored_items = {}
        self.bad_encoding_files = []
        
        self._configuration_incarnation = None
    
    
    @classmethod
    def get_class_for_object_type_name(cls, object_type_name):
        entry = cls.types_creations.get(object_type_name, None)
        if entry is None:
            return None
        klass = entry[0]  # it's ok, all our entries have at least 3 values
        return klass
    
    
    # Set by the arbiter so we know which incarnation/date we are
    def set_configuration_incarnation(self, configuration_incarnation):
        self._configuration_incarnation = configuration_incarnation
    
    
    def get_configuration_incarnation(self):
        return self._configuration_incarnation
    
    
    def get_name(self):
        return 'global configuration file'
    
    
    # We've got macro in the resource file and we want
    # to update our MACRO dict with it
    def fill_resource_macros_names_macros(self):
        """ fill the macro dict will all value
        from self.resource_macros_names"""
        properties = self.__class__.properties
        macros = self.__class__.macros
        static_macros = self.__class__.static_macros
        for macro_name in self.resource_macros_names:
            properties['$' + macro_name + '$'] = StringProp(default='')
            macros[macro_name] = '$' + macro_name + '$'
            # Also fill static macro as global are static by design
            static_macros.add(macro_name)
    
    
    def load_param(self, param, filefrom):
        elts = param.split('=', 1)
        key_name = elts[0]
        if len(elts) == 1:  # error, there is no = !
            self.conf_is_correct = False
            self.configuration_errors.append("In file %s, the parameter %s is malformed! (no = sign)" % (filefrom, key_name))
            logger.error("[config] In file %s, the parameter %s is malformed! (no = sign)" % (filefrom, key_name))
        else:
            if key_name in self.source_configuration_params.get(filefrom, []):
                safe_add_to_dict(self.configuration_duplicate_params, filefrom, key_name, as_set=True)
            
            if not (key_name[0] == '$' and key_name[-1] == '$'):
                safe_add_to_dict(self.source_configuration_params, filefrom, key_name)
            
            value = elts[1]
            self.params[key_name] = value
            setattr(self, key_name, value)
            # Maybe it's a variable as $USER$ or $ANOTHERVATRIABLE$
            # so look at the first character. If it's a $, it's a variable
            # and if it's end like it too
            if key_name[0] == '$' and key_name[-1] == '$':
                macro_name = key_name[1:-1]
                self.resource_macros_names.append(macro_name)
    
    
    def _cut_line(self, line):
        # punct = '"#$%&\'()*+/<=>?@[\\]^`{|}~'
        tmp = re.split("[" + string.whitespace + "]+", line, 1)
        r = [elt for elt in tmp if elt != '']
        return r
    
    
    def read_config(self, files):
        # read all files and put in a buffer
        # Line with # IMPORTEDFROM are file separator
        res = StringIO()
        
        for _file in files:
            # We add a \n (or \r\n) to be sure config files are separated
            # if the previous does not finish with a line return
            res.write('%s# IMPORTEDFROM=%s %s' % (os.linesep, _file, os.linesep))
            if self.read_config_silent == 0:
                logger.info("[config] opening '%s' configuration file" % _file)
            
            try:
                # Open in Universal way for Windows, Mac, Linux
                with open(_file, 'rU') as fd:
                    _buffer = fd.readlines()
                self.config_base_dir = os.path.dirname(_file)
            except IOError as exp:
                self.configuration_errors.append("[config] cannot open config file '%s' for reading: %s" % (_file, exp))
                # The configuration is invalid because we have a bad file!
                self.conf_is_correct = False
                continue
            
            for line_number, line in enumerate(_buffer, 1):
                try:
                    line = line.decode('utf8')
                except UnicodeDecodeError:
                    logger.warning("[config] Some characters could not be read in utf-8 in the file : %s" % _file)
                    self.bad_encoding_files.append(_file)
                    line = line.decode('utf8', 'replace')
                res.write(line)
                if line.endswith('\n'):
                    line = line[:-1]
                line = line.strip()
                if re.search("^cfg_file", line) or re.search("^resource_file", line):
                    self.conf_is_empty = False
                    elts = line.split('=', 1)
                    if os.path.isabs(elts[1]):
                        cfg_file_name = elts[1]
                    else:
                        cfg_file_name = os.path.join(self.config_base_dir, elts[1])
                    cfg_file_name = cfg_file_name.strip()
                    try:
                        if self.read_config_silent == 0:
                            logger.info("Processing object config file '%s'" % cfg_file_name)
                        res.write('%s# IMPORTEDFROM=%s %s' % (os.linesep, cfg_file_name, os.linesep))
                        _file_buffer = self._read_cfg_file(cfg_file_name)
                        res.write(_file_buffer)
                        # Be sure to add a line return so we won't mix files
                        res.write('%s# IMPORTEDFROM=%s=%s %s' % (os.linesep, _file, line_number, os.linesep))
                    except IOError as exp:
                        err = "Cannot open config file '%s' for reading: %s" % (cfg_file_name, exp)
                        if self.read_config_silent == 0:
                            logger.error(err)
                        self.configuration_errors.append(err)
                        # The configuration is invalid because we have a bad file!
                        self.conf_is_correct = False
                elif re.search("^cfg_dir", line):
                    self.conf_is_empty = False
                    elts = line.split('=', 1)
                    if os.path.isabs(elts[1]):
                        cfg_dir_name = elts[1]
                    else:
                        cfg_dir_name = os.path.join(self.config_base_dir, elts[1])
                    # Ok, look if it's really a directory
                    if not os.path.isdir(cfg_dir_name):
                        logger.error("Cannot open config dir '%s' for reading" % cfg_dir_name)
                        self.add_error("Cannot open config dir '%s' for reading" % cfg_dir_name)
                    
                    # Look for .pack file into it :)
                    self.packs_dirs.append(cfg_dir_name)
                    
                    # Now walk for it.
                    # BEWARE : we can follow symlinks only for python 2.6 and higher
                    args = {}
                    if sys.version_info >= (2, 6):
                        args['followlinks'] = True
                    for root, dirs, files in os.walk(cfg_dir_name, **args):
                        for p_file in files:
                            if re.search("\.cfg$", p_file):
                                read_file = os.path.join(root, p_file)
                                if self.read_config_silent == 0:
                                    logger.info("Processing object config file '%s'" % read_file)
                                try:
                                    res.write('%s# IMPORTEDFROM=%s %s' % (os.linesep, read_file, os.linesep))
                                    read_file_buffer = self._read_cfg_file(read_file)
                                    res.write(read_file_buffer)
                                    # Be sure to separate files data
                                    res.write('%s# IMPORTEDFROM=%s=%s %s' % (os.linesep, _file, line_number, os.linesep))
                                except IOError as exp:
                                    self.configuration_errors.append("Cannot open config file '%s' for reading: %s" % (read_file, exp))
                                    # The configuration is invalid
                                    # because we have a bad file!
                                    self.conf_is_correct = False
                elif re.search("^triggers_dir", line):
                    self.conf_is_empty = False
                    elts = line.split('=', 1)
                    if os.path.isabs(elts[1]):
                        trig_dir_name = elts[1]
                    else:
                        trig_dir_name = os.path.join(self.config_base_dir, elts[1])
                    # Ok, look if it's really a directory
                    if not os.path.isdir(trig_dir_name):
                        self.configuration_errors.append("Cannot open triggers dir '%s' for reading" % trig_dir_name)
                        self.conf_is_correct = False
                        continue
        
        config = res.getvalue()
        res.close()
        return config
    
    
    def read_config_buf(self, buf, forbidden_types=None):
        filefrom = "Unknown file"
        objectscfg = {}
        types = self.__class__.configuration_types
        for t in types:
            objectscfg[t] = []
        
        # If not set explicitly, try to have the FORBIDDEN types from the env variables
        if forbidden_types is None:
            forbidden_types = [t.strip() for t in os.environ.get('FORBIDDEN_TYPES', '').split(' ') if t.strip()]
            if forbidden_types:
                logger.debug('Using forbidden_types: %s' % forbidden_types)
        
        tmp = []
        tmp_type = 'void'
        in_define = False
        almost_in_define = False
        continuation_line = False
        tmp_line = ''
        lines = buf.split('\n')
        line_nb = 0  # Keep the line number for the file path
        for line in lines:
            if line.endswith('\r'):
                line = line[:-1]
            if line.startswith("# IMPORTEDFROM="):
                split_ = line.split('=')
                filefrom = split_[1].strip()
                if len(split_) > 2:
                    line_nb = int(split_[2].strip())
                else:
                    line_nb = 0  # reset the line number too
                continue
            
            line_nb += 1
            # Remove comments
            line = split_semicolon(line)[0].strip()
            
            # A backslash means, there is more to come
            if re.search("\\\s*$", line) is not None:
                continuation_line = True
                line = re.sub("\\\s*$", "", line)
                line = re.sub("^\s+", " ", line)
                tmp_line += line
                continue
            elif continuation_line:
                # Now the continuation line is complete
                line = re.sub("^\s+", "", line)
                line = tmp_line + line
                tmp_line = ''
                continuation_line = False
            # } alone in a line means stop the object reading
            if re.search("^\s*}\s*$", line) is not None:
                in_define = False
            
            # { alone in a line can mean start object reading
            if re.search("^\s*{\s*$", line) is not None and almost_in_define:
                almost_in_define = False
                in_define = True
                continue
            
            if re.search("^\s*#|^\s*$|^\s*}", line) is not None:
                pass
            # A define must be catch and the type save
            # The old entry must be save before
            elif re.search("^define", line) is not None:
                if re.search(".*{.*$", line) is not None:
                    in_define = True
                else:
                    almost_in_define = True
                # Get new type
                elts = re.split('\s', line)
                # Maybe there was space before and after the type so we must get all and strip it
                tmp_type = ' '.join(elts[1:]).strip()
                tmp_type = tmp_type.split('{')[0].strip()
                tmp = ['imported_from %s:%d' % (filefrom, line_nb)]
                
                # Do not allow a type that was forbidden in flat files
                if tmp_type in forbidden_types:
                    safe_add_to_dict(self.ignored_items, tmp_type, '%s:%d' % (filefrom, line_nb), as_set=True)
                else:
                    safe_add_to_dict(objectscfg, tmp_type, tmp)
            
            else:
                if in_define:
                    tmp.append(line)
                else:
                    # print "Params", params
                    self.load_param(line, filefrom)
        
        objects = {}
        
        # And then update our MACRO dict
        self.fill_resource_macros_names_macros()
        
        for type in objectscfg:
            objects[type] = []
            for items in objectscfg[type]:
                tmp = {}
                for line in items:
                    elts = self._cut_line(line)
                    if elts == []:
                        continue
                    prop = elts[0]
                    if prop == '[OVERLOAD_FROM]':
                        path = ' '.join(elts[1:])
                        if os.path.isfile(path):
                            c = Config()
                            c.read_config_silent = 1
                            buf = c.read_config([path])
                            c.read_config_buf(buf)
                            for k, v in c.params.iteritems():
                                tmp[k] = [v]
                    else:
                        if prop not in tmp:
                            tmp[prop] = []
                        value = ' '.join(elts[1:])
                        tmp[prop].append(value)
                if tmp != {}:
                    objects[type].append(tmp)
        
        return objects
    
    
    def _read_cfg_file(self, _file):
        try:
            with open(_file, 'rU') as fd:
                _buf = fd.read()
        except IOError as exp:
            self.configuration_errors.append("[config] cannot open config file '%s' for reading: %s" % (_file, exp))
            # The configuration is invalid because we have a bad file!
            self.conf_is_correct = False
            return ''
        try:
            _buf = _buf.decode('utf8')
        except UnicodeDecodeError:
            logger.warning("[config] Some characters could not be read in utf-8 in the file : %s" % _file)
            self.bad_encoding_files.append(_file)
            _buf = _buf.decode('utf8', 'replace')
        return _buf
    
    
    @staticmethod
    def get_internal_commands():
        # type: () -> List[Dict[unicode, unicode]]
        bp_rule_def = {u'command_name': u'bp_rule', u'command_line': u'bp_rule', u'internal': u'1'}
        host_up_def = {u'command_name': u'_internal_host_up', u'command_line': u'_internal_host_up', u'internal': u'1'}
        echo_def = {u'command_name': u'_echo', u'command_line': u'_echo', u'internal': u'1'}
        no_command_def = {u'command_name': u'shinken_no_command', u'command_line': u'shinken_no_command', u'internal': u'1'}
        echo_with_exit_code_def = {u'command_name': u'shinken_echo_with_exit_code_and_skip_macro', u'command_line': u'shinken_echo_with_exit_code_and_skip_macro', u'internal': u'1'}
        return [bp_rule_def, host_up_def, echo_def, no_command_def, echo_with_exit_code_def]
    
    
    # We insert internal_commands : bp_rule, _internal_host_up, _echo, shinken_no_command
    def add_internal_commands(self, raw_objects):
        # type: (Dict[unicode, List[Dict[unicode, unicode]]]) -> None
        raw_objects[u'command'].extend(self.get_internal_commands())
    
    
    # We've got raw objects in string, now create real Instances
    def create_objects(self, raw_objects):
        """ Create real 'object' from dicts of prop/value """
        types_creations = self.__class__.types_creations
        
        # some types are already created in this time
        early_created_types = self.__class__.early_created_types
        
        # Before really create the objects, we add
        # ghost ones like the bp_rule for correlation
        self.add_internal_commands(raw_objects)
        
        for t in types_creations:
            if t not in early_created_types:
                self.create_objects_for_type(raw_objects, t)
    
    
    def create_objects_for_type(self, raw_objects, obj_type):
        types_creations = self.__class__.types_creations
        
        (cls, clss, prop) = types_creations[obj_type]
        # List where we put objects
        lst = []
        for obj_cfg in raw_objects.get(obj_type, []):
            # We create the object
            o = cls(obj_cfg, skip_useless_in_configuration=True)  # in the arbiter, we do not want useless running properties that are used only for scheduler & display
            lst.append(o)
        # we create the objects Class and we set it in prop
        setattr(self, prop, clss(lst))
    
    
    # Here arbiter and modules objects should be prepare and link
    # before all others types
    def early_arbiter_linking(self):
        """ Prepare the arbiter for early operations """
        
        # We can load user default values
        self.load_default_properties_values()
        self.load_properties_display_configuration()
        
        self.modules.create_reversed_list()
        
        if len(self.arbiters) == 0:
            raise ArbiterConfigException('No arbiter defined. Invalid configuration')
        
        # First fill default
        self.arbiters.fill_default()
        self.modules.fill_default()
        self.fill_default()
        
        # Step 4 : fill running properties
        self.modules.fill_running_properties(self)
        
        # print "****************** Pythonize ******************"
        self.arbiters.pythonize()
        
        # print "****************** Linkify ******************"
        self.arbiters.linkify(self.modules)
        self.modules.linkify()
    
    
    # We will load all packs .pack files from all packs_dirs
    def load_packs(self):
        for p in self.packs_dirs:
            self.packs.load_file(p)
    
    
    def load_default_properties_values(self):
        PROPERTIES_MAPPING = {
            'low_host_flap_threshold'    : ('low_flap_threshold', Host),
            'high_host_flap_threshold'   : ('high_flap_threshold', Host),
            'low_service_flap_threshold' : ('low_flap_threshold', Service),
            'high_service_flap_threshold': ('high_flap_threshold', Service),
        }
        
        for config_property, (object_property, cls) in PROPERTIES_MAPPING.iteritems():
            _new_value = getattr(self, config_property, None)
            if _new_value:
                cls.properties[object_property].set_default(_new_value)
        
        for (k, v) in self.__dict__.iteritems():
            # print "Checking for Default key? %s => %s" % (k, v)
            # example: [DEFAULT:host] view_contacts = nobody
            if k.startswith('[DEFAULT:'):
                _elts = k.split(']', 1)
                _type = _elts[0].replace('[DEFAULT:', '')
                _type = 'service' if _type == 'check' else _type
                _key = _elts[1].strip()
                if _type not in self.default_properties_values:
                    self.default_properties_values[_type] = {}
                _value = v.strip()
                self.default_properties_values[_type][_key] = _value
                
                # view_contacts is special : it not overload default in class file
                if _type not in self.types_creations:
                    logger.warning('[default] Type [%s] unknown. Default value [%s]-[%s] : [%s] is ignore' % (_type, _type, _key, _value))
                    continue
                
                if _key == 'view_contacts':
                    # cf #SEF-8624 We must add the default_value to the Host class
                    _class = self.types_creations[_type][0]
                    setattr(_class, u'default_view_contacts_value', _value)
                    logger.debug('[default] new default value for [%s]-[%s] : [%s]' % (_type, _key, _value))
                    continue
                
                _class = self.types_creations[_type][0]
                if not _key in _class.properties:
                    logger.warning('[default] Property [%s] for class [%s] is unknown. Default value [%s]-[%s] : [%s] is ignore' % (_key, _type, _type, _key, _value))
                    continue
                property = _class.properties[_key]
                property.set_default(_value)
                
                logger.debug('[default] new default value for [%s]-[%s] : [%s]' % (_type, _key, _value))
    
    
    @staticmethod
    def _init_properties_display_text():
        # type: () -> Dict[unicode, Dict]
        properties_display_text = {}
        for property_name in PROPERTIES_DISPLAY_ALLOWED_PROPERTIES:
            properties_display_text[property_name] = {}
        return properties_display_text
    
    
    def load_properties_display_configuration(self):
        for (k, v) in self.__dict__.iteritems():
            # print "Checking for Default key? %s => %s" % (k, v)
            # example: [DISPLAY_TEXT:fr] notes_url=Url Externe
            for label_lang in (u'fr', u'en'):
                parameter_tag = u'[DISPLAY_TEXT:%s]' % label_lang
                if k.startswith(parameter_tag):
                    property_name = k.replace(parameter_tag, u'')
                    property_name = property_name.strip()
                    
                    if property_name not in PROPERTIES_DISPLAY_ALLOWED_PROPERTIES:
                        logger.warning(u'[DISPLAY_TEXT] Property [%s] is unknown or you cannot overload its label. Ignoring it' % property_name)
                        continue
                    
                    if property_name not in self.properties_display_text:
                        self.properties_display_text[property_name] = {}
                    _value = v.strip()
                    
                    if re.findall(ur'[^A-Za-z0-9_À-ÿ \']+', _value):
                        logger.warning(u'[DISPLAY_TEXT] Bad value for [ %s ]. Only alphanumeric characters are allowed : [ %s ]. Ignoring it' % (property_name, _value))
                        continue
                    
                    if len(_value) > 80:
                        logger.warning(u'[DISPLAY_TEXT] Bad value for [ %s ]. Value longer than 80 characters : [ %s ]. Ignoring it' % (property_name, _value))
                        continue
                    
                    self.properties_display_text[property_name][label_lang] = _value
    
    
    # We use linkify to make the config more efficient: elements will be
    # linked, like pointers. For example, a host will have it's service,
    # and contacts directly in it's properties
    # REMEMBER: linkify AFTER explode...
    def linkify_realm_only(self):
        """ Make 'links' between elements, like a host got a services list
        with all it's services in it """
        
        # First linkify myself like for some global commands
        self.linkify_one_command_with_commands(self.commands, 'ocsp_command')
        self.linkify_one_command_with_commands(self.commands, 'ochp_command')
        self.linkify_one_command_with_commands(self.commands, 'host_perfdata_command')
        self.linkify_one_command_with_commands(self.commands, 'service_perfdata_command')
        
        # print "Hosts"
        # link hosts with timeperiods and commands
        self.hosts.linkify(
            self.timeperiods,
            self.commands,
            self.contacts,
            self.realms,
            self.resultmodulations,
            self.businessimpactmodulations,
            self.escalations,
            self.hostgroups,
            self.checkmodulations,
            self.macromodulations
        )
        
        self.hostsextinfo.merge(self.hosts)
        
        # Do the simplify AFTER explode groups
        # print "Hostgroups"
        # link hostgroups with hosts
        self.hostgroups.linkify(self.hosts, self.realms)
        
        # print "Services"
        # link services with other objects
        self.services.linkify(self.hosts, self.commands,
                              self.timeperiods, self.contacts,
                              self.resultmodulations, self.businessimpactmodulations,
                              self.escalations, self.servicegroups, self.checkmodulations,
                              self.macromodulations
                              )
        
        self.servicesextinfo.merge(self.services)
        
        # print "Service groups"
        # link servicegroups members with services
        self.servicegroups.linkify(self.services)
        
        # link notificationways with timeperiods and commands
        self.notificationways.linkify(self.timeperiods, self.commands)
        
        # link notificationways with timeperiods and commands
        self.checkmodulations.linkify(self.timeperiods, self.commands)
        
        # Link with timeperiods
        self.macromodulations.linkify(self.timeperiods)
        
        # print "Contactgroups"
        # link contacgroups with contacts
        self.contactgroups.linkify(self.contacts)
        
        # print "Contacts"
        # link contacts with timeperiods and commands
        self.contacts.linkify(self.notificationways)
        
        # print "Timeperiods"
        # link timeperiods with timeperiods (exclude part)
        self.timeperiods.linkify()
        
        # print "Servicedependency"
        self.servicedependencies.linkify(self.hosts, self.services, self.timeperiods)
        
        # print "Hostdependency"
        self.hostdependencies.linkify(self.hosts, self.timeperiods)
        
        # print "Resultmodulations"
        self.resultmodulations.linkify(self.timeperiods)
        
        self.businessimpactmodulations.linkify(self.timeperiods)
        
        # print "Escalations"
        self.escalations.linkify(self.timeperiods, self.contacts, self.services, self.hosts)
        
        # Link discovery commands
        self.discoveryruns.linkify(self.commands)
        
        # print "Realms"
        self.realms.linkify()
        
        # Link all links with realms
        self.schedulers.linkify(self.realms, self.modules)
        
        # Ok, now update all realms with backlinks of satellites
        self.realms.prepare_for_satellites_conf()
    
    
    # same but for architecture objects
    def linkify_common(self):
        # link notificationways with timeperiods and commands
        self.notificationways.linkify(self.timeperiods, self.commands)
        
        # link notificationways with timeperiods and commands
        self.checkmodulations.linkify(self.timeperiods, self.commands)
        
        # Link with timeperiods
        self.macromodulations.linkify(self.timeperiods)
        
        # print "Contactgroups"
        # link contacgroups with contacts
        self.contactgroups.linkify(self.contacts)
        
        # print "Contacts"
        # link contacts with timeperiods and commands
        self.contacts.linkify(self.notificationways)
        
        # print "Timeperiods"
        # link timeperiods with timeperiods (exclude part)
        self.timeperiods.linkify()
        
        # print "Resultmodulations"
        self.resultmodulations.linkify(self.timeperiods)
        
        self.businessimpactmodulations.linkify(self.timeperiods)
        
        # print "Escalations"
        self.escalations.linkify(self.timeperiods, self.contacts, self.services, self.hosts)
        
        # print "Realms"
        self.realms.linkify()
        
        # Link all links with realms
        self.schedulers.linkify(self.realms, self.modules)
        self.brokers.linkify(self.realms, self.modules)
        self.receivers.linkify(self.realms, self.modules)
        self.reactionners.linkify(self.realms, self.modules)
        self.pollers.linkify(self.realms, self.modules)
        
        # Ok, now update all realms with backlinks of satellites
        self.realms.prepare_for_satellites_conf()
    
    
    # Removes service exceptions based on host configuration
    def remove_exclusions(self):
        return self.services.remove_exclusions(self.hosts)
    
    
    # Some elements are maybe set as wrong after a is_correct, so clean them
    # if possible
    def clean(self):
        self.services.clean()
    
    
    # In the scheduler we need to relink the commandCall with
    # the real commands
    def late_linkify(self):
        props = ['ocsp_command', 'ochp_command', 'service_perfdata_command', 'host_perfdata_command']
        for prop in props:
            cc = getattr(self, prop, None)
            if cc:
                cc.late_linkify_with_command(self.commands)
        # But also other objects like hosts and services
        self.hosts.late_linkify_h_by_commands(self.commands)
        self.services.late_linkify_s_by_commands(self.commands)
        self.contacts.late_linkify_c_by_commands(self.commands)
    
    
    # The function that just compute the whole conf pickle string, but n a children
    def __in_subprocess_create_new_monitoring_conf_pack(self, whole_queue):
        try:
            logger_lastest_monitoring_configuration.info("Starting to create the NEW Monitoring Configuration (in a worker process: pid=%s)" % os.getpid())
            whole_queue.append(cPickle.dumps(self, cPickle.HIGHEST_PROTOCOL))
            logger_lastest_monitoring_configuration.info("The NEW Monitoring Configuration is created on the worker process, giving back to the Arbiter process.")
        except:
            logger_lastest_monitoring_configuration.error('FAIL to generate the NEW Monitoring Configuration in the worker process: %s' % traceback.format_exc())
            sys.exit(2)
    
    
    def create_sub_worker_manager(self):
        # If the manager creation is taking 5min or more, it means we are dead lock
        with WatchDogThreadDumper('Worker manager creation', 60 * 5, 60 * 5, fatal_dead_lock_delay=60 * 5):
            self.sub_worker_manager = Manager()
        self.sub_worker_shared_list = {}
        for realm in self.realms:
            self.sub_worker_shared_list[realm.get_name()] = self.sub_worker_manager.list()
    
    
    def get_shards_from_sub_worker_manager(self):
        for realm in self.realms:
            lst = self.sub_worker_shared_list[realm.get_name()]
            sharded_inventories = []
            for (rname, shard_id, serialized_shard, host_names, sharded_hosts_and_checks_receiver_inventory, sharded_hosts_and_checks_broker_inventory) in lst:
                realm.add_shard(shard_id, serialized_shard, host_names)
                sharded_inventories.append((sharded_hosts_and_checks_receiver_inventory, sharded_hosts_and_checks_broker_inventory))
            # And update realm inventories from partial ones
            realm.update_inventory_from_sharded_inventories(sharded_inventories)
    
    
    def shutdown_sub_worker_manager(self):
        self.sub_worker_manager.shutdown()
        # We must delete our manager things before the whole conf pack, because we cannot pickle themm
        del self.sub_worker_manager
        del self.sub_worker_shared_list
    
    
    # Some properties are dangerous to be send like that
    # like realms linked in hosts. Realms are too big to send (too linked)
    # We are also pre-serializing the confs so the sending phase will
    # be quicker.
    def prepare_for_sending(self, filter_realm):
        # Preparing hosts and hostgroups for sending. Some properties
        # should be "flatten" before sent, like .realm object that should
        # be changed into names
        self.hosts.prepare_for_sending()
        self.hostgroups.prepare_for_sending()
        
        logger.info('[Arbiter] Serializing the configurations...')
        
        logger.info('Using the multiprocessing serialization pass')
        
        # We ask a manager to manage the communication with our children
        # The list will got all the strings from the children
        to_master_process_return = self.sub_worker_shared_list[filter_realm.get_name()]
        to_serialize = []
        for r in self.realms:
            if r != filter_realm:
                continue
            for (shard_id, sharded_configuration) in r.sharded_configurations.iteritems():
                to_serialize.append((r.get_name(), shard_id, sharded_configuration))
            # We no more need the shard configurations in the realms as we will load the serialized_version
            # del: to be sure we are not reading it after this point (must crash)
            del r.sharded_configurations
        
        # Spawn process
        receivers_inventory_filters = self.receivers.get_inventory_filters_by_receivers()
        processes = {}
        for (rname, shard_id, sharded_configuration) in to_serialize:
            # Prepare a sub-process that will manage the pickle computation
            p = Process(target=sharded_configuration.parallel_sharded_configuration_serialization, name="serializer-%s-%d" % (rname, shard_id), args=(to_master_process_return, rname, shard_id, receivers_inventory_filters))
            p.start()
            processes[shard_id] = p
        
        # Here all sub-processes are launched for this realm, now wait for them to finish
        while len(processes) != 0:
            to_del = []
            for shard_id, p in processes.iteritems():
                if p.exitcode is not None:
                    to_del.append(shard_id)
                    # remember to join() so the children can die
                    p.join()
                    if p.exitcode != 0:
                        logger.error('The process that serialized the shard %sdid fail. Exiting' % (shard_id))
                        sys.exit(2)
            for shard_id in to_del:
                del processes[shard_id]
            # Don't be too quick to poll!
            time.sleep(0.1)
    
    
    def create_new_monitoring_configuration(self):
        # If the manager creation take more than 5 minutes, it means we are dead locked
        with WatchDogThreadDumper('Total configuration manager', 60 * 5, 60 * 5, fatal_dead_lock_delay=60 * 5):
            whole_conf_manager = Manager()
        # Now pickle the whole configuration into one big pickle object, for the arbiter spares
        whole_conf_queue = whole_conf_manager.list()
        t0 = time.time()
        
        # Go for it
        p = Process(target=self.__in_subprocess_create_new_monitoring_conf_pack, args=(whole_conf_queue,), name='serializer-whole-configuration')
        p.start()
        # Wait for it to die
        while p.exitcode is None:
            time.sleep(0.1)
        p.join()
        # Maybe we don't have our result?
        if len(whole_conf_queue) != 1:
            logger_lastest_monitoring_configuration.error("The NEW Monitoring Configuration FAILED to create, please looks at logs and restart the Shinken Arbiter")
            sys.exit(2)
        
        # Get it and save it
        self.whole_conf_pack = whole_conf_queue.pop()
        logger_lastest_monitoring_configuration.info("The NEW Monitoring Configuration was generated in: %.2fs, size: %.3fMB %s" % ((time.time() - t0), to_mb_size(len(self.whole_conf_pack)), self.get_configuration_incarnation()))
        
        # Shutdown the manager, the sub-process should be gone now
        whole_conf_manager.shutdown()
    
    
    # We are in a retention load configuration in an arbiter spare, and we must be sure this configuration is at least able
    # to have the new communication infos from the configuration_incarnation. If not, the configuration will be dropped and
    # we will wait for a new arbiter master send
    def check_post_02_08_02_satellite_communication(self):
        r = getattr(self, '_configuration_incarnation', None) is not None  # property was not there in previous versions
        for scheduler in self.schedulers:
            r &= scheduler.check_post_02_08_02_satellite_communication()
        return r
    
    
    def dump(self):
        print("Slots", Service.__slots__)
        print('Hosts:')
        for h in self.hosts:
            print('\t', h.get_name(), h.contacts)
        print('Services:')
        for s in self.services:
            print('\t', s.get_name(), s.contacts)
    
    
    # It's used to change Nagios2 names to Nagios3 ones
    # For hosts and services
    def old_properties_names_to_new(self):
        super(Config, self).old_properties_names_to_new()
        self.hosts.old_properties_names_to_new()
        self.services.old_properties_names_to_new()
        self.notificationways.old_properties_names_to_new()
        self.contacts.old_properties_names_to_new()
    
    
    # Overrides specific instances properties in service:
    # * notification_options from host->service that need to be fixed now we have all values
    # * service_override values that need to be apply
    def override_properties(self):
        self.services.fix_service_notification_options_inheritance()
        self.services.override_properties(self.hosts)
    
    
    # Use to fill groups values on hosts and create new services
    # (for host group ones)
    def explode_realm_only(self):
        # first elements, after groups
        # print "Contacts"
        t0 = time.time()
        self.contacts.explode(self.contactgroups, self.notificationways)
        logger.info('[performance] - (detail for support only) explode::contacts:  %.2f' % (time.time() - t0))
        
        # print "Contactgroups"
        t0 = time.time()
        self.contactgroups.explode()
        logger.info('[performance] - (detail for support only) %s::%s:  %.2f' % ('explode', 'contactgroups', (time.time() - t0)))
        
        # print "Hosts"
        t0 = time.time()
        self.hosts.explode(self.hostgroups, self.contactgroups)
        # print "Hostgroups"
        self.hostgroups.explode()
        logger.info('[performance] - (detail for support only) %s::%s:  %.2f' % ('explode', 'hosts', (time.time() - t0)))
        
        # print "Services"
        # print "Initially got nb of services: %d" % len(self.services.items)
        t0 = time.time()
        self.services.explode(self.hosts, self.hostgroups, self.servicegroups, self.servicedependencies)
        
        # print "finally got nb of services: %d" % len(self.services.items)
        # print "Servicegroups"
        self.servicegroups.explode()
        logger.info('[performance] - (detail for support only) %s::%s:  %.2f' % ('explode', 'services', (time.time() - t0)))
        
        # print "Timeperiods"
        t0 = time.time()
        self.timeperiods.explode()
        logger.info('[performance] - (detail for support only) %s::%s:  %.2f' % ('explode', 'timeperiods', (time.time() - t0)))
        
        t0 = time.time()
        self.hostdependencies.explode(self.hostgroups)
        
        # print "Servicedependency"
        self.servicedependencies.explode(self.hostgroups)
        
        # Serviceescalations hostescalations will create new escalations
        self.serviceescalations.explode(self.escalations)
        self.hostescalations.explode(self.escalations)
        self.escalations.explode(self.hosts, self.hostgroups,
                                 self.contactgroups)
        
        self.realms.explode()
        logger.info('[performance] - (detail for support only) %s::%s:  %.2f' % ('explode', 'others', (time.time() - t0)))
    
    
    # This is a special explode phase that is done AFTER the implicit inheritance
    # as the check/service groups can be take from
    def post_inheritance_explode(self):
        self.hosts.post_inheritance_explode(self.contactgroups)
        self.services.post_inheritance_explode(self.contactgroups)
    
    
    def explode_common(self):
        # First common objects
        t0 = time.time()
        self.contacts.explode(self.contactgroups, self.notificationways)
        logger.info('[performance] - (detail for support only) %s::%s:  %.2f' % ('explode', 'contacts', (time.time() - t0)))
        
        # print "Contactgroups"
        t0 = time.time()
        self.contactgroups.explode()
        logger.info('[performance] - (detail for support only) %s::%s:  %.2f' % ('explode', 'contactgroups', (time.time() - t0)))
        
        # print "Timeperiods"
        t0 = time.time()
        self.timeperiods.explode()
        logger.info('[performance] - (detail for support only) %s::%s:  %.2f' % ('explode', 'timeperiods', (time.time() - t0)))
        
        # Now the architecture part
        # print "Realms"
        self.realms.explode()
    
    
    # Remove elements will the same name, so twins :)
    # In fact only services should be acceptable with twins
    def remove_twins(self):
        self.services.remove_twins(self.hosts)
    
    
    # Remove elements with a enabled 0 property, like in satellites
    def remove_disabled(self):
        lsts = [self.arbiters, self.schedulers, self.brokers, self.pollers, self.receivers, self.reactionners, self.synchronizers]
        for lst in lsts:
            to_del = []
            for i in lst:
                if i.enabled not in ('1', True):
                    to_del.append(i.id)
            for id_ in to_del:
                del lst[id_]
    
    
    def remove_orphan_services(self):
        self.services.remove_orphan_services()
    
    
    # return a dict with daemon type and the list of daemon instance
    def get_all_daemons(self):
        daemons = {}
        for sat_type in ('arbiters', 'schedulers', 'reactionners', 'brokers', 'receivers', 'pollers'):
            sats = getattr(self, sat_type)
            daemons[sat_type] = [sat for sat in sats]
        return daemons
    
    
    # Dependencies are important for scheduling
    # This function create dependencies linked between elements.
    def apply_dependencies(self):
        self.hosts.apply_dependencies()
        self.services.apply_dependencies()
    
    
    # Service can have a null value as poller_tag that means in fact:
    # take None and not the host value
    # cause: because the UI is currently not able to set None for this field
    def fix_null_poller_tag_service_into_real_none(self):
        self.services.fix_null_poller_tag_service_into_real_none()
    
    
    # Use to apply inheritance (template and implicit ones)
    # So elements will have their configured properties
    def apply_inheritance(self, parameter_only=None):
        # inheritance properties by template
        # print "Hosts"
        self.hosts.apply_inheritance(parameter_only=parameter_only)
        # print "Contacts"
        self.contacts.apply_inheritance(parameter_only=parameter_only)
        # print "Services"
        self.services.apply_inheritance(parameter_only=parameter_only)
    
    
    # Use to apply implicit inheritance
    def apply_implicit_inheritance(self):
        # type: () -> None
        self.services.apply_implicit_inheritance(self.hosts, self.commands)
        self.hosts.apply_implicit_inheritance(self.commands)
    
    
    def load_value_from_global_conf(self):
        # type: () -> None
        self.services.load_value_from_global_conf()
        self.hosts.load_value_from_global_conf()
    
    
    # will fill properties for elements so they will have all theirs properties
    def fill_default_realm_only(self):
        # Fill default for config (self)
        super(Config, self).fill_default()
        self.hosts.fill_default()
        self.hostgroups.fill_default()
        self.contacts.fill_default()
        self.contactgroups.fill_default()
        self.notificationways.fill_default()
        self.checkmodulations.fill_default()
        self.macromodulations.fill_default()
        self.services.fill_default()
        self.servicegroups.fill_default()
        self.resultmodulations.fill_default()
        self.businessimpactmodulations.fill_default()
        self.hostsextinfo.fill_default()
        self.servicesextinfo.fill_default()
        
        # Now escalations
        self.escalations.fill_default()
        
        # Also fill default of host/servicedep objects
        self.servicedependencies.fill_default()
        self.hostdependencies.fill_default()
        
        # Discovery part
        self.discoveryrules.fill_default()
        self.discoveryruns.fill_default()
        
        # For the is_correct pass of workers, we will need poller tags
        self.realms.fill_default()  # also put default inside the realms themselves
        self.reactionners.fill_default()
        self.pollers.fill_default()
        self.brokers.fill_default()
        self.receivers.fill_default()
        self.schedulers.fill_default()
        self.synchronizers.fill_default()
        
        # Now fill some fields we can predict (like address for hosts)
        self.fill_predictive_missing_parameters()
    
    
    def fill_default_common(self):
        super(Config, self).fill_default()
        # Also common objects
        self.contacts.fill_default()
        self.contactgroups.fill_default()
        self.notificationways.fill_default()
        self.checkmodulations.fill_default()
        self.macromodulations.fill_default()
        self.resultmodulations.fill_default()
        self.businessimpactmodulations.fill_default()
        # Now escalations
        self.escalations.fill_default()
        # Discovery part
        self.discoveryrules.fill_default()
        self.discoveryruns.fill_default()
        
        # now we have all elements, we can create a default
        # realm if need and it will be tagged to sat that do
        # not have an realm
        self.realms.fill_default()  # also put default inside the realms themselves
        self.reactionners.fill_default()
        self.pollers.fill_default()
        self.brokers.fill_default()
        self.receivers.fill_default()
        self.schedulers.fill_default()
        self.synchronizers.fill_default()
    
    
    # Here is a special functions to fill some special
    # properties that are not filled and should be like
    # address for host (if not set, put host_name)
    def fill_predictive_missing_parameters(self):
        self.hosts.fill_predictive_missing_parameters()
    
    
    # Will ask for each host/service if the
    # check_command is a bp rule. If so, it will create
    # a tree structures with the rules
    def create_business_rules(self):
        self.hosts.create_business_rules(self.proxy_items)
        self.services.create_business_rules(self.proxy_items)
    
    
    # Will fill dep list for business rules
    def create_business_rules_dependencies(self):
        self.hosts.create_business_rules_dependencies(self.hosts, self.services)
        self.services.create_business_rules_dependencies(self.hosts, self.services)
    
    
    # Set our timezone value and give it too to unset satellites
    def propagate_option(self, opt_name):
        opt_val = getattr(self, opt_name, '')
        if opt_val != '':
            # first apply myself
            tab = [self.schedulers, self.pollers, self.brokers, self.receivers, self.reactionners]
            for t in tab:
                for s in t:
                    if getattr(s, opt_name, 'NOTSET') == 'NOTSET':
                        setattr(s, opt_name, opt_val)
    
    
    # Link templates with elements
    def linkify_templates(self):
        """ Like for normal object, we link templates with each others """
        self.hosts.linkify_templates()
        self.contacts.linkify_templates()
        self.services.linkify_templates()
    
    
    # Reversed list is a dist with name for quick search by name
    def create_reversed_list(self):
        """ Create quick search lists for objects """
        self.hosts.create_reversed_list()
        self.hostgroups.create_reversed_list()
        self.contacts.create_reversed_list()
        self.contactgroups.create_reversed_list()
        self.notificationways.create_reversed_list()
        self.checkmodulations.create_reversed_list()
        self.macromodulations.create_reversed_list()
        self.services.create_reversed_list()
        self.servicegroups.create_reversed_list()
        self.timeperiods.create_reversed_list()
        # self.modules.create_reversed_list()
        self.resultmodulations.create_reversed_list()
        self.businessimpactmodulations.create_reversed_list()
        self.escalations.create_reversed_list()
        self.discoveryrules.create_reversed_list()
        self.discoveryruns.create_reversed_list()
        self.commands.create_reversed_list()
        
        # For services it's a special case
        # we search for hosts, then for services
        # it's quicker than search in all services
        self.services.optimize_service_search(self.hosts)
    
    
    # Some parameters are just not managed like O*HP commands
    # and regexp capabilities
    # True: OK
    # False: error in conf
    def check_error_on_hard_unmanaged_parameters(self):
        r = True
        if self.use_regexp_matching:
            logger.error("use_regexp_matching parameter is not managed.")
            r &= False
        return r
    
    
    # Default value can be set in file /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
    # There can be incorrect so we checks them here.
    def check_default_values(self):
        r = True
        if Contact.properties['acl_in_tab_history'].default not in ('history_sla', 'history', 'sla'):
            logger.error("Unknown default value for acl_in_tab_history : [%s]. Possible values for acl_in_tab_history are history_sla, history, sla." % Contact.properties['acl_in_tab_history'].default)
            r &= False
        return r
    
    
    # check if elements are correct or not (fill with defaults, etc)
    # Warning: this function call be called from a Arbiter AND
    # from and scheduler. The first one got everything, the second
    # does not have the satellites.
    def is_correct_realm_only(self, check_poller_tag_existance=True):
        """ Check if all elements got a good configuration """
        r = self.conf_is_correct
        
        if not self.check_error_on_hard_unmanaged_parameters():
            r = False
            logger.error("Check global parameters failed")
        if not self.check_default_values():
            r = False
            logger.error("Check default values failed")
        else:
            if self.read_config_silent == 0:
                logger.info('Global parameters are OK')
        
        # NOT architecture one
        for obj_class in ('hosts', 'hostgroups', 'services', 'servicegroups', 'servicedependencies', 'hostdependencies'):
            
            cur = getattr(self, obj_class)
            if not cur.is_correct():
                r = False
                with arbiter_configuration_messages.no_arbiter_message:
                    logger.error("\t- %s configuration is incorrect" % (obj_class))
                for error in cur.configuration_errors:
                    logger.error("\t%s" % error)
            if self.read_config_silent == 0:
                logger.info('\t- Checked %-5d %s' % (len(cur), obj_class))
        
        # Hosts got a special check for loops
        if not self.hosts.no_loop_in_parents():
            r = False
            logger.error("Hosts: detected loop in parents ; conf incorrect")
        
        # Look that all scheduler got a broker that will take brok.
        # If there are no, raise an Error
        # for s in self.schedulers:
        #    rea = s.realm
        
        # Check that for each poller_tag of a host, a poller exists with this tag
        # TODO: need to check that poller are in the good realm too
        hosts_by_tag = {}
        services_by_tag = {}
        pollers_tag = set()
        for h in self.hosts:
            if not h.got_business_rule:  # skip cluster elements, they are not executed into the pollers
                h_list = hosts_by_tag.get(h.poller_tag, [])
                h_list.append(h)
                hosts_by_tag[h.poller_tag] = h_list
        for s in self.services:
            s_list = services_by_tag.get(s.poller_tag, [])
            s_list.append(s)
            services_by_tag[s.poller_tag] = s_list
        for p in self.pollers:
            for t in p.poller_tags:
                pollers_tag.add(t)
        
        if check_poller_tag_existance:
            hosts_tag = set(hosts_by_tag.keys())
            services_tag = set(services_by_tag.keys())
            if not hosts_tag.issubset(pollers_tag):
                for tag in hosts_tag.difference(pollers_tag):
                    error_msg = "Hosts [%s] exist with poller_tag %s but no poller got this tag" % (','.join((h.get_full_name() for h in hosts_by_tag[tag])), ToolsBoxString.escape_XSS(tag))
                    logger.error(error_msg)
                    self.add_error(error_msg)
                    r = False
            if not services_tag.issubset(pollers_tag):
                for tag in services_tag.difference(pollers_tag):
                    error_msg = "Services [%s] exist with poller_tag %s but no poller got this tag" % (','.join((s.get_full_name() for s in services_by_tag[tag])), ToolsBoxString.escape_XSS(tag))
                    logger.error(error_msg)
                    self.add_error(error_msg)
                    r = False
        
        # Check that all hosts involved in business_rules are from the same realm
        for l in [self.services, self.hosts]:
            for e in l:
                if e.got_business_rule:
                    e_ro = e.get_realm()
                    # Something was wrong in the conf, will be raised elsewhere
                    if not e_ro:
                        continue
                    e_r = e_ro.realm_name
                    if e.business_rule is None:  # if there is no ruel, don't care about links
                        continue
                    for proxy in e.business_rule.list_all_elements():
                        if proxy.type == 'check':
                            elt = self.services.find_srv_by_name_and_hostname(proxy.host_name, proxy.name)
                        else:
                            elt = self.hosts.find_by_name(proxy.name)
                        r_o = elt.get_realm()
                        # Something was wrong in the conf, will be raised elsewhere
                        if not r_o:
                            continue
                        elt_r = elt.get_realm().realm_name
                        if not elt_r == e_r:
                            logger.error("Business_rule '%s' got hosts from another realm: %s" % (e.get_full_name(), elt_r))
                            self.add_error("Error: Business_rule '%s' got hosts from another realm: %s" % (e.get_full_name(), elt_r))
                            r = False
        
        self.conf_is_correct = r
    
    
    # check if elements are correct or not (fill with defaults, etc)
    # Warning: this function call be called from a Arbiter AND
    # from and scheduler. The first one got everything, the second
    # does not have the satellites.
    def is_correct_common(self, check_poller_tag_existance=True):
        """ Check if all elements got a good configuration """
        logger.info('Running pre-flight check on configuration data...')
        r = self.conf_is_correct
        
        if not self.check_error_on_hard_unmanaged_parameters():
            r = False
            logger.error("Check global parameters failed")
        if not self.check_default_values():
            r = False
            logger.error("Check default values failed")
        
        # Do not care about objects, they are already managed in workers
        for obj_class in ('contacts', 'contactgroups', 'notificationways',
                          'escalations', 'timeperiods', 'commands',
                          'hostsextinfo', 'servicesextinfo', 'checkmodulations', 'macromodulations', 'resultmodulations',
                          'discoveryrules', 'discoveryruns', 'businessimpactmodulations',
                          'arbiters', 'synchronizers', 'schedulers',
                          'reactionners', 'pollers', 'brokers', 'receivers', 'realms',
                          ):
            
            cur = getattr(self, obj_class)
            if not cur.is_correct():
                r = False
                logger.error("\t%s conf incorrect" % (obj_class.capitalize()))
                for error in cur.configuration_errors:
                    logger.error("\t%s" % error)
            if self.read_config_silent == 0:
                logger.info('\t - checked %d %s' % (len(cur), obj_class))
        
        daemons_list = {}
        for obj_class in ('arbiters', 'synchronizers', 'schedulers', 'reactionners', 'pollers', 'brokers', 'receivers'):
            cur = getattr(self, obj_class)
            for daemon in cur.items.itervalues():
                _name = daemon.get_name()
                if daemons_list.get(_name):
                    daemons_list[_name].append(daemon)
                else:
                    daemons_list[_name] = [daemon]
        
        for name, daemons in daemons_list.iteritems():
            if len(daemons) > 1:
                r = False
                logger.error('\tDaemons names must be unique. The name [%s] is used by more than one daemon. Please check your configuration in files :' % name)
                for daemon in daemons:
                    logger.error('\t- %s' % daemon.imported_from)
        
        if 'host' in self.default_properties_values:
            if 'view_contacts' in self.default_properties_values['host']:
                if not VIEW_CONTACTS_DEFAULT_VALUE.is_valid_default_value(self.default_properties_values['host']['view_contacts']):
                    self.add_error(
                        "Incorrect default value '%s' for [DEFAULT:host] view_contacts : must be '%s' or '%s'" % (self.default_properties_values['host']['view_contacts'], VIEW_CONTACTS_DEFAULT_VALUE.NOBODY, VIEW_CONTACTS_DEFAULT_VALUE.EVERYONE))
                    r = False
        self.conf_is_correct = r
    
    
    # We've got strings (like 1) but we want python elements, like True
    def pythonize_realm_only(self):
        # call item pythonize for parameters
        super(Config, self).pythonize()
        self.hosts.pythonize()
        self.hostgroups.pythonize()
        self.hostdependencies.pythonize()
        self.contactgroups.pythonize()
        self.contacts.pythonize()
        self.notificationways.pythonize()
        self.checkmodulations.pythonize()
        self.macromodulations.pythonize()
        self.servicegroups.pythonize()
        self.services.pythonize()
        self.servicedependencies.pythonize()
        self.resultmodulations.pythonize()
        self.businessimpactmodulations.pythonize()
        self.escalations.pythonize()
        self.discoveryrules.pythonize()
        self.discoveryruns.pythonize()
        self.commands.pythonize()
        
        # We also need poller ones, with poller_tags
        self.synchronizers.pythonize()
        self.schedulers.pythonize()
        self.realms.pythonize()
        self.reactionners.pythonize()
        self.pollers.pythonize()
        self.brokers.pythonize()
        self.receivers.pythonize()
    
    
    def pythonize_common(self):
        super(Config, self).pythonize()
        self.contactgroups.pythonize()
        self.contacts.pythonize()
        self.notificationways.pythonize()
        self.checkmodulations.pythonize()
        self.macromodulations.pythonize()
        self.resultmodulations.pythonize()
        self.businessimpactmodulations.pythonize()
        self.escalations.pythonize()
        self.commands.pythonize()
        
        # The arbiters are already done
        # self.arbiters.pythonize()
        self.synchronizers.pythonize()
        self.schedulers.pythonize()
        self.realms.pythonize()
        self.reactionners.pythonize()
        self.pollers.pythonize()
        self.brokers.pythonize()
        self.receivers.pythonize()
    
    
    # Explode parameters like cached_service_check_horizon in the
    # Service class in a cached_check_horizon manner, o*hp commands
    # , etc
    def explode_global_conf(self):
        clss = [Service, Host, Contact, SchedulerLink,
                PollerLink, ReactionnerLink, BrokerLink,
                ReceiverLink, ArbiterLink, SynchronizerLink, HostExtInfo]
        for cls in clss:
            cls.load_global_conf(self)
    
    
    def change_default_values_into_finals(self):
        self.hosts.change_default_values_into_finals()
        self.services.change_default_values_into_finals()
    
    
    # Clean useless elements like templates because they are not needed anymore
    def remove_templates(self):
        self.hosts.remove_templates()
        self.contacts.remove_templates()
        self.services.remove_templates()
    
    
    def create_proxy_items(self):
        item_proxies = []
        for h in self.hosts:
            prox = ProxyItem(h)
            item_proxies.append(prox)
            for s in h.services:
                prox_s = ProxyItem(s)
                item_proxies.append(prox_s)
        self.proxy_items = ProxyItems(item_proxies)
    
    
    # Add an error in the configuration error list so we can print them
    # all in one place
    def add_error(self, txt):
        self.configuration_errors.append(txt)
        self.conf_is_correct = False
    
    
    # Now it's time to show all configuration errors
    def show_errors(self):
        for err in self.configuration_errors:
            logger.error(err)
    
    
    # Create packs of hosts and services so in a pack all dependencies are resolved.
    # It creates a graph.
    # All hosts are connected to their parents, and hosts without parent are connected to host 'root'.
    # services are link to the host.
    # Dependencies are managed
    # REF: doc/pack-creation.png
    def _create_packs(self, realm_filter):
        # We create a graph with host in nodes
        graph = Graph()
        graph.add_nodes(self.hosts)
        
        # links will be used for relations between hosts
        links = set()
        all_links = set()
        
        # Now the relations
        for host in self.hosts:
            # Add parent relations
            for parent in host.parents:
                if parent is not None:
                    links.add((parent, host))
            # Add the others dependencies
            for dep, _, _, _, _ in host.act_depend_of:
                links.add((dep, host))
            for dep, _, _, _, _ in host.chk_depend_of:
                links.add((dep, host))
        
        # For services: they are linked with their own host,
        # but we need to have the hosts of service dep in the same pack too
        for service in self.services:
            for dep, _, _, _, _ in service.act_depend_of:
                # for all links, we need the element itself
                # For pack I don't care about dep host: they are just the host of the service...
                if hasattr(dep, 'host'):
                    links.add((dep.host, service.host))
            # The other type of dep
            for dep, _, _, _, _ in service.chk_depend_of:
                links.add((dep.host, service.host))
        
        # keep track of the 'clusters', even them without elements
        son_nodes = set()
        # For host/service that are business based,
        # we need to link them too, but only for all links (because they don't need to be set in the same pack anymore)
        for service in self.services:
            if not service.got_business_rule:
                continue
            son_nodes.add(service.get_instance_uuid())
            if getattr(service, 'business_rule', None) is not None:
                for parent in service.business_rule.list_all_elements():
                    all_links.add((parent.uuid, service.get_instance_uuid()))
        
        # Same for hosts of course
        for host in self.hosts:
            if not host.got_business_rule:
                continue
            son_nodes.add(host.get_instance_uuid())
            if getattr(host, 'business_rule', None) is not None:
                for parent in host.business_rule.list_all_elements():
                    all_links.add((parent.uuid, host.get_instance_uuid()))
        
        # We can declare links into the
        proxyitemsgraph.reset_from_all_links(all_links, son_nodes)
        
        # Now we create links in the graph. With links (set)
        # We are sure to call the less add_edge
        for (dep, host) in links:
            graph.add_edge(dep, host)
            graph.add_edge(host, dep)
        
        # Access the list from a node where all nodes are connected : it's a list of ours tmp_packs
        tmp_packs = graph.get_accessibility_packs()
        
        # Now We find the default realm (must be unique or BAD THINGS MAY HAPPEN)
        default_realm = next((realm for realm in self.realms if hasattr(realm, 'default') and realm.default), None)
        
        # Now we look if all elements of all packs have the same realm. If not, not good!
        for pack in tmp_packs:
            tmp_realms = set([item.realm for item in pack if item.realm is not None])
            
            if len(tmp_realms) > 1:
                self.add_error(u'Error: the realm configuration of yours hosts is not good because there a more than one realm in one pack (host relations):')
                for host in pack:
                    if host.realm is None:
                        err = u'   the host %s do not have a realm' % host.get_name()
                        self.add_error(err)
                    else:
                        err = u'   the host %s is in the realm %s' % (host.get_name(), host.realm.get_name())
                        self.add_error(err)
            
            if len(tmp_realms) == 1:  # Ok, good
                realm = tmp_realms.pop()  # There is just one element
                realm.packs.append(pack)
            
            elif len(tmp_realms) == 0:  # Hum.. no realm value? So default Realm
                if default_realm is not None:
                    default_realm.packs.append(pack)
                else:
                    err = u'Error: some hosts do not have a realm and you do not defined a default realm!'
                    self.add_error(err)
                    for host in pack:
                        err = u'    Impacted host: %s ' % host.get_name()
                        self.add_error(err)
        
        # The load balancing is for a loop,
        # so all hosts of a realm (in a pack) will be dispatch in the schedulers of this realm
        # REF: doc/pack-aggregation.png
        
        # Count the numbers of elements in all the realms, to compare it the total number of hosts
        nb_elements_all_realms = 0
        for realm in self.realms:
            # Only look at the realm we want
            if realm != realm_filter:
                continue
            packs = {}
            # create roundrobin iterator for id of cfg
            # So dispatching is load balanced in a realm
            # but add an entry in the roundrobin tourniquet for
            # every weight point schedulers (so Weight round-robin)
            weight_list = []
            no_spare_schedulers = [service for service in realm.schedulers if not service.spare]
            nb_schedulers = len(no_spare_schedulers)
            
            # Maybe there is no scheduler in the realm, it can be a big problem if there are elements in packs
            nb_elements = 0
            for pack in realm.packs:
                nb_elements += len(pack)
                nb_elements_all_realms += len(pack)
            logger.info(u'Number of hosts+clusters in the realm %-15s: %d (distributed in %d un-separated element packs)' % (realm.get_name(), nb_elements, len(realm.packs)))
            
            if nb_schedulers == 0 and nb_elements != 0:
                err = u'The realm %s has hosts but no scheduler!' % realm.get_name()
                self.add_error(err)
                for pack in realm.packs:
                    for host in pack:
                        err = u'  - Host: %-40s    (uuid: %s)' % (host.get_name(), host.get_instance_uuid())
                        self.add_error(err)
                realm.packs = {}  # Dumb pack
                continue
            
            pack_index = 0
            pack_indices = {}
            for service in no_spare_schedulers:
                pack_indices[service.id] = pack_index
                pack_index += 1
                for i in range(0, service.weight):
                    weight_list.append(service.id)
            
            rr = itertools.cycle(weight_list)
            
            # We must have nb_schedulers packs
            for i in range(0, nb_schedulers):
                packs[i] = []
            
            # Now we explode the numerous packs into nb_packs reals packs:
            # we 'load balance' them in a roundrobin way
            for pack in realm.packs:
                i = rr.next()
                
                for item in pack:
                    packs[pack_indices[i]].append(item)
            
            # Now in packs we have the number of packs [h1, h2, etc]
            # equal to the number of schedulers.
            realm.packs = packs
        logger.info(u'Total number of hosts : %d' % nb_elements_all_realms)
        if len(self.hosts) != nb_elements_all_realms:
            msg = u'There are %d hosts defined, and %d hosts dispatched in the realms. Some hosts have been ignored' % (len(self.hosts), nb_elements_all_realms)
            logger.warning(msg)
            self.add_error(msg)
    
    
    # Use the self.conf and make nb_parts new confs.
    # nbparts is equal to the number of schedulerlink
    # New confs are independent with checks. The only communication
    # That can be need is macro in commands
    def cut_into_parts(self, filter_realm):
        # Lazy load of the ShardedConfiguration class to avoid recursive loop
        from .shardedconfiguration import ShardedConfiguration
        
        # print "Scheduler configured:", self.schedulers
        # I do not care about alive or not. User must have set a spare if need it
        nb_parts = len([s for s in self.schedulers if not s.spare if s.realm == filter_realm])
        logger.info(u'SHARDS: need to create %d parts (realm id:%d)' % (nb_parts, filter_realm.id))
        if nb_parts == 0:
            nb_parts = 1
        
        shards_id_base = filter_realm.id * 256  # we allow only 256 scheduler by realm
        # We create dummy configurations for schedulers:
        # they are clone of the master
        # conf but without hosts and services (because they are dispatched between
        # theses configurations)
        for part_id in range(shards_id_base, shards_id_base + nb_parts):
            # print "Create Conf:", i, '/', nb_parts -1
            cur_shard = self.sharded_configurations[part_id] = ShardedConfiguration()
            
            cur_shard.load_from_global_configuration(part_id, self)
        
        # Just create packs. There can be numerous ones
        # In pack we've got hosts and service
        # packs are in the realms
        # REF: doc/pack-creation.png
        self._create_packs(filter_realm)
        
        # We've got all big packs and get elements into configurations
        # REF: doc/pack-agregation.png
        offset = 0
        for realm in self.realms:
            if realm != filter_realm:
                continue
            logger.info('SHARDS: Merging %d packs into shards=%s (packs keys=%s)' % (len(realm.packs), self.sharded_configurations.keys(), realm.packs.keys()))
            # Change packs into real realm configuration/shard
            for i in realm.packs:
                sharded_configuration = self.sharded_configurations[shards_id_base + i + offset]
                pack = realm.packs[i]
                # We can get host and checks from this pack now
                sharded_configuration.load_hosts_and_checks_from_pack(shards_id_base + i, pack)
                # Now the conf can be link in the realm
                realm.sharded_configurations[shards_id_base + i + offset] = sharded_configuration
            offset += len(realm.packs)
            del realm.packs  # packs are no more need
            
            # Now generate proxy items for this realm elements only
            realm_proxy_items = []
            for sharded_configuration in realm.sharded_configurations.itervalues():
                for host in sharded_configuration.hosts:
                    proxy = self.proxy_items[host.uuid]
                    realm_proxy_items.append(proxy)
                    for s in host.services:
                        key = s.get_instance_uuid()
                        proxy = self.proxy_items[key]
                        realm_proxy_items.append(proxy)
            realm_item_proxies = ProxyItems(realm_proxy_items)
            
            # Link the same proxies to all shards of this realm
            for sharded_configuration in realm.sharded_configurations.itervalues():
                sharded_configuration.item_proxies = realm_item_proxies
        
        logger.info('SHARDS realm shards keys: %s' % self.sharded_configurations.keys())
        
        # We've nearly have hosts and services. Now we want REALS hosts (Class)
        # And we want groups too
        for shard_id in self.sharded_configurations:
            # print "Finishing pack Nb:", i
            sharded_configuration = self.sharded_configurations[shard_id]
            
            sharded_configuration.finish_load_and_relink(self)
        
        # We tag conf with instance_id
        for shard_id in self.sharded_configurations:
            self.sharded_configurations[shard_id].instance_id = shard_id
    
    
    # TODO: make this function call mongodb to make all realm share data instead of one file by realm
    def set_monitoring_start_time_realm_only(self, realm):
        now = int(time.time())
        has_retention = False
        monitoring_retention = {}
        
        # by default try to load our realm_retention file, if missing, switch to the old arbiter one (migrating)
        realm_file = REALM_RETENTION_JSON % realm.get_name()
        file_to_load = realm_file
        if not os.path.isfile(realm_file):
            file_to_load = OLD_ARBITER_RETENTION_JSON
            logger.info('The realm monitoring start time retention file %s is not available, try to load the previous arbiter file: %s' % (realm_file, file_to_load))
        
        # load retention file
        if os.path.isfile(file_to_load):
            try:
                with open(file_to_load, 'r') as f:
                    monitoring_retention = json.load(f)
                    has_retention = True
            except Exception as exp:
                logger.error('We cannot load the monitoring start time file at %s: %s' % (file_to_load, exp))
        
        # Now read the old data, and save a new one
        # IMPORTANT: DO NOT CLEAN OLD DATA: we want to keep monitoring start time, even if elements are temporary disabled
        realm_file_tmp = realm_file + '.tmp'
        with open(realm_file_tmp, 'w+') as retention_file:
            logger.debug('[monitoring_start_time] has_retention [%s]' % has_retention)
            # set monitoring start time:
            # * if we did load a file, but it's a new element, set now
            # * if we did fail to load a file (error, etc) then set -1 to let the SLA know it must NOT erase all that was before
            for i in itertools.chain(self.hosts, self.services):
                if has_retention:
                    i.monitoring_start_time = monitoring_retention.get(i.get_instance_uuid(), now)
                else:
                    i.monitoring_start_time = -1
                
                monitoring_retention[i.get_instance_uuid()] = i.monitoring_start_time
            
            # save retention file
            json.dump(monitoring_retention, retention_file)
            logger.info('The realm monitoring start time are saved into the file %s' % realm_file)
        # Move into a one shot, prevent breaking file and loose data at restart
        shutil.move(realm_file_tmp, realm_file)
    
    
    def set_master_arbiter_uuid(self, server_uuid):
        self.master_arbiter_uuid = server_uuid


# ...
def lazy():
    # let's compute the "USER" properties and macros..
    for n in range(1, 256):
        n = str(n)
        Config.properties['$USER' + str(n) + '$'] = StringProp(default='')
        Config.macros['USER' + str(n)] = '$USER' + n + '$'


lazy()
del lazy
