#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

from collections import deque

from item import InheritableItem, InheritableItems
from shinken.log import naglog_result
from shinken.misc.configuration_error_log import log_configuration_error
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.property import BoolProp, IntegerProp, StringProp, AclShareProp, LinkToAnotherObjectProp, LinkToOthersObjectsProp, RawProp

if TYPE_CHECKING:
    from shinken.misc.type_hint import List

_special_properties = ('service_notification_commands', 'host_notification_commands',
                       'service_notification_period', 'host_notification_period',
                       'service_notification_options', 'host_notification_options',
                       'host_notification_commands', 'contact_name')

_simple_way_parameters = ('service_notification_period', 'host_notification_period',
                          'service_notification_options', 'host_notification_options',
                          'service_notification_commands', 'host_notification_commands',
                          'min_business_impact')


class Contact(InheritableItem):
    id = 1  # zero is always special in database, so we do not take risk here
    my_type = 'contact'
    
    properties = InheritableItem.properties.copy()
    properties.update({
        ############################# Simple properties, can be set into a C structure
        'host_notifications_enabled'   : BoolProp(default='1', fill_brok=['full_status']),
        'service_notifications_enabled': BoolProp(default='1', fill_brok=['full_status']),
        'min_business_impact'          : IntegerProp(default='0', fill_brok=['full_status']),
        'can_submit_commands'          : BoolProp(default='1', fill_brok=['full_status']),
        'is_admin'                     : BoolProp(default='0', fill_brok=['full_status']),
        
        'expert'                       : BoolProp(default='0', fill_brok=['full_status']),
        
        'acl_show_history_range'       : BoolProp(default='1', fill_brok=['full_status']),
        'acl_show_sla_range'           : BoolProp(default='1', fill_brok=['full_status']),
        'acl_make_downtime'            : BoolProp(default='1', fill_brok=['full_status']),
        'acl_make_acknowledge'         : BoolProp(default='1', fill_brok=['full_status']),
        'acl_force_result_check'       : BoolProp(default='1', fill_brok=['full_status']),
        'acl_force_retry_check'        : BoolProp(default='1', fill_brok=['full_status']),
        'default_submit_to_staging'    : BoolProp(default='1'),  # for synchronizer by pass working area
        'acl_try_check_on_poller'      : BoolProp(default='1'),  # for synchronizer right
        'acl_try_check_on_synchronizer': BoolProp(default='1'),  # for synchronizer right
        'acl_show_external_url'        : BoolProp(default='1', fill_brok=['full_status']),  # for synchronizer right
        
        ########################### Configuration Structure, like links to others elements
        'contact_name'                 : StringProp(fill_brok=['full_status']),
        'alias'                        : StringProp(default='none', fill_brok=['full_status']),
        'contactgroups'                : LinkToOthersObjectsProp(default='', fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'host_notification_period'     : LinkToAnotherObjectProp(default='24x7', fill_brok=['full_status']),
        'service_notification_period'  : LinkToAnotherObjectProp(default='24x7', fill_brok=['full_status']),
        
        # Theses one can be size bounded
        'host_notification_options'    : StringProp(default='d,u,r,f', fill_brok=['full_status']),
        'service_notification_options' : StringProp(default='w,u,c,r,f,s', fill_brok=['full_status']),
        'host_notification_commands'   : StringProp(fill_brok=['full_status']),
        'service_notification_commands': StringProp(fill_brok=['full_status']),
        
        'email'                        : StringProp(default='none', fill_brok=['full_status']),
        'pager'                        : StringProp(default='none', fill_brok=['full_status']),
        'address1'                     : StringProp(default='none', fill_brok=['full_status']),
        'address2'                     : StringProp(default='none', fill_brok=['full_status']),
        'address3'                     : StringProp(default='none', fill_brok=['full_status']),
        'address4'                     : StringProp(default='none', fill_brok=['full_status']),
        'address5'                     : StringProp(default='none', fill_brok=['full_status']),
        'address6'                     : StringProp(default='none', fill_brok=['full_status']),
        
        'notificationways'             : LinkToOthersObjectsProp(default='', merging='join', fill_brok=['full_status'], handle_additive_inheritance=True),
        'password'                     : StringProp(default='', fill_brok=['full_status']),
        'display_name'                 : StringProp(default=''),
        
        'acl_in_tab_history'           : StringProp(default='history_sla', fill_brok=['full_status']),
        'acl_share_private'            : AclShareProp(default='all', fill_brok=['full_status']),
        'acl_share_group'              : AclShareProp(default='all', fill_brok=['full_status']),
        'acl_share_everybody'          : AclShareProp(default='all', fill_brok=['full_status']),
    })
    
    running_properties = InheritableItem.running_properties.copy()
    running_properties.update({
        'modified_attributes': IntegerProp(default=0, fill_brok=['full_status'], retention=True),
        'downtimes'          : RawProp(default=[], fill_brok=['full_status'], retention=True),
        'customs'            : RawProp(default={}, fill_brok=['full_status']),
    })
    
    # This tab is used to transform old parameters name into new ones
    # so from Nagios2 format, to Nagios3 ones.
    # Or Shinken deprecated names like criticity
    old_properties = {
        'min_criticity' : 'min_business_impact',
        'contact_groups': 'contactgroups',
    }
    
    macros = {
        'CONTACTNAME'      : 'contact_name',
        'CONTACTALIAS'     : 'alias',
        'CONTACTEMAIL'     : 'email',
        'CONTACTPAGER'     : 'pager',
        'CONTACTADDRESS1'  : 'address1',
        'CONTACTADDRESS2'  : 'address2',
        'CONTACTADDRESS3'  : 'address3',
        'CONTACTADDRESS4'  : 'address4',
        'CONTACTADDRESS5'  : 'address5',
        'CONTACTADDRESS6'  : 'address6',
        'CONTACTGROUPNAME' : 'get_groupname',
        'CONTACTGROUPNAMES': 'get_groupnames'
    }
    
    passthrough = InheritableItem.passthrough.copy()
    passthrough.update({
        'template_members': StringProp(default='', merging='join'),  # for synchronizer double link
    })
    
    not_inherited_passthrough = InheritableItem.not_inherited_passthrough.copy()
    not_inherited_passthrough.update({
        'for_all_users': BoolProp(default='1'),
    })
    
    
    def get_name(self):
        return getattr(self, 'contact_name', getattr(self, 'name', 'UnnamedContact'))
    
    
    # Search for notification_options with state and if t is
    # in service_notification_period
    def want_service_notification(self, t, state, type, business_impact, cmd=None):
        if not self.service_notifications_enabled:
            return False
        
        # If we are in downtime, we don't want notification
        for dt in self.downtimes:
            if dt.is_in_effect:
                return False
        
        # Now the rest is for sub notificationways. If one is OK, we are ok
        # We will filter in another phase
        for nw in self.notificationways:
            nw_b = nw.want_service_notification(t, state, type, business_impact, cmd)
            if nw_b:
                return True
        
        # Oh... no one is ok for it? so no, sorry
        return False
    
    
    # Search for notification_options with state and if t is in
    # host_notification_period
    def want_host_notification(self, t, state, type, business_impact, cmd=None):
        if not self.host_notifications_enabled:
            return False
        
        # If we are in downtime, we don't want notification
        for dt in self.downtimes:
            if dt.is_in_effect:
                return False
        
        # Now it's all for sub notificationways. If one is OK, we are OK
        # We will filter in another phase
        for nw in self.notificationways:
            nw_b = nw.want_host_notification(t, state, type, business_impact, cmd)
            if nw_b:
                return True
        
        # Oh, nobody..so NO :)
        return False
    
    
    # Call to get our commands to launch a Notification
    def get_notification_commands(self, type):
        r = []
        # service_notification_commands for service
        notif_commands_prop = type + '_notification_commands'
        for nw in self.notificationways:
            r.extend(getattr(nw, notif_commands_prop))
        return r
    
    
    # Check if the contact is valid.
    # Error message will be logged by log_configuration_error or log_configuration_warning.
    def is_correct(self):
        # type: () -> bool
        state = True
        cls = self.__class__
        
        for property_name, property_definition in cls.properties.iteritems():
            if property_name not in _special_properties and not hasattr(self, property_name) and property_definition.required:
                state = False
                log_configuration_error(self, u'The property %s is not set', property_name)
        
        # If there isn't notificationways it mean that contact is in old format.
        # So we check if all prop in _special_properties are define
        if self.notificationways == []:
            for p in _special_properties:
                if not hasattr(self, p):
                    state = False
                    log_configuration_error(self, u'The property %s is not set', property_name)
        
        # take the alias if we miss the contact_name
        if not hasattr(self, 'contact_name') and hasattr(self, 'alias'):
            self.contact_name = self.alias
        
        if hasattr(self, 'contact_name'):
            for char in cls.illegal_object_name_chars:
                if char in self.contact_name:
                    log_configuration_error(self, u'The character %s is not allowed in name.', char)
                    state = False
        
        return state
    
    
    # Raise a log entry when a downtime begins
    # CONTACT DOWNTIME ALERT: test_contact;STARTED; Contact has entered a period of scheduled downtime
    def raise_enter_downtime_log_entry(self):
        naglog_result('info', "CONTACT DOWNTIME ALERT: %s;STARTED; Contact has "
                              "entered a period of scheduled downtime"
                      % self.get_name())
    
    
    # Raise a log entry when a downtime has finished
    # CONTACT DOWNTIME ALERT: test_contact;STOPPED; Contact has exited from a period of scheduled downtime
    def raise_exit_downtime_log_entry(self):
        naglog_result('info', "CONTACT DOWNTIME ALERT: %s;STOPPED; Contact has "
                              "exited from a period of scheduled downtime"
                      % self.get_name())
    
    
    # Raise a log entry when a downtime prematurely ends
    # CONTACT DOWNTIME ALERT: test_contact;CANCELLED; Contact has entered a period of scheduled downtime
    def raise_cancel_downtime_log_entry(self):
        naglog_result('info', "CONTACT DOWNTIME ALERT: %s;CANCELLED; Scheduled "
                              "downtime for contact has been cancelled."
                      % self.get_name())


class Contacts(InheritableItems):
    name_property = "contact_name"
    inner_class = Contact
    
    
    def linkify(self, notificationways):
        self.linkify_with_notificationways(notificationways)
    
    
    # We've got a notificationways property with , separated contacts names
    # and we want have a list of NotificationWay
    def linkify_with_notificationways(self, notificationways):
        for i in self:
            if not hasattr(i, 'notificationways'):
                continue
            new_notificationways = deque()
            for nw_name in i.notificationways:  # already split & strip & unique
                nw = notificationways.find_by_name(nw_name)
                if nw is not None:
                    new_notificationways.append(nw)
                else:
                    err = "The 'notificationways' of the %s '%s' named '%s' is disabled or does not exist!" % (i.__class__.my_type, i.get_name(), nw_name)
                    i.configuration_errors.append(err)
            # Get the list, but first make elements uniq
            i.notificationways = tuple(new_notificationways)
    
    
    def late_linkify_c_by_commands(self, commands):
        for i in self:
            for nw in i.notificationways:
                nw.late_linkify_nw_by_commands(commands)
    
    
    # We look for contacts property in contacts and
    def explode(self, contactgroups, notificationways):
        # Contactgroups property need to be fullfill for got the informations
        self.apply_partial_inheritance('contactgroups')
        # _special properties maybe came from a template, so
        # import them before grok ourselves
        for prop in _special_properties:
            if prop == 'contact_name':
                continue
            self.apply_partial_inheritance(prop)
        
        # Register ourself into the contactsgroups we are in
        for c in self:
            if c.is_tpl() or not (hasattr(c, 'contact_name') and hasattr(c, 'contactgroups')):
                continue
            for cg in c.contactgroups.split(','):
                contactgroups.add_member(c.contact_name, cg.strip())
        
        # Now create a notification way with the simple parameter of the
        # contacts
        for c in self:
            if not c.is_tpl():
                need_notificationway = False
                params = {}
                for p in _simple_way_parameters:
                    if hasattr(c, p):
                        need_notificationway = True
                        params[p] = getattr(c, p)
                    else:  # put a default text value
                        # Remove the value and put a default value
                        setattr(c, p, '')
                
                if need_notificationway:
                    # print "Create notif way with", params
                    cname = getattr(c, 'contact_name', getattr(c, 'alias', ''))
                    nw_name = cname + '_inner_notificationway'
                    notificationways.new_inner_member(nw_name, params)
                    
                    if not hasattr(c, 'notificationways'):
                        c.notificationways = nw_name
                    else:
                        c.notificationways = c.notificationways + ',' + nw_name
    
    
    def create_reversed_list(self):
        InheritableItems.create_reversed_list(self)
        
        name_property = self.__class__.name_property
        self.reversed_list_name_lower = {}
        for id in self.items:
            if hasattr(self.items[id], name_property):
                name = getattr(self.items[id], name_property)
                self.reversed_list_name_lower[name.lower()] = id
    
    
    def reindex(self, contacts):
        # type: (List[Contact]) -> None
        
        for contact in contacts:
            name_property = self.__class__.name_property
            _id = contact.id
            name = getattr(contact, name_property)
            uuid = contact.uuid
            self.reversed_list[name] = _id
            self.reversed_list_id[uuid] = _id
            self.reversed_list_name_lower[name.lower()] = _id
    
    
    def find_id_by_name_case_insensitive(self, name):
        name_lower = name.lower()
        if hasattr(self, 'reversed_list_name_lower'):
            if name_lower in self.reversed_list_name_lower:
                return self.reversed_list_name_lower[name_lower]
            else:
                return None
        else:  # ok, an early ask, with no reversed list from now...
            name_property = self.__class__.name_property
            for i in self:
                if hasattr(i, name_property):
                    i_name = getattr(i, name_property)
                    if i_name.lower() == name_lower:
                        return i.id
            return None
    
    
    def get_contact_case_insensitive(self, name):
        id = self.find_id_by_name_case_insensitive(name)
        if id is not None:
            return self.items[id]
        else:
            return None
