#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2021:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.toolbox.url_helper import BaseUrl

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Iterator, Union, Optional


class GraphiteBackendException(Exception):
    pass


class GraphiteBackend(object):
    REALM_SEPARATOR = u':'
    BACKEND_SEPARATOR = u','
    
    
    def __init__(self, graphite_backend, strict=True):
        # type: (unicode, bool) -> None
        graphite_backend = graphite_backend.strip()
        self._errors = []  # type: List[unicode]
        if u':' not in graphite_backend:
            exp = GraphiteBackendException(u'Format error detected. It needs at least a realm and a hostname : <REALM>:<HOSTNAME>')
            if strict:
                raise exp
            self._errors.append(exp.message)
            
            self.realm = u''  # type: Optional[unicode]
            self.base_url = None  # type: Optional[BaseUrl]
            self.cfg_url = graphite_backend  # type: Optional[unicode]
            return
        realm, url = graphite_backend.split(self.REALM_SEPARATOR, 1)
        self.realm = realm.strip()
        self.base_url = BaseUrl.from_url(url, strict)
        self.cfg_url = url
        
        self._errors = list(self.base_url.get_error_messages())
    
    
    def __iter__(self):
        # type: () -> Iterator[Union[BaseUrl, unicode]]
        return iter((self.realm, self.base_url, self.cfg_url))
    
    
    @classmethod
    def parse_graphite_backends(cls, graphite_backends, strict=True):
        # type: (unicode, bool) -> List[GraphiteBackend]
        backends = graphite_backends.split(cls.BACKEND_SEPARATOR)
        parsed_backends = []
        for backend in backends:
            parsed_backends.append(cls(backend, strict))
        return parsed_backends
    
    
    def has_errors(self):
        # type: () -> bool
        return bool(self._errors)
    
    
    def get_errors(self):
        # type: () -> List[unicode]
        return self._errors
