import ctypes
import os
import time

try:
    libc = ctypes.CDLL('libc.so.6')
except Exception:  # noqa: yep, will do a exception on windows
    libc = None

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.log import LoggerFactory

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, List

_DO_DEBUG = os.environ.get('SHINKEN_DEBUG_CPU_STATS_THREADS', u'0') == u'1'
raw_logger = LoggerFactory.get_logger()
logger_debug = raw_logger.get_sub_part(u'CPU STATS')


class CpuSnapshotDiff(object):
    def __init__(self, user_t, sys_t, elapsed):
        # type: (float, float, float) -> None
        self._user = user_t
        self._sys = sys_t
        self._elapsed = elapsed
        self._wait_time = max(0.0, self._elapsed - (self._sys + self._user))  # beware of elasped, time can jump
    
    
    def __str__(self):
        return u'[ clock_time=%.3fs  cpu_user=%.3fs  cpu_sys=%.3fs => wait_time=%.3fs ]' % (self._elapsed, self._user, self._sys, self._wait_time)


class ThreadCpuSnapshot(object):
    def __init__(self, abs_user_t, abs_sys_t):
        # type: (Optional[float],Optional[float]) -> None
        self._t = time.time()
        self._abs_user_t = abs_user_t
        self._abs_sys_t = abs_sys_t
    
    
    def get_diff(self):
        other = cpu_stats_helper.get_thread_cpu_snapshot()
        other_user_t = other._abs_user_t
        other_sys_t = other._abs_sys_t
        other_t = other._t
        
        if self._abs_user_t is None or self._abs_sys_t is None or other_user_t is None or other_sys_t is None:
            return CpuSnapshotDiff(0.0, 0.0, other_t - self._t)
        
        return CpuSnapshotDiff(other_user_t - self._abs_user_t, other_sys_t - self._abs_sys_t, other_t - self._t)


class CpuStats(object):
    def __init__(self):
        pass
    
    
    @staticmethod
    def _get_thread_id():
        # type: () -> int
        tid = 0
        if libc:
            tid = libc.syscall(186)  # get the thread id when you are in it :)
        return tid
    
    
    def _get_my_thread_stat_content(self):
        # type: () -> Optional[str]
        my_thread_id = self._get_thread_id()
        if my_thread_id == 0:  # windows like
            if _DO_DEBUG:
                logger_debug.error(u'ERROR: my_thread_id == 0')
            return None
        
        # The /proc/PID/tak/TID/stat have all we need, but beware: it's updated only every 10ms
        pth = u'/proc/%d/task/%d/stat' % (os.getpid(), my_thread_id)
        if not os.path.exists(pth):  # hum... we have bah pid/tid?
            if _DO_DEBUG:
                logger_debug.error(u'ERROR: no path %s' % pth)
            return None
        
        try:  # it's our process, there should not ba any problem, but we never know after all
            with open(pth, 'r') as f:
                buf = f.read()
        except IOError as exp:
            if _DO_DEBUG:
                logger_debug.error(u'ERROR: IO error on %s: %s' % (pth, exp))
            return None
        
        # DEBUG: use this for resting of spaces in names
        # buf = buf.replace('(python)', '(I love to eat spaces)')
        return buf
    
    
    @staticmethod
    def _parse_my_thread_stat_content(buf):
        # type: (str) -> Optional[List[str]]
        elts = [s for s in buf.split(' ') if s.strip() != '']
        if len(elts) < 16:  # we want the 14/15 elements
            if _DO_DEBUG:
                logger_debug.error(u'ERROR: not enough elements: %s' % elts)
            return None
        
        # Note: maybe the name of the thread have a space or more, then we have a bad number of element, if so, get back
        #       to a normal way
        # name is the 2nd elet like (http_thread) so loop until we close it ^^
        name = elts[2 - 1]
        if u')' not in name:  # the name have space, find the ) to close it
            idx = 2 - 1
            try:
                while u')' not in elts[idx]:
                    if _DO_DEBUG:
                        logger_debug.info(u'INFO: space in thread name, removing %s' % elts[idx])
                    del elts[idx]
            except IndexError:  # what? no )
                if _DO_DEBUG:
                    logger_debug.error(u'ERROR: cannot find the end of name %s' % elts)
                return None
        
        if _DO_DEBUG:
            logger_debug.info(u'INFO: final elts %s' % elts)
        return elts
    
    
    @staticmethod
    def _get_user_sys_time_from_stat_str_elements(str_elements):
        # type: (List[str]) -> (float,float)
        idx_user_time = 14 - 1
        idx_sys_time = 15 - 1
        user_v = str_elements[idx_user_time]
        sys_v = str_elements[idx_sys_time]
        # We have times in centi seconds, so we need to int()/100
        try:
            user_time = int(user_v) / 100.0
            sys_time = int(sys_v) / 100.0
        except ValueError:  # not int? what?
            if _DO_DEBUG:
                logger_debug.error(u'ERROR: not int: %s / %ss' % (user_v, sys_v))
            return None, None
        return user_time, sys_time
    
    
    # Get the caller thread user_time, system_time
    # * ABSOLUTE VALUES! make your own diff
    # * with a 10ms precision, deal with it (thanks Linux HZ=100)
    def get_my_thread_cpu_stats(self):
        # type: () -> (float, float)
        
        buf = self._get_my_thread_stat_content()  # Windows like, skip this call, we don't know
        if buf is None:
            return 0.0, 0.0
        
        str_elements = self._parse_my_thread_stat_content(buf)
        if str_elements is None:
            return 0.0, 0.0
        
        user_time, sys_time = self._get_user_sys_time_from_stat_str_elements(str_elements)
        if user_time is None or sys_time is None:
            return 0.0, 0.0
        
        if _DO_DEBUG:
            logger_debug.info(u'%s MY THREAD CPU TIMEs: %s/%s' % (self._get_thread_id(), user_time, sys_time))
        return user_time, sys_time
    
    
    def get_thread_cpu_snapshot(self):
        user_time, sys_time = self.get_my_thread_cpu_stats()
        snapshot = ThreadCpuSnapshot(user_time, sys_time)
        return snapshot


cpu_stats_helper = CpuStats()
