import socket
from shinken.log import logger

try:
    import ssl
except ImportError:
    ssl = None

try:
    # Note: the cherrypy lib in ssl have a bug where socket_errors_to_ignore is not available
    # in the ssl part, so link it to the lib before load the ssl part
    from cherrypy.wsgiserver import wsgiserver2
    from cherrypy import wsgiserver as cheery_wsgiserver
    
    cheery_wsgiserver.socket_errors_to_ignore = wsgiserver2.socket_errors_to_ignore
    
    
    def sendall(self, data):
        """Sendall for non-blocking sockets."""
        total_sent = 0
        data_size = len(data)
        chunk_size = 4096
        # Avoid expensive big size data copy by limiting maximal chunk size
        max_chunk_size = 8 * 1024 * 1024
        while total_sent < data_size:
            try:
                bytes_sent = self.send(data[total_sent:total_sent + chunk_size])
                # If all data was sent, try to optimize chunk size
                if bytes_sent == chunk_size and chunk_size < max_chunk_size:
                    chunk_size = chunk_size * 2
                # logger.debug(u' WSGISERVER2 [%s/%s] sent:%s chunk_size:%s' % (total_sent, data_size, bytes_sent, chunk_size))
                total_sent = total_sent + bytes_sent
            except socket.error as e:
                if e.args[0] not in wsgiserver2.socket_errors_nonblocking:
                    raise
    
    
    wsgiserver2.CP_fileobject.sendall = sendall

except ImportError:
    wsgiserver2 = None
    cheery_wsgiserver = None

import shinken.webui.bottlewebui as bottle


# CherryPy is allowing us to have an HTTP 1.1 server, and so have a KeepAlive
class CherryPyServerHTTP(bottle.ServerAdapter):
    def run(self, handler):  # pragma: no cover
        daemon_thread_pool_size = self.options.get('numthreads', 64)
        server = cheery_wsgiserver.CherryPyWSGIServer((self.host, self.port), handler, numthreads=daemon_thread_pool_size, shutdown_timeout=1)
        logger.info('Initializing a CherryPy backend with %d threads' % daemon_thread_pool_size)
        use_ssl = self.options['use_ssl']
        ssl_cert = self.options['ssl_cert']
        ssl_key = self.options['ssl_key']
        
        if use_ssl:
            server.ssl_certificate = ssl_cert
            server.ssl_private_key = ssl_key
        
        return server
