from .http_client import HTTPExceptions
from .satellitelink import SatelliteLink
from .log import logger, get_chapter_string


class WithInventorySatelliteLink(SatelliteLink):
    properties = SatelliteLink.properties.copy()
    
    
    def __init__(self, *args, **kwargs):
        super(WithInventorySatelliteLink, self).__init__(*args, **kwargs)
        # We will have the list of realms that this receiver can be contacted with
        self._accessible_realms = []
    
    
    def __getstate__(self):
        _link_as_dict = super(WithInventorySatelliteLink, self).__getstate__()
        _link_as_dict['_accessible_realms'] = self._accessible_realms
        return _link_as_dict
    
    
    def __setstate__(self, state):
        super(WithInventorySatelliteLink, self).__setstate__(state)
        self._accessible_realms = state['_accessible_realms']
    
    
    # A realm is giving it's name to know that this receiver can be contacted by it
    def set_accessible_realm(self, realm_name):
        self._accessible_realms.append(realm_name)
    
    
    # The receiver need to have a new entry in the configuration: 'known_realms'
    # that was prepared by the realms just before
    def prepare_for_conf(self):
        super(WithInventorySatelliteLink, self).prepare_for_conf()
        self.cfg['known_realms'] = self._accessible_realms
    
    
    # For a specific realm, we want to know if the receiver have the inventory up-to-date
    # * ask it for its current inventory hash. If matching, also check that the shard index&number of shards are OK too
    # * if not matching, push the new one (inventory + index + shard_size)
    def assert_have_realm_inventory(self, realm_name, realm_inventory):
        try:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                return
            
            if not self.is_alive() or not self.reachable:
                return
            
            self.con.get('ping')
        except HTTPExceptions as exp:  # It's just a ping error, not a big dead, will be catch by the real .ping()
            logger.info('%s Failed to check elements inventory. Will retry as soon as connexion is back. Current error: %s' % (get_chapter_string(self.get_name()), str(exp)))
            # No other actions, ping will take care of bad pings and DEAD things
            return
        
        inventory_hash = realm_inventory.get_hash()
        args = {
            'realm_name'    : realm_name,
            'inventory_hash': inventory_hash
        }
        _logger = self.logger_configuration.get_sub_part(realm_name)
        url = 'already_have_realm_inventory'
        try:
            already_have = self.con.get(url, args=args)
            if already_have:
                _logger.debug('The %s does already have a valid inventory (%s elements)' % (self.my_type, realm_inventory.get_len()))
            else:
                _logger.info('Starting to send realm inventory to the %s ' % (self.my_type))
                payload = {
                    'realm_name'     : realm_name,
                    'realm_inventory': realm_inventory
                }
                url = 'push_realm_inventory'
                self.con.post(url, payload, wait='long')
                _logger.info('The %s is now up-to-date with inventory %s (%s elements)' % (self.my_type, inventory_hash, realm_inventory.get_len()))
                # We did push both inventory and shard_index, it's sure to be OK
        except HTTPExceptions as exp:
            _logger.warning('Cannot check/push elements inventory (url=/%s): %s. We will retry later' % (url, str(exp)))
            # No other actions, ping will take care of bad pings and DEAD things
