#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

import os
import shutil

from shinken.compat import bytes_to_unicode
from shinken.misc.os_utils import make_file_hidden, write_file_set_owner
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.locking.shinken_locking.shinken_interprocess_rlock import ShinkenInterProcessRLock
from shinkensolutions.os_helper import make_dirs_and_chown_shinken

if TYPE_CHECKING:
    from typing import BinaryIO

ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH = u'/var/lib/shinken/arbiter/arbiter_configuration_messages'
COMMON_FILE_NAME = u'shinken_common.json'
arbiter_configuration_messages_uuid = u''
arbiter_configuration_messages_part = u''

lock_arbiter_messages = None
arbiter_messages = []
first_write = True
MESSAGE_BUFFER_SIZE = 10000


def init_message_arbiter():
    # type: () -> None
    global lock_arbiter_messages
    lock_arbiter_messages = ShinkenInterProcessRLock(make_file_hidden(os.path.join(os.path.dirname(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH), u'arbiter_configuration_messages.lock')))
    clean_arbiter_messages()


def set_arbiter_configuration_messages_uuid(uuid):
    # type: (unicode) -> None
    global arbiter_configuration_messages_uuid
    arbiter_configuration_messages_uuid = uuid


def read_arbiter_configuration_messages(uuid):
    # type: (unicode) -> unicode
    message_folder = os.path.join(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, uuid)
    
    common_file_name = os.path.join(message_folder, u'shinken_final_messages.json')
    with open(common_file_name, u'r') as _file:
        return bytes_to_unicode(_file.read())


def add_arbiter_messages(message):
    # type: (unicode) -> None
    
    if not arbiter_configuration_messages_uuid:
        return
    
    with lock_arbiter_messages:
        arbiter_messages.append(message)
        
        if len(arbiter_messages) > MESSAGE_BUFFER_SIZE:
            flush_arbiter_message_buffer()


def flush_arbiter_message_buffer():
    # type: () -> None
    global arbiter_messages, first_write
    
    if not arbiter_configuration_messages_uuid:
        return
    
    message_folder = os.path.join(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, arbiter_configuration_messages_uuid)
    common_file_name = os.path.join(message_folder, arbiter_configuration_messages_part)
    
    with open(common_file_name, u'a') as _file:
        # If we do not have a message, we write an empty file.
        # We keep creating an empty file because when we make the final message, we will need all the files.
        if arbiter_messages:
            if not first_write:
                _file.write(',\n')
            first_write = False
            _file.write(',\n'.join(arbiter_messages))
    arbiter_messages = []


def start_write_arbiter_messages(part):
    # type: (unicode) -> None
    global arbiter_configuration_messages_part, arbiter_messages, first_write
    
    if not arbiter_configuration_messages_uuid:
        return
    
    with lock_arbiter_messages:
        first_write = True
        arbiter_configuration_messages_part = part
        arbiter_messages = []
        
        message_folder = os.path.join(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, arbiter_configuration_messages_uuid)
        common_file_name = os.path.join(message_folder, arbiter_configuration_messages_part)
        
        make_dirs_and_chown_shinken(message_folder)
        write_file_set_owner(common_file_name, u'', set_owner=True)


def end_write_arbiter_messages():
    # type: () -> None
    if not arbiter_configuration_messages_uuid:
        return
    with lock_arbiter_messages:
        flush_arbiter_message_buffer()


def clean_arbiter_messages():
    # type: () -> None
    try:
        shutil.rmtree(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH)
    except Exception:  # Noqa
        pass


def write_final_json():
    # type: () -> None
    if not arbiter_configuration_messages_uuid:
        return
    end_write_arbiter_messages()
    message_folder = os.path.join(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, arbiter_configuration_messages_uuid)
    common_file_name = os.path.join(message_folder, u'shinken_final_messages.json')
    write_file_set_owner(common_file_name, u'[', set_owner=True)
    sort_by_time()
    delete_files_except_final()


def sort_by_time():
    # type: () -> None
    
    if not arbiter_configuration_messages_uuid:
        return
    
    all_files = []
    message_folder = os.path.join(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, arbiter_configuration_messages_uuid)
    total_of_line = 0
    for file_name in os.listdir(message_folder):
        if file_name != u'shinken_final_messages.json':
            common_file_name = os.path.join(message_folder, file_name)
            common_file = open(common_file_name, u'r')
            number_of_line = count_line(common_file)
            if count_line(common_file) != 0:
                total_of_line += number_of_line
                all_files.append(common_file)
    all_current_lines = [common_file.readline() for common_file in all_files]
    lines_to_write = []
    for index_of_line in range(total_of_line):
        next_line_to_write = 0
        for line_number in range(1, len(all_current_lines)):
            if len(all_current_lines[line_number]) == 0:
                continue
            elif len(all_current_lines[next_line_to_write]) == 0:
                next_line_to_write = line_number
            elif float(all_current_lines[line_number][1:].split(u',', 1)[0]) < float(all_current_lines[next_line_to_write][1:].split(u',', 1)[0]):
                next_line_to_write = line_number
        lines_to_write.append(formate_line(u'[%s,%s' % (str(index_of_line + 1), all_current_lines[next_line_to_write][0:].split(u',', 1)[1]), index_of_line + 1 == total_of_line))
        if len(lines_to_write) >= 100:
            write_lines(lines_to_write)
            lines_to_write = []
        all_current_lines[next_line_to_write] = all_files[next_line_to_write].readline()
    lines_to_write.append(u']')
    write_lines(lines_to_write)
    close_files(all_files)


def formate_line(line, is_last_line):
    # type: (unicode, bool) -> unicode
    if len(line) > 0 and line[-1] == u']':
        if not is_last_line:
            line += u',\n'
    return line


def write_lines(lines_to_write):
    # type: (list) -> None
    
    if not arbiter_configuration_messages_uuid:
        return
    message_folder = os.path.join(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, arbiter_configuration_messages_uuid)
    common_file_name = os.path.join(message_folder, u'shinken_final_messages.json')
    
    with open(common_file_name, u'a') as _file:
        _file.writelines(lines_to_write)


def count_line(common_file):
    # type: (BinaryIO) -> int
    numbers_of_line = 0
    for _ in enumerate(common_file):
        numbers_of_line += 1
    common_file.seek(0, 0)
    return numbers_of_line


def close_files(files):
    # type : (arr[]) -> None
    for common_file in files:
        common_file.close()


def delete_files_except_final():
    # type: () -> None
    message_folder = os.path.join(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, arbiter_configuration_messages_uuid)
    for file_name in os.listdir(message_folder):
        if file_name != u'shinken_final_messages.json':
            common_file_name = os.path.join(message_folder, file_name)
            os.remove(common_file_name)
