#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.
import datetime
import json
import os
import re
import threading
import time
from collections import OrderedDict, namedtuple

from shinken.exceptions.business import ShinkenExceptionValueError, ShinkenExceptionKeyError
from shinken.log import logger
from shinken.misc.filter import get_all_my_visible_items, check_user_item_access
from shinken.misc.perfdata import PerfDatas
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.broker.monitoring_item_manager.helper import get_ack_on_item, get_downtime_on_item

try:
    from cStringIO import StringIO
except ImportError:
    from io import StringIO

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Optional, List, Collection, Set, Union
    from shinken.objects.contact import Contact
    from shinken.objects.host import Host
    from shinken.objects.service import Service
    from shinken.webui.bottlewebui import MultiDict
    from shinken.misc.monitoring_item_manager.monitoring_item_manager import MonitoringItemManager

# ## Will be populated by the UI with it's own value
FRONTEND_DATE_FORMAT = '%Y-%m-%d %H:%M:%S'
DEFAULT_SORT = {'tag': 'host_name', 'value': 'asc'}

DateFilter = namedtuple('DateFilter', ['operator', 'delay'])


class DateFilterOperator(object):
    NEWER = 'newer'
    OLDER = 'older'


class StaticCache(object):
    lock = threading.RLock()
    part_configuration_id = -1
    hosts_by_name = {}  # type: Dict[unicode,Host]
    hosts_by_uuid = {}  # type: Dict[unicode,Host]
    checks_by_uuid = {}  # type: Dict[unicode,Service]
    checks_by_name = {}  # # type: Dict[unicode,List[Service]]
    
    
    @staticmethod
    def _init_static_cache(item):
        item_type = 'host'
        extended_item_type = item_type
        if item.__class__.my_type == 'service':
            item_type = 'check'
            if item.host.got_business_rule:
                extended_item_type = 'check_cluster'
            else:
                extended_item_type = 'check_host'
        elif item.got_business_rule:
            item_type = 'cluster'
            extended_item_type = item_type
        
        list_cache_static = {
            'display_name'        : item.display_name,
            'contact_groups'      : [cg.strip() for cg in getattr(item, 'contact_groups', '') if cg.strip()],
            'contacts'            : [c.get_name() for c in item.contacts],
            'max_attempts'        : item.max_check_attempts,
            'notification_period' : item.notification_period.get_name() if item.notification_period else '',
            'notification_options': item.notification_options,
            'type'                : item_type,
            'extended_type'       : extended_item_type,
            'is_unknown'          : False,
        }
        
        if item_type == 'check':
            list_cache_static['host_name'] = item.host.host_name
            list_cache_static['service_description'] = item.service_description
            list_cache_static['name'] = item.service_description
            list_cache_static['realm'] = item.host.realm
            list_cache_static['uuid'] = '%s-%s' % (item.host.uuid, item.uuid)
            list_cache_static['check_uuid'] = '%s-%s' % (item.host.uuid, item.uuid)
            list_cache_static['uuid_parent'] = item.host.uuid
            list_cache_static['host_groups'] = [i.get_name() for i in item.host.hostgroups]
        else:
            if item_type == 'host':
                list_cache_static['address'] = item.address
            
            list_cache_static['host_name'] = item.host_name
            list_cache_static['name'] = item.host_name
            list_cache_static['realm'] = item.realm
            list_cache_static['host_templates'] = [i for i in item.tags if i != 'shinken-host']
            list_cache_static['host_groups'] = [i.get_name() for i in item.hostgroups]
            list_cache_static['uuid'] = item.uuid
            list_cache_static['uuid_parent'] = item.uuid
        item.list_cache_static = list_cache_static
        
        filter_cache_static = {
            'display_name'        : list_cache_static['display_name'].lower(),
            'contact_groups'      : set([i.lower() for i in list_cache_static['contact_groups']]),
            'contacts'            : set([i.lower() for i in list_cache_static['contacts']]),
            'notification_period' : list_cache_static['notification_period'].lower(),
            'notification_options': set(list_cache_static['notification_options']),
            'type'                : set((item_type,)),
            'extended_type'       : set((extended_item_type,)),
            'name'                : set((list_cache_static['name'].lower(),)),
            'uuid'                : set((list_cache_static['uuid'].lower(),)),
            'uuid_parent'         : set((list_cache_static['uuid_parent'].lower(),)),
            'host_name'           : set((list_cache_static['host_name'].lower(),)),
            'host_name_like'      : list_cache_static['host_name'].lower(),
            'realm'               : set((list_cache_static['realm'].lower(),)),
            'host_groups'         : set([i.lower() for i in list_cache_static['host_groups']])
        }
        if list_cache_static['type'] == 'host':
            filter_cache_static['service_description_like'] = ''
            filter_cache_static['service_description'] = set(('',))
            filter_cache_static['check_uuid'] = set(('',))
            filter_cache_static['address'] = item.address.lower()
            filter_cache_static['host_templates'] = set([i.lower() for i in list_cache_static['host_templates']])
        if list_cache_static['type'] == 'cluster':
            filter_cache_static['service_description_like'] = ''
            filter_cache_static['service_description'] = set(('',))
            filter_cache_static['check_uuid'] = set(('',))
            filter_cache_static['address'] = ''
            filter_cache_static['host_templates'] = set([i.lower() for i in list_cache_static['host_templates']])
        if list_cache_static['type'] == 'check':
            filter_cache_static['service_description'] = set((item.service_description.lower(),))
            filter_cache_static['service_description_like'] = item.service_description.lower()
            filter_cache_static['check_uuid'] = set((list_cache_static['uuid'].lower(),))
            filter_cache_static['address'] = item.host.address.lower()
            filter_cache_static['host_templates'] = set([i.lower() for i in item.host.tags if i != 'shinken-host'])
        
        item.filter_cache_static = filter_cache_static
        
        sort_cache_static = {}
        for _tag, _value in filter_cache_static.iteritems():
            sort_cache_static[_tag] = _build_static_sort_key(_value)
        
        item.sort_cache_static = sort_cache_static
    
    
    @classmethod
    def refresh(cls, items, part_configuration_id):
        with cls.lock:
            if cls.part_configuration_id != part_configuration_id:
                cls.hosts_by_name = {}
                cls.hosts_by_uuid = {}
                cls.checks_by_name = {}
                cls.checks_by_uuid = {}
                for i in items:
                    cls._init_static_cache(i)
                    
                    name = i.list_cache_static[u'name'].lower()
                    uuid = i.list_cache_static[u'uuid'].lower()
                    if i.list_cache_static[u'type'] == u'check':
                        cls.checks_by_uuid[uuid] = i
                        if name in cls.checks_by_name:
                            cls.checks_by_name[name].append(i)
                        else:
                            cls.checks_by_name[name] = [i]
                    else:
                        cls.hosts_by_name[name] = i
                        cls.hosts_by_uuid[uuid] = i
                
                cls.part_configuration_id = part_configuration_id
    
    
    @classmethod
    def get_host_by_uuid(cls, uuid, user=None):
        # type: (unicode, Optional[Contact]) -> Optional[Host]
        if user and not check_user_item_access(user, uuid):
            return None
        if uuid in cls.hosts_by_uuid:
            return cls.hosts_by_uuid[uuid]
        return None
    
    
    @classmethod
    def get_check_by_uuid(cls, uuid, user=None):
        # type: (unicode, Optional[Contact]) -> Optional[Service]
        if user and not check_user_item_access(user, uuid):
            return None
        if uuid in cls.checks_by_uuid:
            return cls.checks_by_uuid[uuid]
        return None
    
    
    @classmethod
    def get_host_by_name(cls, name, user=None):
        # type: (unicode, Optional[Contact]) -> Optional[Host]
        if name in cls.hosts_by_name:
            host = cls.hosts_by_name[name]
            if not user or check_user_item_access(user, host.get_uuid()):
                return host
        return None
    
    
    @classmethod
    def get_check_by_name(cls, host, name, user=None):
        # type: (Host, unicode, Optional[Contact]) -> Optional[Service]
        if not host or not name:
            return None
        host_uuid = host.get_uuid()
        if user and not check_user_item_access(user, host_uuid):
            return None
        if name in cls.checks_by_name:
            for s in cls.checks_by_name[name]:
                if s.host.uuid == host_uuid:
                    return s
        return None


# TODO put in helper after patch
def _get_state(item):
    if item.__class__.my_type == 'host' and item.got_business_rule:
        return item.bp_state
    elif item.state == 'OK' or item.state == 'UP':
        return 0
    elif item.state == 'WARNING':
        return 1
    elif item.state == 'CRITICAL' or item.state == 'DOWN':
        return 2
    else:
        return 3


# TODO put in helper after patch
def _get_context(elt):
    # type: (Union[Host,Service]) -> (unicode, unicode)
    class_name = elt.__class__.my_type
    is_host = (class_name == 'host')
    is_service = (class_name == 'service')
    context = 'NOTHING', 'NOTHING'
    
    # DISABLED is for hosts, service can't be like this because an host disabled got no service
    if is_host and 'disabled' in elt.tags:
        context = 'DISABLED', 'DISABLED'
    elif elt.downtimes and next((dt for dt in elt.downtimes if dt.ref == elt and dt.is_in_effect), None):
        context = 'DOWNTIME', 'DOWNTIME'
    elif elt.is_in_inherited_downtime():
        context = 'INHERITED-DOWNTIME', 'DOWNTIME'
    elif elt.in_partial_downtime:
        context = 'PARTIAL-DOWNTIME', 'DOWNTIME'
    elif elt.acknowledgement and not elt.acknowledgement.automatic:
        context = 'ACKNOWLEDGED', 'ACKNOWLEDGED'
    elif (elt.acknowledgement and elt.acknowledgement.automatic) or (is_service and elt.host.acknowledgement and not elt.host.acknowledgement.automatic):
        context = 'INHERITED-ACKNOWLEDGED', 'ACKNOWLEDGED'
    elif elt.is_partial_acknowledged:
        context = 'PARTIAL-ACKNOWLEDGED', 'ACKNOWLEDGED'
    elif elt.is_flapping:
        context = 'FLAPPING', 'FLAPPING'
    elif elt.got_business_rule and elt.is_partial_flapping:
        context = 'PARTIAL-FLAPPING', 'FLAPPING'
    return context


def _build_static_sort_key(_value):
    return [int(s) if s.isdigit() else s.lower() for s in re.split(r'(\d+)', _value.encode('utf-8', 'ignore') if hasattr(_value, 'encode') else str(_value))]


def _build_var_sort_key(item, tag):
    _value = ''
    if tag == 'problem_has_been_inherited_acknowledged':
        _value = 0
        if item.__class__.my_type == 'service':
            if item.host.acknowledgement:
                _value = 1
        elif item.acknowledgement and item.acknowledgement.automatic:
            _value = 2
    elif tag == 'next_check':
        if item.got_business_rule:
            _value = -1
        else:
            _value = item.next_chk
    elif tag == 'last_check':
        if item.got_business_rule:
            _value = -1
        else:
            _value = item.last_chk
    elif tag == 'business_impact':
        _value = item.business_impact
    elif tag == 'status':
        _value = str(_get_state(item))
    elif tag == 'context':
        _value = _get_context(item)[1]
    elif tag == 'status_since':
        _value = int(item.last_state_change)
    elif tag == 'last_hard_state_change':
        _value = int(item.last_hard_state_change if item.state_type == 'HARD' else 0)
    elif tag == 'is_status_confirmed':
        _value = (item.state_type == 'HARD')
    elif tag == 'attempts':
        _value = item.attempt
    elif tag == 'is_root_problem':
        _value = item.is_problem
    elif tag == 'output':
        _value = item.output
    elif tag == 'long_output':
        _value = item.long_output
    elif tag == 'perf_data':
        _value = item.perf_data
    elif tag == 'got_business_rule':
        _value = item.got_business_rule
    elif tag == 'problem_has_been_acknowledged':
        _value = bool(item.acknowledgement and not item.acknowledgement.automatic)
    elif tag == 'in_flapping':
        _value = item.is_flapping
    elif tag == 'inherited_flapping':
        _value = item.inherited_flapping
    elif tag == 'in_partial_flapping':
        _value = item.is_partial_flapping
    
    return [int(s) if s.isdigit() else s.lower() for s in re.split(r'(\d+)', _value.encode('utf-8', 'ignore') if hasattr(_value, 'encode') else str(_value))]


def _sort_list_api_part_all(sorts, items):
    if not sorts:
        sorts = [DEFAULT_SORT]
    for sort in sorts:
        tag = sort['tag']
        reverse = (sort['value'] == 'desc')
        items.sort(key=lambda item: item.sort_cache_static[tag] if tag in item.sort_cache_static else _build_var_sort_key(item, tag), reverse=reverse)


def _list_all_values(items):
    all_values = {
        'host_templates': set(),
        'realm'         : set(),
        'host_groups'   : set(),
        'contact_groups': set(),
        'contacts'      : set(),
    }
    update_realm = all_values['realm'].add
    update_host_template = all_values['host_templates'].update
    update_host_group = all_values['host_groups'].update
    update_contact_group = all_values['contact_groups'].update
    update_contacts = all_values['contacts'].update
    
    for item in items:
        list_cache_static = item.list_cache_static
        update_contact_group(list_cache_static['contact_groups'])
        update_contacts(list_cache_static['contacts'])
        if not list_cache_static['type'] == 'check':
            update_host_template(list_cache_static['host_templates'])
            update_host_group(list_cache_static['host_groups'])
            update_realm(list_cache_static['realm'])
    
    for (_type, values) in all_values.iteritems():
        all_values[_type] = list(values)
    return all_values


class _FilterPagination(object):
    def __init__(self):
        self.pages = []
        self.start_page = 0
        self.counter = 0
        self.current_page = 0
        self.last = None
    
    
    def __str__(self):
        return u'_FilterPagination(pages=[%s], start_page=[%s], counter=[%s], current_page=[%s], last=[%s])' % (self.pages, self.start_page, self.counter, self.current_page, self.last)
    
    
    def __repr__(self):
        return self.__str__()


class _CheckSet(object):
    def __init__(self):
        self.result_set = {}
        self.nb = 0
    
    
    def add(self, host_name, elements):
        # type: (unicode, list) -> None
        self.result_set.update({host_name: elements})
        self.nb += len(elements)


class FilterItem(object):
    FILTER_TYPE = {
        u'display_name'            : u'string',
        u'business_impact'         : u'int',
        u'contact_groups'          : u'array',
        u'contacts'                : u'array',
        u'status'                  : u'int',
        u'context'                 : u'array',
        u'status_since'            : u'date',
        u'last_check'              : u'date',
        u'last_hard_state_change'  : u'date',
        u'is_status_confirmed'     : u'bool',
        u'attempts'                : u'string',
        u'next_check'              : u'date',
        u'is_root_problem'         : u'bool',
        u'output'                  : u'string',
        u'long_output'             : u'string',
        u'perf_data'               : u'string',
        u'notification_period'     : u'string',
        u'notification_options'    : u'array',
        u'type'                    : u'array',
        u'extended_type'           : u'array',
        u'host_name'               : u'array',
        u'host_name_like'          : u'string_array',
        u'service_description'     : u'array',
        u'service_description_like': u'string_array',
        u'realm'                   : u'array',
        u'address'                 : u'string',
        u'host_templates'          : u'array',
        u'host_groups'             : u'array',
        u'uuid_parent'             : u'array',
        u'check_uuid'              : u'array',
        u'uuid'                    : u'array',
        u'name'                    : u'array'
    }
    
    
    def __init__(self, monitoring_item_manager, user):
        # type: (MonitoringItemManager, Optional[Contact]) -> None
        self.page = None
        self.page_size = None
        self.sorts = None
        self.full_sort_for_log = None
        self.result_set_id = None
        self.filters = None
        self.full_filter_for_log = None
        self.flattened_view = True
        self.checks_in_tree = False
        self.monitoring_item_manager = monitoring_item_manager
        self.user = user
        self.ALLOWED_OUTPUT = None
        self.compute_not_found = True
        
        if not self.monitoring_item_manager:
            raise (NotImplementedError(u'monitoring_item_manager not available'))
    
    
    def _translate_filter_value(self, filter_tag, filter_value, only_allowed_values=None):
        filter_value = [v.lower() for v in filter_value]
        
        if filter_tag == u'extended_type' and u'check' in filter_value:
            filter_value = [val for val in filter_value if val != u'check']
            if u'check_host' not in filter_value and (not only_allowed_values or filter_tag not in only_allowed_values or u'check_host' in only_allowed_values[filter_tag]):
                filter_value.append(u'check_host')
            if u'check_cluster' not in filter_value and (not only_allowed_values or filter_tag not in only_allowed_values or u'check_cluster' in only_allowed_values[filter_tag]):
                filter_value.append(u'check_cluster')
        
        valid_values = []
        if only_allowed_values and filter_tag in only_allowed_values:
            valid_values = only_allowed_values[filter_tag]
        elif filter_tag == u'status':
            valid_values = [u'0', u'1', u'2', u'3']
        elif filter_tag == u'context':
            valid_values = [u'nothing', u'acknowledged', u'partial-acknowledged', u'inherited-acknowledged', u'downtime', u'partial-downtime', u'inherited-downtime', u'flapping', u'partial-flapping', u'disabled']
        elif filter_tag == u'is_status_confirmed':
            valid_values = [u'true', u'false']
        elif filter_tag == u'extended_type':
            valid_values = [u'host', u'cluster', u'check', u'check_host', u'check_cluster']
        elif filter_tag == u'type':
            valid_values = [u'host', u'cluster', u'check']
        elif filter_tag == u'business_impact':
            valid_values = [u'0', u'1', u'2', u'3', u'4', u'5']
        
        if valid_values:
            all_values = filter_value
            if not isinstance(all_values, list):
                all_values = [all_values]
            wrong_values = [val for val in all_values if val not in valid_values]
            if wrong_values:
                raise ShinkenExceptionValueError(text=u'wrong value%s %s' % (u's' if len(wrong_values) > 1 else u'', wrong_values))
        
        compare_type = self.FILTER_TYPE[filter_tag]
        if compare_type == u'int':
            filter_value = set([int(v) for v in filter_value])
        elif compare_type == u'bool':
            for v in filter_value:
                if v not in [u'true', u'false']:
                    raise ShinkenExceptionValueError(text=u'value [ %s ] is not a boolean' % v)
            filter_value = set([(u'true' == v) for v in filter_value])
        elif compare_type == u'array':
            filter_value = set(filter_value)
        elif compare_type == u'date':
            val = filter_value[0]
            if val.startswith(u'latest|') or val.startswith(u'old|') or val.startswith(u'in_more_than|') or val.startswith(u'in_less_than|') or val.startswith(u'in-more-than|') or val.startswith(u'in-less-than|') or val.startswith(
                    u'later-than|') or val.startswith(u'older-than|'):
                operator, delay = val.split(u'|')
                try:
                    delay = int(delay)
                except ValueError as e:
                    raise ShinkenExceptionValueError(text=u'[ %s ] => %s' % (operator, str(e)))
                operator = DateFilterOperator.NEWER if operator == u'latest' or operator == u'in_more_than' or operator == u'in-less-than' or operator == u'later-than' else DateFilterOperator.OLDER
                filter_value = [DateFilter(operator, delay)]
            else:
                raise ShinkenExceptionKeyError(text=u'unknown date constraint [ %s ]' % val)
        return filter_value
    
    
    def _parse_filter(self, get_filters, allowed_filter=None, only_allowed_values=None, default_if_empty=None):
        filters = []
        cpt = - 1
        for get_filter in get_filters:
            _filter = {}
            cpt = cpt + 1
            for search_type in get_filter.split(u'~'):
                split = search_type.split(u':', 1)
                filter_tag = split[0].lower()
                filter_tag_name = filter_tag
                if allowed_filter:
                    if filter_tag not in allowed_filter:
                        error_text = u'filtering[%s]: invalid field name [ %s ]' % (cpt, filter_tag_name)
                        if hasattr(self, u'ALLOWED_FILTER'):
                            error_text = u'filtering[%s]: field name [ %s ] not allowed in this route ' % (cpt, filter_tag_name) if filter_tag in getattr(self, u'ALLOWED_FILTER') else error_text
                        raise ShinkenExceptionKeyError(text=error_text)
                    filter_tag = allowed_filter[filter_tag]
                if len(split) > 1:
                    filter_value = split[1].lower().split(u'^^')
                else:
                    filter_value = None
                if not filter_value and default_if_empty and filter_tag in default_if_empty:
                    filter_value = default_if_empty[filter_tag]
                if not filter_value or u'' in filter_value:
                    raise ShinkenExceptionValueError(text=u'filtering[%s]: missing value for field [ %s ]' % (cpt, filter_tag_name))
                
                try:
                    filter_value = self._translate_filter_value(filter_tag, filter_value, only_allowed_values)
                except (ValueError, ShinkenExceptionValueError) as e:
                    raise ShinkenExceptionValueError(text=u'filtering[%s]: field [ %s ] => %s' % (cpt, filter_tag_name, str(e)))
                except ShinkenExceptionKeyError as e:
                    raise ShinkenExceptionKeyError(text=u'filtering[%s]: field [ %s ] %s' % (cpt, filter_tag_name, str(e)))
                
                if filter_tag in _filter and _filter[filter_tag] != filter_value:
                    raise ShinkenExceptionKeyError(text=u'filtering[%s]: value for field [ %s ] is already set, can not redefine to [ %s ]' % (cpt, filter_tag_name, split[1].lower()))
                _filter[filter_tag] = filter_value
            if default_if_empty:
                # extra_default_filter = {k: self._translate_filter_value(k, default_if_empty[k]) for k in default_if_empty if k not in _filter}
                extra_default_filter = {}
                for k in default_if_empty:
                    if k not in _filter:
                        extra_default_filter.update({k: self._translate_filter_value(k, default_if_empty[k])})
                if extra_default_filter:
                    _filter.update(extra_default_filter)
            filters.append(_filter)
        return filters
    
    
    def _filter_root_problems(self, filters, items):
        items = self._filter_elements_with_search(items, [{'is_root_problem': set(['true'])}])
        # remove the occurrence of is_root_problem in filter
        for _filter in filters[:]:
            _filter.pop('is_root_problem', None)
        return items
    
    
    def _filter_elements_with_search(self, items, _filters, counters=None):
        new_elements = []
        # counter on found elements, with this field matching :
        # 0 hosts found,
        # 1 clusters found
        # 2 checks found
        if not isinstance(counters, list) or len(counters) != 3:
            counters = [0, 0, 0]
        
        for item in items:
            add_element = False
            
            if item.__class__.my_type == u'host':
                if item.got_business_rule:
                    type_found = 1
                else:
                    type_found = 0
            else:
                type_found = 2
            
            for _filter in _filters:
                if self._apply_filter(_filter, item):
                    add_element = True
                    break
            
            if add_element:
                counters[type_found] = counters[type_found] + 1
                new_elements.append(item)
        
        return new_elements
    
    
    # Yes this function is ugly but do not refactor without check performance first
    @staticmethod
    def _get_value_for_filter(item, filter_name):
        if filter_name in item.filter_cache_static:
            return item.filter_cache_static[filter_name]
        
        if filter_name == 'business_impact':
            return item.business_impact
        if filter_name == 'status':
            return _get_state(item)
        if filter_name == 'context':
            return set((_get_context(item)[1].lower(),))
        if filter_name == 'status_since':
            return int(item.last_state_change)
        if filter_name == 'last_hard_state_change':
            return int(item.last_hard_state_change if item.state_type == 'HARD' else 0)
        if filter_name == 'last_check':
            return int(item.last_chk)
        if filter_name == 'next_check':
            return item.next_chk
        if filter_name == 'is_status_confirmed':
            return item.state_type == 'HARD'
        if filter_name == 'attempts':
            return '%s/%s' % (item.attempt, item.max_check_attempts)
        if filter_name == 'is_root_problem':
            return item.is_problem
        if filter_name == 'output':
            return item.output.lower()
        if filter_name == 'long_output':
            return item.long_output.lower()
        if filter_name == 'perf_data':
            return item.perf_data.lower()
    
    
    def _apply_filter(self, _filter, item):
        match_filter = True
        for (filter_tag, filter_values) in _filter.iteritems():
            value = self._get_value_for_filter(item, filter_tag)
            compare_type = self.FILTER_TYPE[filter_tag]
            # 0 is a value
            if value != 0 and not value:
                match_filter = False
                break
            
            _match_tag = False
            if compare_type == 'bool' or compare_type == 'int':
                _match_tag = value in filter_values
            elif compare_type == 'array':
                _match_tag = bool(filter_values & value)
            elif compare_type == 'string':
                _match_tag = filter_values[0] in value
            elif compare_type == 'string_array':
                _match_tag = True in [v in value for v in filter_values]
            elif compare_type == 'date':
                filter_value = filter_values[0]
                if isinstance(filter_value, DateFilter):
                    if filter_value.operator == DateFilterOperator.NEWER:
                        _match_tag = filter_value.delay >= abs(time.time() - value)
                    else:
                        _match_tag = filter_value.delay <= abs(time.time() - value)
                else:
                    value = datetime.datetime.fromtimestamp(value).strftime(FRONTEND_DATE_FORMAT)
                    _match_tag = filter_value in value
            else:
                logger.warning('unknown filter type : %s' % compare_type)
                _match_tag = False
            
            if not _match_tag:
                match_filter = False
                break
        return match_filter
    
    
    @staticmethod
    def _api_list_element_dict(item):
        # type: (Union[Host,Service]) -> Dict
        list_cache_var = getattr(item, 'list_cache_var', {})
        if not list_cache_var:
            list_cache_var['business_impact'] = item.business_impact
            list_cache_var['status'] = _get_state(item)
            list_cache_var['context'], _ = _get_context(item)
            list_cache_var['status_since'] = int(item.last_state_change)
            list_cache_var['last_hard_state_change'] = int(item.last_hard_state_change if item.state_type == 'HARD' else 0)
            list_cache_var['is_status_confirmed'] = (item.state_type == 'HARD')
            list_cache_var['attempts'] = item.attempt
            list_cache_var['max_attempts'] = item.max_check_attempts
            list_cache_var['is_root_problem'] = item.is_problem
            list_cache_var['output'] = item.output
            list_cache_var['long_output'] = item.long_output
            list_cache_var['perf_data'] = item.perf_data
            list_cache_var['smart_perf_data'] = PerfDatas(item.perf_data).as_list()
            list_cache_var['got_business_rule'] = item.got_business_rule
            list_cache_var['in_flapping'] = item.is_flapping
            list_cache_var['inherited_flapping'] = item.inherited_flapping
            list_cache_var['in_partial_flapping'] = item.is_partial_flapping
            list_cache_var['notes_multi_url'] = item.notes_multi_url
            list_cache_var['active_checks_enabled'] = item.active_checks_enabled
            
            get_ack_on_item(item, list_cache_var)
            get_downtime_on_item(item, list_cache_var)
            
            if not item.got_business_rule:
                list_cache_var['next_check'] = item.next_chk
                list_cache_var['last_check'] = item.last_chk
            
            list_cache_var.update(item.list_cache_static)
            item.list_cache_var = list_cache_var
        
        return list_cache_var
    
    
    def set_range(self, range_param):
        # type: (unicode) -> None
        start_time = time.time()
        self.page = None
        self.page_size = None
        # range_param = self.app.request.GET.get('range', '').strip()
        if not isinstance(range_param, unicode):
            range_param = range_param.decode(u'UTF-8')
        range_param = range_param.strip()
        if range_param:
            range_split = range_param.split(u'~')
            if len(range_split) != 2:
                raise ShinkenExceptionValueError(text=u'Wrong format for pagination parameters, expected:[name:integer~name:integer], got:[ %s ]' % range_param)
            
            param_start = range_split[0].split(u':')
            param_nb = range_split[1].split(u':')
            
            if len(param_start) != 2:
                raise ShinkenExceptionKeyError(text=u'Missing page number parameter')
            if len(param_nb) != 2:
                raise ShinkenExceptionKeyError(text=u'Missing page size parameter')
            
            try:
                self.page = int(param_start[1])
                if self.page < 0:
                    raise ShinkenExceptionValueError()
            except:
                raise ShinkenExceptionValueError(text=u'Wrong value:[ %s ] for page number parameter' % param_start[1])
            
            try:
                self.page_size = int(param_nb[1])
                if self.page_size <= 0:
                    raise ShinkenExceptionValueError()
            except:
                raise ShinkenExceptionValueError(text=u'Wrong value:[ %s ] for page size parameter' % param_nb[1])
        logger.log_perf(start_time, u'get_list_api_part_all', u'page %s page_size %s' % (self.page, self.page_size))
    
    
    def set_sort(self, sort_param, allowed_sort=None):
        # type: (unicode, Optional[dict]) -> None
        start_time = time.time()
        self.sorts = []
        # get_sorts = self.app.request.GET.get('sort', '').strip()
        if sort_param is None:
            return
        get_sorts = sort_param.strip()
        if not isinstance(get_sorts, unicode):
            get_sorts = get_sorts.decode(u'UTF-8')
        if get_sorts:
            for get_sort in get_sorts.split('~'):
                split = get_sort.split(':', 1)
                sort_tag = split[0].lower()
                if len(split) >= 2:
                    sort_value = split[1].lower()
                else:
                    sort_value = u'asc'
                if allowed_sort and sort_tag not in allowed_sort:
                    raise ShinkenExceptionKeyError(text=u'sort: invalid field name [ %s ]' % sort_tag)
                if sort_value not in [u'asc', u'desc']:
                    raise ShinkenExceptionValueError(text=u'sort: invalid sort direction [ %s ] for field [ %s ]' % (sort_value, sort_tag))
                self.sorts.append({u'tag': allowed_sort[sort_tag], u'value': sort_value})
        self.full_sort_for_log = get_sorts[:]
        logger.log_perf(start_time, u'get_list_api_part_all', u'sorts %s' % self.sorts)
    
    
    def set_result_set_id(self, uuid):
        # type: (unicode) -> None
        # self.result_set_id = self.app.request.GET.get('configuration_id', '-1').strip()
        self.result_set_id = uuid
    
    
    def set_flattened_mode(self, flattened_mode, checks_in_tree=False):
        # type: (bool, Optional[bool]) -> None
        # self.flattened_view = self.app.request.GET.get('flatten_list', '0') == '1'
        self.flattened_view = bool(flattened_mode)
        self.checks_in_tree = bool(checks_in_tree)
    
    
    def get_flattened_mode(self):
        return self.flattened_view, self.checks_in_tree
    
    
    def set_filters(self, filters_json_str, allowed_filters=None, only_allowed_values=None, default_if_empty=None):
        # type: (unicode, Optional[dict], Optional[dict], Optional[dict]) -> None
        # allowed_filters = { "public_field_name": "internal_field_name", ... }
        # only_allowed_values = { "internal_field_nameY1": ["valueY1", "valueY2", ...], ...}
        # default_if_empty = { "internal_field_nameX1": ["valueX1", "valueX2", ...], ...}
        #
        # allowed_filters translates publicly exposed field names to internal field name, and limit the set of usable filters
        # only_allowed_values allows to check for user provided filtering values and silently ignore the ones not in its set
        # default_if_empty defines default values for filtering tags not provided by user
        start_time = time.time()
        self.filters = []
        self.full_filter_for_log = u''
        
        get_filters = []
        if filters_json_str:
            cnt = 0
            filters_json = filters_json_str.strip()
            filters_dict = json.loads(filters_json)
            get_filter = filters_dict.get(u'filter%s' % cnt, u'')
            while get_filter:
                get_filters.append(get_filter.strip())
                cnt += 1
                get_filter = filters_dict.get(u'filter%s' % cnt, u'')
            
            self.full_filter_for_log = get_filters[:]
        
        if get_filters:
            self.filters = self._parse_filter(get_filters, allowed_filters, only_allowed_values, default_if_empty)
        elif default_if_empty:
            # self.filters = [{k: self._translate_filter_value(k, default_if_empty[k]) for k in default_if_empty}]
            self.filters = {}
            for k in default_if_empty:
                self.filters.update({k: self._translate_filter_value(k, default_if_empty[k])})
            self.filters = [self.filters]
        logger.log_perf(start_time, u'get_list_api_part_all', u'filters %s' % self.filters)
    
    
    def set_output_fields(self, fields):
        # type: (dict) -> None
        self.ALLOWED_OUTPUT = fields
    
    
    def get_output_fields(self):
        # type: () -> Optional[dict]
        return self.ALLOWED_OUTPUT
    
    
    def add_output_fields(self, fields):
        # type: (dict) -> None
        if self.ALLOWED_OUTPUT:
            self.ALLOWED_OUTPUT.update(fields)
        else:
            self.ALLOWED_OUTPUT = fields
    
    
    def set_compute_not_found(self, mode):
        # type: (bool) -> None
        self.compute_not_found = mode
    
    
    def _get_elements(self, initial_set, check_element, page_counters, kept_checks, log_name, counters=None):
        # type: (OrderedDict, OrderedDict, _FilterPagination, Optional[_CheckSet], unicode, Optional[List[int,int]]) -> list
        start_time = time.time()
        initial_set = initial_set.values()
        if (self.sorts and self.sorts != [DEFAULT_SORT]) or self.filters:
            _sort_list_api_part_all(self.sorts, initial_set)
        logger.log_perf(start_time, u'get_list_api_part_all', u'sort %s' % log_name)
        
        if not isinstance(counters, list) or len(counters) != 3:
            counters = [0, 0, 0]
        result_set = []
        first = None
        last = None
        if self.page is not None:
            if first is None and self.page == page_counters.current_page:
                first = len(result_set)
            if last is None and self.page < page_counters.current_page:
                last = len(result_set)
        for host in initial_set:
            result_set.append(host)
            page_counters.counter += 1
            if self.page is None or self.page == page_counters.current_page:
                if host.got_business_rule:
                    counters[1] = counters[1] + 1
                else:
                    counters[0] = counters[0] + 1
            check_for_host = check_element.get(host.host_name, [])
            if check_for_host:
                if self.page is None or self.page == page_counters.current_page:
                    counters[2] = counters[2] + len(check_for_host)
                    if self.sorts and self.sorts != [DEFAULT_SORT]:
                        _sort_list_api_part_all(self.sorts, check_for_host)
                    if kept_checks is not None:
                        kept_checks.add(host.host_name, check_for_host)
                    else:
                        result_set.extend(check_for_host)
                        page_counters.counter += len(check_for_host)
            if self.page_size and page_counters.counter >= self.page_size:
                page_counters.pages.append({u'start': page_counters.start_page, u'end': page_counters.start_page + page_counters.counter})
                page_counters.start_page = page_counters.start_page + page_counters.counter
                page_counters.counter = 0
                page_counters.current_page += 1
                if self.page is not None:
                    if first is None and self.page == page_counters.current_page:
                        first = len(result_set)
                    if last is None and self.page < page_counters.current_page:
                        last = len(result_set)
        
        logger.log_perf(start_time, u'get_list_api_part_all', u'sort %s checks' % log_name)
        start_time = time.time()
        if first is not None or last is not None:
            result_set = result_set[first:last]
            page_counters.last = last
        elif self.page is not None:
            result_set = []
        logger.log_perf(start_time, u'get_list_api_part_all', u'Result pagination %s checks' % log_name)
        return result_set
    
    
    def _jsonify_elements(self, host_elements, check_elements, output_string, start_header=None, with_counter=False, item_name=None):
        # type: (list, Optional[dict], StringIO, Optional[unicode], Optional[bool], Optional[unicode]) -> None
        try:
            add_counter = with_counter and (not self.ALLOWED_OUTPUT or u'nb_element_filter' in self.ALLOWED_OUTPUT)
            if not item_name:
                item_name = u'element'
            
            if start_header:
                output_string.write(u'%s' % start_header)
            
            if add_counter:
                output_string.write(u'"nb_%s":%d,' % (item_name, len(host_elements)))
            output_string.write(u'"%s":[' % item_name)
            
            for output_element in host_elements:
                output_element = self._api_list_element_dict(output_element)
                host_name = output_element[u'host_name']
                if self.ALLOWED_OUTPUT:
                    # output_element = {self.ALLOWED_OUTPUT[k]: output_element[k] for k in self.ALLOWED_OUTPUT if k in output_element}
                    tmp_output = {}
                    for k in self.ALLOWED_OUTPUT:
                        if k in output_element:
                            tmp_output.update({self.ALLOWED_OUTPUT[k]: output_element[k]})
                    output_element = tmp_output
                output_string.write(u'%s' % json.dumps(output_element, separators=(',', ':')))
                
                if check_elements is not None:
                    output_string.seek(-1, os.SEEK_END)
                    if add_counter:
                        output_string.write(u',"nb_check":%d' % len(check_elements.get(host_name, [])))
                    output_string.write(u',"checks":[')
                    
                    if host_name in check_elements:
                        checks = check_elements[host_name]
                        for output_check in checks:
                            output_check = self._api_list_element_dict(output_check)
                            if self.ALLOWED_OUTPUT:
                                # output_check = {self.ALLOWED_OUTPUT[k]: output_check[k] for k in self.ALLOWED_OUTPUT if k in output_check}
                                tmp_output = {}
                                for k in self.ALLOWED_OUTPUT:
                                    if k in [u'uuid_parent', u'host_name']:
                                        continue
                                    if k in output_check:
                                        tmp_output.update({self.ALLOWED_OUTPUT[k]: output_check[k]})
                                output_check = tmp_output
                            output_string.write(u'%s,' % json.dumps(output_check, separators=(',', ':')))
                        if checks:
                            output_string.seek(-1, os.SEEK_END)
                    output_string.write(u']}')
                
                output_string.write(u',')
            if host_elements:
                output_string.seek(-1, os.SEEK_END)
            output_string.write(u']')
        except:
            # TODO: remove these logs ?
            logger.print_stack()
            output_string.seek(0, os.SEEK_END)
            logger.error(u'Could not build JSON document from element list. Buffer size was : [%d]' % output_string.tell())
            # if host_elements:
            #     logger.error('Host elements nb : %d' % len(host_elements))
            #     if len(host_elements) < 10:
            #         logger.error("Hosts elements: %s" % str(host_elements))
            #  if check_elements:
            #      logger.error('Check elements nb : %d' % len(check_elements))
            #      if len(check_elements) < 10:
            #         for k, v in check_elements.items():
            #              logger.error("Check elements: [%s] %d elements" % (str(k), len(v)))
            
            # Free memory if we could not complete
            output_string.close()
            raise
    
    
    def get_list_elements(self, jsonify_elements=True, specific_module_pagination=None):
        # type: (bool, Optional[_FilterPagination]) -> Union[OrderedDict,unicode]
        full_start_time = time.time()
        
        start_time = time.time()
        user = self.user
        
        thread_id = u'%s' % threading.current_thread().name
        if user:
            log_prefix = u'%-22s ] [ user= %s ] [ get_data_visualisation_list' % (thread_id, getattr(user, u'uuid', 0))
        else:
            log_prefix = u'%-22s ] [ get_data_visualisation_list' % thread_id
        
        items = self.monitoring_item_manager.get_all_hosts_and_services()
        
        configuration_id = u''
        part_configuration_id = u''
        last_part_configuration_incarnation = self.monitoring_item_manager.get_last_part_configuration_incarnation()
        if last_part_configuration_incarnation:
            configuration_id = last_part_configuration_incarnation.get_uuid()
            part_configuration_id = u'%s-%s' % (last_part_configuration_incarnation.get_uuid(), last_part_configuration_incarnation.get_part_id())
        full_nb_items_in_broker = len(items)
        
        logger.log_perf(start_time, log_prefix, u'get_all_hosts_and_services len [%s]' % full_nb_items_in_broker)
        
        start_time = time.time()
        StaticCache.refresh(items, part_configuration_id)
        logger.log_perf(start_time, log_prefix, u'_init_static_cache len [%s]' % full_nb_items_in_broker)
        
        # Filter with authorization
        # TODO mark filter elements.
        if user:
            start_time = time.time()
            items = get_all_my_visible_items(user, self.monitoring_item_manager)[:]
            logger.log_perf(start_time, log_prefix, u'only_related_to')
        else:
            items = items[:]
        
        nb_elements_total = len(items)
        elements = []
        
        # Values for selector
        all_values = {}
        if self.result_set_id is not None and self.result_set_id != configuration_id:
            start_time = time.time()
            all_values = _list_all_values(items)
            logger.log_perf(start_time, log_prefix, u'_list_all_values')
        
        start_time = time.time()
        hosts = {}
        nb_element_root_problems = 0
        nb_hosts_total = 0
        nb_clusters_total = 0
        nb_checks_total = 0
        for item in items:
            if item.__class__.my_type == u'service':
                nb_checks_total = nb_checks_total + 1
                if item.host is None:
                    logger.error(u'A service without host have been found id: %s  service_description: %s' % (item.instance_uuid, item.service_description))
                    continue
            else:
                if item.got_business_rule:
                    nb_clusters_total = nb_clusters_total + 1
                else:
                    nb_hosts_total = nb_hosts_total + 1
                hosts[item.host_name] = item
            if item.is_problem:
                nb_element_root_problems += 1
        logger.log_perf(start_time, log_prefix, u'count nb_element_root_problems')
        
        # Filter with filter parameters
        timestamp_backend_processing_start = int(time.time())
        start_time = time.time()
        if self.filters:
            filters_contain_root_problems = u'true' in self.filters[0].get(u'is_root_problem', set([u'false']))
            if filters_contain_root_problems:
                items = self._filter_root_problems(self.filters, items)
            
            # normal filters
            nb_found = [0, 0, 0]
            items = self._filter_elements_with_search(items, self.filters, nb_found)
            nb_hosts_found = nb_found[0]
            nb_clusters_found = nb_found[1]
            nb_checks_found = nb_found[2]
        else:
            nb_hosts_found = nb_hosts_total
            nb_clusters_found = nb_clusters_total
            nb_checks_found = nb_checks_total
        logger.log_perf(start_time, log_prefix, u'_filter_elements_with_search ')
        
        nb_items_post_filter = len(items)
        
        host_kept_checks = _CheckSet()
        cluster_kept_checks = _CheckSet()
        elements_hosts = []
        elements_clusters = []
        nb_hosts_in_page = 0
        nb_clusters_in_page = 0
        nb_checks_in_page = 0
        
        if self.flattened_view:
            start_time = time.time()
            if (self.sorts and self.sorts != [DEFAULT_SORT]) or self.filters:
                _sort_list_api_part_all(self.sorts, items)
            logger.log_perf(start_time, log_prefix, u'sort all (flattened view)')
            
            start_time = time.time()
            _pages = []
            if self.page_size:
                if specific_module_pagination:
                    _pages = specific_module_pagination.pages
                else:
                    _pages = [{u'start': start_index, u'end': min(start_index + self.page_size, len(items))} for start_index in range(0, len(items), self.page_size)]
            
            elements = items
            nb_element_return = len(elements)
            
            # Result pagination
            if self.page is not None and _pages:
                if specific_module_pagination:
                    elements = elements[_pages[0][u'start']:_pages[0][u'end']]
                elif len(_pages) > self.page:
                    elements = elements[_pages[self.page][u'start']:_pages[self.page][u'end']]
                else:
                    elements = []
                nb_element_return = len(elements)
                for page_item in elements:
                    if page_item.__class__.my_type == u'host':
                        if page_item.got_business_rule:
                            nb_clusters_in_page = nb_clusters_in_page + 1
                        else:
                            nb_hosts_in_page = nb_hosts_in_page + 1
                    else:
                        nb_checks_in_page = nb_checks_in_page + 1
                logger.log_perf(start_time, log_prefix, u'Result pagination')
        else:
            start_time = time.time()
            host_element = OrderedDict()
            cluster_element = OrderedDict()
            check_element = OrderedDict()
            cluster_check_element = OrderedDict()
            
            if self.checks_in_tree:
                for i in items:
                    if i.__class__.my_type == u'host':
                        if i.got_business_rule:
                            cluster_element[i.host_name] = i
                        else:
                            host_element[i.host_name] = i
                    elif i.__class__.my_type == u'service':
                        check_host_name = i.host.host_name
                        if i.host.got_business_rule:
                            if check_host_name not in cluster_check_element:
                                cluster_check_element[check_host_name] = []
                            cluster_check_element[check_host_name].append(i)
                        else:
                            if check_host_name not in check_element:
                                check_element[check_host_name] = []
                            check_element[check_host_name].append(i)
                
                for check_host_name in check_element:
                    # If the host of the check isn't in host_element we add it
                    # We don't want orphan check
                    if check_host_name not in host_element:
                        host = hosts[check_host_name]
                        host_element[host.host_name] = host
                for check_host_name in cluster_check_element:
                    if check_host_name not in cluster_element:
                        host = hosts[check_host_name]
                        cluster_element[host.host_name] = host
            else:
                host_element = OrderedDict((i.host_name, i) for i in items if (i.__class__.my_type != u'service'))
                for i in items:
                    if i.__class__.my_type == u'service':
                        check_host_name = i.host.host_name
                        if check_host_name not in check_element:
                            check_element[check_host_name] = []
                        check_element[check_host_name].append(i)
                        
                        # If the host of the check isn't in host_element we add it
                        # We don't want orphan check
                        if check_host_name not in host_element:
                            host = hosts[check_host_name]
                            host_element[host.host_name] = host
            
            logger.log_perf(start_time, log_prefix, u'append host')
            
            page_counters = specific_module_pagination if specific_module_pagination else _FilterPagination()
            start_time = time.time()
            elements_in_page_counters = [0, 0, 0]
            
            if self.checks_in_tree:
                host_kept_checks = _CheckSet()
                cluster_kept_checks = _CheckSet()
                
                elements_clusters = self._get_elements(cluster_element, cluster_check_element, page_counters, cluster_kept_checks, u'clusters', elements_in_page_counters)
                elements_hosts = self._get_elements(host_element, check_element, page_counters, host_kept_checks, u'hosts', elements_in_page_counters)
                
                nb_hosts_found = len(elements_hosts)
                nb_clusters_found = len(elements_clusters)
                nb_checks_found = cluster_kept_checks.nb + host_kept_checks.nb
                nb_element_return = len(elements_clusters) + len(elements_hosts) + cluster_kept_checks.nb + host_kept_checks.nb
            else:
                elements = self._get_elements(host_element, check_element, page_counters, None, u'host', elements_in_page_counters)
                nb_element_return = len(elements)
            
            nb_hosts_in_page = elements_in_page_counters[0]
            nb_clusters_in_page = elements_in_page_counters[1]
            nb_checks_in_page = elements_in_page_counters[2]
            
            if page_counters.counter != 0 and not specific_module_pagination:
                page_counters.pages.append({u'start': page_counters.start_page, u'end': page_counters.start_page + page_counters.counter})
            _pages = page_counters.pages
            
            logger.log_perf(start_time, log_prefix, u'Result pagination')
        
        # Extra check on bad filters (element not found) :
        lost_elements = []
        lost_elements_nb = 0
        lost_fathers_nb = 0
        lost_checks_nb = 0
        if self.compute_not_found:
            start_time = time.time()
            lost_element_set = set()
            if self.filters:
                for _filter in self.filters:
                    if u'uuid_parent' in _filter:
                        for uuid in _filter[u'uuid_parent']:
                            if not StaticCache.get_host_by_uuid(uuid, user) and (uuid, u'', u'', u'') not in lost_element_set:
                                lost_element_set.add((uuid, u'', u'', u'',))
                    if u'host_name' in _filter:
                        for name in _filter[u'host_name']:
                            if not StaticCache.get_host_by_name(name, user) and (u'', name, u'', u'') not in lost_element_set:
                                lost_element_set.add((u'', name, u'', u'',))
                    if u'check_uuid' in _filter:
                        for uuid in _filter[u'check_uuid']:
                            if (u'-' not in uuid or not StaticCache.get_check_by_uuid(uuid, user)) and (u'', u'', u'', uuid) not in lost_element_set:
                                lost_element_set.add((u'', u'', u'', uuid,))
                    if u'service_description' in _filter and (u'host_name' in _filter or u'uuid_parent' in _filter):
                        for check_name in _filter[u'service_description']:
                            if u'host_name' in _filter:
                                for host_name in _filter[u'host_name']:
                                    if (u'', host_name, u'', u'') not in lost_element_set:
                                        host = StaticCache.get_host_by_name(host_name, user)
                                        host_uuid = getattr(host, u'uuid', u'')
                                        if (not host or not StaticCache.get_check_by_name(host, check_name, user)) and (host_uuid, host_name, check_name, u'') not in lost_element_set:
                                            lost_element_set.add((host_uuid, host_name, check_name, u'',))
                            if u'uuid_parent' in _filter:
                                for uuid in _filter[u'uuid_parent']:
                                    host = StaticCache.get_host_by_uuid(uuid, user)
                                    host_name = getattr(host, u'host_name', '')
                                    if (not host or not StaticCache.get_check_by_name(host, check_name, user)) and (uuid, host_name, check_name, u'') not in lost_element_set:
                                        lost_element_set.add((uuid, host_name, check_name, u'',))
            if lost_element_set:
                lost_elements = []
                lost_elements_nb = len(lost_element_set)
                for (host_uuid, host_name, check_name, check_uuid) in lost_element_set:
                    lost_element = {}
                    if host_uuid:
                        lost_element.update({u'uuid_parent': host_uuid})
                    if host_name:
                        lost_element.update({u'host_name': host_name})
                    if check_name:
                        lost_element.update({u'service_description': check_name})
                    if check_uuid:
                        lost_element.update({u'check_uuid': check_uuid})
                    if check_name or check_uuid:
                        lost_checks_nb = lost_checks_nb + 1
                    else:
                        lost_fathers_nb = lost_fathers_nb + 1
                    if self.ALLOWED_OUTPUT:
                        tmp_element = {}
                        for k in lost_element:
                            if k in self.ALLOWED_OUTPUT:
                                tmp_element.update({self.ALLOWED_OUTPUT[k]: lost_element[k]})
                        lost_element = tmp_element
                    if lost_element:
                        lost_elements.append(lost_element)
            logger.log_perf(start_time, log_prefix, u'Not found filters analysis')
        
        ret = {
            u'nb_element_total'                  : nb_elements_total,
            u'nb_element_root_problems'          : nb_element_root_problems,
            u'configuration_id'                  : configuration_id,
            u'pages'                             : _pages,
            u'timestamp_backend_processing_start': timestamp_backend_processing_start,
            u'timestamp_backend_processing_end'  : int(time.time())
        }
        
        # if self.flattened_view or not self.checks_in_tree:
        #     ret['nb_element_filter'] = nb_element_return
        
        if all_values:
            ret[u'all_values'] = all_values
        
        elements_count = OrderedDict()
        elements_count.update({u'nb_elements_total': nb_elements_total})
        elements_count.update({u'nb_hosts_total': nb_hosts_total})
        elements_count.update({u'nb_clusters_total': nb_clusters_total})
        elements_count.update({u'nb_checks_total': nb_checks_total})
        elements_count.update({u'nb_elements_filtered': nb_hosts_found + nb_clusters_found + nb_checks_found})
        elements_count.update({u'nb_hosts_filtered': nb_hosts_found})
        elements_count.update({u'nb_clusters_filtered': nb_clusters_found})
        elements_count.update({u'nb_checks_filtered': nb_checks_found})
        if self.page is not None and _pages:
            elements_count.update({u'nb_elements_in_page': nb_hosts_in_page + nb_clusters_in_page + nb_checks_in_page})
            elements_count.update({u'nb_hosts_in_page': nb_hosts_in_page})
            elements_count.update({u'nb_clusters_in_page': nb_clusters_in_page})
            elements_count.update({u'nb_checks_in_page': nb_checks_in_page})
        if lost_elements:
            elements_count.update({u'nb_elements_not_found': lost_elements_nb})
            elements_count.update({u'nb_fathers_not_found': lost_fathers_nb})
            elements_count.update({u'nb_checks_not_found': lost_checks_nb})
        
        filtered_elements = OrderedDict()
        filtered_elements.update({u'request_statistics': elements_count})
        output_string = StringIO()
        try:
            output_start = u'{'
            elements_count_name = u'elements_count'
            if self.ALLOWED_OUTPUT and elements_count_name in self.ALLOWED_OUTPUT:
                elements_count_name = self.ALLOWED_OUTPUT[elements_count_name]
                tmp_count = OrderedDict()
                for k in elements_count:
                    if k in self.ALLOWED_OUTPUT:
                        tmp_count.update({self.ALLOWED_OUTPUT[k]: elements_count[k]})
                
                if tmp_count:
                    if jsonify_elements:
                        output_string.write(u'%s"%s":%s' % (output_start, elements_count_name, json.dumps(tmp_count, separators=(',', ':'))))
                        output_start = u','
                    else:
                        filtered_elements.update({u'request_statistics': tmp_count})
            
            if self.page is not None and _pages:
                pagination = self._set_pagination_info(_pages)
                pagination_name = u'pagination'
                if self.ALLOWED_OUTPUT:
                    if pagination_name in self.ALLOWED_OUTPUT:
                        pagination_name = self.ALLOWED_OUTPUT[pagination_name]
                    else:
                        pagination_name = u''
                if pagination_name:
                    filtered_elements.update({pagination_name: pagination})
                    output_string.write(u'%(start_character)s"%(field_name)s":%(data_set)s' % {u'start_character': output_start,
                                                                                               u'field_name'     : pagination_name,
                                                                                               u'data_set'       : json.dumps(pagination, separators=(',', ':'))})
                    output_start = u','
            
            # Drop UI compatibility ...
            # if self.ALLOWED_OUTPUT:
            #     # ret = {self.ALLOWED_OUTPUT[k]: ret[k] for k in ret if k in self.ALLOWED_OUTPUT}
            #     tmp_ret = {}
            #     for k in ret:
            #         if k in self.ALLOWED_OUTPUT:
            #             tmp_ret.update({self.ALLOWED_OUTPUT[k]: ret[k]})
            #     ret = tmp_ret
            #
            # for k in ret:
            #     output_string.write(u',"%s":%s' % (k, json.dumps(ret[k], separators=(',', ':'))))
            
            if jsonify_elements:
                if self.flattened_view or not self.checks_in_tree:
                    start_time = time.time()
                    self._jsonify_elements(elements, None, output_string, output_start, False, u'elements_found')
                else:
                    start_time = time.time()
                    output_string.write(u'%s"%s":{' % (output_start, u'elements_found'))
                    self._jsonify_elements(elements_clusters, cluster_kept_checks.result_set, output_string, u'', False, u'clusters')
                    self._jsonify_elements(elements_hosts, host_kept_checks.result_set, output_string, u', ', False, u'hosts')
                    output_string.write(u'}')
                
                if lost_elements:
                    self._sort_lost_elements(lost_elements)
                    not_found_name = self._get_not_found_property_name()
                    output_string.write(u',"%s":%s' % (not_found_name, json.dumps(lost_elements, separators=(',', ':'))))
            else:
                if self.flattened_view or not self.checks_in_tree:
                    elements_found = self._get_elements_with_allowed_output(elements)
                else:
                    elements_found = {
                        u'hosts'   : self._get_elements_with_allowed_output(elements_hosts, host_kept_checks.result_set),
                        u'clusters': self._get_elements_with_allowed_output(elements_clusters, cluster_kept_checks.result_set)
                    }
                filtered_elements.update({u'elements_found': elements_found})
                
                if lost_elements:
                    self._sort_lost_elements(lost_elements)
                    not_found_name = self._get_not_found_property_name()
                    filtered_elements.update({not_found_name: lost_elements})
                return filtered_elements
            
            output_string.write(u'}')
            
            logger.log_perf(start_time, log_prefix, u'json.dumps')
            
            logger.log_perf(full_start_time, log_prefix, u'nb item in broker [%s] nb item filter [%s] nb item post add missing host [%s] nb item display [%s] filter [%s] sort [%s]' % (
                full_nb_items_in_broker, nb_items_post_filter, nb_element_return, len(elements), str(self.full_filter_for_log), str(self.full_sort_for_log)))
            
            output_string.seek(0, os.SEEK_END)
            
            return output_string.getvalue()
        finally:
            if not output_string.closed:
                output_string.close()
    
    
    def _sort_lost_elements(self, lost_elements):
        if self.ALLOWED_OUTPUT:
            lost_elements.sort(key=lambda elt: (elt.get(self.ALLOWED_OUTPUT.get(u'host_name', u'None'), u''),
                                                elt.get(self.ALLOWED_OUTPUT.get(u'uuid_parent', u'None'), u''),
                                                elt.get(self.ALLOWED_OUTPUT.get(u'check_name', u'None'), u''),
                                                elt.get(self.ALLOWED_OUTPUT.get(u'check_uuid', u'None'), u'')))
        else:
            lost_elements.sort(key=lambda elt: (elt.get(u'host_name', u''), elt.get(u'uuid_parent', u''), elt.get(u'check_name', u''), elt.get(u'check_uuid', u'')))
    
    
    def _get_not_found_property_name(self, not_found_name=u'element_not_found'):
        # type: (unicode) -> unicode
        if self.ALLOWED_OUTPUT and not_found_name in self.ALLOWED_OUTPUT:
            not_found_name = self.ALLOWED_OUTPUT[not_found_name]
        else:
            available_output = getattr(self, u'AVAILABLE_OUTPUT', {})
            if not_found_name in available_output:
                not_found_name = available_output[not_found_name]
        return not_found_name
    
    
    def _update_element_output(self, output_element, unauthorized_properties):
        # type: (Dict, List[unicode]) -> Dict
        if self.ALLOWED_OUTPUT:
            tmp_output = {}
            for k in self.ALLOWED_OUTPUT:
                if k in unauthorized_properties:
                    continue
                if k in output_element:
                    tmp_output.update({self.ALLOWED_OUTPUT[k]: output_element[k]})
            return tmp_output
    
    
    def _get_elements_with_allowed_output(self, elements, fathers_checks=None):
        # type: (List[Union[Host,Service]], Dict) -> List
        updated_fathers = []
        for output_element in elements:
            output_element = self._api_list_element_dict(output_element)
            host_name = output_element[u'host_name']
            output_element = self._update_element_output(output_element, [])
            updated_fathers.append(output_element)
            
            if fathers_checks is None:
                continue
            if host_name in fathers_checks:
                checks = fathers_checks[host_name]
                for output_check in checks:
                    output_check = self._update_element_output(self._api_list_element_dict(output_check), [u'uuid_parent', u'host_name'])
                    if u'checks' not in output_element and output_check:
                        output_element[u'checks'] = []
                    output_element[u'checks'].append(output_check)
        return updated_fathers
    
    
    def _set_pagination_info(self, pages):
        # type: (List[Dict[unicode, int]]) -> Dict
        if self.page is None:
            return {}
        
        has_next_page = True if self.page < (len(pages) - 1) else False
        pagination = OrderedDict()
        pagination.update({u'has_next_page': has_next_page})
        pagination.update({u'nb_total_page': len(pages)})
        pagination.update({u'page': self.page})
        pagination.update({u'page_size': self.page_size})
        return pagination


class RequestManagerLivedataV2(FilterItem):
    ALLOWED_FILTER = {
        u'description'                : u'display_name',
        u'status'                     : u'status',
        u'context'                    : u'context',
        u'is_status_confirmed'        : u'is_status_confirmed',
        u'type'                       : u'extended_type',
        u'father_name'                : u'host_name',
        u'father_name_contains'       : u'host_name_like',
        u'father_uuid'                : u'uuid_parent',
        u'check_name'                 : u'service_description',
        u'check_name_contains'        : u'service_description_like',
        u'check_uuid'                 : u'check_uuid',
        u'realm'                      : u'realm',
        u'address'                    : u'address',
        u'notification_contacts'      : u'contacts',
        u'notification_contact_groups': u'contact_groups',
        u'father_templates'           : u'host_templates',
        u'host_groups'                : u'host_groups',
        u'business_impact'            : u'business_impact',
        u'status_since'               : u'status_since',
        u'status_confirmed_since'     : u'last_hard_state_change',
        u'last_check'                 : u'last_check',
        u'next_check'                 : u'next_check',
    }
    ALLOWED_SORT = ALLOWED_FILTER
    
    # Can be used to build ALLOWED_OUTPUT (which converts internal names to public names, and select allowed output fields)
    AVAILABLE_OUTPUT = {
        u'element_not_found'     : u'elements_not_found',
        u'elements_count'        : u'request_statistics',
        u'nb_elements_total'     : u'nb_elements_total',
        u'nb_hosts_total'        : u'nb_hosts_total',
        u'nb_clusters_total'     : u'nb_clusters_total',
        u'nb_checks_total'       : u'nb_checks_total',
        u'nb_elements_filtered'  : u'nb_elements_filtered',
        u'nb_hosts_filtered'     : u'nb_hosts_filtered',
        u'nb_clusters_filtered'  : u'nb_clusters_filtered',
        u'nb_checks_filtered'    : u'nb_checks_filtered',
        u'nb_elements_in_page'   : u'nb_elements_in_page',
        u'nb_hosts_in_page'      : u'nb_hosts_in_page',
        u'nb_clusters_in_page'   : u'nb_clusters_in_page',
        u'nb_checks_in_page'     : u'nb_checks_in_page',
        u'nb_elements_not_found' : u'nb_elements_not_found',
        u'nb_fathers_not_found'  : u'nb_fathers_not_found',
        u'nb_checks_not_found'   : u'nb_checks_not_found',
        u'address'               : u'address',
        u'attempts'              : u'attempts',
        u'business_impact'       : u'business_impact',
        u'service_description'   : u'check_name',
        u'contacts'              : u'notification_contacts',
        u'contact_groups'        : u'notification_contact_groups',
        u'context'               : u'context',
        u'display_name'          : u'description',
        u'extended_type'         : u'type',
        u'host_name'             : u'father_name',
        u'host_groups'           : u'host_groups',
        u'host_templates'        : u'father_templates',
        u'is_status_confirmed'   : u'is_status_confirmed',
        u'last_check'            : u'last_check',
        u'long_output'           : u'long_output',
        u'max_attempts'          : u'max_attempts',
        u'next_check'            : u'next_check',
        u'notification_options'  : u'notification_options',
        u'notification_period'   : u'notification_period',
        u'output'                : u'output',
        u'perf_data'             : u'raw_perf_data',
        u'realm'                 : u'realm',
        u'smart_perf_data'       : u'perf_data',
        u'status'                : u'status',
        u'status_since'          : u'status_since',
        u'last_hard_state_change': u'status_confirmed_since',
        u'check_uuid'            : u'check_uuid',
        u'uuid_parent'           : u'father_uuid',
        u'pagination'            : u'pagination'
    }
    POST_AVAILABLE_PARAMETERS = set((u'filter', u'output_field', u'output_format', u'sort', u'with_element_not_found'))
    
    
    def __init__(self, monitoring_item_manager, user, mandatory_output_fields=None, user_allowed_output_field=None, user_filters=None, user_allowed_filters=None, page_settings=u'', bottle_post=None, additional_format=u'', compute_not_found=None):
        # type:(MonitoringItemManager, Optional[Contact], Optional[List[unicode]], Optional[List[unicode]], unicode, List[unicode], unicode, Optional[MultiDict, Dict[unicode, unicode]], unicode, bool) -> None
        super(RequestManagerLivedataV2, self).__init__(monitoring_item_manager, user)
        
        self.update_modified_properties(mandatory_output_fields=mandatory_output_fields, user_allowed_output_field=user_allowed_output_field, user_filters=user_filters, user_allowed_filters=user_allowed_filters,
                                        page_settings=page_settings, bottle_post=bottle_post, additional_format=additional_format, compute_not_found=compute_not_found)
    
    
    def update_modified_properties(self, mandatory_output_fields=None, user_allowed_output_field=None, user_filters=None, user_allowed_filters=None, page_settings=u'', bottle_post=None, additional_format=u'',
                                   compute_not_found=None):
        # type:(Optional[List[unicode]], Optional[List[unicode]], unicode, List[unicode], unicode, Optional[MultiDict, Dict[unicode, unicode]], unicode, bool) -> None
        self.set_allowed_output(bottle_post=bottle_post, mandatory_fields=mandatory_output_fields, user_allowed_output_field=user_allowed_output_field)
        if user_allowed_filters:
            user_allowed_filters = self.build_restricted_allowed_filters(user_allowed_filters)
        
        if user_filters:
            self.set_filters(user_filters, user_allowed_filters)
        
        if page_settings:
            self.set_range(page_settings)
        
        if bottle_post:
            self.set_output_format(bottle_post, additional_format=additional_format)
        
        if compute_not_found:
            self.set_compute_not_found(compute_not_found)
    
    
    @staticmethod
    def validate_bottle_post_parameter(bottle_post, enabled_parameters=None, disabled_parameters=None):
        # type: (MultiDict, Optional[Set], Optional[Set]) -> None
        if not bottle_post:
            return
        for key in bottle_post.iterkeys():
            if not isinstance(key, unicode):
                key = key.decode(u'UTF-8')
            if disabled_parameters and (key in disabled_parameters or (key[0:6] == u'filter' and u'filter' in disabled_parameters)):
                raise ShinkenExceptionKeyError(text=u'POST parameter [ %s ] is not available for this route' % key)
            if enabled_parameters and key not in enabled_parameters and (key[0:6] != u'filter' or u'filter' not in enabled_parameters):
                raise ShinkenExceptionKeyError(text=u'POST parameter [ %s ] is unknown' % key)
    
    
    @staticmethod
    def build_filters_from_bottle_post(bottle_post):
        # type: (MultiDict) -> unicode
        filters = u''
        if not bottle_post:
            return filters
        cpt = 0
        for key in bottle_post.iterkeys():
            if not isinstance(key, unicode):
                key = key.decode(u'UTF-8')
            if len(key) >= 6 and key[0:6] == u'filter':
                for value in bottle_post.getall(key):
                    if not isinstance(value, unicode):
                        value = value.decode(u'UTF-8')
                    new_filters = u'"filter%(cpt)d":"%(value)s"' % {u'cpt': cpt, u'value': value}
                    if len(filters) > 0:
                        filters = u'%(filters)s, %(new_filter)s' % {u'filters': filters, u'new_filter': new_filters}
                    else:
                        filters = new_filters
                    cpt += 1
        return u'{%s}' % filters
    
    
    def build_restricted_allowed_filters(self, restricted_filter_tags):
        # type: (Optional[Collection[unicode]]) -> Dict
        # reduce usable filter fields set
        # input: public filter name, output: dict for FilterItem
        if not restricted_filter_tags:
            return self.ALLOWED_FILTER
        
        allowed_filters = {}
        for tag in restricted_filter_tags:
            if tag not in self.ALLOWED_FILTER:
                raise ShinkenExceptionKeyError(text=u'incorrect filter tag name [ %s ]' % tag)
            allowed_filters.update({tag: self.ALLOWED_FILTER[tag]})
        return allowed_filters
    
    
    def build_allowed_output(self, output_fields, user_allowed_output_field):
        # type: (List, Optional[List[unicode]]) -> Dict
        # input: list of public field names, output: dict for Filter output managing
        new_output = {}
        if output_fields:
            # reversed_output = {v: k for k, v in item_set.get_output_fields().items()}
            reversed_output = {}
            for k, v in self.AVAILABLE_OUTPUT.items():
                reversed_output.update({v: k})
            for field in output_fields:
                if field in reversed_output:
                    if field not in user_allowed_output_field:
                        raise ShinkenExceptionKeyError(text=u'output_field: field name [ %s ] not allowed in this route' % field)
                    new_output.update({reversed_output[field]: field})
                else:
                    raise ShinkenExceptionKeyError(text=u'output_field: invalid field name [ %s ]' % field)
        return new_output
    
    
    def set_allowed_output(self, bottle_post=None, mandatory_fields=None, user_allowed_output_field=None):
        # type: (Optional[MultiDict], Optional[List[unicode]], Optional[List[unicode]]) -> None
        output_fields = None
        
        if user_allowed_output_field:
            user_allowed_output_field.extend(mandatory_fields)
        else:
            user_allowed_output_field = set(self.AVAILABLE_OUTPUT.values())
        
        if bottle_post:
            output_fields = bottle_post.get(u'output_field', None)
            if output_fields and not isinstance(output_fields, unicode):
                output_fields = output_fields.decode(u'UTF-8')
        
        if output_fields:
            output_fields = output_fields.strip().split('~')
        
        if not output_fields:
            output_fields = []
        
        if mandatory_fields:
            output_fields.extend(mandatory_fields)
        
        self.set_output_fields(self.build_allowed_output(output_fields, user_allowed_output_field))
    
    
    def set_output_format(self, bottle_post, additional_format=u''):
        # type: (MultiDict, unicode) -> None
        output_format = bottle_post.get(u'output_format', None)
        if output_format is not None:
            if output_format not in [u'checks_attached_to_father', u'elements_on_same_level', additional_format]:
                if not isinstance(output_format, unicode):
                    output_format = output_format.decode(u'UTF-8')
                raise ShinkenExceptionValueError(text=u'output_format: invalid value [ %s ]' % output_format)
            self.set_flattened_mode((output_format == u'elements_on_same_level' or output_format == additional_format), True)
    
    
    def set_print_element__not_found(self, bottle_post):
        # type: (MultiDict) -> None
        mode = bottle_post.get(u'with_element_not_found', None)
        if mode is not None:
            if mode not in [u'true', u'false']:
                raise ShinkenExceptionValueError(text=u'with_element_not_found: invalid value [ %s ]' % mode)
            self.set_compute_not_found(mode == u'true')
