#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import time

from shinken.misc.type_hint import TYPE_CHECKING, cast
from shinken.thread_helper import Thread
from shinkensolutions.data_hub.data_hub_driver.abstract_data_hub_driver import AbstractDataHubDriver, AbstractDataHubDriverConfig
from shinkensolutions.data_hub.data_hub_factory.data_hub_factory import DataHubFactory

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Any
    from shinken.log import PartLogger
    from shinkensolutions.data_hub.data_hub import DataHubConfig


class DataHubMetaDriverConfigCleanupDirectory(AbstractDataHubDriverConfig):
    def __init__(self, config_main_driver, cleanup_interval):
        # type: (AbstractDataHubDriverConfig, int) -> None
        super(DataHubMetaDriverConfigCleanupDirectory, self).__init__(u'CLEANUP')
        self.config_main_driver = config_main_driver
        self.cleanup_interval = cleanup_interval


def data_hub_meta_driver_cleanup_directory_factory(logger, driver_config, data_hub_config):
    # type: (PartLogger, AbstractDataHubDriverConfig, DataHubConfig) -> DataHubMetaDriverCleanupDirectory
    
    driver_config = cast(DataHubMetaDriverConfigCleanupDirectory, driver_config)
    
    main_driver = DataHubFactory.build_driver(logger, driver_config.config_main_driver, data_hub_config)
    return DataHubMetaDriverCleanupDirectory(logger, driver_config, main_driver)


DataHubFactory.register_driver_factory(DataHubMetaDriverConfigCleanupDirectory, data_hub_meta_driver_cleanup_directory_factory)


class DataHubMetaDriverCleanupDirectory(AbstractDataHubDriver, Thread):
    def __init__(self, logger, driver_config, main_driver):
        # type: (PartLogger, DataHubMetaDriverConfigCleanupDirectory, AbstractDataHubDriver) -> None
        Thread.__init__(self)
        AbstractDataHubDriver.__init__(self, logger, driver_config)
        
        self.logger = self._logger
        self.main_driver = main_driver
        self.cleanup_interval = driver_config.cleanup_interval
    
    
    def get_lock(self, data_id):
        raise NotImplementedError()
    
    
    def loop_turn(self):
        # type: () -> None
        nb_data_cleaned = 0
        start_time = time.time()
        for data_id in self.main_driver.get_all_data_id():
            try:
                if (time.time() - self.main_driver.get_last_modification_date(data_id)) >= self.cleanup_interval:
                    self.main_driver.remove(data_id)
                    nb_data_cleaned += 1
            except:
                continue  # Item not found, can't delete it
        if nb_data_cleaned or time.time() - start_time >= 1:
            self._logger.log_perf(start_time, u'cleanup_directory', u'Cleanup of file older than %s done for %s %s' % (self.cleanup_interval, nb_data_cleaned, self._data_type), min_time=0,
                                  warn_time=1)
    
    
    def get_thread_name(self):
        # type: () -> unicode
        return u'DHub clean %s' % self._name
    
    
    def init(self):
        # type: () -> None
        self.main_driver.init()
        self.start_thread()
    
    
    def get_all_data_id(self):
        # type: () -> List[unicode]
        return self.main_driver.get_all_data_id()
    
    
    def is_data_correct(self, data_id):
        # type: (unicode) -> bool
        return self.main_driver.is_data_correct(data_id)
    
    
    def write(self, data_id, data):
        # type: (unicode, Any) -> None
        self.main_driver.write(data_id, data)
    
    
    def read(self, data_id, log_error=True):
        # type: (unicode, bool) -> Any
        return self.main_driver.read(data_id, log_error=log_error)
    
    
    def remove(self, data_id):
        # type: (unicode) -> None
        self.main_driver.remove(data_id)
    
    
    def get_last_modification_date(self, data_id):
        # type: (unicode) -> int
        return self.main_driver.get_last_modification_date(data_id)
    
    
    def restore(self, data_id, data):
        # type: (unicode, Any) -> None
        self.main_driver.write(data_id, data)
    
    
    def sanitize(self, data_id, data):
        # type: (unicode, Any) -> None
        self.main_driver.write(data_id, data)
    
    
    def destroy(self):
        # type: () -> None
        self.main_driver.destroy()
    
    
    def check_right_access_issues_then_fix_them(self):
        # type: () -> None
        self.main_driver.check_right_access_issues_then_fix_them()
    
    
    def get_total_size(self):
        # type: () -> int
        return self.main_driver.get_total_size()
    
    
    def get_number_of_stored_data(self):
        # type: () -> int
        return self.main_driver.get_number_of_stored_data()
