#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import re

from index import Index
from shinken.log import LoggerFactory

indexed_collection_logger = LoggerFactory.get_logger(u'IndexedCollection')


class SEARCH_FILTER(object):
    IN = u'$in'
    NOT_IN = u'$nin'
    NOT_EQUAL = u'$ne'
    EXISTS = u'$exists'


class SEARCH_OPERAND(object):
    AND = u'$and'
    OR = u'$or'


class IndexedCollection(object):
    DISABLED_OPERATOR_IN_NON_INDEXED_COLLECTION = True
    
    
    def __init__(self, hash_key=None):
        super(IndexedCollection, self).__init__()
        self.counter = 0
        self.indexes = {}
        self.inverse = {}
        self.data = {}
        self.hash_key = hash_key
    
    
    def __iter__(self):
        return self.data.itervalues()
    
    
    def __len__(self):
        return len(self.data)
    
    
    def append(self, item):
        self._append(item)
    
    
    def extend(self, items):
        if items:
            for item in items:
                self._append(item)
    
    
    def remove(self, search):
        item = self._find_one(search)
        if not item:
            raise KeyError(u'Item not found [%s]' % search)
        item_key = self._get_item_key(item)
        key_index = self.inverse[item_key]
        for index in self.indexes.itervalues():
            index.unindex_item(key_index)
        del self.inverse[item_key]
        del self.data[key_index]
    
    
    def regenerate_index(self, item):
        key_index = self.inverse[self._get_item_key(item)]
        for index in self.indexes.itervalues():
            index.unindex_item(key_index)
            index.index_item(item, key_index)
    
    
    def add_index(self, property_name, getter=None, on_list=False):
        self.indexes[property_name] = Index(property_name, getter, on_list)
        self.indexes[property_name].add_index(self.data)
    
    
    def find_one(self, search):
        return self._find_one(search)
    
    
    def find(self, search, only_count=False):
        return self._find(search, only_count)
    
    
    def count(self, search):
        return self._find(search, only_count=True)
    
    
    def _get_item_key(self, item):
        key = id(item)
        if hasattr(item, u'get_key'):
            key = item.get_key()
        elif self.hash_key:
            key = item[self.hash_key]
        return key
    
    
    def _append(self, item):
        key_index = self.counter
        self.data[key_index] = item
        item_key = self._get_item_key(item)
        self.inverse[item_key] = key_index
        for index in self.indexes.itervalues():
            index.index_item(item, key_index)
        self.counter += 1
    
    
    def _find(self, search, only_count=False):
        # This method is use for remap result of _find_and_return_key_indexes to item or count
        if not search:
            return len(self.data) if only_count else self.data.values()
        
        found_key_indexes = self._find_and_return_key_indexes(search)
        if only_count:
            return len(found_key_indexes)
        
        found_items = map(lambda i: self.data[i], found_key_indexes)
        return found_items
    
    
    def _find_and_return_key_indexes(self, search):
        # This method search in data with search parameter and return only item key indexes (id)
        found_key_indexes = set(self.data.keys())
        if not search:
            return found_key_indexes
        
        search = search.copy()
        prop_already_search = []
        for prop_name, value_search in search.iteritems():
            prop_with_index = prop_name in self.indexes
            prop_is_filter_operator = isinstance(value_search, dict)
            value_is_regex = isinstance(value_search, re._pattern_type)
            
            if prop_with_index and prop_is_filter_operator:
                for search_operator, search_value_for_operator in value_search.iteritems():
                    found_key_indexes = self._find_filter_operator_with_index(found_key_indexes, prop_already_search, prop_name, search_operator, search_value_for_operator)
            elif prop_with_index and not value_is_regex:
                prop_already_search.append(prop_name)
                index = self.indexes[prop_name]
                key_indexes = index.find(value_search)
                found_key_indexes &= set(key_indexes)
            elif prop_name == SEARCH_OPERAND.OR:
                value_match = set()
                for value in value_search:
                    value_match |= self._find_and_return_key_indexes(value)
                found_key_indexes &= value_match
            elif prop_name == SEARCH_OPERAND.AND:
                for value in value_search:
                    found_key_indexes &= self._find_and_return_key_indexes(value)
            elif prop_name.startswith(u'$'):
                raise NotImplementedError(u'Operator [%s] is not yet implemented' % prop_name)
            else:
                indexed_collection_logger.warning(u'The property [%s] is not indexed. This will affect the search speed.' % prop_name)
                for found_key_index in found_key_indexes.copy():  # copy here because we remove element from it in loop
                    found_item = self.data[found_key_index]
                    if not self._recursive_search(found_item, prop_name, value_search):
                        found_key_indexes.remove(found_key_index)
        return found_key_indexes
    
    
    def _find_filter_operator_with_index(self, found_key_indexes, prop_already_search, prop_name, search_operator, search_value_for_operator):
        if search_operator == SEARCH_FILTER.IN:
            # The $in will search equality for each search_lookup
            tmp = set()
            prop_already_search.append(prop_name)
            index = self.indexes[prop_name]
            for sub_search_lookup in search_value_for_operator:
                key_indexes = index.find(sub_search_lookup)
                tmp |= set(key_indexes)
            found_key_indexes &= tmp
        elif search_operator == SEARCH_FILTER.NOT_EQUAL:
            # The $ne will exclude the search lookup
            prop_already_search.append(prop_name)
            index = self.indexes[prop_name]
            key_indexes_to_exclude = index.find(search_value_for_operator)
            key_indexes_to_exclude = [item_index for item_index in key_indexes_to_exclude if self.data[item_index].get(prop_name, '') == search_value_for_operator]
            found_key_indexes -= set(key_indexes_to_exclude)
        elif search_operator == SEARCH_FILTER.NOT_IN:
            # The $nin will exclude each the search lookup
            tmp = set()
            prop_already_search.append(prop_name)
            index = self.indexes[prop_name]
            for sub_search_lookup in search_value_for_operator:
                key_indexes = index.find(sub_search_lookup)
                tmp |= set(key_indexes)
            found_key_indexes -= tmp
        elif search_operator == SEARCH_FILTER.EXISTS:
            # The $exists will exclude the search lookup
            prop_already_search.append(prop_name)
            index = self.indexes[prop_name]
            items_with_values = index.reverse_index.keys()
            if search_value_for_operator:
                found_key_indexes = found_key_indexes.intersection(items_with_values)
            else:
                found_key_indexes -= set(items_with_values)
        return found_key_indexes
    
    
    def _find_one(self, search):
        item = self._find(search)
        if len(item) == 1:
            return item[0]
        elif len(item) > 1:
            raise ValueError(u'Too many value found')
        return None
    
    
    def _recursive_search(self, sub_dict, full_prop_name, value_search):
        full_prop_name = full_prop_name.split(u'.', 1)
        prop = full_prop_name[0]
        if len(full_prop_name) > 1:
            sub_props = full_prop_name[1]
            if sub_dict.get(prop, None):
                return self._recursive_search(sub_dict[prop], sub_props, value_search)
            else:
                return False
        else:
            if isinstance(value_search, dict):
                for search_key, search_lookup in value_search.iteritems():
                    if search_key == SEARCH_FILTER.NOT_EQUAL:
                        return sub_dict.get(prop, u'') != search_lookup
                    elif IndexedCollection.DISABLED_OPERATOR_IN_NON_INDEXED_COLLECTION:
                        raise NotImplementedError(u'[%s] filter is not implement in non-indexed collections due to performance' % value_search)
                    elif search_key in (SEARCH_FILTER.IN, SEARCH_FILTER.NOT_IN):
                        item_value_as_list = sub_dict.get(prop, u'').split(u',')
                        values_matches = set(search_lookup).intersection(item_value_as_list)
                        if (values_matches and search_key == SEARCH_FILTER.IN) or (not values_matches and SEARCH_FILTER.NOT_IN):
                            return True
                        else:
                            return False
                    elif search_key == SEARCH_FILTER.EXISTS:
                        item_has_field = sub_dict.get(prop, False) is not False
                        return search_lookup == item_has_field
                    else:
                        raise NotImplementedError(u'"%s" filter is not implement in %s find method' % (value_search, self.__class__.__name__))
            elif isinstance(value_search, re._pattern_type):
                return bool(value_search.search(sub_dict.get(prop, '')))
            else:
                return sub_dict.get(prop, u'') == value_search
