#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.

import logging
import os
import time

from shinken.log import FixedTimedRotatingFileHandler
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.util import to_bool
from shinkensolutions.system_tools import set_ownership, check_right_on_file

if TYPE_CHECKING:
    from typing import Optional, Dict, Any
    from logging import Logger
    from shinken.log import PartLogger


class LoggerAuthentication:
    FORMATS = [
        (u'log_time', u'[ %(log_time)s ]'),
        (u'mode', u'[ %(mode)-5s ]'),
        (u'type_request', u'[ %(type_request)-6s ]'),
        (u'result_code', u'[ RESULT:%(result_code)-3s ]'),
        (u'run_time', u'[ TIME:   %(run_time)6.0fms ]'),
        (u'user_info', u'[ USER:%(user_info)-9s ]'),
        (u'requester_ip', u'[ CALL_BY:%(requester_ip)-15s ]'),
        (u'authentication_status', u'[ AUTHENTICATED:%(authentication_status)-6s ]'),
        (u'requester_module', u'[ BY:%(requester_module)s ]'),
        (u'auth_module', u'[ AUTHENTICATED BY THE MODULE:%(auth_module)s ]'),
        (u'error', u'[ ERROR:%(error)-9s ]'),
    ]
    
    KEY_LOG_USERS__ENABLED = u'log_users__enabled'
    KEY_LOG_USERS__FILE_PATH = u'log_users__file_path'
    KEY_LOG_USERS__ADD_USER_NAME = u'log_users__add_user_name'
    
    
    def __init__(self, requester_prefix, conf, logger_error=None):
        #  type: (unicode, Any, Optional[PartLogger]) -> None
        self.log_enable = to_bool(getattr(conf, u'%s__%s' % (requester_prefix, self.KEY_LOG_USERS__ENABLED), False))
        
        _default_path = u'/var/log/shinken/%s/log_users.log' % requester_prefix
        self.log_file = getattr(conf, u'%s__%s' % (requester_prefix, self.KEY_LOG_USERS__FILE_PATH), _default_path)
        if not self.log_file or not isinstance(self.log_file, basestring):
            self.log_file = _default_path
        
        self.add_user_name = to_bool(getattr(conf, u'%s__%s' % (requester_prefix, self.KEY_LOG_USERS__ADD_USER_NAME), False))
        # This logger are use for log errors of this logger,
        # classically use the default logger of the module or daemon
        self.logger_error = logger_error
        
        self._logger = None  # type: Optional[Logger]
        
        if self.log_enable:
            self._create_path_and_add_right()
            self._init_internal_logger()
    
    
    def _init_internal_logger(self):
        # type: () -> None
        try:
            log_handler = FixedTimedRotatingFileHandler(self.log_file, u'midnight', backupCount=5)
            log_handler.setFormatter(logging.Formatter('%(message)s'))
            self._logger = logging.getLogger(u'AUTHENTICATION LOG')
            self._logger.addHandler(log_handler)
        except Exception as e:
            self.log_enable = False
            if self.logger_error:
                self.logger_error.error(u'Failed to open file [%s] -> Error [%s]' % (self.log_file, e))
            return
    
    
    def _create_path_and_add_right(self):
        # type: () -> None
        
        path_as_array = self.log_file.split(u'/')
        path_of_dirs = path_as_array[:-1]
        
        for i in range(1, len(path_of_dirs)):
            sub_path = u'/'.join(path_as_array[:i + 1])
            if not os.access(sub_path, os.F_OK):
                try:
                    os.makedirs(sub_path)
                except Exception as e:
                    if self.logger_error:
                        self.logger_error.error(u'Could not create directories [%s] -> Error [%s]' % (sub_path, e))
                    return
                
                try:
                    set_ownership(sub_path)
                except Exception as e:
                    if self.logger_error:
                        self.logger_error.error(u'Could not set user right [%s] -> Error [%s]' % (sub_path, e))
                    return
        
        if not os.access(self.log_file, os.F_OK):
            try:
                with open(self.log_file, "a+") as f:
                    f.write('')
            except Exception as e:
                if self.logger_error:
                    self.logger_error.error(u'Could not create file [%s] -> Error [%s]' % (self.log_file, e))
                return
        
        if not os.access(self.log_file, os.W_OK):
            try:
                set_ownership(self.log_file)
            except Exception as e:
                if self.logger_error:
                    self.logger_error.error(u'Could not set user right [%s] -> Error [%s]' % (self.log_file, e))
                return
    
    
    def login(self, user_uuid, user_name, requester_ip, run_time, result_code=200, requester_module=None, auth_module=None):
        # type: (unicode, unicode, unicode, float, int, Optional[unicode], Optional[unicode]) -> None
        if not self.log_enable or not self._logger:
            return
        log_data = self._build_common_log_data(
            user_uuid=user_uuid,
            user_name=user_name,
            requester_ip=requester_ip,
            run_time=run_time,
            result_code=result_code,
            requester_module=requester_module,
            auth_module=auth_module,
        )
        
        log_data.update({
            u'type_request'         : u'LOGIN',
            u'authentication_status': u'OK',
        })
        
        self._log(self._build_message(log_data))
    
    
    def login_failed(self, user_uuid, user_name, requester_ip, run_time, result_code=401, requester_module=None, auth_module=None, error=None):
        # type: (unicode, unicode, unicode, float, int, Optional[unicode], Optional[unicode], Optional[unicode]) -> None
        if not self.log_enable or not self._logger:
            return
        log_data = self._build_common_log_data(
            user_uuid=user_uuid,
            user_name=user_name,
            requester_ip=requester_ip,
            run_time=run_time,
            result_code=result_code,
            requester_module=requester_module,
            auth_module=auth_module,
            error=error,
        )
        
        log_data.update({
            u'type_request'         : u'LOGIN',
            u'authentication_status': u'FAILED',
        })
        
        self._log(self._build_message(log_data))
    
    
    def logout(self, user_uuid, user_name, requester_ip, run_time, result_code=200, requester_module=None):
        # type: (unicode, unicode, unicode, float, int, Optional[unicode]) -> None
        if not self.log_enable or not self._logger:
            return
        log_data = self._build_common_log_data(
            user_uuid=user_uuid,
            user_name=user_name,
            requester_ip=requester_ip,
            run_time=run_time,
            result_code=result_code,
            requester_module=requester_module,
        )
        
        log_data.update({
            u'type_request': u'LOGOUT',
        })
        
        self._log(self._build_message(log_data))
    
    
    def _build_common_log_data(self,
                               user_uuid=None,
                               user_name=None,
                               requester_ip=None,
                               run_time=None,
                               result_code=None,
                               requester_module=None,
                               auth_module=None,
                               error=None
                               ):
        # type: (Optional[unicode], Optional[unicode], Optional[unicode], Optional[float], Optional[int], Optional[unicode], Optional[unicode], Optional[unicode]) -> Dict
        _run_time = None
        if run_time:
            _run_time = run_time * 1000  # we want ms
        return {
            u'log_time'        : time.strftime('%Y-%m-%d %T'),
            u'user_info'       : self._build_user_info(user_uuid, user_name),
            u'mode'            : u'READ',
            u'run_time'        : _run_time,
            u'result_code'     : result_code,
            u'requester_ip'    : requester_ip,
            u'requester_module': requester_module,
            u'auth_module'     : auth_module,
            u'error'           : error
        }
    
    
    def _build_user_info(self, user_uuid, user_name):
        # type: (Optional[unicode], Optional[unicode]) -> Optional[unicode]
        
        user_info = user_uuid
        if self.add_user_name and user_name:
            if user_info:
                user_info += u'/%-6s' % user_name
            else:
                user_info = user_name
        if user_info:
            return user_info
        return None
    
    
    @staticmethod
    def _build_message(log_data):
        # type: (Dict) -> unicode
        
        res = (_format % log_data for key_data, _format in LoggerAuthentication.FORMATS if log_data.get(key_data, None))
        return u' '.join(res)
    
    
    def _log(self, txt):
        try:
            check_right_on_file(self.log_file, u'write')
            self._logger.info(txt)
        except Exception:
            self._create_path_and_add_right()
            self._init_internal_logger()
            try:
                self._logger.info(txt)
            except Exception as e:
                if self.logger_error:
                    self.logger_error.error(u'Could not write in file [%s] -> Error [%s]' % (self.log_file, e))
