#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

# ################ IMPORTANT #################
# THIS FILE IS LOADED IN APACHE WORKER PROCESS
# ############################################

#  __          __     _____  _   _ _____ _   _  _____
#  \ \        / /\   |  __ \| \ | |_   _| \ | |/ ____|
#   \ \  /\  / /  \  | |__) |  \| | | | |  \| | |  __
#    \ \/  \/ / /\ \ |  _  /| . ` | | | | . ` | | |_ |
#     \  /\  / ____ \| | \ \| |\  |_| |_| |\  | |__| |
#      \/  \/_/    \_\_|  \_\_| \_|_____|_| \_|\_____|
# This file is here for merge it was not use in this version. It will be use with new graphite module with http inventory (see http://jira.shinken-solutions.com:8080/browse/SEF-9642)

import time
import os
import ssl
import json

from graphite.logger import log
from shinkensolutions.data_hub.data_hub_factory.data_hub_factory import DataHubFactory
from shinkensolutions.http_helper import urlopen, URLError


# Fake PartLogger for datahub
class FakePartLogger(object):
    def __getattr__(self, item):
        try:
            return getattr(log, item)
        except:
            return self
    
    
    def __str__(self):
        return u''
    
    
    def __call__(self, *args, **kwargs):
        return self


class InApacheHTTPInventoryReader(object):
    def __init__(self, elements_mapping, elements_mapping_lock, log_about_f, graphite_banner):
        self._elements_mapping = elements_mapping
        self._elements_mapping_lock = elements_mapping_lock
        self._log_about = log_about_f
        self._graphite_banner = graphite_banner
        
        self._shinken_inventory_uri = u'http://localhost:52000/inventory/'
        self._shinken_inventory_timeout = 10
        self._shinken_inventory_configuration_file = u'/opt/graphite/conf/shinken_inventory.conf'
        self._inventory_update_chapter = u'%s [ INVENTORY CACHE UPDATE ]' % graphite_banner
        
        self._shared_data_hub = None
    
    
    def log_info(self, message):
        log.info(u'%s %s' % (self._inventory_update_chapter, message))
    
    
    def log_fatal(self, message):
        message = u'%s %s' % (self._inventory_update_chapter, message)
        log.info(message)
        log.exception(message)
    
    
    # We do not care about cache reset as we dump full every time
    def cache_was_reset(self):
        pass
    
    
    def inventory_data_from_shinken_lookup(self, cache_invalidation_date):
        if not self._shared_data_hub:
            datahub_log = FakePartLogger()
            try:
                self._shared_data_hub = DataHubFactory.build_in_memory_exchange_pickle_data_hub(
                    datahub_log,  # noqa: we know it's a mock here
                    data_type=u'data',
                    data_id_key_name=u'key_name',
                    daemon_name=u'httpd_graphite',
                    module_name=u''
                )
            except Exception:
                import traceback
                formatted_lines = traceback.format_exc().splitlines()
                first_line = formatted_lines[0]
                if first_line == u'None':
                    for line in traceback.format_stack():
                        self.log_fatal(u'%s' % line)
                else:
                    self.log_fatal(u'ERROR stack : %s' % first_line)
                    for line in formatted_lines[1:]:
                        self.log_fatal(u'%s' % line)
                return False
        data_id = u'inventory'
        data_hub = self._shared_data_hub
        with data_hub.get_lock(data_id):
            try:
                data_last_update = data_hub.get_last_modification_date(data_id)
            except:
                data_last_update = 0
            if data_last_update >= cache_invalidation_date:
                before = time.time()
                self._elements_mapping.update(data_hub.get_data(data_id))
                elapsed = time.time() - before
                self.log_info(u'Update host/check mapping:: the inventory was loaded from the data_hub in memory in %.3fs' % elapsed)
                return True
            
            lookup_result = self._inventory_data_from_shinken_lookup()
            if lookup_result:
                before = time.time()
                data_hub.save_data(data_id, self._elements_mapping)
                elapsed = time.time() - before
                self.log_info(u'Update host/check mapping:: the inventory was stored in the data_hub in %.3fs' % elapsed)
            return lookup_result
    
    
    def _inventory_data_from_shinken_lookup(self):
        # type: () -> bool
        t0 = time.time()
        if os.path.isfile(self._shinken_inventory_configuration_file):
            from ConfigParser import ConfigParser, NoOptionError
            parser = ConfigParser()
            parser.read(self._shinken_inventory_configuration_file)
            
            try:
                enable = parser.get('inventory', 'ENABLE') != '0'
            except NoOptionError:
                enable = True
            
            if not enable:
                return True
            
            try:
                self._shinken_inventory_uri = parser.get('inventory', 'URI').split(',')
            except NoOptionError:
                pass
            
            try:
                self._shinken_inventory_timeout = float(parser.get('inventory', 'TIMEOUT'))
            except (NoOptionError, ValueError):
                pass
        
        # Whatever we have as conf, we are working on a list of URLs
        if not isinstance(self._shinken_inventory_uri, list):
            self._shinken_inventory_uri = [self._shinken_inventory_uri]
        
        self.log_info(u'parameters : URI %s, timeout [%s]' % (self._shinken_inventory_uri, self._shinken_inventory_timeout))
        
        success = False
        sup = {}
        for url in self._shinken_inventory_uri:
            if self._get_inventory_data_from_shinken(url, sup):
                success = True
        if sup:
            self._shared_data_hub.save_data(u'sup_data', sup)
        if not success:
            return False
        
        nb_hosts = len(self._elements_mapping)
        nb_checks = 0
        for host_entry in self._elements_mapping.itervalues():
            nb_checks += (len(host_entry[u'checks']) - 1)  # remove the __host__ entry
        self.log_info(u'Update host/check mapping:: Total number of hosts/checks in mapping cache: %s/%s, Cache update took: %.3fs' % (nb_hosts, nb_checks, time.time() - t0))
        return True
    
    
    def _get_inventory_data_from_shinken(self, url, sup):
        start_time = time.time()
        url = url.strip()
        try:
            result = None
            while result is None:
                try:
                    context = ssl._create_unverified_context()  # noqa: no choice here
                    result = urlopen(url, timeout=self._shinken_inventory_timeout, context=context)
                    break
                except URLError as error:
                    if getattr(error.reason, 'errno', 0) == 111 and (time.time() - start_time) < self._shinken_inventory_timeout:
                        time.sleep(0.5)
                        continue
                    raise
        except Exception as error:
            attempt_duration = time.time() - start_time
            self.log_info(u'Unable to access URL [ %s ] after %.3fs with error [ %s ]' % (url, attempt_duration, error))
            sup.update({url: (1, start_time, u'Unable to connect after .%3fs [ %s ]' % (attempt_duration, error))})
            return False
        
        # log.info('%s Requesting URI [ %s ] to fetch inventory data' % (graphite_banner, url))
        
        try:
            result = json.load(result)
        except Exception as error:
            self.log_info(u'Unable to decode received json data from URL [ %s ] with error [ %s ]' % (url, error))
            sup.update({url: (1, start_time, u'Unable to decode data [ %s ]' % error)})
            return False
        
        if u'inventory' not in result:
            self.log_info(u'missing inventory data in reply from URL [ %s ]' % url)
            sup.update({url: (1, start_time, u'missing inventory data in server reply')})
            return False
        nb_check = nb_host = 0
        for element in result[u'inventory']:
            uuid, name, status, last_update = element
            host_uuid, check_uuid = uuid.split(u'.', 1)
            host_name, check_name = name.split(u'.', 1)
            host_entry = self._elements_mapping.get(host_name, None)
            if host_entry is None:
                host_entry = {u'uuid': host_uuid, u'checks': {}}
                self._elements_mapping[host_name] = host_entry
                self._log_about(host_name, u'%s Update host/check mapping:: new host detected: %s, uuid=%s' % (self._graphite_banner, host_name, host_uuid))
            if host_uuid != host_entry[u'uuid']:
                self._log_about(host_name, u'%s Update host/check mapping:: Old check name is detected, skipping it: %s/%s %s %s' % (self._graphite_banner, host_name, check_name, host_uuid, check_uuid))
                continue
            
            check_entry = {u'uuid': check_uuid}
            host_entry[u'checks'][check_name] = check_entry
            if check_uuid != u'__HOST__':
                nb_check = nb_check + 1
                self._log_about(host_name, u'%s Update host/check mapping:: New check add: %s/%s, uuid=%s-%s' % (self._graphite_banner, host_name, check_name, host_uuid, check_uuid))
            else:
                nb_host = nb_host + 1
        self.log_info(u'successfully updated inventory data with %s hosts / %s checks from URL [ %s ] in %.3fs' % (nb_host, nb_check, url, time.time() - start_time))
        sup.update({url: (0, start_time, u'Inventory fetched in %.3fs' % (time.time() - start_time))})
        return True
