#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import errno
import os
import socket
import struct
import subprocess
from stat import S_ISREG, S_IRUSR, S_IWUSR, S_IXUSR, S_IRGRP, S_IWGRP, S_IXGRP, S_IROTH, S_IWOTH, S_IXOTH

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.util import make_unicode
from shinkensolutions.os_helper import get_user_id_from_name, get_cur_user_id, get_cur_group_id, get_group_id_from_name, set_ownership

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Dict, List, Union
    from shinken.log import PartLogger

try:
    import fcntl
except ImportError:
    fcntl = None


class FILE_RIGHT_MODE(object):
    READ = u'read'
    WRITE = u'write'
    EXECUTE = u'execute'


class FILE_RIGHT_TARGET(object):
    USER = u'user'
    GROUP = u'group'
    OTHER = u'other'


class PATH_TYPE(object):
    DIRECTORY = u'directory'
    FILE = u'file'


# This dict use the unix mask to compute the rights
unix_rights = {
    FILE_RIGHT_TARGET.USER : {
        FILE_RIGHT_MODE.READ   : S_IRUSR,
        FILE_RIGHT_MODE.WRITE  : S_IWUSR,
        FILE_RIGHT_MODE.EXECUTE: S_IXUSR
    },
    FILE_RIGHT_TARGET.GROUP: {
        FILE_RIGHT_MODE.READ   : S_IRGRP,
        FILE_RIGHT_MODE.WRITE  : S_IWGRP,
        FILE_RIGHT_MODE.EXECUTE: S_IXGRP
        
    },
    FILE_RIGHT_TARGET.OTHER: {
        FILE_RIGHT_MODE.READ   : S_IROTH,
        FILE_RIGHT_MODE.WRITE  : S_IWOTH,
        FILE_RIGHT_MODE.EXECUTE: S_IXOTH
    }
}


def check_right_on_path(path, mode=FILE_RIGHT_MODE.READ, path_type=PATH_TYPE.DIRECTORY, user=None, group=None):
    # type: (unicode, unicode, unicode, Optional[unicode], Optional[unicode]) -> bool
    # Use this function to check the rights on folder and file.
    # path: the path to check
    # mode: which mode you want for your path : 'read', 'write', 'execute'
    # path_type: is your path is a directory or a file
    # user: on which user check right ? if None, it will use the current process’s effective user id
    # group: on which group check right ? if None, it will use the current process’s effective group id
    # return: True if rights are OK. Raise if you can access to your file
    
    path = os.path.join(path)
    # First check if the path exists
    if not os.path.exists(path):
        raise OSError(errno.ENOENT, u'The path doesn\'t exists', path)
    
    # Is the target is the good one ?
    if (path_type == PATH_TYPE.DIRECTORY and not os.path.isdir(path)) or \
            (path_type == PATH_TYPE.FILE and not os.path.isfile(path)):
        raise OSError(errno.ENOENT, u'This is not a %s  : ' % path_type, path)
    
    # Will check if we use the user, the group or other
    user_uid = get_user_id_from_name(user) if user else get_cur_user_id()
    
    if user_uid == 0:
        # You're root ? ... ok you can go
        return True
    
    group_gid = get_group_id_from_name(group) if group else get_cur_group_id()
    file_stat = os.stat(path)
    
    if user_uid == file_stat.st_uid:
        key_used = FILE_RIGHT_TARGET.USER
    elif group_gid == file_stat.st_gid:
        key_used = FILE_RIGHT_TARGET.GROUP
    else:
        key_used = FILE_RIGHT_TARGET.OTHER
    
    # Now will check the rights with the good mask
    if bool(file_stat.st_mode & unix_rights[key_used][mode]):
        return True
    else:
        raise OSError(errno.EACCES, u'User %s can not access to the %s with the %s mode ' % (user, path_type, mode), path)


def check_right_on_file(path, mode=FILE_RIGHT_MODE.READ, user=None, group=None):
    # type: (unicode, unicode, Optional[unicode], Optional[unicode]) -> bool
    # Use this function to check the rights on file.
    # path: the path to check
    # mode: which mode you want for your path : 'read', 'write', 'execute'
    # user: on which user check right ? if None, it will use the current process’s effective user id
    # group: on which group check right ? if None, it will use the current process’s effective group id
    # return: True if rights are OK. Raise if you can access to your file
    return check_right_on_path(path, mode, PATH_TYPE.FILE, user, group)


def check_right_on_directory(path, mode=FILE_RIGHT_MODE.READ, user=None, group=None):
    # type: (unicode, unicode, Optional[unicode], Optional[unicode]) -> bool
    # Use this function to check the rights on folder.
    # path: the path to check
    # mode: which mode you want for your path : 'read', 'write', 'execute'
    # user: on which user check right ? if None, it will use the current process’s effective user id
    # group: on which group check right ? if None, it will use the current process’s effective group id
    # return: True if rights are OK. Raise if you can access to your folder
    return check_right_on_path(path, mode, PATH_TYPE.DIRECTORY, user, group)


def create_tree(path, user=u'shinken', group=u'shinken', mode=0o755):
    # type: (unicode, unicode, unicode, int) -> None
    # Warning! the mode will be overwritten by os.umask if set
    drive, path = os.path.splitdrive(path)
    full_path = [p for p in path.split(os.sep) if p]
    # to manage relative path or not
    parents = (u'%s%s' % (drive, os.sep)) if os.path.isabs(path) else ''
    for folder_name in full_path:
        current_path = os.path.join(parents, folder_name)
        
        if not os.path.exists(current_path):
            try:
                check_right_on_directory(parents, FILE_RIGHT_MODE.WRITE)
            except OSError:
                raise OSError(errno.EACCES, u'Cannot create the full tree. Permission denied ', parents)
            os.mkdir(current_path, mode)
            set_ownership(current_path, user, group)
        
        try:
            check_right_on_directory(current_path, FILE_RIGHT_MODE.READ, user, group)
            check_right_on_directory(current_path, FILE_RIGHT_MODE.EXECUTE, user, group)
        except OSError:
            raise OSError(errno.EACCES, u'The path already exists but cannot jump in. Permission denied ', current_path)
        
        # Now, next level
        parents = current_path


def run_command(command):
    # type: (unicode) -> unicode
    _, stdout, stderr = run_command_with_return_code(command)
    return u'%s%s' % (stdout, stderr)


def run_command_with_return_code(command, shell=True, env=None):
    # type: (Union[List[str], unicode], bool, Optional[Dict]) -> (int, unicode, unicode)
    if env is None:
        _process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=shell)
    else:
        _process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=shell, env=env)
    stdout, stderr = _process.communicate()
    return _process.returncode, make_unicode(stdout), make_unicode(stderr)


def is_regular_file(path):
    # type: (unicode) -> bool
    if os.path.exists(path):
        mode = os.stat(path).st_mode
        return S_ISREG(mode)
    return False


def get_linux_local_addresses(_logger):
    # type: (PartLogger) -> List[unicode]
    
    try:
        rc, stdout, stderr = run_command_with_return_code(u'hostname -I')
    except Exception as exp:
        _logger.info(u'Cannot use the hostname -I call for linux (%s), trying to guess local addresses' % exp)
        stdout = ''
    buf = stdout.decode(u'utf8', u'ignore').strip()
    res = [s.strip() for s in buf.split(' ') if s.strip()]
    
    # Some system like in alpine linux that don't have hostname -I call so try to guess
    if len(res) == 0:
        _logger.debug(u'Cannot use the hostname -I call for linux, trying to guess local addresses')
        for interface_prefix in (u'bond', u'eth', u'venet', u'wlan'):
            for interface_counter in range(0, 10):
                interface_name = u'%s%d' % (interface_prefix, interface_counter)
                try:
                    addr = _get_ip_address(interface_name)
                    res.append(addr)
                except IOError:  # no such interface
                    pass
    res = sorted(res, cmp=_sort_local_addresses)
    return res


def _get_ip_address(interface_name):
    # type: (unicode) -> unicode
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    return socket.inet_ntoa(fcntl.ioctl(
        s.fileno(),
        0x8915,  # SIOCGIFADDR
        struct.pack(u'256s', interface_name[:15])
    )[20:24])


def _sort_local_addresses(addr1, addr2):
    # type: (unicode, unicode) -> int
    addr1_is_192 = addr1.startswith(u'192.')
    addr2_is_192 = addr2.startswith(u'192.')
    addr1_is_10 = addr1.startswith(u'10.')
    addr2_is_10 = addr2.startswith(u'10.')
    addr1_is_172 = addr1.startswith(u'172.')
    addr2_is_172 = addr2.startswith(u'172.')
    addr1_is_127 = addr1.startswith(u'127.')
    addr2_is_127 = addr2.startswith(u'127.')
    
    # lower is better
    addr1_order = 4
    if addr1_is_192:
        addr1_order = 1
    elif addr1_is_172:
        addr1_order = 2
    elif addr1_is_10:
        addr1_order = 3
    if addr1_is_127:
        addr1_order = 5
    addr2_order = 4
    if addr2_is_192:
        addr2_order = 1
    elif addr2_is_172:
        addr2_order = 2
    elif addr2_is_10:
        addr2_order = 3
    if addr2_is_127:
        addr2_order = 5
    
    if addr1_order > addr2_order:
        return 1
    elif addr1_order < addr2_order:
        return -1
    return 0


def create_symlink_from_os_version(symlink_path, os_version, user=u'shinken'):
    # type: (unicode, unicode, unicode) -> bool
    target_path = u'%s_%s' % (symlink_path, os_version)
    # The link target dos not exists
    if not os.path.exists(target_path):
        raise OSError(u'The link target does not exists')
    
    if os.path.islink(symlink_path):
        old_target_name = os.path.basename(os.readlink(symlink_path))
        target_name = os.path.realpath(target_path)
        # There is already a link pointing the wanted destination
        if target_name == old_target_name:
            return False
    
    # We want to override the existing file / link
    if os.path.exists(symlink_path):
        # Remove the symlink_path, but handle the symlinks. Will NOT remove the target if the symlink_path is a symbolic link
        os.unlink(symlink_path)
    
    os.symlink(target_path, symlink_path)
    set_ownership(symlink_path, user, user, is_link=True)
    return True
