#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.


import copy
import time
from threading import RLock
from shinken.webui import bottlecore as bottle

from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Optional, List


class ArbiterTraceManager(object):
    def __init__(self):
        # type: () -> None
        self._arbiter_traces = []
        self._arbiter_traces_lock = RLock()
        self._current_arbiter_trace = None
        
        self.raw_logger = LoggerFactory.get_logger()
        self.logger_configuration = self.raw_logger.get_sub_part(u'CONFIGURATION')
        self.logger_about_arbiters = self.logger_configuration.get_sub_part(u'ARBITERS')
    
    
    def get_current_arbiter_trace(self):
        # type: () -> Dict
        return self._current_arbiter_trace
    
    
    @staticmethod
    def _trace_have_expire(trace, current_time):
        # type: (Dict, int) -> bool
        expire_period = trace[u'expire_period']
        return current_time >= trace[u'insert_time'] + expire_period
    
    
    # see SEF-5766
    def _oldest_trace_from_spare_and_master(self, trace):
        # type: (Dict) -> Optional[Dict]
        return next((i for i in self._arbiter_traces if not trace[u'identifier'] != i[u'identifier'] and trace.get(u'master_arbiter_uuid', u'--') == i.get(u'master_arbiter_uuid', u'-') and trace[u'insert_time'] < i[u'insert_time']), None)
    
    
    # Look at SEF-8279: get traces, and put into it the now entry so all the check can be done
    # with only look at our entry, and not look at the "checker/requester" local time (can be wrong)
    def get_up_to_date_arbiter_traces(self):
        # type: () -> List[Dict]
        with self._arbiter_traces_lock:
            up_to_date_traces = self._arbiter_traces[:]  # copy it
            for trace in up_to_date_traces:
                trace[u'now'] = int(time.time())
        return up_to_date_traces
    
    
    def set_arbiter_trace(self, arbiter_trace, static_context):
        # type: (Dict, Dict) -> Dict
        with self._arbiter_traces_lock:
            ##
            
            # 'insert_time' is used from the web API to delete entries inserted more than 3 x check_interval seconds ago
            # (See self._arbiter_traces_cleanup())
            now = time.time()
            arbiter_trace[u'insert_time'] = int(now)
            arbiter_trace[u'diff_time_with_arbiter'] = now - arbiter_trace[u'arbiter_time']
            arbiter_trace[u'daemon_version'] = static_context[u'current_version']
            arbiter_trace[u'ip'] = bottle.request.environ.get('REMOTE_ADDR', '(unknown)')
            # Maybe the trace is from a too old arbiter, skip this
            if u'expire_period' not in arbiter_trace:
                return copy.deepcopy(arbiter_trace)
            
            identifier = arbiter_trace[u'identifier']
            arbiter_name = arbiter_trace[u'name']
            spare = arbiter_trace[u'spare']
            master_arbiter_uuid = arbiter_trace[u'master_arbiter_uuid']
            expire_period = arbiter_trace[u'expire_period']
            architecture_name = arbiter_trace.get(u'architecture_name', u'(unknown)')
            
            old_index, old_entry = next(((index, trace) for index, trace in enumerate(self._arbiter_traces) if trace[u'identifier'] == identifier), (None, None))
            self.logger_about_arbiters.debug(u'Arbiter is pinging us with a trace %s' % arbiter_trace)
            if old_entry is not None:
                old_name = old_entry[u'name']
                old_was_spare = old_entry[u'spare']
                old_architecture_name = old_entry.get(u'architecture_name', u'(unknown)')
                # Warn about some special cases:
                # * same name, but not same spare => arbiter just changed
                # * other name, same spare => just changed name maybe
                # * other name, other spare => WARNING, spare seems to have the same server.uuid
                if old_name == arbiter_name and spare != old_was_spare:
                    self.logger_about_arbiters.info(
                        u'The Arbiter %s ( Architecture=%s / MASTER arbiter server uuid=%s ) did contact us and did change it\'s spare role (spare=%s => spare=%s)' % (architecture_name, arbiter_name, master_arbiter_uuid, old_was_spare, spare))
                elif spare == old_was_spare and old_name != arbiter_name:
                    spare_str = u'spare' if spare else u'not spare'
                    self.logger_about_arbiters.info(u'The %s arbiter ( Architecture=%s / MASTER arbiter server uuid=%s ) did change its name %s => %s' % (architecture_name, spare_str, master_arbiter_uuid, old_name, arbiter_name))
                elif spare != old_was_spare and old_name != arbiter_name:
                    self.logger_about_arbiters.warning(
                        u'Two arbiters ( Architecture=%s name=%s spare=%s ) & ( Architecture=%s name=%s spare=%s) have the same server_uuid ( %s on the file "/var/lib/shinken/server.uuid" ), this can lead to undetected arbiter conflicts.' % (
                            old_architecture_name, old_name, old_was_spare, architecture_name, arbiter_name, spare, identifier))
                # In all cases, we are updating the entry
                self._arbiter_traces[old_index] = arbiter_trace
            else:
                self.logger_about_arbiters.info(u'A new arbiter did connect ( Architecture=%s name=%s spare=%s expiration=%ds) ( MASTER arbiter server uuid=%s )' % (architecture_name, arbiter_name, spare, expire_period, master_arbiter_uuid))
                # It is a new entry.
                self._arbiter_traces.append(arbiter_trace)
            
            # before exit, be sure to clean it
            self.arbiter_traces_cleanup()
            
            # Caller may need to give to arbiter what we did finally set locally
            # IMPORTANT: as we will json it in HTTP, we NEED to deepcopy it
            #            while we have the lock
            self._current_arbiter_trace = arbiter_trace
            return copy.deepcopy(arbiter_trace)
    
    
    ##
    # Will remove every entry of arbiters older than the arbiter 2 * check_interval period.
    # If all entries have expired, keep the last one.
    def arbiter_traces_cleanup(self):
        # type: () -> None
        with self._arbiter_traces_lock:
            if len(self._arbiter_traces) < 1:
                # Not configured (0) or configured by one (1), no need to go further.
                return
            
            len_before_loop = len(self._arbiter_traces)
            
            current_time = int(time.time())
            new_traces = []
            for trace in self._arbiter_traces:
                master_arbiter_uuid = trace[u'master_arbiter_uuid']
                if self._trace_have_expire(trace, current_time):
                    expire_period = trace[u'expire_period']
                    self.logger_about_arbiters.info(u'Last connexion from arbiter %s ( spare=%s ) ( MASTER arbiter server uuid=%s ) is too old (%.1fs ago > %ds for expiration)' % (
                        trace[u'name'], trace[u'spare'], master_arbiter_uuid, current_time - trace[u'insert_time'], expire_period))
                    continue
                
                # If it's our spare (same master_arbiter_uuid), clean if oldest
                if self._oldest_trace_from_spare_and_master(trace):
                    self.logger_about_arbiters.info(
                        u'Last connexion from arbiter %s ( spare=%s ) ( MASTER arbiter server uuid=%s ) is older than the earlier arbiter with the master configuration' % (trace['name'], trace['spare'], master_arbiter_uuid))
                    continue
                
                # trace seems to be still valid
                new_traces.append(trace)
            
            ##
            # Now new_arbiters may be empty because all arbiters may have expired.
            # However, we take for good the very last element updated, and keep it.
            if len(new_traces) == 0:
                arbiters_sorted = sorted(self._arbiter_traces, key=lambda k: k[u'insert_time'])
                kept_trace = arbiters_sorted.pop()
                new_traces.append(kept_trace)
                # Only show log if we did really remove some trace
                if len_before_loop != 1:
                    master_arbiter_uuid = kept_trace[u'master_arbiter_uuid']
                    self.logger_about_arbiters.info(u'The connexion from arbiter %s ( spare=%s ) ( MASTER arbiter server uuid=%s ) is the last one we have, so we are keeping it ( age=%.3fs )' % (
                        kept_trace[u'name'], kept_trace[u'spare'], master_arbiter_uuid, current_time - kept_trace[u'insert_time']))
            
            self._arbiter_traces = new_traces
