#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import os

import time

from shinken.action import Action, ACTION_TYPES
from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.property import BoolProp, IntegerProp, FloatProp, StringProp

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, List, Dict
    from shinken.objects.schedulingitem import SchedulingItem

NO_END_VALIDITY = -1
MONITORING_CHECK_CONSUME_DEBUG_FLAG = os.environ.get('MONITORING_CHECK_CONSUME_DEBUG_FLAG', '0') == '1'


class CHECK_CAUSE(object):
    SCHEDULE = u'schedule'  # type: unicode
    FORCE = u'force'  # type: unicode
    RETRY = u'retry'  # type: unicode
    DEPENDENCY = u'dependency'  # type: unicode


# IMPORTANT : if you change the values, SANATIZE the retention data!
# IMPORTANT2: if you are adding a new state, FILL THE RETENTION DATA COMMENT about
#        what you are doing with this state. If you don't know, don't commit!
class CHECK_STATUS(object):
    SCHEDULED = u'scheduled'  # type: unicode
    INPOLLER = u'inpoller'  # type: unicode
    TIMEOUT = u'timeout'  # type: unicode
    WAITCONSUME = u'waitconsume'  # type: unicode
    WAITDEP = u'waitdep'  # type: unicode
    HAVETORESOLVEDEP = u'havetoresolvedep'  # type: unicode
    ALL_DEP_ARE_FINISH = u'all_dep_are_finish'  # type: unicode
    MUST_RECREATE_DEP = u'must_recreate_dep'  # type: unicode
    ZOMBIE = u'zombie'  # type: unicode
    # Only in POLLER/REACTIONNER:
    READY_TO_RUN = u'ready_to_run'  # type: unicode
    LAUNCHED = u'launched'  # type: unicode


class Check(Action):
    # AutoSlots create the __slots__ with properties and
    # running_properties names
    
    # ## DO NOT ENABLE AUTOSLOTS, we need the __dict__
    # __metaclass__ = AutoSlots
    
    my_type = ACTION_TYPES.CHECK  # type: unicode
    
    # ########### WARNING: DO NOT ADD a property without look at retention part!
    # if you don't know what it means, ask Jean and don't commit.
    properties = {
        'is_a'                                 : StringProp(default=ACTION_TYPES.CHECK),
        'type'                                 : StringProp(default=''),
        '_in_timeout'                          : BoolProp(default=False),
        'status'                               : StringProp(default=''),
        'exit_status'                          : IntegerProp(default=3),
        'state'                                : IntegerProp(default=0),
        'output'                               : StringProp(default=''),
        'long_output'                          : StringProp(default=''),
        'ref'                                  : IntegerProp(default=-1),
        't_to_go'                              : IntegerProp(default=0),
        'original_t_to_go'                     : IntegerProp(default=0),
        'depend_on'                            : StringProp(default=[]),
        'dep_check'                            : StringProp(default=[]),
        'check_time'                           : IntegerProp(default=0),
        'execution_time'                       : FloatProp(default=0.0),
        'u_time'                               : FloatProp(default=0.0),
        's_time'                               : FloatProp(default=0.0),
        'cpu_time'                             : FloatProp(default=0.0),
        'perf_data'                            : StringProp(default=''),
        'check_type'                           : IntegerProp(default=0),
        'poller_tag'                           : StringProp(default='None'),
        'reactionner_tag'                      : StringProp(default='None'),
        'env'                                  : StringProp(default={}),
        'internal'                             : BoolProp(default=False),
        'module_type'                          : StringProp(default='fork'),
        'executor_id'                          : StringProp(default='none'),
        'from_trigger'                         : BoolProp(default=False),
        'check_interval'                       : IntegerProp(default=0),
        'shell_execution'                      : BoolProp(default=False),
        'cause'                                : StringProp(default=CHECK_CAUSE.SCHEDULE),
        'state_validity_period'                : IntegerProp(default=None),
        'state_validity_period_when_go_in_soft': IntegerProp(default=None),
        'warning_threshold_cpu_usage'          : IntegerProp(default=-1),
        'timeout'                              : IntegerProp(default=10),
        'is_dummy_check'                       : BoolProp(default=False),
    }
    
    # these fields have links to other objects, so we CANNOT serialize/retention them, and they must be recreated
    properties_to_remove_for_retention = ('ref', 'depend_on', 'depend_on_me')
    
    
    def __init__(self, status, command, ref, t_to_go,
                 depend_on_me=None,
                 uuid=None,
                 timeout=10,
                 timeout_from=None,
                 poller_tag=u'None',
                 reactionner_tag=u'None',
                 env=None,
                 module_type=u'fork',
                 from_trigger=False,
                 dependency_check=False,
                 shell_execution=False,
                 command_name=u'MISSING_NAME',
                 cause=CHECK_CAUSE.SCHEDULE,
                 warning_threshold_cpu_usage=-1,
                 is_dummy_check=False,
                 is_internal=False,
                 ):
        # type: (unicode, unicode, Optional[SchedulingItem], Optional[float], Optional[Check], Optional[int], float, Optional[unicode], unicode, unicode, Dict[unicode], unicode, bool, bool, bool, unicode, unicode, float, bool, bool) -> None
        # print "IN CHECK id", id
        if env is None:
            env = {}
        super(Check, self).__init__(uuid, command, command_name, timeout, module_type, timeout_from=timeout_from, t_to_go=t_to_go)
        self.is_a = ACTION_TYPES.CHECK  # type: unicode
        self.type = u''  # type: unicode
        self.cause = cause  # type: unicode
        
        self.warning_threshold_cpu_usage = warning_threshold_cpu_usage
        self.check_interval = getattr(ref, 'check_interval', 0)  # type: int
        self._in_timeout = False  # type: bool
        self.timeout = timeout  # type: float
        self.status = status  # type: unicode
        self.exit_status = 3  # type: int
        self.command = command  # type: unicode
        self.output = u''  # type: unicode
        self.long_output = u''  # type: unicode
        self.ref = ref  # type: Optional[SchedulingItem]
        # self.ref_type = ref_type
        
        # Father check
        self.depend_on = []  # type: List[Check]
        if depend_on_me is None:
            self.depend_on_me = []  # type: List[Check]
        else:
            self.depend_on_me = [depend_on_me]  # type: List[Check]
        self.check_time = 0  # type: float
        self.execution_time = 0  # type: float
        # user cpu time
        self.u_time = 0  # type: float
        # system cpu time
        self.s_time = 0  # type: float
        # system + user cpu time
        self.cpu_time = 0  # type: float
        self.perf_data = ''  # type: unicode
        # which kind of check result? 0=active 1=passive
        self.check_type = 0  # type: int
        self.poller_tag = poller_tag  # type: unicode
        self.reactionner_tag = reactionner_tag  # type: unicode
        self.module_type = module_type  # type: unicode
        self.env = env
        
        # If it's a business rule, manage it as a special check
        self.internal = is_internal  # type: bool
        self.from_trigger = from_trigger
        self.dependency_check = dependency_check
        self.shell_execution = shell_execution
        self.state_validity_period = None
        self.state_validity_period_when_go_in_soft = None
        
        self._result_modulations_are_applied = False  # do not apply result modulation twice
        self.is_dummy_check = is_dummy_check
        
        if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
            if ref:
                logger.info('[monitoring] Creating a check for %s (check uuid=%s)' % (self.ref.get_full_name(), self.get_uuid()))
            else:
                logger.info('[monitoring] Creating a dummy check from check uuid=%s' % (self.get_uuid()))
        
        # We must save in the Check at which epoch is the Check inserted in the epoch index,
        # so we can quickly add to a new epoch, WITHOUT FORGETTING TO REMOVE THE OLD PLACE ^^
        self._current_indexed_at_epoch = None
        
        # ########### WARNING: DO NOT ADD a property without look at retention part!
        # if you don't know what it means, ask Jean and don't commit.
    
    
    # Call by pickle when load retention
    def __setstate__(self, state):
        #  IMPORTANT: #SEF-8804 19/07/2021: maybe the object came from an old retention without uuid (was id)
        #                                   so we generate a new one
        if not hasattr(self, '_uuid'):
            self._generate_my_uuid()
        
        for prop_name, value in state.iteritems():
            if prop_name == 'dep_check':
                prop_name = 'depend_on_me'
            setattr(self, prop_name, value)
        
        # For manage revert on patch v02.07.06/02.08.01
        if not hasattr(self, 'id'):
            self._generate_old_id()
        
        if not hasattr(self, 'state_validity_period'):
            self.state_validity_period = 5 * 60
        if not hasattr(self, 'state_validity_period_when_go_in_soft'):
            self.state_validity_period_when_go_in_soft = 5 * 60
        if not hasattr(self, 'cpu_time'):
            self.cpu_time = 0
        # In all cases, a retention value for _current_indexed_at_epoch
        # is not interesting
        self._current_indexed_at_epoch = None
    
    
    def get_printable_name(self):
        # type: () -> str
        if self.ref is not None:
            return '(check-execution uuid=%s on %s=%s, on cluster=%s, status=%s, index_time=%s)' % (self.get_uuid(), self.ref.my_type, self.ref.get_full_name(), self.internal, self.status, self._current_indexed_at_epoch)
        else:
            return '(check-execution uuid=%s on no Host or Check, on cluster=%s, status=%s, index_time=%s)' % (self.get_uuid(), self.internal, self.status, self._current_indexed_at_epoch)
    
    
    def can_be_indexed_for_execution(self):
        # type: () -> bool
        return self.status == CHECK_STATUS.SCHEDULED
    
    
    def set_cause(self, cause):
        # type: (unicode) -> None
        self.cause = cause
    
    
    def get_current_index_at_epoch(self):
        return self._current_indexed_at_epoch
    
    
    # We are inserted in the global check index with our time, so save it
    def set_current_index_at_epoch(self):
        # type: () -> None
        self._current_indexed_at_epoch = int(self.t_to_go)
    
    
    def unset_current_index_at_epoch(self):
        # type: () -> None
        self._current_indexed_at_epoch = None
    
    
    def is_indexed_at_epoch(self):
        # type: () -> bool
        return self._current_indexed_at_epoch is not None
    
    
    def copy_shell(self):
        """return a copy of the check but just what is important for execution
        So we remove the ref and all
        """
        # We create a dummy check with nothing in it, just defaults values
        dummy_check = Check('', '', None, None, uuid=self.get_uuid(), shell_execution=self.shell_execution, command_name=self.command_name, cause=self.cause, is_dummy_check=True)
        return self.minimal_copy_for_exec(dummy_check)
    
    
    def get_time_to_be_start(self):
        # type: () -> int
        return int(self.t_to_go)
    
    
    def __str__(self):
        return u'Check %s status:%s command:%s exit_status:%s t_to_go:%s ref:%s' % \
               (self.get_uuid(), self.status, self.command.strip(), self.exit_status, self.t_to_go, self.ref)
    
    
    def set_type_passive(self):
        # type: () -> None
        self.check_type = 1
    
    
    def is_dependent(self):
        return self.dependency_check
    
    
    # Look if all output* strings are in unicode-utf8
    def assert_unicode_strings(self):
        # if str, go in unicode
        properties = ('output', 'long_output', 'perf_data')
        for prop in properties:
            value = getattr(self, prop)
            if isinstance(value, str):
                new_value = value.decode('utf8', 'ignore')
                setattr(self, prop, new_value)
    
    
    def apply_result_modulations(self, result_modulations):
        if self._result_modulations_are_applied:
            return
        
        self._result_modulations_are_applied = True  # do not apply result twice
        # Before setting state, modulate them
        for rm in result_modulations:
            if rm is not None:
                module_return = rm.module_return(self.exit_status, self.output, self.long_output)
                if module_return != self.exit_status:
                    self.exit_status = module_return
                    break
    
    
    # A check can be consumed only if:
    # * it's a waitconsume (obvious)
    # * it's a waitdep that all dependant checks are done (so depend_on is now void)
    def can_be_consume(self):
        # type: () -> bool
        return self.status == CHECK_STATUS.WAITCONSUME or self.status == CHECK_STATUS.ALL_DEP_ARE_FINISH
    
    
    def consume_is_waiting_for_dependencies(self):
        return self.status == CHECK_STATUS.WAITDEP or self.status == CHECK_STATUS.ALL_DEP_ARE_FINISH
    
    
    def launch_consume(self):
        # type: () -> None
        item = self.ref
        item.consume_result(self)
    
    
    # We need to launch dependency checks if:
    # * we are in a bad state
    # * we are not already in a waiting dependency phase
    def need_to_launch_dependency_checks(self):
        # type: () -> bool
        return self.exit_status != 0 and self.status == CHECK_STATUS.WAITCONSUME
    
    
    # if another check is asking for a dependency, maybe we are a valid check that can be executed
    # SCHEDULED => OK, change the time to go now
    # INPOLLER => already exiting, just register the check as a son
    # TIMEOUT/WAITCONSUME => just in time!  just register the check as a son
    # WAITDEP => no problem, will be consumed when all dep wil be done, so can register as a son, don't have to modify time
    # HAVETORESOLVEDEP => transient state, cannot be here
    # ALL_DEP_ARE_FINISH => just in time! just register the check as a son
    # MUST_RECREATE_DEP => possible, just register the check, don't have to modify time
    # ZOMBIE => NO! will die
    # READY_TO_RUN/LAUNCHED => impossible, only available in POLLER/REACTIONNER
    def can_be_hot_hook_for_dependency(self):
        # type: () -> bool
        return self.status in (CHECK_STATUS.SCHEDULED, CHECK_STATUS.INPOLLER, CHECK_STATUS.TIMEOUT,
                               CHECK_STATUS.WAITCONSUME, CHECK_STATUS.WAITDEP, CHECK_STATUS.HAVETORESOLVEDEP,
                               CHECK_STATUS.ALL_DEP_ARE_FINISH, CHECK_STATUS.MUST_RECREATE_DEP)
    
    
    # Someone is asking us to go ASAP, and as the scheduler have a
    # time indexed list of checks, we must give back our old time
    # so the scheduler will be able to find the old index
    def force_time_change(self, new_time):
        # type: (float) -> float
        # be sure the new time is an int
        new_time = int(new_time)
        old_time = self.t_to_go
        self.t_to_go = new_time
        self.original_t_to_go = new_time
        self.set_cause(CHECK_CAUSE.FORCE)
        
        return old_time
    
    
    # Set our values like if the poller really give us something
    def simulate_execution_return(self, exit_status, output, check_time, execution_time, long_output=''):
        # type: (int, unicode, float, float, unicode) -> None
        self.exit_status = exit_status
        self.output = output
        self.long_output = long_output
        self.check_time = check_time
        self.execution_time = execution_time
        self.status = CHECK_STATUS.WAITCONSUME
    
    
    def hot_hook_for_dependency(self, son_check):
        # type: (Check) -> bool
        self.depend_on_me.append(son_check)
        was_rescheduled = False
        if self.status == CHECK_STATUS.SCHEDULED:
            new_time = int(time.time())
            self.force_time_change(new_time)
            was_rescheduled = True
            # This check is rescheduled because of dependency check, so we must flag it as dependency_check and so it won't count for retry
            # See SEF-9343
            self.dependency_check = True
        return was_rescheduled
    
    
    def register_dependency_checks(self, checks_uuids):
        self.status = CHECK_STATUS.WAITDEP
        for check_uuid in checks_uuids:
            self.depend_on.append(check_uuid)
            # Ok, no more need because checks are not take by host/service, and not returned
    
    
    # We did have a father check that just finish, remove it from our list
    def remove_dependency_check(self, dependency_check_uuid):
        if self.status != CHECK_STATUS.WAITDEP:
            logger.error('ERROR: trying to remove a dependency on a check %s that is not waiting for dependency (current state=%s).' % (dependency_check_uuid, self.status))
            return
        if dependency_check_uuid in self.depend_on:
            self.depend_on.remove(dependency_check_uuid)
        if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
            logger.info("[monitoring] The check id=%s on %s is removing the father check %s and now still have %s father checks" % (self.get_uuid(), self.ref.get_full_name(), dependency_check_uuid, len(self.depend_on)))
        # No more deps? congrats, we will be able to be consumed
        if len(self.depend_on) == 0:
            self.status = CHECK_STATUS.ALL_DEP_ARE_FINISH
            if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
                logger.info("[monitoring] The check id=%s on %s do not have any more dependency. Going to is state ALL_DEP_ARE_FINISH." % (self.get_uuid(), self.ref.get_full_name()))
    
    
    # Now our host/service did fully consume us, we need to :
    # * go zombie if no-one is waiting for us
    # * go have to resolve dep is son checks are waiting for us (so they can be unlocked)
    #   -> so we are removing ourselves from them, and we go zombie
    def set_fully_consumed(self):
        # we must warn our sons that we did finish
        if len(self.depend_on_me) != 0:
            my_id = self.get_uuid()
            self.status = CHECK_STATUS.HAVETORESOLVEDEP  # yes it won't last, but it's better for reader to understand
            for dependent_checks in self.depend_on_me:
                # Ok, now dependent will no more wait c
                dependent_checks.remove_dependency_check(my_id)
        # In all case we did finish this check, go zombie
        # REMOVE OLD DEP CHECK -> zombie
        self.status = CHECK_STATUS.ZOMBIE
        
        # Protection/ASSERT about a bug in check index: maybe it was still in the index when go zombie, if so, should not be possible
        if self._current_indexed_at_epoch is not None:
            logger.error(
                'JOB-EXECUTION FAST INDEX :: The check %s is set as zombie (to be deleted) without being cleaned in the time index (name=%s, was indexed at %s). The index will automatically clean the check in maximum 60s, but this is not a normal situation. Please report to your support.' % (
                    self.get_uuid(), self.ref.get_full_name(), self._current_indexed_at_epoch))
    
    
    def is_zombie(self):
        return self.status == CHECK_STATUS.ZOMBIE
    
    
    def _get_core_object_for_retention(self):
        new_object = self.copy_shell()  # already manage creator call
        # now copy interesting property, but not the one we know we cannot serialize
        for prop, value in self.__dict__.iteritems():
            if prop in self.properties_to_remove_for_retention:
                continue
            setattr(new_object, prop, value)
        
        # Be sure the problematic properties are removed
        for prop in self.properties_to_remove_for_retention:
            delattr(new_object, prop)
        return new_object
    
    
    # #### Retention part:
    # For each state, what we are doing for retention:
    # => in all cases, remove the 'ref', we are not able to serialize it
    #     SCHEDULED => drop, we are able to recreate it
    #     INPOLLER => drop, we will recreate it, poller will give us back, but it's not a problem, it will be dropped
    #     TIMEOUT/WAITCONSUME => just returned, can be consumed but remove dependency if there are (son will recreate it anyway)
    #     WAITDEP  => we did consume it once, so cannot be lost. Must be reconsume but:
    #                  - do not take output/result from it (already done)
    #                  - recreate the dependency (like we did for go in WAITDEP)
    #                  => so we are setting it to a new state "MUST_RECREATE_DEP"
    #     ALL_DEP_ARE_FINISH => => we did consume it once, so cannot be lost. Must be reconsume but:
    #                  - do not take output/result from it (already done)
    #     MUST_RECREATE_DEP => can be in retention if the daemon is stopped very just before the first turn, so nothing to do
    #     HAVETORESOLVEDEP => transient case (just for workflow to be clear in a draft), cannot be access in retention
    #     ZOMBIE => just drop, we already did finish with it
    #     READY_TO_RUN/LAUNCHED => impossible, only available in POLLER/REACTIONNER
    def get_object_for_retention(self):
        # First look at state, because a lot of them just return nothing
        to_drop_status = (CHECK_STATUS.SCHEDULED, CHECK_STATUS.INPOLLER, CHECK_STATUS.ZOMBIE)
        to_set_in_recreate_status = (CHECK_STATUS.WAITDEP, CHECK_STATUS.MUST_RECREATE_DEP)
        impossible_status = (CHECK_STATUS.HAVETORESOLVEDEP, CHECK_STATUS.READY_TO_RUN, CHECK_STATUS.LAUNCHED)  # BUG!
        just_drop_dependency = (CHECK_STATUS.WAITCONSUME, CHECK_STATUS.TIMEOUT, CHECK_STATUS.ALL_DEP_ARE_FINISH,)
        
        status = self.status
        if status in to_drop_status:
            return None
        elif status in to_set_in_recreate_status:
            retention_object = self._get_core_object_for_retention()
            # drop all about dependency
            retention_object.depend_on = []
            retention_object.depend_on_me = []
            retention_object.status = CHECK_STATUS.MUST_RECREATE_DEP
            return retention_object
        elif status in just_drop_dependency:
            retention_object = self._get_core_object_for_retention()
            # drop all about dependency
            retention_object.depend_on = []
            retention_object.depend_on_me = []
            # No need to set a new status
            return retention_object
        elif status in impossible_status:
            logger.error('The check state %s should not be present in retention. Please fill a bug with your log and retention data.' % status)
            return None
        else:
            logger.error('The check state %s is not managed. Please fill a bug with your log and retention data.' % status)
            return None
    
    
    # We are at the end of the retention load, we must check:
    # * WAITCONSUME/TIMEOUT/ALL_DEP_ARE_FINISH => nothing to do, we will just consume them
    # * MUST_RECREATE_DEP => ask our ref to recreate dependency checks, and go in WAITDEP status
    #                        but beware: if there are no more dependency, cannot stay in MUST_RECREATE_DEP and go ALL_DEP_ARE_FINISH
    def restore_from_retention(self, logger_retention_analyse):
        # Just returned checks are ready to consume, they are already clean from dep in the retention save part
        if self.status == CHECK_STATUS.WAITCONSUME or self.status == CHECK_STATUS.TIMEOUT or self.status == CHECK_STATUS.ALL_DEP_ARE_FINISH:
            if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
                logger_retention_analyse.info(u'[%s] the check was reload from the retention with state %s so its results can be load.' % (self.ref.get_full_name(), self.status))
            return
        elif self.status == CHECK_STATUS.MUST_RECREATE_DEP:
            are_dep_checks_launched = self.ref.raise_dependencies_check(self)
            if not are_dep_checks_launched:
                self.status = CHECK_STATUS.ALL_DEP_ARE_FINISH
            if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
                logger_retention_analyse.info(u'[%s] the check was reload from the retention and ask for check father dependency in order to finish the check return analysis.' % self.ref.get_full_name())
            # protect against bad states here
            if self.status not in (CHECK_STATUS.WAITDEP, CHECK_STATUS.ALL_DEP_ARE_FINISH):
                logger_retention_analyse.error(u'The check state %s was unable to be restored in a valid state. Please fill a bug with your log and retention.' % self.status)
                self.status = CHECK_STATUS.ALL_DEP_ARE_FINISH
            return
        else:
            logger_retention_analyse.warning(u'The check-execution have the status %s that is not managed for retention loading, it will be automatically deleted. Please fill a bug.' % self.status)
            self.status = CHECK_STATUS.ALL_DEP_ARE_FINISH
            return
    
    
    def empty_perf_data(self):
        # type: () -> None
        if self.ref.process_perf_data:
            return
        self.perf_data = u''


# For the retention we are looking at the list of checks of a host/service, and return
# retention object that are ok with the retention status
def get_for_retention_callback(ref, checks):
    if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
        logger.info('[retention][%s] Asking retention for %s' % (ref.get_full_name(), [check.get_uuid() for check in checks]))
    retention_objects = []
    for check in checks:
        if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
            logger.info('[retention][%s]  - check id=%s status=%s (is dummy=%s)' % (ref.get_full_name(), check.get_uuid(), check.status, check.is_dummy_check))
        check_for_retention = check.get_object_for_retention()
        if check_for_retention:
            retention_objects.append(check_for_retention)
    if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
        logger.info('[retention][%s] Giving back %s checks for retention' % (ref.get_full_name(), len(retention_objects)))
    return retention_objects
