#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2021:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import itertools
import json
import os
import shutil
import time

from shinken.inter_daemon_message import InterDaemonMessage, MessageRef
from shinken.log import LoggerFactory, PartLogger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.config import OLD_ARBITER_RETENTION_JSON, REALM_RETENTION_JSON
from shinkensolutions.system_tools import set_ownership

if TYPE_CHECKING:
    from shinken.brokerlink import BrokerLink
    from shinken.log import Log
    from shinken.misc.type_hint import Any, Dict, Generator, List, Number, Optional, Tuple, Union
    from shinken.objects.config import Config
    from shinken.objects.itemsummary import HostSummary, CheckSummary
    from shinken.objects.realm import Realm
    from sla.sla_module_broker import SLAModuleBroker


class MonitoringStartTimeComponent(object):
    RETENTION_TYPE = {u'GLOBAL': 0, u'REALM': 1, u'REALM_WITH_SLA_UPDATE': 2}
    
    
    def __init__(self, logger, conf, compute_module_list=False):
        # type: (Union[PartLogger, Log], Config, bool) -> None
        self.logger = logger
        if not isinstance(logger, PartLogger):
            self.logger = LoggerFactory.get_logger(u'MONITORING START TIME COMPONENT')
        
        self.conf = conf
        self.asked_migration_is_need = True
        self.sla_info_migration_launched = False
        self.sla_info_migrated = False
        self.module_list = []
        self.check_interval = 300  # ask only each 5 minutes
        self.sla_module_to_ask = []
        self.sla_writer_modules_by_realms = {}
        if compute_module_list:
            self._compute_sla_module_writer_list()
    
    
    def _compute_sla_module_writer_list(self):
        log_computed_list_of_module_to_query = []
        for broker in self.conf.brokers:  # type: BrokerLink
            some_realm_need_update = False
            
            # Build a list of realm needing update from SLA
            known_realms = broker.get_all_my_managed_realms()
            for realm in known_realms:
                if realm.get_name_hash() in self.sla_writer_modules_by_realms:
                    continue
                
                realm_name_hash = realm.get_name_hash()
                realm_retention_file = REALM_RETENTION_JSON % realm_name_hash
                arbiter_have_monitoring_start_time_from_sla = os.path.isfile(realm_retention_file)
                if not arbiter_have_monitoring_start_time_from_sla:
                    some_realm_need_update = True
                    self.sla_writer_modules_by_realms[realm_name_hash] = {
                        u'sla_modules_writer_info': [],
                        u'last_ask_time'          : 0,
                    }
            
            if not some_realm_need_update:
                continue
            
            # Then search sla_module_writer and add it in appropriate list
            sla_modules = (b for b in broker.modules if b.get_type() in (u'sla', u'sla_broker_writer'))  # type: Generator[SLAModuleBroker]
            
            broker_name = broker.get_name()
            
            for sla_module in sla_modules:
                sla_module_name = sla_module.get_name()
                
                sla_module_writer_info = {
                    u'inter_daemon_destination': MessageRef(broker_name, sla_module_name),
                    u'broker_name'             : broker_name
                }
                
                for realm in known_realms:
                    realm_name_hash = realm.get_name_hash()
                    if realm_name_hash not in self.sla_writer_modules_by_realms:
                        continue
                    self.sla_writer_modules_by_realms[realm_name_hash][u'sla_modules_writer_info'].append(sla_module_writer_info)
                    log_computed_list_of_module_to_query.append((realm.get_name(), sla_module_name, broker_name))
        for realm_name, sla_module_name, broker_name in log_computed_list_of_module_to_query:
            self.logger.info(u'Will query configuration update of elements of realm 〖 %s 〗 from Module 〖 %s 〗 of Broker 〖 %s 〗' % (realm_name, sla_module_name, broker_name))
    
    
    def init(self):
        pass
    
    
    def get_inter_daemon_message_to_send(self, arbiter_name):
        # type: (unicode) -> List[InterDaemonMessage]
        messages = []
        
        if not self.sla_writer_modules_by_realms:
            return messages
        
        inter_daemon_sender = MessageRef(arbiter_name, u'')
        modules_to_ask = set()
        
        for realm_name_hash in self.sla_writer_modules_by_realms.iterkeys():
            realm_info = self.sla_writer_modules_by_realms[realm_name_hash]
            
            if (realm_info[u'last_ask_time'] + self.check_interval) > time.time():
                # do not hammer this module
                continue
            
            sla_modules_writer_info = realm_info.get(u'sla_modules_writer_info', [])
            
            # the realm is not handle by at least one broker : no data to get
            if not sla_modules_writer_info:
                continue
            
            # simple case : one realm = one broker
            if len(sla_modules_writer_info) == 1:
                modules_to_ask.add(sla_modules_writer_info[0][u'inter_daemon_destination'])
            
            # Many sla_writer for the same realm : we need to choose one
            else:
                sorted_modules = sorted(sla_modules_writer_info, key=self._sort_sla_modules)
                modules_to_ask.add(sorted_modules[0][u'inter_daemon_destination'])
            
            realm_info[u'last_ask_time'] = time.time()
        
        for inter_daemon_receiver in modules_to_ask:
            messages.append(InterDaemonMessage(u'ask_monitoring_start_time', message_to=inter_daemon_receiver, message_from=inter_daemon_sender, data={}))
        return messages
    
    
    def _sort_sla_modules(self, sla_module):
        # type: (Dict[unicode, Any]) -> int
        
        # Use this method to sort sla_module in sorted functions
        sort_score = 0
        
        satellite = next((broker for broker in self.conf.brokers if broker.get_name() == sla_module[u'broker_name']), None)
        if not satellite:
            return -1
        if satellite.alive and satellite.is_activated():
            sort_score += 100
        if satellite.manage_sub_realms:
            sort_score += 10
        if not satellite.spare:
            sort_score += 1
        
        return sort_score
    
    
    def handle_response_monitoring_start_time(self, inter_daemon_message):
        # type: (InterDaemonMessage) -> None
        
        from_broker_name = inter_daemon_message.message_from[0]
        from_broker = next((broker for broker in self.conf.brokers if broker.get_name() == from_broker_name), None)
        
        if not from_broker:
            self.logger.error(u'Discarding monitoring_start_time message from unknown Broker 〖 %s 〗' % from_broker_name)
            return
        
        # We received data from a Broker
        # As SLA module does not record realm of its items, we have to find it back for each item
        # Discard items no more in realms handled by this Broker
        
        message_data = inter_daemon_message.data
        to_save_realm_data = {}
        
        for realm in from_broker.get_all_my_managed_realms():  # type: Realm
            realm_name_hash = realm.get_name_hash()
            
            if realm_name_hash not in to_save_realm_data:
                to_save_realm_data[realm_name_hash] = (realm, {})
            
            for item in realm.get_inventory_for_broker().get_all():  # type: Union[HostSummary, CheckSummary]
                uuid = item.get_uuid()
                
                if uuid in message_data:
                    item.monitoring_start_time = message_data[uuid]
                    
                    if self.logger.is_debug():
                        self.logger.debug(u'realm〖 %s 〗 updating monitoring start time from Module〖 %s 〗 of Broker〖 %s 〗  for item〖 %s 〗' % (realm.get_name(), inter_daemon_message.message_from[1], from_broker_name, uuid))
                    
                    to_save_realm_data[realm_name_hash][1].update({uuid: message_data[uuid]})
                    del message_data[uuid]
        
        for realm, retention_data in to_save_realm_data.itervalues():
            realm_name_hash = realm.get_name_hash()
            
            updated_nb = len(retention_data)
            # Update previously saved retention, if any, with received data
            _, saved_retention = self._load_retention(realm)
            if saved_retention:
                saved_retention.update(retention_data)
                retention_data = saved_retention
            
            if self._save_retention(realm, retention_data, retention_type=self.RETENTION_TYPE[u'REALM_WITH_SLA_UPDATE']):
                # Another Broker may already have provided data for this realm
                if realm_name_hash in self.sla_writer_modules_by_realms:
                    # This realm is up-to-date with data from SLA, no need to query Broker anymore
                    del self.sla_writer_modules_by_realms[realm_name_hash]
                if updated_nb > 0:
                    self.logger.info(u'Updated retention data of %d elements for realm 〖 %s 〗 from Module 〖 %s 〗 of Broker 〖 %s 〗' % (updated_nb, realm.get_name(), inter_daemon_message.message_from[1], from_broker_name))
                else:
                    self.logger.info(u'Retention data for realm 〖 %s 〗 did not need update from Module 〖 %s 〗 of Broker 〖 %s 〗' % (realm.get_name(), inter_daemon_message.message_from[1], from_broker_name))
        
        # Remaining message_data is lost, as we don't know which realm these elements belong to
        if message_data:
            self.logger.info(u'Discarding data received from Module 〖 %s 〗 of Broker 〖 %s 〗 for %d elements no more in configuration' % (inter_daemon_message.message_from[1], from_broker_name, len(message_data)))
            if self.logger.is_debug():
                for uuid, monitoring_start_time in message_data.iteritems():
                    self.logger.debug(u'Discarded elements〖 %s 〗 with monitoring start time 〖 %s 〗' % (uuid, monitoring_start_time))
    
    
    # Called in arbiter sub process who handle this realm only, after all main computation.
    # Configuration (self.conf) only contains hosts and services from this realm
    def set_monitoring_start_time_realm_only(self, realm):
        # type: (Realm) -> None
        now = int(time.time())
        retention_was_updated = False
        
        retention_type, monitoring_retention_data = self._load_retention(realm)
        has_retention = bool(monitoring_retention_data)
        
        if not has_retention:
            monitoring_retention_data = {}
        
        # Update elements with data from retention, and add new elements to retention
        # IMPORTANT: DO NOT CLEAN OLD DATA: we want to keep monitoring start time, even if elements are temporary disabled
        for item in itertools.chain(self.conf.hosts, self.conf.services):
            
            item_instance_uuid = item.get_instance_uuid()
            
            # Get monitoring start time from retention
            if item_instance_uuid in monitoring_retention_data:
                item.monitoring_start_time = monitoring_retention_data[item_instance_uuid]
                
                # self.logger.debug(u' realm〖 %s 〗 retention: loaded item〖 %s 〗 monitoring start time:%s' % (realm.get_name(), item_instance_uuid, item.monitoring_start_time))
            
            # If element does not exist in retention (new element), set monitoring start time to now, and add it to retention
            else:
                item.monitoring_start_time = now
                monitoring_retention_data[item_instance_uuid] = now
                retention_was_updated = True
                
                # self.logger.debug(u' realm〖 %s 〗 retention: added item〖 %s 〗' % (realm.get_name(), item_instance_uuid))
        
        # NOTICE: in case of a global old retention file, elements may be duplicated in each realm retention file as we have no way to know disabled/deleted elements realm
        if not has_retention or retention_was_updated:
            self._save_retention(realm, monitoring_retention_data, retention_type)
    
    
    def _load_retention_file(self, file_to_load):
        # type: (unicode) -> Dict
        monitoring_retention = {}
        try:
            with open(file_to_load, u'r') as f:
                monitoring_retention = json.load(f)
            
            if not monitoring_retention:
                try:
                    if os.path.isfile(file_to_load):
                        os.unlink(file_to_load)
                        self.logger.info(u'Removed empty retention data file 〖 %s 〗' % file_to_load)
                
                except Exception as exp:
                    self.logger.warning(u'Remove of empty retention data file 〖 %s 〗 returned 〖 %s 〗' % (file_to_load, exp))
        except Exception as exp:
            self.logger.error(u'Loading of retention data file 〖 %s 〗 raised error 〖 %s 〗' % (file_to_load, exp))
            self.logger.print_stack()
        return monitoring_retention
    
    
    def _load_retention(self, realm):
        # type: (Realm) -> Tuple[int, Optional[Dict]]
        monitoring_retention = {}
        found_a_retention = False
        retention_type = self.RETENTION_TYPE[u'GLOBAL']
        
        # First try to load historical Arbiter realm file
        old_realm_file = REALM_RETENTION_JSON % realm.get_name()
        if os.path.isfile(old_realm_file):
            found_a_retention = True
            retention_type = self.RETENTION_TYPE[u'REALM']
            self.logger.info(u'Old realm retention file 〖 %s 〗 found, loading its data' % old_realm_file)
            monitoring_retention.update(self._load_retention_file(old_realm_file))
        
        # Second, try to update with realm file containing Brokers updates
        realm_file = REALM_RETENTION_JSON % realm.get_name_hash()
        if os.path.isfile(realm_file):
            if found_a_retention:
                self.logger.info(u'Realm retention file 〖 %s 〗 found, updating monitoring start times with its data' % realm_file)
            else:
                self.logger.info(u'Realm retention file 〖 %s 〗 found, loading its data' % realm_file)

            found_a_retention = True
            retention_type = self.RETENTION_TYPE[u'REALM_WITH_SLA_UPDATE']
            monitoring_retention.update(self._load_retention_file(realm_file))
        
        # Third, if there was no realm retention file, try to load obsolete global file
        if not found_a_retention:
            file_to_load = OLD_ARBITER_RETENTION_JSON
            retention_type = self.RETENTION_TYPE[u'GLOBAL']
            if os.path.isfile(file_to_load):
                found_a_retention = True
                self.logger.info(u'The realm  retention data file 〖 %s 〗 is not available, trying to load the global retention file 〖 %s 〗' % (realm_file, file_to_load))
                monitoring_retention.update(self._load_retention_file(file_to_load))
        
        if not found_a_retention or not monitoring_retention:
            self.logger.info(u'No retention data was available for realm 〖 %s 〗 tried in 〖 %s 〗, 〖 %s 〗, 〖 %s 〗' % (
                realm.get_name(),
                REALM_RETENTION_JSON % realm.get_name_hash(),
                REALM_RETENTION_JSON % realm.get_name(),
                OLD_ARBITER_RETENTION_JSON))
        
        return retention_type, monitoring_retention
    
    
    def _save_retention(self, realm, monitoring_retention_data, retention_type=0):
        # type: (Realm, Dict[unicode,Number], int) -> bool
        if retention_type == self.RETENTION_TYPE[u'REALM_WITH_SLA_UPDATE']:
            realm_name_hash = realm.get_name_hash()
            realm_file = REALM_RETENTION_JSON % realm_name_hash
            obsolete_file_cleanup = True
        else:
            realm_name = realm.get_name()
            realm_file = REALM_RETENTION_JSON % realm_name
            obsolete_file_cleanup = False
        
        realm_file_tmp = realm_file + u'.tmp'
        try:
            with open(realm_file_tmp, u'w+') as retention_file:
                json.dump(monitoring_retention_data, retention_file)
            shutil.move(realm_file_tmp, realm_file)
            set_ownership(realm_file)
            self.logger.info(u'Saved retention data file〖 %s 〗 successfully' % realm_file)
        
        except Exception as exp:
            self.logger.error(u'Save of retention data file 〖 %s 〗 raised error 〖 %s 〗' % (realm_file, exp))
            self.logger.print_stack()
            return False
        
        # Cleanup of obsolete files
        if obsolete_file_cleanup:
            try:
                realm_file = REALM_RETENTION_JSON % realm.get_name()
                if os.path.isfile(realm_file) and os.access(realm_file, os.R_OK):
                    os.unlink(realm_file)
                    self.logger.info(u'Removed obsolete retention data file 〖 %s 〗' % realm_file)
                else:
                    self.logger.warning(u'Remove of obsolete retention data file 〖 %s 〗 canceled due to wrong access rights' % realm_file)
            except Exception as exp:
                self.logger.warning(u'Remove of obsolete retention data file 〖 %s 〗 returned 〖 %s 〗' % (realm_file, exp))
        return True
