#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.


import os
import shutil
import sys
import time
import uuid

from shinken.compat import cPickle

from shinken.misc.os_utils import safe_write_binary_file_and_force_mtime, make_file_hidden
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.locking.shinken_locking.shinken_interprocess_rlock import ShinkenInterProcessRLock, set_ownership

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Optional, Any

INTERNAL_KEY = (u'_lock_ex', u'_lock_shared', u'_data', u'_data_file_name', u'_last_load_time', u'_in_update_context')

if sys.platform.startswith('win'):
    import tempfile
    
    SHARE_ITEM_SYNC_FILE_PATH = os.path.join(tempfile.gettempdir(), u'share_item')

else:
    SHARE_ITEM_SYNC_FILE_PATH = '/dev/shm/share_item'


class MockShareItem(object):
    @staticmethod
    def acquire():
        return
    
    
    @staticmethod
    def release():
        return
    
    
    def __enter__(self):
        self.acquire()
    
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        self.release()


class ShareItem(object):
    @classmethod
    def get_dir_path(cls):
        return os.path.join(SHARE_ITEM_SYNC_FILE_PATH, cls.__name__)
    
    
    @classmethod
    def delete_dir(cls):
        folder_path = cls.get_dir_path()
        if os.path.exists(folder_path):
            shutil.rmtree(folder_path)
    
    
    def destroy(self):
        try:
            os.unlink(self._data_file_name)
        except:
            pass
        try:
            self._lock_ex.destroy()
        except:
            pass
        try:
            self._lock_shared.destroy()
        except:
            pass
        try:
            os.rmdir(os.path.dirname(self._data_file_name))
        except:
            pass
    
    
    def __init__(self, key_name=None, reinit=False):
        # type: (Optional[unicode], bool) -> None
        if key_name:
            key_name = key_name.replace(os.sep, u'')
            key_name = key_name.replace(u'/', u'')
            key_name = os.path.splitdrive(key_name)[1]
        if not key_name:
            key_name = u'%s.%s-%s' % (type(self).__module__, type(self).__name__, uuid.uuid4().hex)
        self._data = {}
        self._data_file_name = os.path.join(SHARE_ITEM_SYNC_FILE_PATH, type(self).__name__, key_name)
        self._last_load_time = 0
        self._in_update_context = False
        # self.dir_create()
        self._lock_ex, self._lock_shared = self._create_lock(self._data_file_name)
        if reinit:
            self.data_file_remove()
        self.init_data_file()
    
    
    @staticmethod
    def _create_lock(data_file_name):
        lock_filename = make_file_hidden(u'%s.lock' % data_file_name)
        return ShinkenInterProcessRLock(lock_filename), ShinkenInterProcessRLock(lock_filename, True)
    
    
    def dir_create(self):
        # type: () -> None
        if not os.path.exists(SHARE_ITEM_SYNC_FILE_PATH):
            os.mkdir(SHARE_ITEM_SYNC_FILE_PATH, 0o777)
            set_ownership(SHARE_ITEM_SYNC_FILE_PATH)
        
        sub_folder = self.get_dir_path()
        if not os.path.exists(sub_folder):
            os.mkdir(sub_folder, 0o777)
            set_ownership(sub_folder)
    
    
    def init_data_file(self):
        # type: () -> None
        if not os.path.exists(self._data_file_name):
            self._save()
    
    
    def get_lock(self, shared=False):
        return self._lock_shared if shared else self._lock_ex
    
    
    def data_file_remove(self):
        # type: () -> None
        try:
            with self.get_lock():
                # On Windows the clock is not enough precise, so we must wait here to force a change on file
                if os.name == 'nt':
                    time.sleep(0.01)
                
                # the open / close as wb will clean the file.
                with open(self._data_file_name, u'wb'):
                    now = time.time()
                # Forcing the modification time to now, this was benched and we lost ~4% process time for 1M executions
                # The loss is acceptable because without this, the share_item would have update issues when really fast accesses
                os.utime(self._data_file_name, (now, now))
        except:
            pass
    
    
    def __str__(self):
        # type: () -> unicode
        return u'ShareItem %s/%s: [%s]' % (type(self).__name__, self._data_file_name, self._data)
    
    
    def __repr__(self):
        # type: () -> unicode
        return self.__str__()
    
    
    def __getattr__(self, key):
        # type: (unicode) -> Any
        if key in INTERNAL_KEY:
            return super(ShareItem, self).__getattribute__(key)
        self._reload(need_lock=True, shared_lock=True)
        try:
            return self._data[key]
        except KeyError:
            raise AttributeError(u'%s object has no attribute [%s]' % (type(self).__name__, key))
    
    
    def __setattr__(self, key, value):
        # type: (unicode, unicode) -> None
        if key in INTERNAL_KEY:
            return super(ShareItem, self).__setattr__(key, value)
        with self.get_lock():
            self._reload()
            self._data[key] = value
            if not self._in_update_context:
                self._save()
    
    
    def __enter__(self):
        # type: () -> None
        self.get_lock().acquire()
        self._in_update_context = True
    
    
    def __exit__(self, _type, value, traceback):
        # type: (unicode, Any, unicode) -> None
        self._in_update_context = False
        self._save()
        self.get_lock().release()
    
    
    def get_all_attr_read_only(self):
        # type: () -> Dict
        with self.get_lock(shared=True):
            self._reload()
            return self._data.copy()
    
    
    def clear_all_attr(self):
        # type: () -> None
        with self.get_lock():
            self._data.clear()
            self._save()
    
    
    def _reload(self, need_lock=False, shared_lock=False):
        # type: (bool, bool) -> None
        last_modification_time = os.stat(self._data_file_name).st_mtime
        if last_modification_time == self._last_load_time:
            return
        if need_lock:
            self.get_lock(shared=shared_lock).acquire()
        try:
            # Be careful, on Windows the modification date on files is not precise enough and this condition could fail
            # when accessing the same file too quickly (like in unit tests)
            if self._last_load_time < last_modification_time:
                if os.stat(self._data_file_name).st_size > 0:
                    with open(self._data_file_name, u'rb') as data_file:
                        self._deserialize(data_file)
                else:
                    self._data = {}
                self._last_load_time = last_modification_time
        finally:
            if need_lock:
                self.get_lock(shared=shared_lock).release()
    
    
    def _deserialize(self, file_descriptor):
        self._data = cPickle.load(file_descriptor)
    
    
    def _serialize(self):
        # type: () -> unicode
        return cPickle.dumps(self._data, cPickle.HIGHEST_PROTOCOL)
    
    
    def _save(self):
        # type: () -> None
        # self.dir_create()
        with self.get_lock():
            file_content = self._serialize()
            safe_write_binary_file_and_force_mtime(self._data_file_name, file_content)
            set_ownership(self._data_file_name)
            self._last_load_time = os.stat(self._data_file_name).st_mtime
