#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

import os
import socket
from Queue import Empty, Full

from shinken.ipc.shinken_queue.shinken_queue_lib import ShinkenThreadedSocket, socket_pair
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.safepickle import cPickle, SafeUnpickler

if TYPE_CHECKING:
    from shinken.misc.type_hint import Number, Optional


class ShinkenQueue(object):
    def __init__(self, queue_name, father_name=None, child_name=None):
        # noinspection SpellCheckingInspection
        if hasattr(socket, u'socketpair'):
            s1, s2 = socket.socketpair()
        else:
            s1, s2 = socket_pair()
        self.socket_for_parent_process = ShinkenThreadedSocket(s1, queue_name, father_name, child_name)
        self.socket_for_child_process = ShinkenThreadedSocket(s2, queue_name, child_name, father_name)
        self.parent_pid = os.getpid()
    
    
    def get_receiver_name(self):
        return self.socket_for_parent_process.get_receiver_name() if os.getpid() == self.parent_pid else self.socket_for_child_process.get_receiver_name()
    
    
    def get_sender_name(self):
        return self.socket_for_parent_process.get_sender_name() if os.getpid() == self.parent_pid else self.socket_for_child_process.get_sender_name()
    
    
    def get_queues_size(self):
        # just for stats, to be reviewed  # type: () -> Tuple[int,int,int,int]
        threaded_socket = self.socket_for_parent_process if os.getpid() == self.parent_pid else self.socket_for_child_process
        return threaded_socket.get_queues_size()
    
    
    def recv(self):
        # type: () -> bytes
        return self.socket_for_parent_process.recv() if os.getpid() == self.parent_pid else self.socket_for_child_process.recv()
    
    
    def poll(self, timeout=0.0):
        # type: (Optional[Number]) -> bool
        return self.socket_for_parent_process.poll(timeout) if os.getpid() == self.parent_pid else self.socket_for_child_process.poll(timeout)
    
    
    def write_poll(self, timeout=0.0):
        # type: (Optional[Number]) -> bool
        return self.socket_for_parent_process.write_poll(timeout) if os.getpid() == self.parent_pid else self.socket_for_child_process.write_poll(timeout)
    
    
    def flush(self, timeout=0.0):
        # type: (Optional[Number]) -> bool
        return self.socket_for_parent_process.flush(timeout) if os.getpid() == self.parent_pid else self.socket_for_child_process.flush(timeout)
    
    
    def send(self, data):
        # type: (bytes) -> int
        return self.socket_for_parent_process.send(data) if os.getpid() == self.parent_pid else self.socket_for_child_process.send(data)
    
    
    def close(self):
        # type: () -> None
        self.socket_for_parent_process.close()
        self.socket_for_child_process.close()
    
    
    # Python Queue API
    
    def qsize(self):
        read_nb, read_size, write_nb, write_size = self.get_queues_size()
        return read_nb + write_nb
    
    
    def empty(self):
        return self.poll(0) is False
    
    
    @staticmethod
    def full():
        return False
    
    
    def put(self, item, block=True, timeout=None):
        timeout = 0 if not block else (timeout if timeout >= 0 else None)
        if timeout is not None:
            if not self.flush(timeout):
                raise Full
        
        item = cPickle.dumps(item, cPickle.HIGHEST_PROTOCOL)
        self.send(item)
        # Give some time to send data, regardless of timeout
        self.flush(None)
    
    
    def put_nowait(self, item):
        self.put(item, False)
    
    
    def get(self, block=True, timeout=None):
        timeout = 0 if not block else (timeout if timeout >= 0 else None)
        if not self.poll(timeout):
            raise Empty
        return SafeUnpickler.loads(self.recv(), self.get_receiver_name())
    
    
    def get_nowait(self):
        return self.get(False)
    
    
    def task_done(self):
        self.flush(None)
    
    
    def join(self):
        # type: () -> bool
        return self.socket_for_parent_process.join(None) if os.getpid() == self.parent_pid else self.socket_for_child_process.join(None)
