#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

""" This is the main class for the Host. In fact, it's mainly
about the configuration part. for the running one, it's better
to look at the SchedulingItem class that manage all
scheduling/consume check smart things :)
"""

import re
import time

from item import InheritableItems, MAX_INHERITANCE_LEVEL, _format_inherited_plus_value
from schedulingitem import SchedulingItem, PROPERTIES_RETRIEVABLE_FROM_COMMAND_TO_HOST, CUSTOM_DATA_FORMAT
from shinken.autoslots import AutoSlots
from shinken.eventhandler import EventHandler
from shinken.graph import Graph
from shinken.log import logger, naglog_result
from shinken.macroresolver import MacroResolver
from shinken.misc.configuration_error_log import log_configuration_error, log_configuration_warning
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.proxyitem import proxyitemsmgr
from shinken.objects.service import Service
from shinken.property import BoolProp, IntegerProp, CharProp, StringProp, ListProp, EpochProp, RawProp, LinkToOthersObjectsProp
from shinken.rawdata import TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL
from shinken.util import \
    format_t_into_dhms_format, \
    to_hostnames_list, \
    get_obj_name, \
    to_list_string_of_names, \
    get_in_inherited_acknowledged, \
    get_in_inherited_downtime, \
    get_brok_instance_uuid, \
    get_real_state_id, \
    to_cluster_members_uuid_list_and_do_cache_result
from shinkensolutions.service_override_parser import parse_service_override_property, unparse_service_override, SyntaxServiceOverrideError

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from shinken.objects.command import Commands
    from shinken.objects.hostgroup import Hostgroups

SECONDS_MAX_DATE_VALUE = 157852800
MINUTES_MAX_DATE_VALUE = 2630880
MINUTE_PROPERTY = set(('check_interval', 'retry_interval', 'notification_interval', 'first_notification_delay'))
SECOND_PROPERTY = set(('warning_threshold_cpu_usage', 'freshness_threshold', 'check_running_timeout'))
VALID_RETURN_CODE = (0, 1, 2, 3)


class EXCLUDE_FROM_TEMPLATES_RESULTS(object):
    ALLOW = 'allow'
    DONT_HAVE = 'dont-have'
    EXCLUDE = 'exclude'


class VIEW_CONTACTS_DEFAULT_VALUE(object):
    NOBODY = 'nobody'
    EVERYONE = 'everyone'
    
    
    @staticmethod
    def is_valid_default_value(value):
        return value in (VIEW_CONTACTS_DEFAULT_VALUE.NOBODY, VIEW_CONTACTS_DEFAULT_VALUE.EVERYONE)


class Host(SchedulingItem):
    # AutoSlots create the __slots__ with properties and
    # running_properties names
    __metaclass__ = AutoSlots
    
    id = 1  # zero is reserved for host (primary node for parents)
    ok_up = 'UP'
    my_type = 'host'
    
    # name of the type when display log for end users
    my_type_log_display = 'host'
    
    # cf #SEF-8624 this will be used to determine contacts rights for inheritance
    # the config.py "load_default_properties_values" method will fill it
    default_view_contacts_value = None
    
    if TYPE_CHECKING:
        max_check_attempts = 0
        check_interval = 0
        retry_interval = 0
        obsess_over_host = False
        low_flap_threshold = 25
        high_flap_threshold = 50
        flap_detection_enabled = True
        host_name = u''
        display_name = u''
        realm = u''
        # parents = None   # type: Optional[??]
        hostgroups = None  # type: Optional[Hostgroups]
        flap_detection_options = u''
        notification_options = u''
        # service_excludes = None   # type: Optional[??]
        # service_excludes_by_id = None   # type: Optional[??]
        service_overrides = u''

        retain_status_information = True
        retain_nonstatus_information = True
        stalking_options = u''
        notes = u''
        action_url = u''
        icon_image = u''
        icon_image_alt = u''
        icon_set = u''
        vrml_image = u''
        statusmap_image = u''
        initial_state = u''
        # 2d_coords
        # 3d_coords
        failure_prediction_enabled = False
        labels = u''
        business_rule_output_template = u''
        business_rule_smart_notifications = False
        business_rule_downtime_as_ack = False
        business_rule_host_notification_options = u''
        business_rule_service_notification_options = u''
        time_to_orphanage = 300
        in_inherited_downtime = False
        childs = []
        
        
        def __init__(self, params=None, skip_useless_in_configuration=False):
            super(Host, self).__init__(params, skip_useless_in_configuration)
            self.address = u''
            self.alias = u''
            self.last_time_unreachable = 0.0
            self.last_time_up = 0.0
            self.last_time_down = 0.0
            self._list_of_service_uuids = []
            self.templates = []
    
    # properties defined by configuration
    # *required: is required in conf
    # *default: default value if no set in conf
    # *pythonize: function to call when transforming string to python object
    # *fill_brok: if set, send to broker. there are two categories: full_status for initial and update status, check_result for check results
    # *no_slots: do not take this property for __slots__
    #  Only for the initial call
    # conf_send_preparation: if set, will pass the property to this function. It's used to "flatten"
    #  some dangerous properties like realms that are too 'linked' to be sent like that.
    # brok_transformation: if set, will call the function with the value of the property
    #  the major times it will be to flatten the data (like realm_name instead of the realm object).
    properties = SchedulingItem.properties.copy()
    properties.update({
        # Simple properties, can be set into a C structure
        u'max_check_attempts'    : IntegerProp(default=u'2', fill_brok=[u'full_status']),
        u'check_interval'        : IntegerProp(default=u'1', fill_brok=[u'full_status', u'check_result']),
        u'retry_interval'        : IntegerProp(default=u'1', fill_brok=[u'full_status', u'check_result']),
        u'obsess_over_host'      : BoolProp(default=u'0', fill_brok=[u'full_status']),
        
        u'low_flap_threshold'    : IntegerProp(default=u'25', fill_brok=[u'full_status']),
        u'high_flap_threshold'   : IntegerProp(default=u'50', fill_brok=[u'full_status']),
        u'flap_detection_enabled': BoolProp(default=u'1', fill_brok=[u'full_status']),
        
        # Configuration Structure, like links to others elements
        u'host_name'             : StringProp(fill_brok=[u'full_status', u'check_result', u'next_schedule']),
        u'display_name'          : StringProp(default='', fill_brok=[u'full_status']),
        u'address'               : StringProp(fill_brok=[u'full_status'], inherit=False),
        u'realm'                 : StringProp(default=None, fill_brok=[u'full_status', u'check_result', u'next_schedule'], conf_send_preparation=get_obj_name),
        u'parents'               : LinkToOthersObjectsProp(brok_transformation=to_hostnames_list, default='', fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        u'hostgroups'            : LinkToOthersObjectsProp(brok_transformation=to_list_string_of_names, default='', fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        u'check_command'         : StringProp(default=u'check-host-alive', fill_brok=[u'full_status']),
        
        # These one can be size bounded
        u'flap_detection_options': ListProp(default=u'o,d,u', fill_brok=[u'full_status'], merging=u'join'),
        u'notification_options'  : ListProp(default=u'd,u,r,f', fill_brok=[u'full_status'], merging=u'join'),
        
        u'service_excludes'      : LinkToOthersObjectsProp(default=u'', merging=u'join', inherit=False),
        u'service_excludes_by_id': LinkToOthersObjectsProp(default=u'', merging=u'join', inherit=False),
        u'service_overrides'     : ListProp(default=u'', merging=u'duplicate', split_on_coma=False, separator="-=#=-"),
        
    })
    
    # properties keep but we don't show differences
    passthrough = SchedulingItem.passthrough.copy()
    passthrough.update({
        # ## Remove from the properties list.
        'retain_status_information'                 : BoolProp(default='1', fill_brok=['full_status']),
        'retain_nonstatus_information'              : BoolProp(default='1', fill_brok=['full_status']),
        'stalking_options'                          : ListProp(default='', fill_brok=['full_status']),
        'notes'                                     : StringProp(default='', fill_brok=['full_status']),
        'action_url'                                : StringProp(default='', fill_brok=['full_status']),
        'icon_image'                                : StringProp(default='', fill_brok=['full_status']),
        'icon_image_alt'                            : StringProp(default='', fill_brok=['full_status']),
        'icon_set'                                  : StringProp(default='', fill_brok=['full_status']),
        'vrml_image'                                : StringProp(default='', fill_brok=['full_status']),
        'statusmap_image'                           : StringProp(default='', fill_brok=['full_status']),
        'initial_state'                             : CharProp(default='u', fill_brok=['full_status']),
        
        # No slots for this 2 because begin property by a number seems bad
        # it's stupid!
        '2d_coords'                                 : StringProp(default='', fill_brok=['full_status'], no_slots=True),
        '3d_coords'                                 : StringProp(default='', fill_brok=['full_status'], no_slots=True),
        'failure_prediction_enabled'                : BoolProp(default='0', fill_brok=['full_status']),
        
        'labels'                                    : ListProp(default='', fill_brok=['full_status'], merging='join'),
        
        # BUSINESS CORRELATOR PART
        # Business rules output format template
        'business_rule_output_template'             : StringProp(default='', fill_brok=['full_status']),
        # Business rules notifications mode
        'business_rule_smart_notifications'         : BoolProp(default='0', fill_brok=['full_status']),
        # Treat downtimes as acknowledgements in smart notifications
        'business_rule_downtime_as_ack'             : BoolProp(default='0', fill_brok=['full_status']),
        # Enforces child nodes notification options
        'business_rule_host_notification_options'   : ListProp(default='', fill_brok=['full_status']),
        'business_rule_service_notification_options': ListProp(default='', fill_brok=['full_status']),
        # ## still exists in the properties list.
        'time_to_orphanage'                         : IntegerProp(default='300', fill_brok=['full_status']),
        
    })
    
    # properties prohibited
    blacklist = {
        # Load some triggers
        'trigger'                     : StringProp(default=''),
        'trigger_name'                : ListProp(default=''),
        'trigger_broker_raise_enabled': BoolProp(default='0'),
        'checkmodulations'            : ListProp(default='', fill_brok=['full_status'], merging='ordered'),
    }
    
    # properties set only for running purpose
    # retention: save/load this property from retention
    running_properties = SchedulingItem.running_properties.copy()
    running_properties.update({
        # #################### Purely scheduler or arbiter internal
        # no brok, too many links
        'services'                             : RawProp(default=[]),
        'got_default_realm'                    : BoolProp(default=False),
        'last_cluster_state_id'                : IntegerProp(default=-1, retention=True, useless_in_configuration=True),
        'pack_id'                              : IntegerProp(default=-1),  # Keep in mind our pack id after the cutting phase
        'state_before_hard_unknown_reach_phase': StringProp(default='UP', retention=True, useless_in_configuration=True),
        
        # Cluster sons, exported for raw data only
        'cluster_members'                      : RawProp(default=[], fill_raw_data=[TYPE_RAW_DATA_FULL], brok_transformation=to_cluster_members_uuid_list_and_do_cache_result),
        
        # ################### Need brok sending
        'host_uuid'                            : StringProp(default='', fill_brok=['full_status', 'check_result', 'next_schedule'], brok_transformation=(lambda ref, value: ref.uuid)),
        'instance_uuid'                        : StringProp(default='', fill_brok=['full_status'], brok_transformation=get_brok_instance_uuid),
        'state_id'                             : IntegerProp(default=3, fill_brok=['full_status', 'check_result'], raw_data_transformation=get_real_state_id, fill_raw_data=[TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL], retention=True,
                                                             useless_in_configuration=True),
        'last_time_up'                         : EpochProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'last_time_down'                       : EpochProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'last_time_unreachable'                : EpochProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        
        'problem_has_been_acknowledged'        : BoolProp(default=False, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'in_inherited_acknowledged'            : BoolProp(default=False, fill_brok=['full_status', 'check_result'], brok_transformation=get_in_inherited_acknowledged, useless_in_configuration=True),
        
        'customs'                              : RawProp(default={}),
        
        'in_scheduled_downtime'                : BoolProp(default=False, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'in_inherited_downtime'                : BoolProp(default=False, fill_brok=['full_status', 'check_result'], brok_transformation=get_in_inherited_downtime, useless_in_configuration=True),
        'in_partial_downtime'                  : BoolProp(default=False, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        
        'childs'                               : RawProp(default=[], brok_transformation=to_hostnames_list, fill_brok=['full_status']),
        
    })
    
    not_inherited_passthrough = SchedulingItem.not_inherited_passthrough.copy()
    not_inherited_passthrough.update({
        'for_all_users': BoolProp(default='1'),
    })
    
    # Hosts macros and prop that give the information
    # the prop can be callable or not
    macros = {
        'HOSTUUID'                  : 'uuid',
        'HOSTNAME'                  : 'host_name',
        'HOSTDISPLAYNAME'           : 'display_name',
        'HOSTALIAS'                 : 'display_name',
        'HOSTADDRESS'               : 'address',
        'HOSTSTATE'                 : 'state',
        'HOSTSTATEID'               : 'state_id',
        'LASTHOSTSTATE'             : 'last_state',
        'LASTHOSTSTATEID'           : 'last_state_id',
        'HOSTSTATETYPE'             : 'state_type',
        'HOSTATTEMPT'               : 'attempt',
        'MAXHOSTATTEMPTS'           : 'max_check_attempts',
        'HOSTEVENTID'               : 'current_event_id',
        'LASTHOSTEVENTID'           : 'last_event_id',
        'HOSTPROBLEMID'             : 'current_problem_id',
        'LASTHOSTPROBLEMID'         : 'last_problem_id',
        'HOSTLATENCY'               : 'latency',
        'HOSTEXECUTIONTIME'         : 'execution_time',
        'HOSTDURATION'              : 'get_duration',
        'HOSTDURATIONSEC'           : 'get_duration_sec',
        'HOSTDOWNTIMECOMMENT'       : 'get_downtime_comment',
        'HOSTDOWNTIMEAUTHOR'        : 'get_downtime_author',
        'HOSTPERCENTCHANGE'         : 'percent_state_change',
        'HOSTGROUPNAME'             : 'get_groupname',
        'HOSTGROUPNAMES'            : 'get_groupnames',
        'LASTHOSTCHECK'             : 'last_chk',
        'LASTHOSTSTATECHANGE'       : 'last_state_change',
        'LASTHOSTUP'                : 'last_time_up',
        'LASTHOSTDOWN'              : 'last_time_down',
        'LASTHOSTUNREACHABLE'       : 'last_time_unreachable',
        'HOSTOUTPUT'                : 'output',
        'LONGHOSTOUTPUT'            : 'long_output',
        'HOSTPERFDATA'              : 'perf_data',
        'LASTHOSTPERFDATA'          : 'last_perf_data',
        'HOSTCHECKCOMMAND'          : 'get_check_command',
        'HOSTACKAUTHOR'             : 'get_ack_author_name',
        'HOSTACKAUTHORNAME'         : 'get_ack_author_name',
        'HOSTACKAUTHORALIAS'        : 'get_ack_author_name',
        'HOSTACKCOMMENT'            : 'get_ack_comment',
        'HOSTACTIONURL'             : 'action_url',
        'HOSTFIRSTNOTIFICATIONDELAY': 'first_notification_delay',
        'HOSTNOTESURL'              : 'notes_url',
        'HOSTNOTESMULTIURL'         : 'notes_multi_url',
        'HOSTNOTES'                 : 'notes',
        'HOSTREALM'                 : 'get_realm',
        'TOTALHOSTSERVICES'         : 'get_total_services',
        'TOTALHOSTSERVICESOK'       : 'get_total_services_ok',
        'TOTALHOSTSERVICESWARNING'  : 'get_total_services_warning',
        'TOTALHOSTSERVICESUNKNOWN'  : 'get_total_services_unknown',
        'TOTALHOSTSERVICESCRITICAL' : 'get_total_services_critical',
        'HOSTBUSINESSIMPACT'        : 'business_impact',
        'HOSTNOTIFICATIONNUMBER'    : 'current_notification_number',
    }
    
    # Manage ADDRESSX macros by adding them dynamically
    for _i in range(32):
        macros['HOSTADDRESS%d' % _i] = 'address%d' % _i
    # Some macros can be cached, and should be computed only once as a cache
    static_macros = set(['HOSTUUID', 'HOSTNAME', 'HOSTDISPLAYNAME', 'HOSTALIAS', 'HOSTADDRESS', 'HOSTNOTESURL', 'HOSTREALM, HOSTFIRSTNOTIFICATIONDELAY'])
    # This tab is used to transform old parameters name into new ones
    # so from Nagios2 format, to Nagios3 ones.
    # Or Shinken deprecated names like criticity
    old_properties = {
        'normal_check_interval': 'check_interval',
        'retry_check_interval' : 'retry_interval',
        'criticity'            : 'business_impact',
        'hostgroup'            : 'hostgroups',
    }
    
    
    #######
    #                   __ _                       _   _
    #                  / _(_)                     | | (_)
    #   ___ ___  _ __ | |_ _  __ _ _   _ _ __ __ _| |_ _  ___  _ __
    #  / __/ _ \| '_ \|  _| |/ _` | | | | '__/ _` | __| |/ _ \| '_ \
    # | (_| (_) | | | | | | | (_| | |_| | | | (_| | |_| | (_) | | | |
    #  \___\___/|_| |_|_| |_|\__, |\__,_|_|  \__,_|\__|_|\___/|_| |_|
    #                         __/ |
    #                        |___/
    ######
    
    # Fill address with host_name if not already set
    def fill_predictive_missing_parameters(self):
        # type: () -> None
        if hasattr(self, u'host_name') and not hasattr(self, u'address'):
            self.address = self.host_name
        if hasattr(self, u'host_name') and not hasattr(self, u'alias'):
            self.alias = self.host_name
    
    
    def get_item_type_for_synchronizer(self):
        if self.got_business_rule:
            if self.is_tpl():
                return u'clustertpls'
            else:
                return u'clusters'
        else:
            if self.is_tpl():
                return u'hosttpls'
            else:
                return u'hosts'
    
    
    # Check if the host is valid.
    # Error message will be logged by log_configuration_error or log_configuration_warning.
    def is_correct(self):
        # type: () -> bool
        state = True
        if next((parent for parent in self.parents if parent.get_uuid() == self.get_uuid()), None):
            state = False
            log_configuration_error(self, u'Invalid parent value, it cannot be parent of itself')
        
        cls = self.__class__
        special_properties = (u'check_period', u'notification_interval', u'notification_period')
        for property_name, property_definition in cls.properties.iteritems():
            if property_name not in special_properties and not hasattr(self, property_name) and property_definition.required:
                state = False
                log_configuration_error(self, u'The property %s is not set', property_name)
        
        for _warning in self.configuration_warnings:
            log_configuration_warning(self, _warning)

        if self.configuration_errors:
            state = False
            for _error in self.configuration_errors:
                log_configuration_error(self, _error)
        
        if not hasattr(self, 'notification_period'):
            self.notification_period = None
        
        low_flap = self.low_flap_threshold if self.low_flap_threshold != -1 else self.__class__.global_low_flap_threshold
        high_flap = self.high_flap_threshold if self.high_flap_threshold != -1 else self.__class__.global_high_flap_threshold
        if high_flap < low_flap:
            log_configuration_error(self, u'Flapping enter threshold must not be lower than exit threshold (respectively %s%% and %s%%)', low_flap, high_flap)
            state = False
        
        if self.sla_warning_threshold < self.sla_critical_threshold:
            log_configuration_error(self, u'Critical SLA thresholds must be lower than warning (respectively critical %s%% and warning %s%%)', self.sla_warning_threshold, self.sla_critical_threshold)
            state = False
        
        for custom in self.customs.iterkeys():
            if CUSTOM_DATA_FORMAT.search(custom):
                log_configuration_error(self, u'The DATA name %s is malformed. Special characters are forbidden.', custom)
                state = False
        
        if getattr(self, 'event_handler', None) and not self.event_handler.is_valid():
            log_configuration_error(self, u'The event_handler %s set on this host is not valid.', self.event_handler.get_name())
            state = False
        
        if getattr(self, 'check_command', None) is None:
            log_configuration_error(self, u'The check_command is missing.')
            state = False
        # Ok got a command, but maybe it's invalid
        else:
            if not self.check_command.is_valid():
                log_configuration_error(self, u'The check_command %s set on this host is not valid.', self.check_command.get_name())
                state = False
            if self.got_business_rule:
                if self.business_rule and self.business_rule.configuration_errors:
                    log_configuration_error(self, u'The definition is invalid : ')
                    for bp_error in self.business_rule.configuration_errors:
                        log_configuration_error(self, u'%s' % bp_error)
                    
                    state = False
                elif self.business_rule is None:
                    log_configuration_warning(self, u'There is no definition')
        
        if hasattr(self, 'host_name'):
            for char in cls.illegal_object_name_chars:
                if char in self.host_name:
                    log_configuration_error(self, u'The character %s is not allowed in name.', char)
                    state = False
        
        for property_name in MINUTE_PROPERTY:
            value = getattr(self, property_name, None)
            if value and value > MINUTES_MAX_DATE_VALUE:
                log_configuration_error(self, u'The value in %s property is too high. Value is %s, the maximum is %s', property_name, value, MINUTES_MAX_DATE_VALUE)
                state = False
        
        for property_name in SECOND_PROPERTY:
            value = getattr(self, property_name, None)
            if value and value > SECONDS_MAX_DATE_VALUE:
                log_configuration_error(self, u'The value in %s property is too high. Value is %s, the maximum is %s', property_name, value, SECONDS_MAX_DATE_VALUE)
                state = False
        
        return state
    
    
    # Search in my service if I've got the service
    def find_service_by_name(self, service_description):
        # type: (unicode) -> Optional[Service]
        
        # service on host can be indexed (it will in broker and scheduler)
        service_index = getattr(self, 'service_index', {})
        if service_index:
            service_index_by_name = getattr(self, 'service_index_by_name', {})
            service_id = service_index_by_name.get(service_description, None)
            if service_id is not None:
                return service_index.get(service_id, None)
        else:
            return next((service for service in self.services if getattr(service, 'service_description', '__UNNAMED_SERVICE__') == service_description), None)
        return None
    
    
    # Search in my service if I've got the service
    def find_service_by_uuid(self, uuid):
        # type: (unicode) -> Optional[Service]
        
        # service on host can be indexed (it will in broker and scheduler)
        service_index = getattr(self, 'service_index', {})
        if service_index:
            service_index_by_name = getattr(self, 'service_index_by_uuid', {})
            service_id = service_index_by_name.get(uuid, None)
            if service_id is not None:
                return service_index.get(service_id, None)
        else:
            return next((service for service in self.services if getattr(service, 'uuid', '__NO_UUID__') == uuid), None)
        return None
    
    
    # For get a nice name
    def get_name(self):
        if not self.is_tpl():
            try:
                return self.host_name
            except AttributeError:  # ouch, no hostname
                return 'UNNAMEDHOST'
        else:
            try:
                return self.name
            except AttributeError:  # ouch, no name for this template
                return 'UNNAMEDHOSTTEMPLATE'
    
    
    def get_groupname(self):
        groupname = ''
        for hg in self.hostgroups:
            # naglog_result('info', 'get_groupname : %s %s %s' % (hg.id, hg.alias, hg.get_name()))
            # groupname = "%s [%s]" % (hg.alias, hg.get_name())
            groupname = "%s" % hg.alias
        return groupname
    
    
    def get_groupnames(self):
        groupnames = ''
        for hg in self.hostgroups:
            # naglog_result('info', 'get_groupnames : %s' % (hg.get_name()))
            if groupnames == '':
                groupnames = hg.get_name()
            else:
                groupnames = "%s, %s" % (groupnames, hg.get_name())
        return groupnames
    
    
    # For debugging purpose only
    def get_dbg_name(self):
        return getattr(self, 'host_name', '') or 'tpl:%s' % getattr(self, 'name', '(no name)')
    
    
    # Same but for clean call, no debug
    def get_full_name(self):
        return self.host_name
    
    
    def get_instance_uuid(self):
        return self.uuid
    
    
    # Get our realm
    def get_realm(self):
        return self.realm
    
    
    def get_hostgroups(self):
        return self.hostgroups
    
    
    def get_host_tags(self):
        return self.tags
    
    
    # Say if we got the other in one of your dep list
    def is_linked_with_host(self, other):
        for (h, status, _type, timeperiod, inherits_parent) in self.act_depend_of:
            if h == other:
                return True
        return False
    
    
    # Delete all links in the act_depend_of list of self and other
    def del_host_act_dependency(self, other):
        to_del = []
        # First we remove in my list
        for (h, status, _type, timeperiod, inherits_parent) in self.act_depend_of:
            if h == other:
                to_del.append((h, status, _type, timeperiod, inherits_parent))
        for t in to_del:
            self.act_depend_of.remove(t)
        
        # And now in the father part
        to_del = []
        for (h, status, _type, timeperiod, inherits_parent) in other.act_depend_of_me:
            if h == self:
                to_del.append((h, status, _type, timeperiod, inherits_parent))
        for t in to_del:
            other.act_depend_of_me.remove(t)
        
        # Remove in child/parents deps too
        # Me in father list
        other.child_dependencies.remove(self)
        # and father list in mine
        self.parent_dependencies.remove(other)
    
    
    # Add a dependency for action event handler, notification, etc.)
    # and add ourselves in its dep list
    def add_host_act_dependency(self, h, status, timeperiod, inherits_parent):
        # I add him in MY list
        self.act_depend_of.append((h, status, 'logic_dep', timeperiod, inherits_parent))
        # And I add me in its list
        h.act_depend_of_me.append((self, status, 'logic_dep', timeperiod, inherits_parent))
        
        # And the parent/child dep lists too
        h.register_son_in_parent_child_dependencies(self)
    
    
    # Register the dependency between 2 service for action (notification etc.)
    # but based on a BUSINESS rule, so on fact:
    # ERP depends on database, so we fill just database.act_depend_of_me
    # because we will want ERP mails to go on! So call this
    # on the database service with the srv=ERP service
    def add_business_rule_act_dependency(self, h, status, timeperiod, inherits_parent):
        # first I add the other I depend on in MY list
        # I only register, so he knows that I WILL be an impact
        self.act_depend_of_me.append((h, status, 'business_dep',
                                      timeperiod, inherits_parent))
        
        # And the parent/child dep lists too
        self.register_son_in_parent_child_dependencies(h)
    
    
    # Add a dependency for check (so before launch)
    def add_host_chk_dependency(self, h, status, timeperiod, inherits_parent):
        # I add him in MY list
        self.chk_depend_of.append((h, status, 'logic_dep', timeperiod, inherits_parent))
        # And I add me in its list
        h.chk_depend_of_me.append((self, status, 'logic_dep', timeperiod, inherits_parent))
        
        # And we fill parent/childs dep for brok purpose
        # Here self depend on h
        h.register_son_in_parent_child_dependencies(self)
    
    
    # Add one of our service to services (at linkify)
    def add_service_link(self, service):
        self.services.append(service)
    
    
    #####
    #                         _
    #                        (_)
    #  _ __ _   _ _ __  _ __  _ _ __   __ _
    # | '__| | | | '_ \| '_ \| | '_ \ / _` |
    # | |  | |_| | | | | | | | | | | | (_| |
    # |_|   \__,_|_| |_|_| |_|_|_| |_|\__, |
    #                                  __/ |
    #                                 |___/
    ####
    
    # Set unreachable: all our parents are down!
    # We have a special state, but state was already set, we just need to
    # update it. We are no DOWN, we are UNREACHABLE and
    # got a state id is 2
    def set_unreachable(self):
        now = time.time()
        old_state_id = self.state_id
        old_state_as_string = self.state
        self.state_id = 2
        self.state = 'UNREACHABLE'
        if old_state_id != self.state_id or old_state_as_string != self.state:
            self.last_state_id = old_state_id
            self.last_state_as_string = old_state_as_string
            self.last_state_change = int(now)
        
        self.last_time_unreachable = int(now)
        proxyitemsmgr.update_state(self.get_instance_uuid(), self.state_id)
    
    
    # We just go an impact, so we go unreachable
    # But only if we enable this state change in the conf
    def set_impact_state(self):
        cls = self.__class__
        if cls.enable_problem_impacts_states_change:
            # Keep a trace of the old state (problem came back before a new checks)
            self.state_before_impact = self.state
            self.state_id_before_impact = self.state_id
            # This flag will know if we override the impact state
            self.state_changed_since_impact = False
            self.state = 'UNREACHABLE'  # exit code UNDETERMINED
            self.state_id = 2
    
    
    # Ok, we are no more an impact, if no news checks
    # override the impact state, we came back to old
    # states
    # And only if impact state change is set in configuration
    def unset_impact_state(self):
        cls = self.__class__
        if cls.enable_problem_impacts_states_change and not self.state_changed_since_impact:
            self.state = self.state_before_impact
            self.state_id = self.state_id_before_impact
            proxyitemsmgr.update_state(self.get_instance_uuid(), self.state_id)
    
    
    # Set state with exit code return by the check. Also update last_state.
    def set_state_from_exit_status(self, return_code):
        now = int(time.time())
        self.last_state_update = now
        
        # we should put in last_state the good last state:
        # if not just change the state by a problem/impact
        # we can take current state. But if it's the case, the
        # real old state is self.state_before_impact (it's the TRUE
        # state in fact)
        # And only if we enable the impact state change
        # Note: do not apply this for bp_rule, we don't care about their last state
        cls = self.__class__
        if not self.got_business_rule and cls.enable_problem_impacts_states_change and self.is_impact and not self.state_changed_since_impact:
            self.last_state = self.state_before_impact
        else:
            self.last_state = self.state
        
        if return_code == 0:
            self.state = 'UP'
            self.state_id = 0
            self.last_time_up = int(self.last_state_update)
        elif self.are_all_parents_down():
            self.state = 'UNREACHABLE'
            self.state_id = 2
        else:
            self.state = 'DOWN'  # exit code UNDETERMINED
            self.state_id = 1
            self.last_time_down = int(self.last_state_update)
        
        self.output = Host.get_unknown_return_code_output(return_code, self.output)
        
        # If business rule for host, save the raw result from bp_rule computation somewhere else
        if self.got_business_rule:
            self.bp_state = return_code
            # Update the proxy objects with the new value
            proxyitemsmgr.update_state(self.get_instance_uuid(), self.bp_state)
        else:
            proxyitemsmgr.update_state(self.get_instance_uuid(), self.state_id)
    
    
    @staticmethod
    def get_unknown_return_code_output(return_code, current_output):
        if return_code not in VALID_RETURN_CODE:
            return u'''Shinken ERROR: The exit code returned by the command [ %s ] is not valid, we set the host to CRITICAL. Valid values are : %s.<br><br>The output of the command is :<br>----------------------------<br>%s''' % (
                return_code, u','.join(map(str, VALID_RETURN_CODE)), current_output)
        else:
            return current_output
    
    
    # See if status is status. Can be low of high format (o/UP, d/DOWN, ...)
    def is_state(self, status):
        if status == self.state:
            return True
        # Now low status
        elif status == 'o' and self.state == 'UP':
            return True
        elif status == 'd' and self.state == 'DOWN':
            return True
        elif status == 'u' and self.state == 'UNREACHABLE':
            return True
        return False
    
    
    # The last time when the state was not UP
    def last_time_non_ok_or_up(self):
        if self.last_time_down > self.last_time_up:
            last_time_non_up = self.last_time_down
        else:
            last_time_non_up = 0
        return last_time_non_up
    
    
    # Add a log entry with a HOST ALERT like:
    # HOST ALERT: server;DOWN;HARD;1;I don't know what to say...
    def raise_alert_log_entry(self):
        naglog_result('critical', 'HOST ALERT: %s;%s;%s;%d;%s'
                      % (self.get_name(),
                         self.state, self.state_type,
                         self.attempt, self.output))
    
    
    # If the configuration allow it, raise an initial log like
    # CURRENT HOST STATE: server;DOWN;HARD;1;I don't know what to say...
    def raise_initial_state(self):
        if self.__class__.log_initial_states:
            naglog_result('info', 'CURRENT HOST STATE: %s;%s;%s;%d;%s'
                          % (self.get_name(),
                             self.state, self.state_type,
                             self.attempt, self.output))
    
    
    # Add a log entry with a Freshness alert like:
    # Warning: The results of host 'Server' are stale by 0d 0h 0m 58s (threshold=0d 1h 0m 0s).
    # I'm forcing an immediate check of the host.
    def raise_freshness_log_entry(self, t_stale_by, t_threshold):
        logger.warning("The results of host '%s' are stale by %s "
                       "(threshold=%s).  I'm forcing an immediate check "
                       "of the host."
                       % (self.get_name(),
                          format_t_into_dhms_format(t_stale_by),
                          format_t_into_dhms_format(t_threshold)))
    
    
    # Raise a log entry with a Notification alert like
    # HOST NOTIFICATION: super admin;server;UP;notify-by-rss;no output
    def raise_notification_log_entry(self, n):
        contact = n.contact
        command = n.command_call
        if n.type in ('DOWNTIMESTART', 'DOWNTIMEEND', 'CUSTOM',
                      'ACKNOWLEDGEMENT', 'FLAPPINGSTART', 'FLAPPINGSTOP',
                      'FLAPPINGDISABLED'):
            state = '%s (%s)' % (n.type, self.state)
        else:
            state = self.state
        if self.__class__.log_notifications:
            naglog_result('critical', "HOST NOTIFICATION: %s;%s;%s;%s;%s"
                          % (contact.get_name(), self.get_name(),
                             state, command.get_name(), self.output))
    
    
    # Raise a log entry with an Eventhandler alert like
    # HOST NOTIFICATION: super admin;server;UP;notify-by-rss;no output
    def raise_event_handler_log_entry(self, command):
        if self.__class__.log_event_handlers:
            naglog_result('critical', "HOST EVENT HANDLER: %s;%s;%s;%s;%s"
                          % (self.get_name(),
                             self.state, self.state_type,
                             self.attempt, command.get_name()))
    
    
    # Raise a log entry with FLAPPING START alert like
    # HOST FLAPPING ALERT: server;STARTED; Host appears to have started flapping (50.6% change >= 50.0% threshold)
    def raise_flapping_start_log_entry(self, change_ratio, threshold):
        naglog_result('critical', "HOST FLAPPING ALERT: %s;STARTED; "
                                  "Host appears to have started flapping "
                                  "(%.1f%% change >= %.1f%% threshold)"
                      % (self.get_name(), change_ratio, threshold))
    
    
    # Raise a log entry with FLAPPING STOP alert like
    # HOST FLAPPING ALERT: server;STOPPED; host appears to have stopped flapping (23.0% change < 25.0% threshold)
    def raise_flapping_stop_log_entry(self, change_ratio, threshold):
        naglog_result('critical', "HOST FLAPPING ALERT: %s;STOPPED; "
                                  "Host appears to have stopped flapping "
                                  "(%.1f%% change < %.1f%% threshold)"
                      % (self.get_name(), change_ratio, threshold))
    
    
    # If there is no valid time for next check, raise a log entry
    def raise_no_next_check_log_entry(self):
        logger.warning("I cannot schedule the check for the host '%s' "
                       "because there is not future valid time"
                       % (self.get_name()))
    
    
    # Raise a log entry when a downtime begins
    # HOST DOWNTIME ALERT: test_host_0;STARTED; Host has entered a period of scheduled downtime
    def raise_enter_downtime_log_entry(self):
        naglog_result('critical', "HOST DOWNTIME ALERT: %s;STARTED; "
                                  "Host has entered a period of scheduled downtime"
                      % (self.get_name()))
    
    
    # Raise a log entry when a downtime has finished
    # HOST DOWNTIME ALERT: test_host_0;STOPPED; Host has exited from a period of scheduled downtime
    def raise_exit_downtime_log_entry(self):
        naglog_result('critical', "HOST DOWNTIME ALERT: %s;STOPPED; Host has "
                                  "exited from a period of scheduled downtime"
                      % (self.get_name()))
    
    
    # Raise a log entry when a downtime prematurely ends
    # HOST DOWNTIME ALERT: test_host_0;CANCELLED; Service has entered a period of scheduled downtime
    def raise_cancel_downtime_log_entry(self):
        naglog_result('critical', "HOST DOWNTIME ALERT: %s;CANCELLED; "
                                  "Scheduled downtime for host has been cancelled."
                      % (self.get_name()))
    
    
    # Is stalking?
    # Launch if check is wait consume==first time
    # and if c.status is in self.stalking_options
    def manage_stalking(self, c):
        need_stalk = False
        if c.status == 'waitconsume':
            if c.exit_status == 0 and 'o' in self.stalking_options:
                need_stalk = True
            elif c.exit_status == 1 and 'd' in self.stalking_options:
                need_stalk = True
            elif c.exit_status == 2 and 'd' in self.stalking_options:
                need_stalk = True
            elif c.exit_status == 3 and 'u' in self.stalking_options:
                need_stalk = True
            if c.output != self.output:
                need_stalk = False
        if need_stalk:
            logger.info("Stalking %s: %s" % (self.get_name(), self.output))
    
    
    # A host is in downtime only by itself, don't care about its services
    def is_in_downtime(self):
        return self.in_scheduled_downtime
    
    
    def is_in_inherited_downtime(self):
        return self.in_inherited_downtime
    
    
    def get_downtime_comment(self):
        return next((downtime.comment for downtime in self.downtimes if downtime.is_in_effect), '')
    
    
    def get_downtime_author(self):
        return next((downtime.author for downtime in self.downtimes if downtime.is_in_effect), '')
    
    
    # A host is in downtime only by itself, don't care about its services
    def is_acknowledged(self):
        return self.problem_has_been_acknowledged
    
    
    def is_in_inherited_acknowledged(self):
        return self.acknowledgement.automatic if self.acknowledgement else False
    
    
    def get_acknowkledge_id(self):
        if self.acknowledgement:
            return self.acknowledgement.id
        elif self.partial_acknowledge:
            return self.partial_acknowledge.id
        else:
            return None
    
    
    # Host number of downtimes is only itself
    def get_number_of_active_downtimes(self):
        return self.scheduled_downtime_depth
    
    
    # fill act_depend_of with my parents (so network dep)
    # and say parents they impact me, no timeperiod and follow parents of course
    def fill_parents_dependency(self):
        for parent in self.parents:
            if parent is not None:
                # I add my parent in my list
                self.act_depend_of.append((parent, ['d', 'u', 's', 'f'], 'network_dep', None, True))
                
                # And I register myself in my parent list too
                parent.register_child(self)
                
                # And add the parent/child dep filling too, for broking
                parent.register_son_in_parent_child_dependencies(self)
    
    
    def fill_default(self):
        # Catch special cluster case, has different flap options
        if not hasattr(self, 'flap_detection_options'):
            if getattr(self, 'check_command', '').startswith('bp_rule'):
                self.flap_detection_options = Service.properties['flap_detection_options'].default
            else:
                self.flap_detection_options = Host.properties['flap_detection_options'].default
        
        super(Host, self).fill_default()
    
    
    # Register a child in our lists
    def register_child(self, child):
        # We've got 2 list: a list for our child
        # where we just put the pointer, it's just for broking
        # and another with all data, useful for 'running' part
        self.childs.append(child)
        self.act_depend_of_me.append((child, ['d', 'u', 's', 'f'], 'network_dep', None, True))
    
    
    # Give data for checks' macros
    def get_data_for_checks(self):
        return [self]
    
    
    # Give data for event handler's macro
    def get_data_for_event_handler(self):
        return [self]
    
    
    # Give data for notifications' macros
    def get_data_for_notifications(self, contact, n):
        return [self, contact, n]
    
    
    # See if the notification is launchable (time is OK and contact is OK too)
    def notification_is_blocked_by_contact(self, n, contact):
        return not contact.want_host_notification(self.last_chk, self.state, n.type, self.business_impact, n.command_call)
    
    
    # MACRO PART
    def get_duration_sec(self):
        return str(int(self.duration_sec))
    
    
    def get_duration(self):
        m, s = divmod(self.duration_sec, 60)
        h, m = divmod(m, 60)
        return "%02dh %02dm %02ds" % (h, m, s)
    
    
    # Check if a notification for this host is suppressed at this time
    # This is a check at the host level. Do not look at contacts here
    def notification_is_blocked_by_item(self, notification_type):
        # type: (unicode) -> bool
        
        # Block if notifications are program-wide disabled
        if not self.enable_notifications:
            return True
        
        # Block if notifications are disabled for this host
        if not self.notifications_enabled:
            return True
        
        # Block if the current status is in the notification_options d,u,r,f,s
        if 'n' in self.notification_options:
            return True
        
        if notification_type in (u'PROBLEM', u'RECOVERY'):
            if self.state == u'DOWN' and not (u'd' in self.notification_options or u'c' in self.notification_options):
                return True
            if self.state == u'UP' and u'r' not in self.notification_options:
                return True
            if self.state == u'UNREACHABLE' and u'u' not in self.notification_options:
                return True
        
        if notification_type in (u'FLAPPINGSTART', u'FLAPPINGSTOP', u'FLAPPINGDISABLED') and u'f' not in self.notification_options:
            return True
        if notification_type in (u'DOWNTIMESTART', u'DOWNTIMEEND', u'DOWNTIMECANCELLED') and u's' not in self.notification_options:
            return True
        
        # Acknowledgements make no sense when the status is ok/up
        if notification_type == u'ACKNOWLEDGEMENT' and self.state == self.ok_up:
            return True
        
        # We need to know how much active downtimes we have
        nb_active_downtimes = self.get_number_of_active_downtimes()
        
        # Flapping
        if notification_type in (u'FLAPPINGSTART', u'FLAPPINGSTOP', u'FLAPPINGDISABLED') and nb_active_downtimes > 0:
            # todo    block if not notify_on_flapping
            return True
        
        # When in deep downtime, only allow end-of-downtime notifications
        # In depth 1 the downtime just started and can be notified
        if nb_active_downtimes > 1 and notification_type in (u'DOWNTIMEEND', u'DOWNTIMECANCELLED'):
            return True
        
        # Block if in a scheduled downtime and a problem arises
        if nb_active_downtimes > 0 and notification_type in (u'PROBLEM', u'RECOVERY'):
            return True
        
        # Block if the status is SOFT
        if self.state_type == u'SOFT' and notification_type == u'PROBLEM':
            return True
        
        # Block if the problem has already been acknowledged
        if self.problem_has_been_acknowledged and notification_type != u'ACKNOWLEDGEMENT':
            return True
        
        # Block if flapping
        if self.is_flapping and notification_type not in (u'FLAPPINGSTART', u'FLAPPINGSTOP', u'FLAPPINGDISABLED'):
            return True
        
        return False
    
    
    # Get an oc*p command if item has obsess_over_*
    # command. It must be enabled locally and globally
    def get_obsessive_compulsive_processor_command(self):
        cls = self.__class__
        if not cls.obsess_over or not self.obsess_over_host:
            return
        
        m = MacroResolver()
        data = self.get_data_for_event_handler()
        cmd = m.resolve_command(cls.ochp_command, data)
        e = EventHandler(cmd, timeout=cls.ochp_timeout, command_name=self.get_check_command())
        
        # ok we can put it in our temp action queue
        self.actions.append(e)
    
    
    # Macro part
    def get_total_services(self):
        return str(len(self.services))
    
    
    def get_total_services_ok(self):
        return str(len([s for s in self.services if s.state_id == 0]))
    
    
    def get_total_services_warning(self):
        return str(len([s for s in self.services if s.state_id == 1]))
    
    
    def get_total_services_critical(self):
        return str(len([s for s in self.services if s.state_id == 2]))
    
    
    def get_total_services_unknown(self):
        return str(len([s for s in self.services if s.state_id == 3]))
    
    
    def get_ack_author_name(self):
        if self.acknowledgement is None:
            return ''
        return self.acknowledgement.author
    
    
    def get_ack_comment(self):
        if self.acknowledgement is None:
            return ''
        return self.acknowledgement.comment
    
    
    def get_check_command(self):
        return self.check_command.get_name()
    
    
    # For service override we have special inheritance
    def get_service_overrides_by_inheritance(self, items, only_one_level, level=0, already_seen_templates=None):
        # Protect against too much recursion
        if level >= MAX_INHERITANCE_LEVEL:
            return None
        
        if already_seen_templates is None:
            already_seen_templates = set()
        
        if not (only_one_level and level >= 1):
            for template in self.templates:
                # If we have a template loop, we should not lose time here
                if template in already_seen_templates:
                    continue
                already_seen_templates.add(template)
                
                tpl_value = template.get_service_overrides_by_inheritance(items, only_one_level, level=level + 1, already_seen_templates=already_seen_templates)
                if tpl_value is None:
                    continue
                
                if hasattr(self, 'service_overrides'):
                    current_value = getattr(self, 'service_overrides')
                    try:
                        current_value_parse = parse_service_override_property(current_value)
                    except SyntaxServiceOverrideError as e:
                        # Service override must be validated before. This log shouldn't be called.
                        logger.warning(u'Fail to parse service overrides in %s with error : %s' % (self.get_name(), e))
                        continue
                    try:
                        tpl_value_parse = parse_service_override_property(tpl_value)
                    except SyntaxServiceOverrideError as e:
                        # Service override must be validated before. This log shouldn't be called.
                        logger.warning(u'Fail to parse service overrides in %s with error : %s' % (self.get_name(), e))
                        continue
                    for check, so_value in tpl_value_parse.iteritems():
                        so_value.update(current_value_parse.get(check, {}))
                        current_value_parse[check] = so_value
                    tpl_value_parse.update(current_value_parse)
                    
                    tpl_value = unparse_service_override(tpl_value_parse)
                
                setattr(self, 'service_overrides', tpl_value)
        return getattr(self, 'service_overrides', None)
    
    
    def combine_two_view_contacts(self, first_contact, second_contact):
        # type: (unicode, unicode) -> unicode
        real_first_contact = self.default_view_contacts_value if first_contact in (u'__DEFAULT_NO_TEMPLATE__', u'null') else first_contact
        real_second_contact = self.default_view_contacts_value if second_contact in (u'__DEFAULT_NO_TEMPLATE__', u'null') else second_contact
        
        if not first_contact or real_first_contact == u'nobody':
            return second_contact
        
        if not second_contact or real_second_contact == u'nobody':
            return first_contact
        
        if real_first_contact == u'everyone' or real_second_contact == u'everyone':
            return u'null'
        
        return u'%s,%s' % (first_contact, second_contact)
    
    
    def get_view_contacts_by_inheritance(self, items, only_one_level, level=0, already_seen_templates=None):
        # Protect against too much recursion
        if level >= MAX_INHERITANCE_LEVEL:
            return None
        
        if already_seen_templates is None:
            already_seen_templates = set()
        
        if getattr(self, u'view_contacts', None) and not self.has_plus(u'view_contacts'):
            return getattr(self, u'view_contacts', None)
        
        if not (only_one_level and level >= 1):
            previous_tpl_value = None
            for template in self.templates:
                still_loop = False
                # If we have a template loop, we should not lose time here
                if template in already_seen_templates:
                    continue
                already_seen_templates.add(template)
                
                tpl_value = template.get_view_contacts_by_inheritance(items, only_one_level, level=level + 1, already_seen_templates=already_seen_templates)
                
                if tpl_value is None:
                    continue
                
                if tpl_value.startswith(u'+'):
                    tpl_value = tpl_value[1:]
                    still_loop = True
                
                tpl_value = self.combine_two_view_contacts(previous_tpl_value, tpl_value)
                if tpl_value:
                    setattr(self, u'view_contacts', _format_inherited_plus_value(tpl_value))
                
                if not still_loop:
                    break
                
                previous_tpl_value = tpl_value
        
        value = getattr(self, u'view_contacts', None)
        plus_value = None
        if self.has_plus(u'view_contacts'):
            plus_value = self.get_plus_and_delete(u'view_contacts')
            value = self.combine_two_view_contacts(value, plus_value)
        
        if (not plus_value or plus_value not in (u'__DEFAULT_NO_TEMPLATE__', u'null')) and value and value in (u'__DEFAULT_NO_TEMPLATE__', u'null') and self.default_view_contacts_value == u'everyone':
            # SEF-8624, TODO DELETE THIS LOG AFTER FEW MONTHS, IT IS JUST SECURITY
            logger.debug(u'Host with name [ %s ] inherited [ everyone ] from template' % self.get_name())
        
        # Template should keep their '+'
        if value and self.is_tpl() and not value.startswith(u'+'):
            value = u'+%s' % value
        
        if value:
            setattr(self, u'view_contacts', _format_inherited_plus_value(value))
        
        return getattr(self, u'view_contacts', None)
    
    
    # Make all a template tree that it does manage a check/service
    def set_templates_with_service_uuid_recursive(self, all_hosts, service_uuid):
        self.set_in_template_service_uuid(service_uuid)
        # logger.debug('SET:: %s in %s' % (service_uuid, self.get_name()))
        all_my_sons = all_hosts.get_my_son_templates(self)
        for template in all_my_sons:
            template.set_in_template_service_uuid(service_uuid)
            # logger.debug('SET SON:: %s in %s' % (service_uuid, template.get_name()))
    
    
    def set_in_template_service_uuid(self, service_uuid):
        if not hasattr(self, '_list_of_service_uuids'):
            self._list_of_service_uuids = []
        if service_uuid not in self._list_of_service_uuids:
            self._list_of_service_uuids.append(service_uuid)
    
    
    def get_in_template_service_uuids(self):
        return getattr(self, '_list_of_service_uuids', [])
    
    
    # We need to have our service exclude property, but beware: at this point
    # templates are not pythonize, so we still have a string here
    def _get_my_service_excludes(self):
        service_excludes = getattr(self, 'service_excludes', '')
        service_excludes = self.__class__.properties['service_excludes'].pythonize(service_excludes)
        return service_excludes
    
    
    # same but for by id exclude
    def _get_my_service_excludes_by_id(self):
        service_excludes_by_id = getattr(self, 'service_excludes_by_id', '')
        service_excludes_by_id = self.__class__.properties['service_excludes_by_id'].pythonize(service_excludes_by_id)
        return service_excludes_by_id
    
    
    # NOTE: the synchronizer can give us the name in 2 ways:
    # UUID => it found the real check, so we are sure that we must take this one
    #           We check for service duplicate for each if the dfe_key match
    # NAME => it only has its name, like in duplicate foreach, so we fall back to this
    @staticmethod
    def parse_service_exclusion_expr(name_expr):
        # name_expr for link format is :  uuid!UUID!dfe_key!DEF_KEY or uuid!UUID
        name_expr_split = name_expr.split('!')
        uuid_to_search = name_expr_split[1]
        dfe_key = name_expr_split[3] if len(name_expr_split) == 4 else '-'  # For service excludes, there is no DFE key
        return uuid_to_search, dfe_key
    
    
    def is_service_excluded_by_template(self, service, level=0):
        # type: (Service,int)->str
        if level > MAX_INHERITANCE_LEVEL:
            return EXCLUDE_FROM_TEMPLATES_RESULTS.ALLOW
        check_definition_on_this_template = self.get_in_template_service_uuids()
        # logger.debug('EXCLUSION: %s/%s is from this template %s?' % (service.host_name, service.get_name(), self.get_name()))
        # We need to check 2 cases:
        # * the check is a genuine template check, so have its own uuid
        # * the check came from a duplicate foreach, so its uuid is a new, and the template know only about the original one
        # so take the definition uuid because the template only know about this one, not the final object
        if service.get_uuid_from_definition_object() not in check_definition_on_this_template:
            # logger.debug('EXCLUSION: %s/%s is from this template %s? NON (%s != %s)' % (service.host_name, service.get_name(), self.get_name(), service.get_uuid_from_definition_object(), check_definition_on_this_template))
            return EXCLUDE_FROM_TEMPLATES_RESULTS.DONT_HAVE  # don't care
        
        # V02.07.05 : DFE check can't be excluded on host by template because there are generated by the host
        # V02.07.06 : If the template exclude by name the common part of the check, the check should be excluded (SEF-7341)
        if service.is_dfe():
            service_description_no_dfe_name = service.get_name_without_dfe()
            by_pattern_template_service_excludes = self._get_my_service_excludes()
            for template_service_exclude in by_pattern_template_service_excludes:
                if self._is_matching_name_regex_star_only(service_description_no_dfe_name, template_service_exclude):
                    return EXCLUDE_FROM_TEMPLATES_RESULTS.EXCLUDE
            return EXCLUDE_FROM_TEMPLATES_RESULTS.DONT_HAVE
        
        # WE DO HAVE the check, so cannot send DONT_HAVE now
        if self.is_check_exclude(service):
            return EXCLUDE_FROM_TEMPLATES_RESULTS.EXCLUDE
        else:
            result = EXCLUDE_FROM_TEMPLATES_RESULTS.ALLOW
            for sub_template in self.templates:
                sub_result = sub_template.is_service_excluded_by_template(service, level=level + 1)
                if sub_result == EXCLUDE_FROM_TEMPLATES_RESULTS.DONT_HAVE:
                    continue
                elif sub_result == EXCLUDE_FROM_TEMPLATES_RESULTS.ALLOW:
                    result = EXCLUDE_FROM_TEMPLATES_RESULTS.ALLOW
                elif sub_result == EXCLUDE_FROM_TEMPLATES_RESULTS.EXCLUDE:
                    return EXCLUDE_FROM_TEMPLATES_RESULTS.EXCLUDE
                else:
                    raise Exception('No such sub-result')
            return result
    
    
    def is_check_exclude(self, service):
        service_description = getattr(service, 'service_description', '__NO_SERVICE_DESCRIPTION__')
        
        # We have patterns' name, and uuids list
        by_pattern_template_service_excludes = self._get_my_service_excludes()
        by_id_template_service_excludes = self._get_my_service_excludes_by_id()
        if not by_pattern_template_service_excludes and not by_id_template_service_excludes:
            return False
        
        # Name pattern
        for template_service_exclude in by_pattern_template_service_excludes:
            if self._is_matching_name_regex_star_only(service_description, template_service_exclude):
                return True
        
        # By uuids
        for by_id_template_service_exclude in by_id_template_service_excludes:
            # parse uuid!UUID => UUID to know which uuid we need to erase
            uuid_to_exclude, _ = self.parse_service_exclusion_expr(by_id_template_service_exclude)
            if service.uuid == uuid_to_exclude:
                return True
        
        # no exclusion founded
        return False
    
    
    @staticmethod
    def _is_matching_name_regex_star_only(name_service, exclude_by_name_label):
        exclude_by_name_label_escape = re.escape(exclude_by_name_label)
        exclude_by_name_label_escape = exclude_by_name_label_escape.replace('\\*', ".*")
        return re.match(r'^%s$' % exclude_by_name_label_escape, name_service)
    
    
    def build_service_index(self):
        service_index = {}
        service_index_by_name = {}
        service_index_by_uuid = {}
        for service in self.services:
            service_id = service.id
            service_name = getattr(service, 'service_description', '__UNNAMED_SERVICE__')
            service_uuid = getattr(service, 'uuid', '__NO_UUID__')
            service_index[service_id] = service
            service_index_by_name[service_name] = service_id
            service_index_by_uuid[service_uuid] = service_id
        setattr(self, 'service_index', service_index)
        setattr(self, 'service_index_by_name', service_index_by_name)
        setattr(self, 'service_index_by_uuid', service_index_by_uuid)


# CLass for the hosts lists. It's mainly for configuration
# part
class Hosts(InheritableItems):
    name_property = "host_name"  # use for the search by name
    inner_class = Host  # use for know what is in items
    
    is_cluster_pattern = re.compile("^bp_rule(!|$)")
    
    
    def __init__(self, items):
        super(Hosts, self).__init__(items)
        self.__template_to_host_cache = {}  # cache for looking for hosts that match a specific template
    
    
    # Create link between elements:
    # hosts -> time periods
    # hosts -> hosts (parents, etc)
    # hosts -> commands (check_command)
    # hosts -> contacts
    def linkify(self, timeperiods, commands, contacts, realms, resultmodulations, businessimpactmodulations, escalations, hostgroups, checkmodulations, macromodulations):
        self.linkify_with_timeperiods(timeperiods, 'notification_period')
        self.linkify_with_timeperiods(timeperiods, 'check_period')
        self.linkify_with_timeperiods(timeperiods, 'maintenance_period')
        self.linkify_h_by_realms(realms)  # note: realm before host parents, as we need the name into it
        self.linkify_h_by_h()
        self.linkify_h_by_hg(hostgroups)
        self.linkify_one_command_with_commands(commands, 'check_command')
        self.linkify_one_command_with_commands(commands, 'event_handler')
        
        self.linkify_with_contacts(contacts)
        self.linkify_with_resultmodulations(resultmodulations)
        self.linkify_with_business_impact_modulations(businessimpactmodulations)
        # WARNING: all escalations will not be linked here
        # (just the escalation here, not service esca or host esca).
        # This last one will be link in escalations linkify.
        self.linkify_with_escalations(escalations)
        # self.linkify_with_triggers(triggers)
        self.linkify_with_checkmodulations(checkmodulations)
        self.linkify_with_macromodulations(macromodulations)
    
    
    # Fill address by host_name if not set
    def fill_predictive_missing_parameters(self):
        for h in self:
            h.fill_predictive_missing_parameters()
    
    
    # Links host with hosts (parents)
    def linkify_h_by_h(self):
        for h in self:
            parents = h.parents
            # The new member list
            new_parents = []
            for parent in parents:
                parent = parent.strip()
                if parent == '':
                    continue
                p = self.find_by_name(parent)
                if p is not None:
                    new_parents.append(p)
                else:
                    h_realm = h.realm
                    if h_realm is None:
                        realm_name = '(missing realm for this element)'
                    else:
                        realm_name = h_realm.get_name()
                    err = "the network dependency '%s' on host '%s' is disabled or does not exist in the realm '%s'" % (parent, h.get_name(), realm_name)
                    self.configuration_errors.append(err)
            # print "Me,", h.host_name, "define my parents", new_parents
            # We find the id, we replace the names
            h.parents = list(new_parents)
    
    
    # Link with realms and set a default realm if none
    def linkify_h_by_realms(self, realms):
        default_realm = None
        for r in realms:
            if getattr(r, 'default', False):
                default_realm = r
        # if default_realm is None:
        #    print "Error: there is no default realm defined!"
        for h in self:
            if h.realm is not None:
                p = realms.find_by_name(h.realm.strip())
                if p is None:
                    err = "the host %s got an invalid realm (%s)." % (h.get_name(), h.realm)
                    h.configuration_errors.append(err)
                h.realm = p
            else:
                # print "Notice: applying default realm %s to host %s" % (default_realm.get_name(), h.get_name())
                h.realm = default_realm
                h.got_default_realm = True
    
    
    # We look for hostgroups property in hosts and
    # link them
    def linkify_h_by_hg(self, hostgroups):
        # Register host in the hostgroups
        for h in self:
            is_cluster = self.is_cluster_pattern.search(h.check_command)
            if not h.is_tpl() and not is_cluster:
                new_hostgroups = []
                if hasattr(h, 'hostgroups') and len(h.hostgroups) != 0:
                    hgs = h.hostgroups
                    for hg_name in hgs:
                        hg_name = hg_name.strip()
                        hg = hostgroups.find_by_name(hg_name)
                        if hg is not None:
                            new_hostgroups.append(hg)
                        else:
                            err = "the hostgroup '%s' of the host '%s' is disabled or does not exist" % (hg_name, h.host_name)
                            h.configuration_errors.append(err)
                h.hostgroups = list(new_hostgroups)
            elif is_cluster:
                # Clusters have no business being in host groups
                h.hostgroups = []
    
    
    # We look for host groups property in hosts and
    # noinspection PyUnusedLocal, SpellCheckingInspection
    def explode(self, hostgroups, contactgroups):
        # Register host in the hostgroups
        for h in self:
            if not h.is_tpl() and hasattr(h, 'host_name'):
                hname = h.host_name
                if hasattr(h, 'hostgroups'):
                    if isinstance(h.hostgroups, list):
                        h.hostgroups = ','.join(h.hostgroups)
                    hgs = h.hostgroups.split(',')
                    for hg in hgs:
                        hostgroups.add_member(hname, hg.strip())
    
    
    # In the scheduler we need to relink the commandCall with the real commands
    def late_linkify_h_by_commands(self, commands):
        props = ['check_command', 'event_handler']
        for h in self:
            for prop in props:
                cc = getattr(h, prop, None)
                if cc:
                    cc.late_linkify_with_command(commands)
            
            # Ok also link checkmodulations
            for cw in h.checkmodulations:
                cw.late_linkify_cw_by_commands(commands)
    
    
    # Create dependencies:
    # Dependencies at the host level: host parent
    def apply_dependencies(self):
        for h in self:
            h.fill_parents_dependency()
    
    
    # Parent graph: use to find quickly relations between all host, and loop
    # return True if there is a loop
    def no_loop_in_parents(self):
        # Ok, we say "from now, no loop :) "
        r = True
        
        # Create parent graph
        parents = Graph()
        
        # With all hosts as nodes
        for h in self:
            if h is not None:
                parents.add_node(h)
        
        # And now fill edges
        for h in self:
            for p in h.parents:
                if p is not None:
                    self._add_item_in_parents_list(parents, p, h)
        
        # Now get the list of all hosts in a loop
        host_in_loops = parents.loop_check()
        
        # and raise errors about it
        for h in host_in_loops:
            log_configuration_error(h, u'The host \'%s\' is part of a circular parent/child chain!', h.get_name())
            r = False
        
        return r
    
    
    def _add_item_in_parents_list(self, parents, item_to_add, myself):
        parents.add_edge(item_to_add, myself)
        if item_to_add.got_business_rule:
            for item in item_to_add.business_rule.list_all_elements():
                if item.type in ['host', 'cluster']:
                    self._add_item_in_parents_list(parents, self.find_by_uuid(item.uuid), item_to_add)
                elif item.type == 'check':
                    self._add_item_in_parents_list(parents, self.find_by_uuid(item.host_uuid), item_to_add)
    
    
    # Return a list of the host_name of the hosts
    # that got the template with name=tpl_name or inherit from
    # a template that use it
    def find_hosts_that_use_template(self, tpl_name):
        # Cache hit?
        if tpl_name in self.__template_to_host_cache:
            return self.__template_to_host_cache[tpl_name]
        res = set()
        tpl_name = tpl_name.strip()
        
        # First find the template
        tpl = self.find_template(tpl_name)
        
        # If we find none, we should manually look up all hosts to find this 'tag'
        if tpl is None:
            # NOP => DISABLING TAG research because it's too difficult for users to understand
            self.__template_to_host_cache[tpl_name] = []  # update the cache
            return []
        
        # Ok, we find the tpl. We should find its father template too
        for t in self.templates.itervalues():
            t.dfs_loop_status = 'DFS_UNCHECKED'
        all_tpl_searched = self.templates_graph.dfs_get_all_childs(tpl)
        # Clean the search tag
        # TODO: better way?
        for t in self.templates.itervalues():
            del t.dfs_loop_status
        
        # Now we got all the templates we are looking for (so the template
        # and all its own templates too, we search for the hosts that are
        # using them
        for h in self:
            # If the host is a not valid one, skip it
            if not hasattr(h, 'host_name'):
                continue
            # look if there is a match between host templates
            # and the ones we are looking for
            for t in h.templates:
                if t in all_tpl_searched:
                    res.add(h.host_name)
                    continue
        
        res = list(res)  # need a list, not a set()
        # update the cache, so we don't do this research again
        self.__template_to_host_cache[tpl_name] = res
        return res
    
    
    def find_template(self, tpl_name):
        tpl = None
        for h in self.templates.itervalues():
            # Look for template with the good name
            if hasattr(h, 'name') and h.name.strip() == tpl_name:
                tpl = h
        return tpl
    
    
    def get_my_son_templates(self, from_template, level=0):
        if level > MAX_INHERITANCE_LEVEL:
            return set()
        sons = set()
        for template in self.templates.itervalues():
            for sub_template in template.templates:
                if sub_template == from_template:
                    sons.add(template)
                    sons.update(self.get_my_son_templates(template, level + 1))
        return sons
    
    
    # Will create all business tree for the
    # services
    def create_business_rules(self, proxy_items):
        for h in self:
            h.create_business_rules(proxy_items)
    
    
    # Will link all business service/host with theirs
    # dep for problem/impact link
    def create_business_rules_dependencies(self, hosts, services):
        for h in self:
            h.create_business_rules_dependencies(hosts, services)
    
    
    # The contactgroups -> contacts explode must be done only AFTER
    def post_inheritance_explode(self, contactgroups):
        t0 = time.time()
        # take all contacts from our contact_groups into our contact property
        self.explode_contact_groups_into_contacts(contactgroups)
        logger.info('[performance] - (detail for support only) explode::hosts::explode_contact_groups_into_contacts   %.2f' % (time.time() - t0))
    
    
    def create_reversed_list(self):
        # type: () -> None
        InheritableItems.create_reversed_list(self)
        
        name_property = self.__class__.name_property
        self.reversed_list_name_lower = {}
        for _id, item in self.items.iteritems():
            if hasattr(item, name_property):
                name = getattr(item, name_property)
                self.reversed_list_name_lower[name.lower()] = _id
    
    
    def find_id_by_name_case_insensitive(self, name):
        # type: (unicode) -> Optional[int]
        
        name_lower = name.lower()
        if hasattr(self, 'reversed_list_name_lower'):
            if name_lower in self.reversed_list_name_lower:
                return self.reversed_list_name_lower[name_lower]
            else:
                return None
        else:  # ok, an early ask, with no reversed list from now...
            name_property = self.__class__.name_property
            for i in self:
                if hasattr(i, name_property):
                    i_name = getattr(i, name_property)
                    if i_name.lower() == name_lower:
                        return i.id
            return None
    
    
    def get_host_case_insensitive(self, name):
        # type: (unicode) -> Optional[Host]
        _id = self.find_id_by_name_case_insensitive(name)
        if _id is not None:
            return self.items[_id]
        else:
            return None
    
    
    @staticmethod
    def _apply_inheritance_from_command(host, commands):
        # type: (Host, Commands) -> None
        for property_name in PROPERTIES_RETRIEVABLE_FROM_COMMAND_TO_HOST:
            if host.apply_inheritance_from_command(host, commands, property_name):
                continue
            
            if hasattr(host, property_name):
                setattr(host, u'%s_from' % property_name, u'host')
                continue
            
            setattr(host, u'%s_from' % property_name, u'global')
    
    
    def apply_implicit_inheritance(self, commands):
        # type:(Commands) -> None
        for host in self:
            if not host.is_tpl():
                self._apply_inheritance_from_command(host, commands)
    
    
    def load_value_from_global_conf(self):
        # type:() -> None
        for host in self:
            host.load_value_from_global_conf()
