#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import Set, NoReturn, Dict
from shinken.log import PartLogger
from shinkensolutions.api.synchronizer import ITEM_TYPE, DEF_ITEMS
from shinkensolutions.api.synchronizer.source.item.source_item import SourceItem

SyncKey = str  # Type for SyncKey. Used only for sync_key


class SyncKeysManager(object):
    
    def __init__(self, logger, properties_used_as_synckey):
        # type: (PartLogger, Dict) -> None
        self.properties_used_as_synckey = properties_used_as_synckey
        self.logger = logger
    
    
    @staticmethod
    def _compute_minimal_sync_keys(item):
        # type: (SourceItem) -> Set[SyncKey]
        item_type = item.item_type
        is_template = ITEM_TYPE.is_template(item_type)
        new_sync_keys = []
        
        if item_type in ITEM_TYPE.ALL_DEDICATED_SERVICES:
            item_sync_keys_uuid = getattr(item, '_SE_UUID', None)
            if item_sync_keys_uuid:
                new_sync_keys.append(item_sync_keys_uuid.lower())
        else:
            item_name = getattr(item, DEF_ITEMS[item_type]['key_name'], None)
            item_sync_keys_name = ('%s-tpl' % item_name if is_template else item_name)
            if item_sync_keys_name:
                new_sync_keys.append(item_sync_keys_name.lower())
            
            item_sync_keys_uuid = getattr(item, '_SE_UUID', None)
            if item_sync_keys_uuid:
                new_sync_keys.append(item_sync_keys_uuid.lower())
        
        return set(new_sync_keys)
    
    
    def compute_sync_keys(self, item):
        # type: (SourceItem) -> NoReturn
        sync_keys = set()
        _properties_used_as_synckey = self.properties_used_as_synckey.get(item.item_type, [])
        if not _properties_used_as_synckey:
            return
        
        for _property_used_as_synckey in _properties_used_as_synckey:
            sync_key_value = getattr(item, _property_used_as_synckey, '')
            if not sync_key_value:
                continue
            
            sync_key_value = sync_key_value.lower()
            
            must_add_tpl_suffix = ITEM_TYPE.is_template(item.item_type) and 'name' == _property_used_as_synckey
            if must_add_tpl_suffix:
                sync_key_value = '%s-tpl' % sync_key_value
            
            sync_keys.add(sync_key_value)
        
        sync_keys |= self._compute_minimal_sync_keys(item)
        item._SYNC_KEYS = (','.join(sync_keys))
