#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved

from shinken.basesubprocess import EventHandler
from shinken.log import PartLogger
from shinken.misc.type_hint import Optional

from shinken.basemodule import SOURCE_STATE
from shinkensolutions.api.synchronizer import SourceTranslatePart, component_manager
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.distributor.option_distributor import SELECTOR_CONST
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.distributor.selector_distributor import instance_selector_distributor
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.html_distributor import HtmlDistributor
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.object.html.label import LabelHtml
from shinkensolutions.api.synchronizer.source.route.route import AbstractRoute
from shinkensolutions.api.synchronizer.source.rules_application_template import SourceRulesManager
from shinkensolutions.api.synchronizer.source.tab.tab import AbstractTab
from shinkensolutions.api.synchronizer.source.source_info import SourceInfo



class TabHostTemplateBindingRules(AbstractTab):
    
    def __init__(self, logger, translator, source_name, host_template_binding_rule_manager, source_info):
        # type: (PartLogger, SourceTranslatePart, str, SourceRulesManager, SourceInfo ) -> None
        super(TabHostTemplateBindingRules, self).__init__(logger, translator, source_name, 'host_template_binding_rules', translator.translate('host_template_binding_rules.tab_display_name'), source_info)
        self.host_template_binding_rule_manager = host_template_binding_rule_manager
        self.html_distributor = None  # type: Optional[HtmlDistributor]
    
    
    def controller(self):
        common_translator = component_manager.get_translate_component().translator()
        for rule in self.host_template_binding_rule_manager.rules:
            rule.compute_validation_message_to_html()
        return {
            '__shinken_lang__'   : self.translator.get_lang(),
            'tab_info'           : self,
            'source_name'        : self.source_name,
            'translate'          : self.translator.translate,
            'common_translator'  : common_translator,
            'rules'              : self.host_template_binding_rule_manager.rules,
            'html_distributor'   : self.init_html_object(),
            'validation_state'   : self.host_template_binding_rule_manager.validation_state.as_html(),
            'files_configuration': self.host_template_binding_rule_manager.user_rules_files
        }
    
    
    def init_html_object(self):
        # type: ()-> HtmlDistributor
        if not self.html_distributor:
            _to_return = HtmlDistributor(self.translator.translate)
            # BUTTONS
            _to_return.add_button('reload', LabelHtml('''<span class="shinicon-undo"></span>'''))
            # FILTERS
            _to_return.add_list_filter('define_by', type_filter='select-single', content=instance_selector_distributor.find_element(SELECTOR_CONST.FILTER_DEFINE_BY))
            _to_return.add_list_filter('index', placeholder='--')
            _to_return.add_list_filter('name', placeholder='--')
            _to_return.add_list_filter('conditions', placeholder='--')
            _to_return.add_list_filter('templates', placeholder='--')
            _to_return.add_list_filter('counters', type_filter='select-single', content=instance_selector_distributor.find_element(SELECTOR_CONST.FILTER_COUNTERS))
            self.html_distributor = _to_return
        return self.html_distributor


class RouteReloadTemplateBindingRules(AbstractRoute):
    
    def __init__(self, logger, source_name, host_template_binding_rule_manager, host_template_binding_rule_event, source_info):
        # type: (PartLogger, str, SourceRulesManager, EventHandler, SourceInfo) -> None
        super(RouteReloadTemplateBindingRules, self).__init__(logger, source_name, 'template_binding_rules/reload', source_info, have_http_template=False)
        self.host_template_binding_rule_manager = host_template_binding_rule_manager
        self.host_template_binding_rule_event = host_template_binding_rule_event
    
    
    def controller(self):
        if not self.share_info.import_in_progressed:
            self.host_template_binding_rule_manager.reload_rule()
            self.host_template_binding_rule_event.send_event()
            return SOURCE_STATE.OK
        else:
            return SOURCE_STATE.RUNNING
