#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import os.path
import shutil
import sys
from os import listdir
from os.path import dirname, abspath, join, exists
from shutil import copy

# If DEBUG = False, it will override existing configuration folders
DEBUG = False

# Useful folders
SCRIPT_FOLDER = dirname(abspath(__file__))
GENERATED_CFG_FOLDER = join(SCRIPT_FOLDER, u'cfg')
ROOT_FOLDER = dirname(dirname(dirname(dirname(SCRIPT_FOLDER))))

# Destinations folders
ETC_SHINKEN = join(ROOT_FOLDER, u'sources', u'shinken-patched', u'etc')
SOURCES_ORIGINAL_CONF = join(ROOT_FOLDER, u'sources', u'modules-original-configurations')

SOURCES_MODULES = join(ETC_SHINKEN, u'modules')
SOURCES_MODULES_SOURCES = join(ETC_SHINKEN, u'sources')
SOURCES_DEFAULT_MODULES = join(ETC_SHINKEN, u'_default', u'daemons')
TOOLS_USED_BY_SHINKEN = join(ETC_SHINKEN, u'tools_used_by_shinken')

SKELETONS = join(ROOT_FOLDER, u'tarball', u'shinken-enterprise-tarball', u'skeletons')
SKELETONS_ARBITER_MODULES = join(SKELETONS, u'configuration', u'daemons', u'arbiters', u'modules')
SKELETONS_BROKER_MODULES = join(SKELETONS, u'configuration', u'daemons', u'brokers', u'modules')
SKELETONS_SCHEDULERS_MODULES = join(SKELETONS, u'configuration', u'daemons', u'schedulers', u'modules')
SKELETONS_SYNCHRONIZER_SOURCES = join(SKELETONS, u'configuration', u'daemons', u'synchronizers', u'sources')

SKELETONS_EXAMPLES = join(SKELETONS, u'examples')
SKELETONS_TOOLS_USED_BY_SHINKEN_EXAMPLES = join(SKELETONS_EXAMPLES, u'configuration', u'tools_used_by_shinken')
SKELETONS_BROKER_MODULES_EXAMPLES = join(SKELETONS_EXAMPLES, u'configuration', u'daemons', u'brokers', u'modules')
SKELETONS_ARBITER_MODULES_EXAMPLES = join(SKELETONS_EXAMPLES, u'configuration', u'daemons', u'arbiters', u'modules')
SKELETONS_SYNCHRONIZER_SOURCES_EXAMPLES = join(SKELETONS_EXAMPLES, u'configuration', u'daemons', u'synchronizers', u'sources')


def _get_destination_file_name(file_name):
    return file_name if not DEBUG else u'%s.DEBUG' % file_name


def _get_module_original_configuration(module_name, file_name):
    return join(SOURCES_ORIGINAL_CONF, module_name, u'etc', u'modules', _get_destination_file_name(file_name))


# Change this to add / remove destination folders
CFG_FILE_DEST_MAPPING = {
    u'architecture-export.cfg'         : [
        join(SOURCES_MODULES, _get_destination_file_name(u'architecture-export.cfg')),
        join(SKELETONS_ARBITER_MODULES_EXAMPLES, u'architecture-export', _get_destination_file_name(u'architecture-export-example.cfg'))
    ],
    u'broker-module-livedata.cfg'      : [
        join(SOURCES_MODULES, _get_destination_file_name(u'broker-module-livedata.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, u'broker-module-livedata', _get_destination_file_name(u'broker-module-livedata-example.cfg'))
    ],
    u'discovery.cfg'                   : [
        join(SOURCES_MODULES_SOURCES, _get_destination_file_name(u'discovery.cfg')),
        join(SKELETONS_SYNCHRONIZER_SOURCES_EXAMPLES, u'discovery-import', _get_destination_file_name(u'discovery-example.cfg'))
    ],
    u'event_manager_reader.cfg'        : [
        join(SOURCES_MODULES, _get_destination_file_name(u'event_manager_reader.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, u'webui', u'modules', u'event_manager_reader', _get_destination_file_name(u'event_manager_reader-example.cfg'))
    ],
    u'event_manager_writer.cfg'        : [
        join(SOURCES_MODULES, _get_destination_file_name(u'event_manager_writer.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, u'event_manager_writer', _get_destination_file_name(u'event_manager_writer-example.cfg'))
    ],
    u'graphite.cfg': [
        join(SOURCES_MODULES, _get_destination_file_name(u'graphite.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, u'graphite_perfdata', _get_destination_file_name(u'graphite_perfdata-example.cfg'))
    ],
    u'livedata-module-sla-provider.cfg': [
        join(SOURCES_MODULES, _get_destination_file_name(u'livedata-module-sla-provider.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, u'broker-module-livedata', u'modules', u'livedata-module-sla-provider', _get_destination_file_name(u'livedata-module-sla-provider-example.cfg'))
    ],
    u'mongodb.cfg'                     : [
        join(SOURCES_MODULES, _get_destination_file_name(u'mongodb.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, u'webui', u'modules', u'mongodb', _get_destination_file_name(u'mongodb-example.cfg'))
    ],
    u'mongos_socket_keep_alive.cfg'    : [
        join(TOOLS_USED_BY_SHINKEN, u'shinken_mongo', _get_destination_file_name(u'mongos_socket_keep_alive.cfg')),
        join(SKELETONS_TOOLS_USED_BY_SHINKEN_EXAMPLES, u'shinken_mongo', _get_destination_file_name(u'mongos_socket_keep_alive-example.cfg'))
    ],
    u'retention_mongodb.cfg'           : [
        join(SOURCES_MODULES, _get_destination_file_name(u'retention-mongodb.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, u'retention-mongodb', _get_destination_file_name(u'retention-mongodb-example.cfg'))
    ],
    u'sla.cfg'                         : [
        join(SOURCES_MODULES, _get_destination_file_name(u'sla.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, u'sla', _get_destination_file_name(u'sla-example.cfg')),
        join(ROOT_FOLDER, u'sources', u'modules-original-configurations', u'sla', u'etc', u'modules', _get_destination_file_name(u'sla.cfg'))
    ],
    u'syncui.cfg' : [
        join(SKELETONS_SYNCHRONIZER_SOURCES_EXAMPLES, u'syncui', _get_destination_file_name(u'syncui-example.cfg'))
    ]
}


def copy_files():
    if not exists(GENERATED_CFG_FOLDER):
        print u'━━━━━━━━\033[31mERROR\033[0m : Generated configuration files not found at location :━━━━━━━━\n\t%s' % GENERATED_CFG_FOLDER
        sys.exit(1)
    for cfg_file in listdir(GENERATED_CFG_FOLDER):
        if cfg_file not in CFG_FILE_DEST_MAPPING:
            continue
        print u'Found %s' % cfg_file
        for destination in CFG_FILE_DEST_MAPPING[cfg_file]:
            source_file = join(GENERATED_CFG_FOLDER, cfg_file)
            print u'\t➞ \033[36m%s\033[0m' % destination
            if destination.endswith(u'example.cfg') and cfg_file != u'syncui.cfg':
                _create_example_file(source_file, destination)
            else:
                if not os.path.isdir(os.path.dirname(destination)):
                    os.makedirs(os.path.dirname(destination))
                copy(source_file, destination)
        print u''
    print u'━━━━━━━━ \033[32mDone\033[0m : Configuration deployment finished ʕ•ᴥ•ʔ ━━━━━━━━'


def _create_example_file(source, destination):
    with open(source, u'r') as f:
        source_content = f.readlines()
    
    for idx, line in enumerate(source_content):
        if line.strip().startswith('module_name') or line.strip().startswith('source_name'):
            name = line.strip().split()[1]
            new_name = '%s-example' % name
            source_content[idx] = line.replace(name, new_name)
    
    tmp_destination = u'%s.tmp' % destination
    if not os.path.isdir(os.path.dirname(tmp_destination)):
        os.makedirs(os.path.dirname(tmp_destination))
    
    if not os.path.isdir(os.path.dirname(destination)):
        os.makedirs(os.path.dirname(destination))
    
    with open(tmp_destination, u'wb+') as f:
        for line in source_content:
            f.write(line)
    
    shutil.move(tmp_destination, destination)


