#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.data_hub.data_hub import DataHub, DataHubConfig
from shinkensolutions.data_hub.data_hub_exception.data_hub_exception import DataHubFatalException

if TYPE_CHECKING:
    from shinkensolutions.data_hub.data_hub_driver.abstract_data_hub_driver import AbstractDataHubDriver, AbstractDataHubDriverConfig
    from shinkensolutions.data_hub.data_hub_driver.data_hub_driver_cPickle import DataHubDriverPickle
    
    from shinken.log import PartLogger
    from shinken.misc.type_hint import Callable, Dict, Type, List
    
    DriverFactory = Callable[[PartLogger, AbstractDataHubDriverConfig, DataHubConfig], AbstractDataHubDriver]


class DataHubFactory(object):
    _register_driver_factory = {}  # type: Dict[unicode, DriverFactory]
    
    
    @staticmethod
    def build_data_hub(logger, data_hub_config):
        # type: (PartLogger, DataHubConfig) -> DataHub
        
        data_hub = DataHub(logger, data_hub_config.data_hub_id, data_hub_config.data_type, data_hub_config.data_id_key_name, DataHubFactory.build_driver(logger, data_hub_config.driver_config, data_hub_config))
        
        if data_hub_config.must_save_configuration:
            save_configuration_driver = DataHubFactory._build_save_configuration_driver(logger)
            save_configuration_driver.write(data_hub_config.data_hub_id, data_hub_config)
        
        return data_hub
    
    
    @staticmethod
    def build_and_init_data_hub(logger, data_hub_config):
        # type: (PartLogger, DataHubConfig) -> DataHub
        
        data_hub = DataHubFactory.build_data_hub(logger, data_hub_config)
        data_hub.init()
        return data_hub
    
    
    @staticmethod
    def build_in_memory_exchange_pickle_data_hub(logger, data_type, data_id_key_name, daemon_name=u'', module_name=u'', submodule_name=u''):
        # type: (PartLogger, unicode, unicode, unicode, unicode, unicode) -> DataHub
        # lazy import for avoid recursion
        from shinkensolutions.data_hub.data_hub_driver.data_hub_driver_cPickle import DataHubDriverConfigPickle
        from shinkensolutions.data_hub.data_hub_driver.abstract_data_hub_driver_file import DATA_HUB_FILE_DEFAULT_DIRECTORY
        
        driver_config = DataHubDriverConfigPickle(
            base_directory=DATA_HUB_FILE_DEFAULT_DIRECTORY.DEV_SHM,
            data_location_name=data_type,
            daemon_name=daemon_name,
            module_name=module_name,
            submodule_name=submodule_name
        )
        data_hub_config = DataHubConfig(
            data_hub_id=u'data hub for %s' % data_type,
            data_hub_category=u'NONE',
            data_type=data_type,
            data_id_key_name=data_id_key_name,
            driver_config=driver_config,
            must_save_configuration=False
        )
        
        return DataHubFactory.build_and_init_data_hub(logger, data_hub_config)
    
    
    @staticmethod
    def build_in_memory_exchange_json_data_hub(logger, data_type, data_id_key_name, daemon_name=u'', module_name=u'', submodule_name=u''):
        # type: (PartLogger, unicode, unicode, unicode, unicode, unicode) -> DataHub
        # lazy import for avoid recursion
        from shinkensolutions.data_hub.data_hub_driver.data_hub_driver_json import DataHubDriverConfigJson
        from shinkensolutions.data_hub.data_hub_driver.abstract_data_hub_driver_file import DATA_HUB_FILE_DEFAULT_DIRECTORY
        
        driver_config = DataHubDriverConfigJson(
            base_directory=DATA_HUB_FILE_DEFAULT_DIRECTORY.DEV_SHM,
            data_location_name=data_type,
            daemon_name=daemon_name,
            module_name=module_name,
            submodule_name=submodule_name
        )
        data_hub_config = DataHubConfig(
            data_hub_id=u'data hub for %s' % data_type,
            data_hub_category=u'NONE',
            data_type=data_type,
            data_id_key_name=data_id_key_name,
            driver_config=driver_config,
            must_save_configuration=False
        )
        
        return DataHubFactory.build_and_init_data_hub(logger, data_hub_config)
    
    
    @staticmethod
    def build_data_hub_from_configuration_driver(logger, data_hub_id):
        # type: (PartLogger, unicode) -> DataHub
        save_configuration_driver = DataHubFactory._build_save_configuration_driver(logger)
        data_hub_configuration = save_configuration_driver.read(data_hub_id)
        return DataHubFactory.build_data_hub(logger, data_hub_configuration)
    
    
    @staticmethod
    def list_by_category_data_hub_configuration(logger, data_hub_category):
        # type: (PartLogger, unicode) -> List[DataHubConfig]
        data_hub_confs = []
        save_configuration_driver = DataHubFactory._build_save_configuration_driver(logger)
        for data_hub_id in save_configuration_driver.get_all_data_id():
            data_hub_conf = save_configuration_driver.read(data_hub_id)  # type: DataHubConfig
            if data_hub_conf.data_hub_category == data_hub_category:
                
                if data_hub_conf.data_hub_id != data_hub_id:
                    logger.warning(u'The DataHub configuration id [ %s ] does not match the configuration filename [ %s ]. Ignoring it.' % (data_hub_id, data_hub_conf.data_hub_id))
                    continue

                data_hub_confs.append(data_hub_conf)
        
        return data_hub_confs
    
    
    @staticmethod
    def list_data_hub_configuration(logger):
        # type: (PartLogger) -> List[DataHubConfig]
        data_hub_confs = []
        save_configuration_driver = DataHubFactory._build_save_configuration_driver(logger)
        for data_hub_id in save_configuration_driver.get_all_data_id():
            data_hub_conf = save_configuration_driver.read(data_hub_id)  # type: DataHubConfig

            if data_hub_conf.data_hub_id != data_hub_id:
                logger.warning(u'The DataHub configuration id [ %s ] does not match the configuration filename [ %s ]. Ignoring it.' % (data_hub_id, data_hub_conf.data_hub_id))
                continue
            
            data_hub_confs.append(data_hub_conf)
        
        return data_hub_confs
    
    
    @staticmethod
    def register_driver_factory(data_hub_driver_config_class, driver_factory):
        # type: (Type, DriverFactory) -> None
        data_hub_driver_qualify_name = u'%s.%s' % (data_hub_driver_config_class.__module__, data_hub_driver_config_class.__name__)
        DataHubFactory._register_driver_factory[data_hub_driver_qualify_name] = driver_factory
    
    
    @staticmethod
    def build_driver(logger, data_hub_driver_config, data_hub_config):
        # type: (PartLogger, AbstractDataHubDriverConfig, DataHubConfig) -> AbstractDataHubDriver
        
        data_hub_driver_config.data_type = data_hub_config.data_type
        data_hub_driver_config.data_hub_id = data_hub_config.data_hub_id
        
        data_hub_driver_qualify_name = DataHubFactory._get_class_qualify_name(data_hub_driver_config)
        driver_factory = DataHubFactory._register_driver_factory.get(data_hub_driver_qualify_name, None)
        
        if not driver_factory:
            logger.error(u'Unknown data hub driver type : %s. You must register your driver to DataHubFactory.' % data_hub_driver_qualify_name)
            raise DataHubFatalException(u'Unknown data hub driver type : %s. You must register your driver to DataHubFactory.' % data_hub_driver_qualify_name)
        
        data_hub_driver = driver_factory(logger, data_hub_driver_config, data_hub_config)
        
        return data_hub_driver
    
    
    @staticmethod
    def _get_class_qualify_name(instance):
        # type: (object) -> unicode
        return u'%s.%s' % (instance.__class__.__module__, instance.__class__.__name__)
    
    
    @staticmethod
    def _build_save_configuration_driver(logger):
        # type: (PartLogger) -> DataHubDriverPickle
        # lazy import for avoid recursion
        from shinkensolutions.data_hub.data_hub_driver.abstract_data_hub_driver_file import DATA_HUB_FILE_DEFAULT_DIRECTORY
        from shinkensolutions.data_hub.data_hub_driver.data_hub_driver_data_hub_config_file import DataHubDriverConfigDataHubConfigFile, DataHubDriverDataHubConfigFile
        
        save_configuration_driver_config = DataHubDriverConfigDataHubConfigFile(base_directory=DATA_HUB_FILE_DEFAULT_DIRECTORY.SHINKEN_LIB_PERSISTENT_DATA, data_location_name=u'.data_hub_config')
        
        return DataHubDriverDataHubConfigFile(logger, save_configuration_driver_config)
