#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import time

from shinken.misc.type_hint import TYPE_CHECKING, cast
from shinken.thread_helper import DelayedThread
from shinkensolutions.data_hub.data_hub_driver.abstract_data_hub_driver import AbstractDataHubDriver, AbstractDataHubDriverConfig
from shinkensolutions.data_hub.data_hub_factory.data_hub_factory import DataHubFactory

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Any
    from shinken.log import PartLogger
    from shinkensolutions.data_hub.data_hub import DataHubConfig


class DataHubMetaDriverConfigRetention(AbstractDataHubDriverConfig):
    def __init__(self, config_main_driver, config_retention_driver, delay_to_wait, retention_interval):
        # type: (AbstractDataHubDriverConfig, AbstractDataHubDriverConfig, int, int) -> None
        super(DataHubMetaDriverConfigRetention, self).__init__(u'RETENTION')
        self.config_main_driver = config_main_driver
        self.config_retention_driver = config_retention_driver
        self.delay_to_wait = delay_to_wait
        self.retention_interval = retention_interval


def data_hub_meta_driver_retention_factory(logger, driver_config, data_hub_config):
    # type: (PartLogger, AbstractDataHubDriverConfig, DataHubConfig) -> DataHubMetaDriverRetention
    
    driver_config = cast(DataHubMetaDriverConfigRetention, driver_config)
    
    main_driver = DataHubFactory.build_driver(logger, driver_config.config_main_driver, data_hub_config)
    retention_driver = DataHubFactory.build_driver(logger, driver_config.config_retention_driver, data_hub_config)
    return DataHubMetaDriverRetention(logger, driver_config, main_driver, retention_driver)


DataHubFactory.register_driver_factory(DataHubMetaDriverConfigRetention, data_hub_meta_driver_retention_factory)


class DataHubMetaDriverRetention(AbstractDataHubDriver, DelayedThread):
    def __init__(self, logger, driver_config, main_driver, retention_driver):
        # type: (PartLogger, DataHubMetaDriverConfigRetention, AbstractDataHubDriver, AbstractDataHubDriver) -> None
        DelayedThread.__init__(self, delay_to_wait=driver_config.delay_to_wait, loop_speed=driver_config.retention_interval)
        AbstractDataHubDriver.__init__(self, logger, driver_config)
        
        # it is self.logger in DelayedThread
        self.logger = self._logger
        
        self.main_driver = main_driver
        self.retention_driver = retention_driver
        self.retention_interval = driver_config.retention_interval
    
    
    def get_size_of(self, data_id):
        raise NotImplementedError()
    
    
    def get_lock(self, data_id):
        raise NotImplementedError()
    
    
    def loop_turn(self):
        # type: () -> None
        self._logger.debug(u'Its been more than [ %d ] seconds since the last retention update. We will write it' % self.retention_interval)
        
        nb_data_saved = 0
        start = time.time()
        for data_id in self.main_driver.get_all_data_id():
            try:
                data = self.main_driver.read(data_id, log_error=False)
            except Exception:
                continue  # Item not found, can't add it to retention
            self.retention_driver.write(data_id, data)
            nb_data_saved += 1
        
        elapsed = time.time() - start
        self._logger.info(u'Retention done in [ %0.3f ] seconds for %s %s' % (elapsed, nb_data_saved, self._data_type))
    
    
    def get_thread_name(self):
        # type: () -> unicode
        return u'DHub ret %s' % self._name
    
    
    def init(self):
        # type: () -> None
        self.main_driver.init()
        self.retention_driver.init()
        self.start_thread()
    
    
    def get_all_data_id(self):
        # type: () -> List[unicode]
        return list(set().union(self.main_driver.get_all_data_id(), self.retention_driver.get_all_data_id()))
    
    
    def is_data_correct(self, data_id):
        # type: (unicode) -> bool
        
        if self.main_driver.is_data_correct(data_id):
            return True
        elif self.retention_driver.is_data_correct(data_id):
            data = self.retention_driver.read(data_id, log_error=False)
            # We found it in retention, so we need to save it in main driver, so we do not have to check retention again
            self.main_driver.write(data_id, data)
            return True
        return False
    
    
    def write(self, data_id, data):
        # type: (unicode, Any) -> None
        self.main_driver.write(data_id, data)
    
    
    def read(self, data_id, log_error=True):
        # type: (unicode, bool) -> Any
        try:
            data = self.main_driver.read(data_id, log_error=log_error)
        except Exception:
            data = self.retention_driver.read(data_id, log_error=log_error)
            # We found it in retention, so we need to save it in main driver, so we do not have to check retention again
            self.main_driver.write(data_id, data)
        return data
    
    
    def remove(self, data_id):
        # type: (unicode) -> None
        self.main_driver.remove(data_id)
        self.retention_driver.remove(data_id)
    
    
    def get_last_modification_date(self, data_id):
        # type: (unicode) -> int
        try:
            last_file_modification = self.main_driver.get_last_modification_date(data_id)
            return last_file_modification
        except Exception:
            last_file_modification = self.retention_driver.get_last_modification_date(data_id)
            return last_file_modification
    
    
    def restore(self, data_id, data):
        # type: (unicode, Any) -> None
        self.main_driver.write(data_id, data)
        self.retention_driver.write(data_id, data)
    
    
    def sanitize(self, data_id, data):
        # type: (unicode, Any) -> None
        self.main_driver.write(data_id, data)
        self.retention_driver.write(data_id, data)
    
    
    def destroy(self):
        # type: () -> None
        self.stop()
        self.main_driver.destroy()
        self.retention_driver.destroy()
    
    
    def stop(self):
        # type: () -> None
        DelayedThread.stop(self)
        self.main_driver.stop()
        self.retention_driver.stop()
    
    
    def check_right_access_issues_then_fix_them(self):
        # type: () -> None
        self.main_driver.check_right_access_issues_then_fix_them()
        self.retention_driver.check_right_access_issues_then_fix_them()
    
    
    def get_total_size(self):
        # type: () -> int
        try:
            return self.main_driver.get_total_size()
        except Exception:
            return self.retention_driver.get_total_size()
    
    
    def get_number_of_stored_data(self):
        # type: () -> int
        try:
            return self.main_driver.get_number_of_stored_data()
        except Exception:
            return self.retention_driver.get_number_of_stored_data()
