#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import calendar
import time
from collections import namedtuple
from datetime import datetime, timedelta, time as datetime_time, tzinfo

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.memoized import memoize_lru

if TYPE_CHECKING:
    from shinken.misc.type_hint import Number, Tuple, Union

_MAX_END_OF_DAY_CACHE = 10
_end_of_day_cache = {}
_start_of_day_cache = {}

Date = namedtuple('Date', ['yday', 'year'])

ZERO = timedelta(0)
STDOFFSET = timedelta(seconds=-time.timezone)
DSTOFFSET = timedelta(seconds=-time.altzone) if time.daylight else STDOFFSET
DSTDIFF = DSTOFFSET - STDOFFSET


# A class capturing the platform's idea of local time.
# from https://docs.python.org/fr/2.7/library/datetime.html#datetime.tzinfo
class LocalTimezone(tzinfo):
    
    def utcoffset(self, dt):
        if self._isdst(dt):
            return DSTOFFSET
        else:
            return STDOFFSET
    
    
    def dst(self, dt):
        if self._isdst(dt):
            return DSTDIFF
        else:
            return ZERO
    
    
    def tzname(self, dt):
        return time.tzname[self._isdst(dt)]
    
    
    def _isdst(self, dt):
        # type: (datetime) -> bool
        return self._compute_dst_with_cache(dt.year, dt.month, dt.day, dt.hour, dt.weekday())
    
    
    @staticmethod
    @memoize_lru(256)
    def _compute_dst_with_cache(year, month, day, hour, weekday):
        # type: (int, int, int, int, int) -> bool
        tt = (year, month, day,
              hour, 0, 0,
              weekday, 0, 0)
        stamp = time.mktime(tt)
        tt = time.localtime(stamp)
        return tt.tm_isdst > 0


TZINFO_LOCAL = LocalTimezone()


class UTC(tzinfo):
    
    def utcoffset(self, dt):
        return ZERO
    
    
    def tzname(self, dt):
        return "UTC"
    
    
    def dst(self, dt):
        return ZERO


TZINFO_UTC = UTC()


def get_datetime_with_local_time_zone(_time=None):
    # type: (Union[Number,basestring]) -> datetime
    if _time is None:
        _time = get_now()
    elif isinstance(_time, basestring):
        _time = date_string_to_timestamp(_time)
    return datetime.fromtimestamp(_time, TZINFO_LOCAL)


def get_datetime_from_utc_to_local_time_zone(_datetime):
    # type: (datetime) -> datetime
    return _datetime.replace(tzinfo=TZINFO_UTC).astimezone(TZINFO_LOCAL)


class DATE_COMPARE(object):
    IS_BEFORE = -1
    IS_AFTER = 1
    IS_EQUAL = 0


# Need for mocking in TU
class InternalTime(object):
    @staticmethod
    def get_now():
        return int(time.time())
    
    
    __original_get_now = get_now
    
    
    @staticmethod
    def remove_mock():
        # type: () -> None
        if InternalTime.get_now is not InternalTime.__original_get_now:
            InternalTime.get_now = staticmethod(InternalTime.__original_get_now)
    
    
    @staticmethod
    def mock_time_set_value(_time):
        # type: (int) -> int
        InternalTime.get_now = staticmethod(lambda: _time)
        return _time
    
    
    @staticmethod
    def mock_time_set_value_from_date_string(_time):
        # type: (unicode) -> int
        return InternalTime.mock_time_set_value(date_string_to_timestamp(_time))
    
    
    # Compatibility shims
    mock_time = mock_time_set_value_from_date_string
    reset = remove_mock


_internal_time = InternalTime()


def get_now():
    return _internal_time.get_now()


def print_time(_time):
    try:
        return datetime.fromtimestamp(_time).strftime('%j-%Y %H:%M:%S')
    except ValueError:
        return '%s' % _time


def parse_string_to_date(date_str, str_format='%d_%m_%Y'):
    tm = datetime.strptime(date_str, str_format).timetuple()
    return Date(tm.tm_yday, tm.tm_year)


def format_date_to_string(_date, str_format='%d_%m_%Y'):
    try:
        return datetime.strptime('%s %s' % _date, '%j %Y').strftime(str_format)
    except ValueError:
        return '%s' % _date


def date_string_to_timestamp(_date_text, day_format=None):
    end_of_day = False
    if _date_text.startswith('+'):
        end_of_day = True
        _date_text = _date_text[1:]
    
    if day_format:
        _datetime = datetime.strptime(_date_text, day_format + ' %H:%M:%S')
    elif '_' in _date_text:
        _datetime = datetime.strptime(_date_text, '%j_%Y %H:%M:%S')
    elif '-' in _date_text:
        _datetime = datetime.strptime(_date_text, '%j-%Y %H:%M:%S')
    else:
        _datetime = datetime.strptime(_date_text, '%j %Y %H:%M:%S')
    
    if end_of_day:
        _datetime = datetime.combine(_datetime, datetime_time.max)
    ret = int(time.mktime(_datetime.timetuple()))
    if end_of_day:
        ret += 1
    return ret


def parse_range_text(_date, _range):
    if isinstance(_date, Date):
        _date = '%s_%s' % (_date.yday, _date.year)
    split = _range.split(' ')
    _start = date_string_to_timestamp('%s %s' % (_date, split[0]))
    if split[1].startswith('+'):
        _end = date_string_to_timestamp('+%s %s' % (_date, split[1][1:]))
    else:
        _end = date_string_to_timestamp('%s %s' % (_date, split[1]))
    return _end, _start


def timestamp_from_datetime(_datetime):
    return int(time.mktime(_datetime.timetuple()))


def date_from_timestamp(timestamp):
    tm = time.localtime(timestamp)
    return Date(tm.tm_yday, tm.tm_year)


def timestamp_from_date(date):
    return int(time.mktime(time.strptime('%d %d 00:00:00' % date, '%j %Y %H:%M:%S')))


def datetime_from_date(date):
    return datetime.strptime('%s %s 00:00:00' % date, '%j %Y %H:%M:%S')


def date_now():
    tm = time.localtime(get_now())
    return Date(tm.tm_yday, tm.tm_year)


# compare_date(date1, date2) == IS_BEFORE means date2 is before date1
# compare_date(date1, date2) == IS_AFTER means date2 is after date1
def compare_date(date1, date2):
    if not date1 or not date2:
        return None
    tmp = (date2.year * 1000 + date2.yday) - (date1.year * 1000 + date1.yday)
    tmp = DATE_COMPARE.IS_AFTER if tmp > 0 else tmp
    tmp = DATE_COMPARE.IS_BEFORE if tmp < 0 else tmp
    return tmp


def get_end_of_day(date):
    # type: (Date) -> int
    global _end_of_day_cache
    end_of_day = _end_of_day_cache.get(date, None)
    
    if not end_of_day:
        datetime_end_of_day = datetime.combine(datetime_from_date(date), datetime_time.max)
        end_of_day = int(time.mktime(datetime_end_of_day.timetuple())) + 1
        _end_of_day_cache[date] = end_of_day
        
        if len(_end_of_day_cache) > _MAX_END_OF_DAY_CACHE:
            new_end_of_day_cache = {}
            now = datetime.fromtimestamp(get_now())
            for i in xrange(-3, 3, 1):
                datetime_end_of_day = datetime.combine(now + timedelta(days=i), datetime_time.max)
                tm = datetime_end_of_day.timetuple()
                d = Date(tm.tm_yday, tm.tm_year)
                _end_of_day = int(time.mktime(tm)) + 1  # use localtime  and not UTC calendar.timegm(_day)
                new_end_of_day_cache[d] = _end_of_day
            
            _end_of_day_cache = new_end_of_day_cache
    
    return end_of_day


def get_start_of_day(date):
    # type: (Date) -> int
    global _start_of_day_cache
    start_of_day = _start_of_day_cache.get(date, None)
    
    if not start_of_day:
        _day = time.strptime('%d %d' % date, '%j %Y')
        start_of_day = int(time.mktime(_day))  # use localtime  and not UTC calendar.timegm(_day)
        _start_of_day_cache[date] = start_of_day
        
        if len(_start_of_day_cache) > _MAX_END_OF_DAY_CACHE:
            new_start_of_day_cache = {}
            tm = time.localtime()
            tm = time.struct_time((tm.tm_year, tm.tm_mon, tm.tm_mday, 0, 0, 0, tm.tm_wday, tm.tm_yday, tm.tm_isdst))
            d = Date(tm.tm_yday, tm.tm_year)
            new_start_of_day_cache[d] = int(time.mktime(tm))
            _start_of_day_cache = new_start_of_day_cache
    
    return start_of_day


def get_start_of_previous_day_and_day(today_date):
    # type: (Date) -> Tuple[int,int]
    return get_start_of_day(get_previous_date(today_date)), get_start_of_day(today_date)


def get_diff_date(date01, date02):
    datetime_date01 = datetime_from_date(date01)
    datetime_date02 = datetime_from_date(date02)
    diff_days = (datetime_date01 - datetime_date02).days
    return diff_days


def get_next_date(date):
    next_date = Date(date.yday + 1, date.year)
    if next_date.yday > 365 and not calendar.isleap(date.year):
        next_date = Date(1, date.year + 1)
    elif next_date.yday > 366 and calendar.isleap(date.year):
        next_date = Date(1, date.year + 1)
    return next_date


def get_previous_date(date):
    previous_date = Date(date.yday - 1, date.year)
    if previous_date.yday < 1:
        previous_day = 366 if calendar.isleap(date.year - 1) else 365
        previous_date = Date(previous_day, date.year - 1)
    return previous_date


def from_to_date_generator(start_date, end_date):
    curr_date = start_date
    while True:
        yield curr_date
        curr_date = get_next_date(curr_date)
        if compare_date(end_date, curr_date) == DATE_COMPARE.IS_AFTER:
            break


def countdown_date_generator(older_date, newer_date):
    if compare_date(older_date, newer_date) == DATE_COMPARE.IS_BEFORE:
        return
    cur_date = newer_date
    while True:
        yield cur_date
        cur_date = get_previous_date(cur_date)
        if compare_date(older_date, cur_date) == DATE_COMPARE.IS_BEFORE:
            break


def date_from_datetime(_datetime):
    # type: (datetime) -> Date
    tm = _datetime.timetuple()
    return Date(tm.tm_yday, tm.tm_year)
